/**
 * WildCAT: A Generic Framework for Context-Aware Applications.
 * Copyright (C) 2008-2009 Bull S.A.S.
 * Copyright (C) 2008 EMN
 * Contact: wildcat@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * Authors:
 *		- Pierre-Charles David (initial)
 *		- Nicolas Loriant
 * --------------------------------------------------------------------------
 * $Id: POJOAttribute.java 403 2009-01-08 16:26:46Z loris $
 * --------------------------------------------------------------------------
 */
package org.ow2.wildcat.hierarchy.attribute;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.ow2.wildcat.Context;
import org.ow2.wildcat.event.WAttributeEvent;
import org.ow2.wildcat.event.WEventInterface.Scope;
import org.ow2.wildcat.hierarchy.Mountable;
import org.ow2.wildcat.hierarchy.Path;
import org.ow2.wildcat.util.Messages;

/**
 * Attribute holding an object.
 */
public class POJOAttribute implements Attribute {

    /**
     * Logger.
     */
    private static Log logger = LogFactory.getLog(POJOAttribute.class);

    protected Context context = null;

    protected Path path = null;

    protected Object value;

    /**
     * Scope of the events generated when setting the value.
     */
    private Scope scope;

    /**
     * Construct a new {@link POJOAttribute} with a given initial value and a given scope.
     * @param scope the scope of the events generated when setting the value
     * @param value the initial value
     */
    public POJOAttribute(final Scope scope, final Object value) {
        this.scope = scope;
        this.value = value;
    }

    /**
     * Construct a new global {@link POJOAttribute} with a given initial value.
     * @param value the initial value
     */
    public POJOAttribute(final Object value) {
        this.scope = Scope.GLOBAL;
        this.value = value;
    }

    /*
     * (non-Javadoc)
     *
     * @see org.ow2.wildcat.hierarchy.Attribute#getValue()
     */
    public Object getValue() {
        return this.value;
    }

    /*
     * (non-Javadoc)
     *
     * @see org.ow2.wildcat.hierarchy.Attribute#setValue(java.lang.Object)
     */
    public Object setValue(final Object value) {
        Object old = this.value;
        this.value = value;
        if (this.isMount()) {
            this.context.emitEvent(new WAttributeEvent(this.path, scope, this.value));
        }
        return old;
    }

    /**
     * @return the scope of the events generated when setting the value
     */
    public Scope getScope() {
        return scope;
    }

    /**
     * Set the scope of the events generated when setting the value.
     * @param scope the scope of the events generated when setting the value
     */
    public void setScope(final Scope scope) {
        this.scope = scope;
    }

    /*
     * (non-Javadoc)
     *
     * @see org.ow2.wildcat.hierarchy.Mountable#isMount()
     */
    public boolean isMount() {
        return this.context != null && this.path != null;
    }

    /*
     * (non-Javadoc)
     *
     * @see org.ow2.wildcat.hierarchy.Mountable#mount(org.ow2.wildcat.Context,
     *      org.ow2.wildcat.hierarchy.Path)
     */
    public void mount(final Context context, final Path mountPoint) {
        logger.debug(Messages.message(Mountable.class.getName() + ".mount",
                mountPoint));

        if (this.isMount()) {
            logger.info(Messages.message(Mountable.class.getName()
                    + ".alreadymount", mountPoint));
        } else {
            this.context = context;
            this.path = mountPoint;
        }
    }

    /*
     * (non-Javadoc)
     *
     * @see org.ow2.wildcat.hierarchy.Mountable#unMount()
     */
    public void unMount() {
        logger.debug(Messages.message(Mountable.class.getName() + ".unmount",
                (Object[]) null));

        this.context = null;
        this.path = null;
    }

}
