/**
 * WildCAT: A Generic Framework for Context-Aware Applications.
 * Copyright (C) 2008-2009 Bull S.A.S.
 * Copyright (C) 2008 EMN
 * Contact: wildcat@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * Authors:
 *		- Pierre-Charles David (initial)
 *		- Nicolas Loriant
 * --------------------------------------------------------------------------
 * $Id: QueryAttribute.java 406 2009-01-09 12:59:52Z loris $
 * --------------------------------------------------------------------------
 */
package org.ow2.wildcat.hierarchy.attribute;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.ow2.wildcat.Context;
import org.ow2.wildcat.Query;
import org.ow2.wildcat.event.WAttributeEvent;
import org.ow2.wildcat.event.WEventInterface.Scope;
import org.ow2.wildcat.hierarchy.Mountable;
import org.ow2.wildcat.hierarchy.Path;
import org.ow2.wildcat.util.Messages;

import com.espertech.esper.client.UpdateListener;
import com.espertech.esper.event.EventBean;

/**
 * An attribute holding the result of a query.
 */
public class QueryAttribute implements Attribute, UpdateListener {

    /**
     * Logger.
     */
    private static Log logger = LogFactory.getLog(QueryAttribute.class);

    /**
     * The string representing a query.
     */
    private final String queryString;

    /**
     * The scope of generated events.
     */
    private final Scope scope;

    /**
     * The query.
     */
    private Query query;

    /**
     * The current value.
     */
    private Object value;

    /**
     * The context where is mount this attribute.
     */
    private Context context;

    /**
     * The path of this attribute.
     */
    private Path path;


    /**
     * Construct a new query attribute whose generated events have the global scope.
     *
     * @param queryString
     *            the associated query
     */
    public QueryAttribute(final String queryString) {
        this(queryString, Scope.GLOBAL);
    }

    /**
     * Construct a new query attribute.
     * @param queryString the associated query
     * @param scope the scope of generated events
     */
    public QueryAttribute(final String queryString, final Scope scope) {
        this.queryString = queryString;
        this.scope = scope;
    }

    /*
     * (non-Javadoc)
     *
     * @see org.ow2.wildcat.hierarchy.attribute.Attribute#getValue()
     */
    public Object getValue() {
        return this.value;
    }

    /*
     * (non-Javadoc)
     *
     * @see org.ow2.wildcat.hierarchy.attribute.Attribute#setValue(java.lang.Object)
     */
    public Object setValue(final Object value) {
        return this.value;
    }

    /**
     * @return the scope of generated events
     */
    protected Scope getScope() {
        return scope;
    }

    /**
     * @return the context where is mount this attribute
     */
    protected Context getContext() {
        return context;
    }

    /**
     * @return the path of this attribute
     */
    protected Path getPath() {
        return path;
    }

    /*
     * (non-Javadoc)
     *
     * @see org.ow2.wildcat.hierarchy.Mountable#isMount()
     */
    public boolean isMount() {
        return this.context != null && this.path != null;
    }

    /*
     * (non-Javadoc)
     *
     * @see org.ow2.wildcat.hierarchy.Mountable#mount(org.ow2.wildcat.Context,
     *      org.ow2.wildcat.hierarchy.Path)
     */
    public void mount(final Context context, final Path mountPoint) {
        logger.debug(Messages.message(Mountable.class.getName() + ".mount",
                mountPoint));

        if (this.isMount()) {
            logger.info(Messages.message(Mountable.class.getName()
                    + ".alreadymount", mountPoint));
        } else {
            this.context = context;
            this.path = mountPoint;

            this.query = this.context.registerListeners(this.queryString, this);
        }
    }

    /*
     * (non-Javadoc)
     *
     * @see org.ow2.wildcat.hierarchy.Mountable#unMount()
     */
    public void unMount() {
        logger.debug(Messages.message(Mountable.class.getName() + ".unmount",
                (Object[]) null));

        if (this.isMount() && this.query != null) {
            this.context.destroyQuery(this.query);
        }
        query = null;
        this.context = null;
        this.path = null;
    }

    public void update(final EventBean[] newEvents, final EventBean[] oldEvents) {

        if (newEvents == null || newEvents.length == 0) {
            this.value = null;
        } else if (newEvents.length == 1) {
            this.value = newEvents[0].getUnderlying();
        } else {
            Object[] oa = new Object[newEvents.length];
            for (int i = 0; i < newEvents.length; i++) {
                oa[i] = newEvents[i].getUnderlying();
            }
            this.value = oa;
        }

        if (this.isMount()) {
            this.context.emitEvent(new WAttributeEvent(this.path, scope, this.value));
        }
    }

}
