/**
 * WildCAT: A Generic Framework for Context-Aware Applications.
 * Copyright (C) 2008 Bull S.A.S.
 * Copyright (C) 2008 EMN
 * Contact: wildcat@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * --------------------------------------------------------------------------
 * $Id: AttributeListener.java 366 2008-12-10 18:50:44Z loris $
 * --------------------------------------------------------------------------
 */
package org.ow2.wildcat.osgi;

import java.util.HashMap;
import java.util.Map;

import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceEvent;
import org.osgi.framework.ServiceListener;
import org.osgi.framework.ServiceReference;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;
import org.ow2.wildcat.Context;
import org.ow2.wildcat.ContextException;
import org.ow2.wildcat.osgi.api.IAttributeAttacher;

/**
 * Listen for attaching and detaching WildCAT attributes.
 * @author Loris Bouzonnet
 */
public class AttributeListener implements ServiceListener {

    /**
     * Logger.
     */
    private static Log logger = LogFactory.getLog(AttributeListener.class);

    /**
     * OSGi bundle context to retrieve the incoming {@link IAttributeAttacher}.
     */
    private final BundleContext context;

    /**
     * WildCAT Context.
     */
    private final Context ctx;

    /**
     * Paths.
     */
    private final Map<ServiceReference, String> paths = new HashMap<ServiceReference, String>();

    /**
     * @param bundleContext
     * @param ctx
     */
    public AttributeListener(final BundleContext bundleContext, final Context ctx) {
        this.context = bundleContext;
        this.ctx = ctx;
    }

    /**
     * Attach or detach a WildCAT attribute according to the incoming event.
     * @param event a service event
     */
    public void serviceChanged(final ServiceEvent event) {
        ServiceReference ref = event.getServiceReference();
        // React on REGISTERED|UNREGISTERING events
        switch (event.getType()) {
            case ServiceEvent.REGISTERED:
                IAttributeAttacher attacher = (IAttributeAttacher) context.getService(ref);
                try {
                    ctx.attachAttribute(attacher.getPath(), attacher.getAttribute());
                    logger.info("WildCAT Attribute succesfully attached at path {0}", attacher.getPath());
                } catch (ContextException e) {
                logger.warn("Unable to attach a WildCAT attribute at path " + attacher.getPath(), e);
                }
                paths.put(ref, attacher.getPath());
                break;
            case ServiceEvent.UNREGISTERING:
                String path = paths.remove(ref);
                if (path != null) {
                    try {
                        ctx.detachAttribute(path);
                        logger.info("WildCAT Attribute succesfully detached at path {0}", path);
                    } catch (ContextException e) {
                        logger.warn("Unable to detach a WildCAT attribute at path " + path, e);
                    }
                }
                break;
            default:
                break;
        }
    }

}
