/**
 * WildCAT: A Generic Framework for Context-Aware Applications.
 * Copyright (C) 2008 Bull S.A.S.
 * Copyright (C) 2008 EMN
 * Contact: wildcat@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * --------------------------------------------------------------------------
 * $Id: QueryListener.java 366 2008-12-10 18:50:44Z loris $
 * --------------------------------------------------------------------------
 */
package org.ow2.wildcat.osgi;

import java.util.HashMap;
import java.util.Map;

import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceEvent;
import org.osgi.framework.ServiceListener;
import org.osgi.framework.ServiceReference;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;
import org.ow2.wildcat.Context;
import org.ow2.wildcat.Query;
import org.ow2.wildcat.hierarchy.ListenerRegistrationException;
import org.ow2.wildcat.osgi.api.IQueryAdder;

/**
 * Listen for adding and removing of queries.
 * @author Loris Bouzonnet
 */
public class QueryListener implements ServiceListener {

    /**
     * Logger.
     */
    private static Log logger = LogFactory.getLog(QueryListener.class);

    /**
     * OSGi bundle context to retrieve the incoming {@link IQueryAdder}.
     */
    private final BundleContext context;

    /**
     * WildCAT Context.
     */
    private final Context ctx;

    /**
     * Queries.
     */
    private final Map<ServiceReference, Query> queries = new HashMap<ServiceReference, Query>();

    /**
     * @param bundleContext
     * @param ctx
     */
    public QueryListener(final BundleContext bundleContext, final Context ctx) {
        this.context = bundleContext;
        this.ctx = ctx;
    }

    /**
     * Add or remove a WildCAT query according to the incoming event.
     * @param event a service event
     */
    public void serviceChanged(final ServiceEvent event) {
        ServiceReference ref = event.getServiceReference();
        // React on REGISTERED|UNREGISTERING events
        switch (event.getType()) {
            case ServiceEvent.REGISTERED:
                IQueryAdder queryAdder = (IQueryAdder) context.getService(ref);
                try {
                    queries.put(ref, queryAdder.registerListener(ctx));
                    logger.info("WildCAT Query succesfully registered: {0}", queryAdder.getQuery());
                } catch (ListenerRegistrationException e) {
                    logger.warn("Unable to add the WildCAT query: {0}", queryAdder.getQuery());
                }
                break;
            case ServiceEvent.UNREGISTERING:
                Query query = queries.remove(ref);
                if (query != null) {
                    ctx.destroyQuery(query);
                    logger.info("WildCAT Query succesfully unregistered: {0}", query);
                }
                break;
            default:
                break;
        }
    }

}
