/**
 * WildCAT: A Generic Framework for Context-Aware Applications.
 * Copyright (C) 2008 Bull S.A.S.
 * Copyright (C) 2008 EMN
 * Contact: wildcat@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * --------------------------------------------------------------------------
 * $Id: Activator.java 350 2008-11-10 09:41:14Z loris $
 * --------------------------------------------------------------------------
 */
package org.ow2.wildcat.osgi;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.Properties;

import javax.management.MBeanServer;
import javax.management.MBeanServerFactory;
import javax.management.MBeanServerNotification;
import javax.management.Notification;
import javax.management.NotificationListener;
import javax.management.ObjectName;

import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.osgi.framework.BundleActivator;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceListener;
import org.osgi.framework.ServiceRegistration;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;
import org.ow2.wildcat.Context;
import org.ow2.wildcat.ContextException;
import org.ow2.wildcat.ContextFactory;
import org.ow2.wildcat.osgi.api.IAttributeAttacher;
import org.ow2.wildcat.osgi.api.IQueryAdder;


/**
 * Create a {@link Context} and register it as service.
 * @author Loris Bouzonnet
 */
public class Activator implements BundleActivator, NotificationListener {

    /**
     * Logger.
     */
    private static Log logger = LogFactory.getLog(Activator.class);

    /**
     * The WildCAT {@link Context}.
     */
    private Context ctx;

    /**
     * Keep a trace of the registration service in order to unregister it before stopping.
     */
    private ServiceRegistration ctxRegistration;

    /**
     * Listener to dynamically add and remove event queries.
     */
    private ServiceListener queryListener;

    /**
     * Listener to dynamically attach and detach attributes.
     */
    private ServiceListener attributeListener;

    private volatile ObjectName objectNameToListen;

    private ObjectName delegate;

    private MBeanServer mbeaBeanServer;

    private BundleContext bundleContext;

    private Properties conf;


    public void start(final BundleContext bundleContext) throws Exception {

        this.bundleContext = bundleContext;

        conf = new Properties();

        // Try to load the configuration file
        URL configurationURL = new File(System.getProperty("jonas.base") + "/conf/wildcat.properties").toURI().toURL();
        InputStream instream = null;
        try {
            instream = configurationURL.openStream();
            conf.load(instream);
        } catch (IOException e) {
            logger.debug("Unable to open the file wildcat.properties", e);
        } finally {
            if(instream != null) {
                instream.close();
            }
        }
        String contextName = System.getProperty("jonas.name", "jonas") + "_wcat";

        conf.setProperty(ContextFactory.KEY_CONTEXT_NAME, contextName);

        String oname = conf.getProperty("wildcat.osgi.oname");

        if(oname != null) {
            objectNameToListen = new ObjectName(oname);

            mbeaBeanServer = (MBeanServer) MBeanServerFactory.findMBeanServer(null).get(0);
        }

        if(mbeaBeanServer != null
                && !mbeaBeanServer.isRegistered(objectNameToListen)) {
            delegate = new ObjectName("JMImplementation:type=MBeanServerDelegate");
            mbeaBeanServer.addNotificationListener(
                    delegate, this, null, null);
        } else {
            doEffectiveStart();
        }
    }

    private void doEffectiveStart() throws Exception {
        if(ctx == null) {

            ClassLoader lastClassLoader = Thread.currentThread().getContextClassLoader();
            Thread.currentThread().setContextClassLoader(Activator.class.getClassLoader());
            try {
                ctx = createContext();
            } finally {
                Thread.currentThread().setContextClassLoader(lastClassLoader);
            }

            // Listen for incoming queries
            queryListener = new QueryListener(bundleContext, ctx);
            bundleContext.addServiceListener(
                    queryListener, "(objectClass=" + IQueryAdder.class.getName() + ")");

            // Listen for incoming attributes
            attributeListener = new AttributeListener(bundleContext, ctx);
            bundleContext.addServiceListener(
                    attributeListener, "(objectClass=" + IAttributeAttacher.class.getName() + ")");

            // Register the WildCAT context as service
            ctxRegistration = bundleContext.registerService(
                    Context.class.getName(), ctx, null);
        }
    }

    private Context createContext() {
        Logger.getRootLogger().setLevel(Level.WARN);
        ContextFactory factory = new ContextFactory(conf);
        Context ctx = factory.createContext();

        try {
            ctx.export();
        } catch (ContextException e) {
            logger.warn("Unable to register the WildCAT context for remote connection!");
        }
        logger.info("WildCAT context successfully created");
        return ctx;
    }

    public void stop(final BundleContext context) throws Exception {
        if(delegate != null) {
            if(mbeaBeanServer.isRegistered(delegate)) {
                mbeaBeanServer.removeNotificationListener(delegate, this);
            }
        }

        ctx.unexport();

        if(ctxRegistration != null) {
            ctxRegistration.unregister();
        }
        if(attributeListener != null) {
            context.removeServiceListener(attributeListener);
        }
        if(queryListener != null) {
            context.removeServiceListener(queryListener);
        }
        ctx.destroy();
    }

    /**
     * @param notification
     * @param handback
     */
    public void handleNotification(final Notification notification, final Object handback) {
        if (notification.getType().equals(
                MBeanServerNotification.REGISTRATION_NOTIFICATION)) {
            if(objectNameToListen.equals(((MBeanServerNotification) notification).getMBeanName())) {
                try {
                    doEffectiveStart();
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
        } else if(notification.getType().equals(
                MBeanServerNotification.UNREGISTRATION_NOTIFICATION)) {
            if(objectNameToListen.equals(((MBeanServerNotification) notification).getMBeanName())) {
            }
        }

    }

}
