/*
 * Decompiled with CFR 0.152.
 */
package org.pepsoft.util;

import com.google.common.base.Strings;
import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.Charset;
import java.nio.file.Files;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.text.MessageFormat;
import java.util.AbstractCollection;
import java.util.AbstractList;
import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.ListIterator;
import java.util.Map;
import java.util.RandomAccess;
import java.util.Set;
import java.util.SortedMap;
import java.util.SortedSet;
import java.util.TreeMap;
import java.util.TreeSet;
import java.util.function.Consumer;
import org.jetbrains.annotations.NonNls;
import org.pepsoft.util.Checksum;
import org.pepsoft.util.ProgressReceiver;
import org.pepsoft.util.StreamUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileUtils {
    private static final int BUFFER_SIZE = 32768;
    private static final String ILLEGAL_CHARS = "<>:\"/\\|?*\t\r\n\b\f";
    private static final Set<String> RESERVED_NAMES = new HashSet<String>(Arrays.asList("CON", "PRN", "AUX", "NUL", "COM1", "COM2", "COM3", "COM4", "COM5", "COM6", "COM7", "COM8", "COM9", "LPT1", "LPT2", "LPT3", "LPT4", "LPT5", "LPT6", "LPT7", "LPT8", "LPT9"));
    private static final char REPLACEMENT_CHAR = '_';
    private static final Logger logger = LoggerFactory.getLogger(FileUtils.class);

    private FileUtils() {
    }

    public static Checksum getMD5(File file) throws IOException {
        try {
            MessageDigest md5Digest = MessageDigest.getInstance("MD5");
            byte[] buffer = new byte[32768];
            try (FileInputStream in = new FileInputStream(file);){
                int read;
                while ((read = in.read(buffer)) != -1) {
                    md5Digest.update(buffer, 0, read);
                }
            }
            return new Checksum(md5Digest.digest());
        }
        catch (NoSuchAlgorithmException e) {
            throw new InternalError("MD5 message digest not supported by Java runtime");
        }
    }

    public static String load(File file, Charset charset) throws IOException {
        long length = file.length();
        if (length > Integer.MAX_VALUE) {
            throw new UnsupportedOperationException("File too large (" + length + " bytes)");
        }
        StringBuilder sb = new StringBuilder((int)length);
        try (InputStreamReader in = new InputStreamReader((InputStream)new BufferedInputStream(new FileInputStream(file)), charset);){
            int read;
            char[] buffer = new char[32768];
            while ((read = in.read(buffer)) != -1) {
                sb.append(buffer, 0, read);
            }
        }
        return sb.toString();
    }

    public static String load(InputStream inputStream, Charset charset) throws IOException {
        StringBuilder sb = new StringBuilder();
        try (InputStreamReader in = new InputStreamReader((InputStream)new BufferedInputStream(inputStream), charset);){
            int read;
            char[] buffer = new char[32768];
            while ((read = in.read(buffer)) != -1) {
                sb.append(buffer, 0, read);
            }
        }
        return sb.toString();
    }

    public static void copyDir(File dir, File destDir) throws IOException {
        File[] files;
        if (!dir.isDirectory()) {
            throw new IllegalArgumentException("Source directory " + dir + " does not exist or is not a directory");
        }
        if (destDir.isDirectory()) {
            throw new IllegalStateException("Destination directory " + destDir + " already exists");
        }
        if (!destDir.mkdirs()) {
            throw new IOException("Could not create " + destDir);
        }
        for (File file : files = dir.listFiles()) {
            if (file.isDirectory()) {
                FileUtils.copyDir(file, new File(destDir, file.getName()));
                continue;
            }
            if (file.isFile()) {
                FileUtils.copyFileToDir(file, destDir);
                continue;
            }
            logger.warn("Not copying " + file + "; not a regular file or directory");
        }
        long lastModified = dir.lastModified();
        if (lastModified > 0L) {
            destDir.setLastModified(lastModified);
        }
    }

    public static void copyFileToFile(File file, File destFile, boolean overwrite) throws IOException {
        if (!overwrite && destFile.isFile()) {
            throw new IllegalStateException("Destination file " + destFile + " already exists");
        }
        if (destFile.isDirectory()) {
            throw new IllegalStateException("Destination file is an existing directory");
        }
        StreamUtils.copy(new FileInputStream(file), new FileOutputStream(destFile));
        long lastModified = file.lastModified();
        if (lastModified > 0L) {
            destFile.setLastModified(lastModified);
        }
    }

    public static void copyFileToDir(File file, File destDir) throws IOException {
        try {
            FileUtils.copyFileToDir(file, destDir, null);
        }
        catch (ProgressReceiver.OperationCancelled e) {
            throw new InternalError();
        }
    }

    public static void copyFileToDir(File file, File destDir, ProgressReceiver progressReceiver) throws IOException, ProgressReceiver.OperationCancelled {
        File destFile = new File(destDir, file.getName());
        if (destFile.isFile()) {
            throw new IllegalStateException("Destination file " + destFile + " already exists");
        }
        StreamUtils.copy(new FileInputStream(file), new FileOutputStream(destFile), progressReceiver, file.length());
        long lastModified = file.lastModified();
        if (lastModified > 0L) {
            destFile.setLastModified(lastModified);
        }
    }

    public static boolean deleteDir(File dir) {
        File[] contents;
        if (!dir.isDirectory()) {
            throw new IllegalArgumentException(dir + " does not exist or is not a directory");
        }
        for (File file : contents = dir.listFiles()) {
            if (file.isDirectory()) {
                FileUtils.deleteDir(file);
                continue;
            }
            file.delete();
        }
        return dir.delete();
    }

    public static boolean emptyDir(File dir) {
        File[] contents;
        if (!dir.isDirectory()) {
            throw new IllegalArgumentException(dir + " does not exist or is not a directory");
        }
        boolean success = true;
        for (File file : contents = dir.listFiles()) {
            if (file.isDirectory()) {
                success &= FileUtils.deleteDir(file);
                continue;
            }
            success &= file.delete();
        }
        return success;
    }

    public static String sanitiseName(@NonNls String filename) {
        if (Strings.isNullOrEmpty((String)filename)) {
            return filename;
        }
        StringBuilder sb = new StringBuilder(filename.length());
        for (char c : filename.toCharArray()) {
            if (c < ' ' || ILLEGAL_CHARS.indexOf(c) != -1) {
                sb.append('_');
                continue;
            }
            sb.append(c);
        }
        if (sb.charAt(sb.length() - 1) == '.' || sb.charAt(sb.length() - 1) == ' ') {
            sb.setCharAt(sb.length() - 1, '_');
        }
        String uppercaseVersion = sb.toString().toUpperCase();
        for (String reservedName : RESERVED_NAMES) {
            if (!uppercaseVersion.startsWith(reservedName) || uppercaseVersion.length() != reservedName.length() && uppercaseVersion.charAt(reservedName.length()) != '.') continue;
            sb.setCharAt(2, '_');
            break;
        }
        return sb.toString();
    }

    public static File absolutise(File file) {
        return file != null && file.getClass() != File.class ? new File(file.getAbsolutePath()) : file;
    }

    public static <T extends Collection<File>> T absolutise(T collection) {
        if (collection == null) {
            return null;
        }
        if (collection instanceof List) {
            ListIterator<File> i = ((List)collection).listIterator();
            while (i.hasNext()) {
                File file = (File)i.next();
                try {
                    i.set(FileUtils.absolutise(file));
                }
                catch (UnsupportedOperationException e) {
                    AbstractList newCollection = collection instanceof RandomAccess ? new ArrayList(collection.size()) : new LinkedList();
                    for (File file2 : collection) {
                        newCollection.add(FileUtils.absolutise(file2));
                    }
                    return (T)newCollection;
                }
            }
            return collection;
        }
        AbstractCollection newCollection = collection instanceof SortedSet ? new TreeSet() : (collection instanceof Set ? new HashSet(collection.size()) : new ArrayList(collection.size()));
        for (File file : collection) {
            newCollection.add(FileUtils.absolutise(file));
        }
        return (T)newCollection;
    }

    public static <T extends Map<?, ?>> T absolutise(T map) {
        if (map == null) {
            return null;
        }
        for (Map.Entry<?, ?> entry : map.entrySet()) {
            if (entry.getKey() instanceof File && entry.getKey() != File.class) {
                AbstractMap newMap = map instanceof SortedMap ? new TreeMap() : new HashMap();
                for (Map.Entry<?, ?> entry2 : map.entrySet()) {
                    Object key = entry2.getKey();
                    Object value = entry2.getValue();
                    if (key instanceof File) {
                        key = FileUtils.absolutise((File)key);
                    }
                    if (value instanceof File) {
                        value = FileUtils.absolutise((File)value);
                    }
                    newMap.put(key, value);
                }
                return (T)newMap;
            }
            Object value = entry.getValue();
            if (!(value instanceof File)) continue;
            entry.setValue(FileUtils.absolutise((File)value));
        }
        return map;
    }

    public static void rotateFile(File file, String namePattern, int index, int maxIndex) throws IOException {
        if (file.isFile()) {
            if (index >= maxIndex) {
                if (!file.delete()) {
                    throw new IOException("Could not delete file " + file);
                }
            } else {
                File nextFile = new File(file.getParentFile(), MessageFormat.format(namePattern, index + 1));
                FileUtils.rotateFile(nextFile, namePattern, index + 1, maxIndex);
                if (!file.renameTo(nextFile)) {
                    throw new IOException("Could not rename file " + file + " to " + nextFile);
                }
            }
        }
    }

    public static void assertEquals(File expected, File actual) throws IOException {
        block20: {
            if (expected.isFile()) {
                if (!actual.isFile()) {
                    throw new AssertionError((Object)(expected + " is a file but " + actual + " is not"));
                }
                if (expected.length() != actual.length()) {
                    throw new AssertionError((Object)("Size of " + expected + " is " + expected.length() + " bytes but size of " + actual + " is " + actual.length() + " bytes"));
                }
                try (BufferedInputStream expIn = new BufferedInputStream(new FileInputStream(expected));
                     BufferedInputStream actIn = new BufferedInputStream(new FileInputStream(actual));){
                    int expByte;
                    long count = 0L;
                    while ((expByte = expIn.read()) != -1) {
                        int actByte = actIn.read();
                        if (expByte != actByte) {
                            throw new AssertionError((Object)("Byte " + count + " is " + expByte + " in " + expected + " but " + actByte + " in " + actual));
                        }
                    }
                    break block20;
                }
            }
            if (expected.isDirectory()) {
                File[] actFiles;
                File[] expFiles = expected.listFiles();
                if (expFiles.length != (actFiles = actual.listFiles()).length) {
                    throw new AssertionError((Object)(expected + " has " + expFiles.length + " entries but " + actual + " has " + actFiles.length + " entries"));
                }
                for (File expFile : expFiles) {
                    File actFile = new File(actual, expFile.getName());
                    if (!actFile.exists()) {
                        throw new AssertionError((Object)(expFile + " does not exist in " + actual));
                    }
                    FileUtils.assertEquals(expFile, actFile);
                }
            } else {
                throw new IllegalArgumentException("Don't know how to compare type of " + expected);
            }
        }
    }

    public static long getTreeSize(File dir) {
        long totalSize = 0L;
        for (File entry : dir.listFiles()) {
            if (entry.isFile()) {
                totalSize += entry.length();
                continue;
            }
            if (!entry.isDirectory()) continue;
            totalSize += FileUtils.getTreeSize(entry);
        }
        return totalSize;
    }

    public static long getFreeSpace(File path) throws IOException {
        return Files.getFileStore(path.toPath()).getUsableSpace();
    }

    public static String stripExtension(String filename) {
        int p = filename.lastIndexOf(46);
        if (p != -1) {
            return filename.substring(0, p);
        }
        return filename;
    }

    public static String stripDirectory(String path) {
        int p = Math.max(path.lastIndexOf(47), path.lastIndexOf(92));
        if (p != -1) {
            return path.substring(p + 1);
        }
        return path;
    }

    public static void visitFilesRecursively(File directory, Consumer<File> visitor) throws IOException {
        if (!directory.isDirectory()) {
            throw new IllegalArgumentException(directory + " is not a directory");
        }
        FileUtils.visitFilesRecursively(directory, visitor, new HashSet<File>());
    }

    private static void visitFilesRecursively(File directory, Consumer<File> visitor, Set<File> visitedDirectories) throws IOException {
        visitedDirectories.add(directory.getCanonicalFile());
        for (File file : directory.listFiles()) {
            if (file.isDirectory()) {
                File canonicalFile = file.getCanonicalFile();
                if (visitedDirectories.contains(canonicalFile)) {
                    logger.warn("Cycle detected on filesystem: {} has already been visited (as {})", (Object)file, (Object)canonicalFile);
                    continue;
                }
                FileUtils.visitFilesRecursively(file, visitor, visitedDirectories);
                continue;
            }
            visitor.accept(file);
        }
    }

    public static void main(String[] args) throws IOException {
        Checksum md5 = FileUtils.getMD5(new File(args[0]));
        System.out.print('{');
        byte[] bytes = md5.getBytes();
        for (int i = 0; i < bytes.length; ++i) {
            if (i > 0) {
                System.out.print(", ");
            }
            System.out.print("(byte) ");
            System.out.print(bytes[i]);
        }
        System.out.println('}');
    }
}

