/*
 * Decompiled with CFR 0.152.
 */
package org.pepsoft.util.undo;

import java.lang.ref.Reference;
import java.lang.ref.WeakReference;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.ListIterator;
import java.util.Map;
import java.util.Set;
import java.util.WeakHashMap;
import javax.swing.Action;
import org.pepsoft.util.MemoryUtils;
import org.pepsoft.util.ObjectUtils;
import org.pepsoft.util.undo.BufferKey;
import org.pepsoft.util.undo.Snapshot;
import org.pepsoft.util.undo.UndoListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class UndoManager {
    private Action undoAction;
    private Action redoAction;
    private final int maxFrames;
    private final LinkedList<Map<BufferKey<?>, Object>> history = new LinkedList();
    private int currentFrame;
    private final Map<BufferKey<?>, Object> readOnlyBufferCache = new WeakHashMap();
    private final Map<BufferKey<?>, Object> writeableBufferCache = new WeakHashMap();
    private final List<UndoListener> listeners = new ArrayList<UndoListener>();
    private final Map<BufferKey<?>, UndoListener> keyListeners = new WeakHashMap();
    private boolean savePointArmed;
    private final Set<Reference<Snapshot>> snapshots = new HashSet<Reference<Snapshot>>();
    private Set<Class<?>> stopAt;
    private static final int DEFAULT_MAX_FRAMES = 25;
    private static final Logger logger = LoggerFactory.getLogger(UndoManager.class);

    public UndoManager() {
        this(null, null, 25);
    }

    public UndoManager(int maxFrames) {
        this(null, null, maxFrames);
    }

    public UndoManager(Action undoAction, Action redoAction) {
        this(undoAction, redoAction, 25);
    }

    public UndoManager(Action undoAction, Action redoAction, int maxFrames) {
        this.maxFrames = maxFrames;
        this.history.add(new WeakHashMap());
        this.registerActions(undoAction, redoAction);
    }

    public synchronized void registerActions(Action undoAction, Action redoAction) {
        this.undoAction = undoAction;
        this.redoAction = redoAction;
        this.updateActions();
    }

    public synchronized void unregisterActions() {
        this.disableActions();
        this.undoAction = null;
        this.redoAction = null;
    }

    public int getMaxFrames() {
        return this.maxFrames;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void armSavePoint() {
        boolean savePointWasArmed = true;
        UndoManager undoManager = this;
        synchronized (undoManager) {
            if (!this.savePointArmed) {
                this.savePointArmed = true;
                savePointWasArmed = false;
            }
        }
        if (!savePointWasArmed) {
            this.listeners.forEach(UndoListener::savePointArmed);
            if (logger.isDebugEnabled()) {
                logger.debug("Save point armed");
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void savePoint() {
        UndoManager undoManager = this;
        synchronized (undoManager) {
            this.clearRedo();
            this.history.add(new WeakHashMap());
            ++this.currentFrame;
            this.pruneHistory();
            this.writeableBufferCache.clear();
            this.savePointArmed = false;
        }
        this.listeners.forEach(UndoListener::savePointCreated);
        this.updateActions();
        if (logger.isDebugEnabled()) {
            logger.debug("Save point set; new current frame: " + this.currentFrame);
            if (logger.isTraceEnabled()) {
                this.dumpBuffer();
            }
        }
    }

    public synchronized Snapshot getSnapshot() {
        Snapshot snapshot = new Snapshot(this, this.currentFrame);
        this.snapshots.add(new WeakReference<Snapshot>(snapshot));
        return snapshot;
    }

    public synchronized boolean isDirty() {
        return !this.writeableBufferCache.isEmpty();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean undo() {
        boolean undoPerformed;
        Map<BufferKey<?>, Object> previousHistoryFrame;
        UndoManager undoManager = this;
        synchronized (undoManager) {
            if (this.currentFrame > 0) {
                --this.currentFrame;
                this.readOnlyBufferCache.clear();
                this.writeableBufferCache.clear();
                previousHistoryFrame = this.history.get(this.currentFrame + 1);
                undoPerformed = true;
            } else {
                undoPerformed = false;
                previousHistoryFrame = null;
            }
        }
        if (undoPerformed) {
            this.listeners.forEach(UndoListener::undoPerformed);
            for (BufferKey key : previousHistoryFrame.keySet()) {
                UndoListener listener = this.keyListeners.get(key);
                if (listener == null) continue;
                listener.bufferChanged(key);
            }
            this.updateActions();
            if (logger.isDebugEnabled()) {
                logger.debug("Undo requested; now at frame " + this.currentFrame + " (total: " + this.history.size() + ")");
                if (logger.isTraceEnabled()) {
                    this.dumpBuffer();
                }
            }
            return true;
        }
        if (logger.isDebugEnabled()) {
            logger.debug("Undo requested, but no more frames available");
        }
        return false;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean redo() {
        boolean redoPerformed;
        Map<BufferKey<?>, Object> currentHistoryFrame;
        UndoManager undoManager = this;
        synchronized (undoManager) {
            if (this.currentFrame < this.history.size() - 1) {
                ++this.currentFrame;
                this.readOnlyBufferCache.clear();
                this.writeableBufferCache.clear();
                currentHistoryFrame = this.history.get(this.currentFrame);
                redoPerformed = true;
            } else {
                redoPerformed = false;
                currentHistoryFrame = null;
            }
        }
        if (redoPerformed) {
            this.listeners.forEach(UndoListener::redoPerformed);
            for (BufferKey key : currentHistoryFrame.keySet()) {
                UndoListener listener = this.keyListeners.get(key);
                if (listener == null) continue;
                listener.bufferChanged(key);
            }
            this.updateActions();
            if (logger.isDebugEnabled()) {
                logger.debug("Redo requested; now at frame " + this.currentFrame + " (total: " + this.history.size() + ")");
                if (logger.isTraceEnabled()) {
                    this.dumpBuffer();
                }
            }
            return true;
        }
        if (logger.isDebugEnabled()) {
            logger.debug("Redo requested, but no more frames available");
        }
        return false;
    }

    public synchronized void clear() {
        this.clearRedo();
        int deletedFrames = 0;
        while (this.history.size() > 1) {
            this.shrinkHistory();
            ++deletedFrames;
        }
        this.updateSnapshots(-deletedFrames);
        this.updateActions();
        this.savePointArmed = false;
        if (logger.isTraceEnabled()) {
            this.dumpBuffer();
        }
    }

    public synchronized void clearRedo() {
        if (this.currentFrame < this.history.size() - 1) {
            do {
                this.history.removeLast();
            } while (this.currentFrame < this.history.size() - 1);
            this.updateSnapshots(0);
            this.updateActions();
        }
    }

    public <T> void addBuffer(BufferKey<T> key, T buffer) {
        this.addBuffer(key, buffer, null);
    }

    public synchronized <T> void addBuffer(BufferKey<T> key, T buffer, UndoListener listener) {
        this.clearRedo();
        this.history.getLast().put(key, buffer);
        this.writeableBufferCache.put(key, buffer);
        if (listener != null) {
            this.keyListeners.put(key, listener);
        }
        if (logger.isTraceEnabled()) {
            logger.trace("Buffer added: " + key);
        }
    }

    public synchronized void removeBuffer(BufferKey<?> key) {
        this.writeableBufferCache.remove(key);
        this.readOnlyBufferCache.remove(key);
        for (Map map : this.history) {
            map.remove(key);
        }
        this.keyListeners.remove(key);
        if (logger.isTraceEnabled()) {
            logger.trace("Buffer removed: " + key);
        }
    }

    public synchronized <T> T getBuffer(BufferKey<T> key) {
        if (this.writeableBufferCache.containsKey(key)) {
            if (logger.isTraceEnabled()) {
                logger.trace("Getting buffer " + key + " for reading from writeable buffer cache");
            }
            return (T)this.writeableBufferCache.get(key);
        }
        if (this.readOnlyBufferCache.containsKey(key)) {
            if (logger.isTraceEnabled()) {
                logger.trace("Getting buffer " + key + " for reading from read-only buffer cache");
            }
            return (T)this.readOnlyBufferCache.get(key);
        }
        if (logger.isTraceEnabled()) {
            logger.trace("Getting buffer " + key + " for reading from history");
        }
        T buffer = this.findMostRecentCopy(key);
        this.readOnlyBufferCache.put(key, buffer);
        return buffer;
    }

    public synchronized <T> T getBufferForEditing(BufferKey<T> key) {
        Map<BufferKey<?>, Object> currentHistoryFrame;
        if (this.savePointArmed) {
            this.savePoint();
        }
        if (this.writeableBufferCache.containsKey(key)) {
            if (logger.isTraceEnabled()) {
                logger.trace("Getting buffer " + key + " for writing from writeable buffer cache");
            }
            return (T)this.writeableBufferCache.get(key);
        }
        this.clearRedo();
        if (this.readOnlyBufferCache.containsKey(key)) {
            if (logger.isTraceEnabled()) {
                logger.trace("Copying buffer " + key + " for writing from read-only buffer cache");
            }
            Object buffer = this.readOnlyBufferCache.remove(key);
            Object copy = ObjectUtils.copyObject(buffer);
            this.history.getLast().put(key, copy);
            this.writeableBufferCache.put(key, copy);
            return (T)copy;
        }
        if (logger.isTraceEnabled()) {
            logger.trace("Copying buffer " + key + " for writing from history");
        }
        if ((currentHistoryFrame = this.history.getLast()).containsKey(key)) {
            Object buffer = currentHistoryFrame.get(key);
            this.writeableBufferCache.put(key, buffer);
            return (T)buffer;
        }
        T buffer = this.findMostRecentCopy(key);
        T copy = ObjectUtils.copyObject(buffer);
        currentHistoryFrame.put(key, copy);
        this.writeableBufferCache.put(key, copy);
        return copy;
    }

    public synchronized void addListener(UndoListener listener) {
        if (logger.isTraceEnabled()) {
            logger.trace("Adding listener " + listener);
        }
        this.listeners.add(listener);
    }

    public synchronized void removeListener(UndoListener listener) {
        if (logger.isTraceEnabled()) {
            logger.trace("Removing listener " + listener);
        }
        this.listeners.remove(listener);
    }

    public synchronized Class<?>[] getStopAtClasses() {
        return this.stopAt.toArray(new Class[this.stopAt.size()]);
    }

    public synchronized void setStopAtClasses(Class<?> ... stopAt) {
        this.stopAt = new HashSet(Arrays.asList(stopAt));
    }

    public synchronized long getDataSize() {
        return MemoryUtils.getSize(this.history, this.stopAt);
    }

    private void updateSnapshots(int delta) {
        if (logger.isDebugEnabled()) {
            logger.debug("Updating snapshots");
        }
        int frameCount = this.history.size();
        Iterator<Reference<Snapshot>> i = this.snapshots.iterator();
        while (i.hasNext()) {
            Snapshot snapshot = i.next().get();
            if (snapshot == null) {
                if (logger.isDebugEnabled()) {
                    logger.debug("Removing garbage collected snapshot");
                }
                i.remove();
                continue;
            }
            snapshot.frame += delta;
            if (snapshot.frame >= 0 && snapshot.frame < frameCount) continue;
            if (logger.isDebugEnabled()) {
                logger.debug("Disabling and removing snapshot with invalid frame reference");
            }
            snapshot.frame = -1;
            i.remove();
        }
    }

    private void pruneHistory() {
        int deletedFrames = 0;
        while (this.history.size() > this.maxFrames) {
            this.shrinkHistory();
            ++deletedFrames;
        }
        if (deletedFrames > 0) {
            this.updateSnapshots(-deletedFrames);
        }
        if (logger.isTraceEnabled()) {
            this.dumpBuffer();
        }
    }

    private void shrinkHistory() {
        if (logger.isDebugEnabled()) {
            logger.debug("Removing oldest history frame; moving contents to next oldest frame");
        }
        Map<BufferKey<?>, Object> oldestFrame = this.history.removeFirst();
        Map<BufferKey<?>, Object> nextOldestFrame = this.history.getFirst();
        oldestFrame.entrySet().stream().filter(entry -> !nextOldestFrame.containsKey(entry.getKey())).forEach(entry -> nextOldestFrame.put((BufferKey<?>)entry.getKey(), entry.getValue()));
        if (this.currentFrame > 0) {
            --this.currentFrame;
        }
    }

    private <T> T findMostRecentCopy(BufferKey<T> key) {
        return this.findMostRecentCopy(key, this.currentFrame);
    }

    synchronized <T> T findMostRecentCopy(BufferKey<T> key, int frame) {
        ListIterator<Map<BufferKey<?>, Object>> i = this.history.listIterator(frame + 1);
        while (i.hasPrevious()) {
            Map<BufferKey<?>, Object> historyFrame = i.previous();
            if (historyFrame.containsKey(key)) {
                if (logger.isTraceEnabled()) {
                    logger.trace("Most recent copy of buffer " + key + " found in frame " + frame + " of history");
                }
                return (T)historyFrame.get(key);
            }
            --frame;
        }
        if (logger.isTraceEnabled()) {
            logger.trace("Buffer " + key + " not present in undo history");
        }
        return null;
    }

    private void dumpBuffer() {
        int index = 0;
        long totalDataSize = 0L;
        for (Map map : this.history) {
            long frameSize = MemoryUtils.getSize(map, this.stopAt);
            totalDataSize += frameSize;
            logger.debug((index == this.currentFrame ? "* " : "  ") + " " + (index < 10 ? "0" : "") + index + ": " + map.size() + " buffers (size: " + frameSize / 1024L + " KB)");
            ++index;
        }
        logger.debug("   Total data size: " + totalDataSize / 1024L + " KB");
    }

    private void updateActions() {
        if (this.undoAction != null) {
            this.undoAction.setEnabled(this.currentFrame > 0);
        }
        if (this.redoAction != null) {
            this.redoAction.setEnabled(this.currentFrame < this.history.size() - 1);
        }
    }

    private void disableActions() {
        if (this.undoAction != null) {
            this.undoAction.setEnabled(false);
        }
        if (this.redoAction != null) {
            this.redoAction.setEnabled(false);
        }
    }
}

