/*
 * Decompiled with CFR 0.152.
 */
package org.pgpainless.key.util;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import java.util.NoSuchElementException;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import org.bouncycastle.bcpg.S2K;
import org.bouncycastle.openpgp.PGPException;
import org.bouncycastle.openpgp.PGPKeyRing;
import org.bouncycastle.openpgp.PGPPrivateKey;
import org.bouncycastle.openpgp.PGPPublicKey;
import org.bouncycastle.openpgp.PGPPublicKeyRing;
import org.bouncycastle.openpgp.PGPPublicKeyRingCollection;
import org.bouncycastle.openpgp.PGPSecretKey;
import org.bouncycastle.openpgp.PGPSecretKeyRing;
import org.bouncycastle.openpgp.PGPSecretKeyRingCollection;
import org.bouncycastle.openpgp.PGPSignature;
import org.bouncycastle.openpgp.PGPUserAttributeSubpacketVector;
import org.bouncycastle.openpgp.operator.PBESecretKeyDecryptor;
import org.bouncycastle.openpgp.operator.PBESecretKeyEncryptor;
import org.bouncycastle.util.Strings;
import org.pgpainless.PGPainless;
import org.pgpainless.implementation.ImplementationFactory;
import org.pgpainless.key.protection.SecretKeyRingProtector;
import org.pgpainless.key.protection.UnlockSecretKey;
import org.pgpainless.key.protection.fixes.S2KUsageFix;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class KeyRingUtils {
    private static final Logger LOGGER = LoggerFactory.getLogger(KeyRingUtils.class);

    private KeyRingUtils() {
    }

    @Nonnull
    public static PGPSecretKey requirePrimarySecretKeyFrom(@Nonnull PGPSecretKeyRing secretKeys) {
        PGPSecretKey primarySecretKey = KeyRingUtils.getPrimarySecretKeyFrom(secretKeys);
        if (primarySecretKey == null) {
            throw new NoSuchElementException("Provided PGPSecretKeyRing has no primary secret key.");
        }
        return primarySecretKey;
    }

    @Nullable
    public static PGPSecretKey getPrimarySecretKeyFrom(@Nonnull PGPSecretKeyRing secretKeys) {
        PGPSecretKey secretKey = secretKeys.getSecretKey();
        if (secretKey.isMasterKey()) {
            return secretKey;
        }
        return null;
    }

    @Nonnull
    public static PGPPublicKey requirePrimaryPublicKeyFrom(@Nonnull PGPKeyRing keyRing) {
        PGPPublicKey primaryPublicKey = KeyRingUtils.getPrimaryPublicKeyFrom(keyRing);
        if (primaryPublicKey == null) {
            throw new NoSuchElementException("Provided PGPKeyRing has no primary public key.");
        }
        return primaryPublicKey;
    }

    @Nullable
    public static PGPPublicKey getPrimaryPublicKeyFrom(@Nonnull PGPKeyRing keyRing) {
        PGPPublicKey primaryPublicKey = keyRing.getPublicKey();
        if (primaryPublicKey.isMasterKey()) {
            return primaryPublicKey;
        }
        return null;
    }

    @Nullable
    public static PGPPublicKey getPublicKeyFrom(@Nonnull PGPKeyRing keyRing, long subKeyId) {
        return keyRing.getPublicKey(subKeyId);
    }

    @Nonnull
    public static PGPPublicKey requirePublicKeyFrom(@Nonnull PGPKeyRing keyRing, long subKeyId) {
        PGPPublicKey publicKey = KeyRingUtils.getPublicKeyFrom(keyRing, subKeyId);
        if (publicKey == null) {
            throw new NoSuchElementException("KeyRing does not contain public key with keyID " + Long.toHexString(subKeyId));
        }
        return publicKey;
    }

    @Nonnull
    public static PGPSecretKey requireSecretKeyFrom(@Nonnull PGPSecretKeyRing keyRing, long subKeyId) {
        PGPSecretKey secretKey = keyRing.getSecretKey(subKeyId);
        if (secretKey == null) {
            throw new NoSuchElementException("KeyRing does not contain secret key with keyID " + Long.toHexString(subKeyId));
        }
        return secretKey;
    }

    @Nonnull
    public static PGPPublicKeyRing publicKeys(@Nonnull PGPKeyRing keys) {
        if (keys instanceof PGPPublicKeyRing) {
            return (PGPPublicKeyRing)keys;
        }
        if (keys instanceof PGPSecretKeyRing) {
            return KeyRingUtils.publicKeyRingFrom((PGPSecretKeyRing)keys);
        }
        throw new IllegalArgumentException("Unknown keys class: " + keys.getClass().getName());
    }

    @Nonnull
    public static PGPPublicKeyRing publicKeyRingFrom(@Nonnull PGPSecretKeyRing secretKeys) {
        ArrayList<PGPPublicKey> publicKeyList = new ArrayList<PGPPublicKey>();
        Iterator<PGPPublicKey> publicKeyIterator = secretKeys.getPublicKeys();
        while (publicKeyIterator.hasNext()) {
            publicKeyList.add(publicKeyIterator.next());
        }
        PGPPublicKeyRing publicKeyRing = new PGPPublicKeyRing(publicKeyList);
        return publicKeyRing;
    }

    @Nonnull
    public static PGPPublicKeyRingCollection publicKeyRingCollectionFrom(@Nonnull PGPSecretKeyRingCollection secretKeyRings) {
        ArrayList<PGPPublicKeyRing> certificates = new ArrayList<PGPPublicKeyRing>();
        for (PGPSecretKeyRing secretKey : secretKeyRings) {
            certificates.add(PGPainless.extractCertificate(secretKey));
        }
        return new PGPPublicKeyRingCollection(certificates);
    }

    @Nonnull
    public static PGPPrivateKey unlockSecretKey(@Nonnull PGPSecretKey secretKey, @Nonnull SecretKeyRingProtector protector) throws PGPException {
        return UnlockSecretKey.unlockSecretKey(secretKey, protector);
    }

    @Nonnull
    public static PGPPublicKeyRingCollection keyRingsToKeyRingCollection(PGPPublicKeyRing ... rings) {
        return new PGPPublicKeyRingCollection(Arrays.asList(rings));
    }

    @Nonnull
    public static PGPSecretKeyRingCollection keyRingsToKeyRingCollection(PGPSecretKeyRing ... rings) {
        return new PGPSecretKeyRingCollection(Arrays.asList(rings));
    }

    public static boolean keyRingContainsKeyWithId(@Nonnull PGPPublicKeyRing ring, long keyId) {
        return ring.getPublicKey(keyId) != null;
    }

    @Nonnull
    public static <T extends PGPKeyRing> T injectCertification(@Nonnull T keyRing, @Nonnull PGPSignature certification) {
        return KeyRingUtils.injectCertification(keyRing, keyRing.getPublicKey(), certification);
    }

    @Nonnull
    public static <T extends PGPKeyRing> T injectCertification(@Nonnull T keyRing, @Nonnull PGPPublicKey certifiedKey, @Nonnull PGPSignature certification) {
        PGPPublicKeyRing publicKeys;
        PGPSecretKeyRing secretKeys = null;
        if (keyRing instanceof PGPSecretKeyRing) {
            secretKeys = (PGPSecretKeyRing)keyRing;
            publicKeys = PGPainless.extractCertificate(secretKeys);
        } else {
            publicKeys = (PGPPublicKeyRing)keyRing;
        }
        certifiedKey = PGPPublicKey.addCertification(certifiedKey, certification);
        ArrayList<PGPPublicKey> publicKeyList = new ArrayList<PGPPublicKey>();
        Iterator<PGPPublicKey> publicKeyIterator = publicKeys.iterator();
        boolean added = false;
        while (publicKeyIterator.hasNext()) {
            PGPPublicKey key = publicKeyIterator.next();
            if (key.getKeyID() == certifiedKey.getKeyID()) {
                added = true;
                publicKeyList.add(certifiedKey);
                continue;
            }
            publicKeyList.add(key);
        }
        if (!added) {
            throw new NoSuchElementException("Cannot find public key with id " + Long.toHexString(certifiedKey.getKeyID()) + " in the provided key ring.");
        }
        publicKeys = new PGPPublicKeyRing(publicKeyList);
        if (secretKeys == null) {
            return (T)publicKeys;
        }
        secretKeys = PGPSecretKeyRing.replacePublicKeys(secretKeys, publicKeys);
        return (T)secretKeys;
    }

    @Nonnull
    public static <T extends PGPKeyRing> T injectCertification(@Nonnull T keyRing, @Nonnull String userId, @Nonnull PGPSignature certification) {
        PGPPublicKeyRing publicKeys;
        PGPSecretKeyRing secretKeys = null;
        if (keyRing instanceof PGPSecretKeyRing) {
            secretKeys = (PGPSecretKeyRing)keyRing;
            publicKeys = PGPainless.extractCertificate(secretKeys);
        } else {
            publicKeys = (PGPPublicKeyRing)keyRing;
        }
        Iterator<PGPPublicKey> publicKeyIterator = publicKeys.iterator();
        PGPPublicKey primaryKey = publicKeyIterator.next();
        primaryKey = PGPPublicKey.addCertification(primaryKey, userId, certification);
        ArrayList<PGPPublicKey> publicKeyList = new ArrayList<PGPPublicKey>();
        publicKeyList.add(primaryKey);
        while (publicKeyIterator.hasNext()) {
            publicKeyList.add(publicKeyIterator.next());
        }
        publicKeys = new PGPPublicKeyRing(publicKeyList);
        if (secretKeys == null) {
            return (T)publicKeys;
        }
        secretKeys = PGPSecretKeyRing.replacePublicKeys(secretKeys, publicKeys);
        return (T)secretKeys;
    }

    @Nonnull
    public static <T extends PGPKeyRing> T injectCertification(@Nonnull T keyRing, @Nonnull PGPUserAttributeSubpacketVector userAttributes, @Nonnull PGPSignature certification) {
        PGPPublicKeyRing publicKeys;
        PGPSecretKeyRing secretKeys = null;
        if (keyRing instanceof PGPSecretKeyRing) {
            secretKeys = (PGPSecretKeyRing)keyRing;
            publicKeys = PGPainless.extractCertificate(secretKeys);
        } else {
            publicKeys = (PGPPublicKeyRing)keyRing;
        }
        Iterator<PGPPublicKey> publicKeyIterator = publicKeys.iterator();
        PGPPublicKey primaryKey = publicKeyIterator.next();
        primaryKey = PGPPublicKey.addCertification(primaryKey, userAttributes, certification);
        ArrayList<PGPPublicKey> publicKeyList = new ArrayList<PGPPublicKey>();
        publicKeyList.add(primaryKey);
        while (publicKeyIterator.hasNext()) {
            publicKeyList.add(publicKeyIterator.next());
        }
        publicKeys = new PGPPublicKeyRing(publicKeyList);
        if (secretKeys == null) {
            return (T)publicKeys;
        }
        secretKeys = PGPSecretKeyRing.replacePublicKeys(secretKeys, publicKeys);
        return (T)secretKeys;
    }

    @Nonnull
    public static <T extends PGPKeyRing> T keysPlusPublicKey(@Nonnull T keyRing, @Nonnull PGPPublicKey publicKey) {
        PGPPublicKeyRing publicKeys;
        PGPSecretKeyRing secretKeys = null;
        if (keyRing instanceof PGPSecretKeyRing) {
            secretKeys = (PGPSecretKeyRing)keyRing;
            publicKeys = PGPainless.extractCertificate(secretKeys);
        } else {
            publicKeys = (PGPPublicKeyRing)keyRing;
        }
        publicKeys = PGPPublicKeyRing.insertPublicKey(publicKeys, publicKey);
        if (secretKeys == null) {
            return (T)publicKeys;
        }
        secretKeys = PGPSecretKeyRing.insertOrReplacePublicKey(secretKeys, publicKey);
        return (T)secretKeys;
    }

    @Nonnull
    public static PGPSecretKeyRing keysPlusSecretKey(@Nonnull PGPSecretKeyRing secretKeys, @Nonnull PGPSecretKey secretKey) {
        return PGPSecretKeyRing.insertSecretKey(secretKeys, secretKey);
    }

    @Nonnull
    public static PGPSecretKey secretKeyPlusSignature(@Nonnull PGPSecretKey secretKey, @Nonnull PGPSignature signature) {
        PGPPublicKey publicKey = secretKey.getPublicKey();
        publicKey = PGPPublicKey.addCertification(publicKey, signature);
        PGPSecretKey newSecretKey = PGPSecretKey.replacePublicKey(secretKey, publicKey);
        return newSecretKey;
    }

    @Nonnull
    public static PGPSecretKeyRing stripSecretKey(@Nonnull PGPSecretKeyRing secretKeys, long secretKeyId) throws IOException, PGPException {
        if (secretKeys.getPublicKey().getKeyID() == secretKeyId) {
            throw new IllegalArgumentException("Bouncy Castle currently cannot deal with stripped secret primary keys.");
        }
        if (secretKeys.getSecretKey(secretKeyId) == null) {
            throw new NoSuchElementException("PGPSecretKeyRing does not contain secret key " + Long.toHexString(secretKeyId));
        }
        ByteArrayOutputStream encoded = new ByteArrayOutputStream();
        for (PGPSecretKey secretKey : secretKeys) {
            if (secretKey.getKeyID() == secretKeyId) {
                secretKey.getPublicKey().encode(encoded);
                continue;
            }
            secretKey.encode(encoded);
        }
        Iterator<PGPPublicKey> it = secretKeys.getExtraPublicKeys();
        while (it.hasNext()) {
            PGPPublicKey extra = it.next();
            extra.encode(encoded);
        }
        return new PGPSecretKeyRing(encoded.toByteArray(), ImplementationFactory.getInstance().getKeyFingerprintCalculator());
    }

    public static PGPPublicKey getStrippedDownPublicKey(PGPPublicKey bloatedKey) throws PGPException {
        return new PGPPublicKey(bloatedKey.getPublicKeyPacket(), ImplementationFactory.getInstance().getKeyFingerprintCalculator());
    }

    public static List<String> getUserIdsIgnoringInvalidUTF8(PGPPublicKey key) {
        ArrayList<String> userIds = new ArrayList<String>();
        Iterator<byte[]> it = key.getRawUserIDs();
        while (it.hasNext()) {
            byte[] rawUserId = it.next();
            try {
                userIds.add(Strings.fromUTF8ByteArray(rawUserId));
            }
            catch (IllegalArgumentException e) {
                LOGGER.warn("Invalid UTF-8 user-ID encountered: " + new String(rawUserId));
            }
        }
        return userIds;
    }

    public static PGPSecretKeyRing changePassphrase(Long keyId, PGPSecretKeyRing secretKeys, SecretKeyRingProtector oldProtector, SecretKeyRingProtector newProtector) throws PGPException {
        Iterator<PGPSecretKey> secretKeyIterator;
        ArrayList<PGPSecretKey> secretKeyList = new ArrayList<PGPSecretKey>();
        if (keyId == null) {
            secretKeyIterator = secretKeys.getSecretKeys();
            while (secretKeyIterator.hasNext()) {
                PGPSecretKey secretKey = secretKeyIterator.next();
                secretKey = KeyRingUtils.reencryptPrivateKey(secretKey, oldProtector, newProtector);
                secretKeyList.add(secretKey);
            }
        } else {
            secretKeyIterator = secretKeys.getSecretKeys();
            while (secretKeyIterator.hasNext()) {
                PGPSecretKey secretKey = secretKeyIterator.next();
                if (secretKey.getPublicKey().getKeyID() == keyId.longValue()) {
                    secretKey = KeyRingUtils.reencryptPrivateKey(secretKey, oldProtector, newProtector);
                }
                secretKeyList.add(secretKey);
            }
        }
        PGPSecretKeyRing newRing = new PGPSecretKeyRing(secretKeyList);
        newRing = KeyRingUtils.s2kUsageFixIfNecessary(newRing, newProtector);
        return newRing;
    }

    public static PGPSecretKey reencryptPrivateKey(PGPSecretKey secretKey, SecretKeyRingProtector oldProtector, SecretKeyRingProtector newProtector) throws PGPException {
        S2K s2k = secretKey.getS2K();
        if (s2k == null || s2k.getType() != 101) {
            long secretKeyId = secretKey.getKeyID();
            PBESecretKeyDecryptor decryptor = oldProtector.getDecryptor(secretKeyId);
            PBESecretKeyEncryptor encryptor = newProtector.getEncryptor(secretKeyId);
            secretKey = PGPSecretKey.copyWithNewPassword(secretKey, decryptor, encryptor);
        }
        return secretKey;
    }

    public static PGPSecretKeyRing s2kUsageFixIfNecessary(PGPSecretKeyRing secretKeys, SecretKeyRingProtector protector) throws PGPException {
        boolean hasS2KUsageChecksum = false;
        for (PGPSecretKey secKey : secretKeys) {
            if (secKey.getS2KUsage() != 255) continue;
            hasS2KUsageChecksum = true;
            break;
        }
        if (hasS2KUsageChecksum) {
            secretKeys = S2KUsageFix.replaceUsageChecksumWithUsageSha1(secretKeys, protector, true);
        }
        return secretKeys;
    }
}

