/*
 * Decompiled with CFR 0.152.
 */
package org.pgpainless.decryption_verification;

import java.io.BufferedInputStream;
import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import javax.annotation.Nonnull;
import org.bouncycastle.bcpg.ArmoredInputStream;
import org.bouncycastle.openpgp.PGPCompressedData;
import org.bouncycastle.openpgp.PGPEncryptedData;
import org.bouncycastle.openpgp.PGPEncryptedDataList;
import org.bouncycastle.openpgp.PGPException;
import org.bouncycastle.openpgp.PGPKeyRing;
import org.bouncycastle.openpgp.PGPLiteralData;
import org.bouncycastle.openpgp.PGPObjectFactory;
import org.bouncycastle.openpgp.PGPOnePassSignature;
import org.bouncycastle.openpgp.PGPOnePassSignatureList;
import org.bouncycastle.openpgp.PGPPBEEncryptedData;
import org.bouncycastle.openpgp.PGPPrivateKey;
import org.bouncycastle.openpgp.PGPPublicKey;
import org.bouncycastle.openpgp.PGPPublicKeyEncryptedData;
import org.bouncycastle.openpgp.PGPPublicKeyRing;
import org.bouncycastle.openpgp.PGPSecretKey;
import org.bouncycastle.openpgp.PGPSecretKeyRing;
import org.bouncycastle.openpgp.PGPSignature;
import org.bouncycastle.openpgp.PGPSignatureList;
import org.bouncycastle.openpgp.PGPUtil;
import org.bouncycastle.openpgp.operator.KeyFingerPrintCalculator;
import org.bouncycastle.openpgp.operator.PBEDataDecryptorFactory;
import org.bouncycastle.openpgp.operator.PGPContentVerifierBuilderProvider;
import org.bouncycastle.openpgp.operator.PublicKeyDataDecryptorFactory;
import org.pgpainless.PGPainless;
import org.pgpainless.algorithm.CompressionAlgorithm;
import org.pgpainless.algorithm.EncryptionPurpose;
import org.pgpainless.algorithm.StreamEncoding;
import org.pgpainless.algorithm.SymmetricKeyAlgorithm;
import org.pgpainless.decryption_verification.ConsumerOptions;
import org.pgpainless.decryption_verification.DecryptionStream;
import org.pgpainless.decryption_verification.OpenPgpMetadata;
import org.pgpainless.decryption_verification.SignatureInputStream;
import org.pgpainless.exception.MessageNotIntegrityProtectedException;
import org.pgpainless.exception.MissingDecryptionMethodException;
import org.pgpainless.exception.MissingLiteralDataException;
import org.pgpainless.exception.UnacceptableAlgorithmException;
import org.pgpainless.exception.WrongConsumingMethodException;
import org.pgpainless.implementation.ImplementationFactory;
import org.pgpainless.key.SubkeyIdentifier;
import org.pgpainless.key.info.KeyRingInfo;
import org.pgpainless.key.protection.UnlockSecretKey;
import org.pgpainless.signature.DetachedSignature;
import org.pgpainless.signature.OnePassSignatureCheck;
import org.pgpainless.signature.SignatureUtils;
import org.pgpainless.util.CRCingArmoredInputStreamWrapper;
import org.pgpainless.util.IntegrityProtectedInputStream;
import org.pgpainless.util.Passphrase;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class DecryptionStreamFactory {
    private static final Logger LOGGER = LoggerFactory.getLogger(DecryptionStreamFactory.class);
    private static final int MAX_RECURSION_DEPTH = 16;
    private final ConsumerOptions options;
    private final OpenPgpMetadata.Builder resultBuilder = OpenPgpMetadata.getBuilder();
    private final List<OnePassSignatureCheck> onePassSignatureChecks = new ArrayList<OnePassSignatureCheck>();
    private final List<DetachedSignature> detachedSignatureChecks = new ArrayList<DetachedSignature>();
    private static final PGPContentVerifierBuilderProvider verifierBuilderProvider = ImplementationFactory.getInstance().getPGPContentVerifierBuilderProvider();
    private static final KeyFingerPrintCalculator keyFingerprintCalculator = ImplementationFactory.getInstance().getKeyFingerprintCalculator();
    private IntegrityProtectedInputStream integrityProtectedEncryptedInputStream;

    public static DecryptionStream create(@Nonnull InputStream inputStream, @Nonnull ConsumerOptions options) throws PGPException, IOException {
        DecryptionStreamFactory factory = new DecryptionStreamFactory(options);
        return factory.parseOpenPGPDataAndCreateDecryptionStream(inputStream);
    }

    public DecryptionStreamFactory(ConsumerOptions options) {
        this.options = options;
        this.initializeDetachedSignatures(options.getDetachedSignatures());
    }

    private void initializeDetachedSignatures(Set<PGPSignature> signatures) {
        for (PGPSignature signature : signatures) {
            long issuerKeyId = SignatureUtils.determineIssuerKeyId(signature);
            PGPPublicKeyRing signingKeyRing = this.findSignatureVerificationKeyRing(issuerKeyId);
            if (signingKeyRing == null) continue;
            PGPPublicKey signingKey = signingKeyRing.getPublicKey(issuerKeyId);
            SubkeyIdentifier signingKeyIdentifier = new SubkeyIdentifier((PGPKeyRing)signingKeyRing, signingKey.getKeyID());
            try {
                signature.init(verifierBuilderProvider, signingKey);
                DetachedSignature detachedSignature = new DetachedSignature(signature, (PGPKeyRing)signingKeyRing, signingKeyIdentifier);
                this.detachedSignatureChecks.add(detachedSignature);
            }
            catch (PGPException e) {
                LOGGER.warn("Cannot verify detached signature made by {}. Reason: {}", new Object[]{signingKeyIdentifier, e.getMessage(), e});
            }
        }
    }

    private DecryptionStream parseOpenPGPDataAndCreateDecryptionStream(InputStream inputStream) throws IOException, PGPException {
        ArmoredInputStream armor;
        BufferedInputStream bufferedIn = new BufferedInputStream(inputStream);
        bufferedIn.mark(200);
        InputStream decoderStream = PGPUtil.getDecoderStream((InputStream)bufferedIn);
        decoderStream = CRCingArmoredInputStreamWrapper.possiblyWrap(decoderStream);
        if (decoderStream instanceof ArmoredInputStream && (armor = (ArmoredInputStream)decoderStream).isClearText()) {
            throw new WrongConsumingMethodException("Message appears to be using the Cleartext Signature Framework. Use PGPainless.verifyCleartextSignedMessage() to verify this message instead.");
        }
        PGPObjectFactory objectFactory = new PGPObjectFactory(decoderStream, keyFingerprintCalculator);
        try {
            inputStream = this.processPGPPackets(objectFactory, 1);
        }
        catch (EOFException e) {
            throw e;
        }
        catch (MissingLiteralDataException e) {
            LOGGER.debug("The message appears to not be an OpenPGP message. This is probably data signed with detached signatures?");
            bufferedIn.reset();
            inputStream = this.wrapInVerifySignatureStream(bufferedIn);
        }
        catch (IOException e) {
            if (e.getMessage().contains("invalid armor")) {
                LOGGER.debug("The message is apparently not armored.");
                bufferedIn.reset();
                inputStream = this.wrapInVerifySignatureStream(bufferedIn);
            }
            throw e;
        }
        return new DecryptionStream(inputStream, this.resultBuilder, this.integrityProtectedEncryptedInputStream, decoderStream instanceof ArmoredInputStream ? decoderStream : null);
    }

    private InputStream wrapInVerifySignatureStream(InputStream bufferedIn) {
        return new SignatureInputStream.VerifySignatures(bufferedIn, this.onePassSignatureChecks, this.detachedSignatureChecks, this.options, this.resultBuilder);
    }

    private InputStream processPGPPackets(@Nonnull PGPObjectFactory objectFactory, int depth) throws IOException, PGPException {
        Object nextPgpObject;
        if (depth >= 16) {
            throw new PGPException("Maximum recursion depth of packages exceeded.");
        }
        while ((nextPgpObject = objectFactory.nextObject()) != null) {
            if (nextPgpObject instanceof PGPEncryptedDataList) {
                return this.processPGPEncryptedDataList((PGPEncryptedDataList)nextPgpObject, depth);
            }
            if (nextPgpObject instanceof PGPCompressedData) {
                return this.processPGPCompressedData((PGPCompressedData)nextPgpObject, depth);
            }
            if (nextPgpObject instanceof PGPOnePassSignatureList) {
                return this.processOnePassSignatureList(objectFactory, (PGPOnePassSignatureList)nextPgpObject, depth);
            }
            if (!(nextPgpObject instanceof PGPLiteralData)) continue;
            return this.processPGPLiteralData(objectFactory, (PGPLiteralData)nextPgpObject, depth);
        }
        throw new MissingLiteralDataException("No Literal Data Packet found");
    }

    private InputStream processPGPEncryptedDataList(PGPEncryptedDataList pgpEncryptedDataList, int depth) throws PGPException, IOException {
        LOGGER.debug("Depth {}: Encountered PGPEncryptedDataList", (Object)depth);
        InputStream decryptedDataStream = this.decryptSessionKey(pgpEncryptedDataList);
        InputStream decodedDataStream = PGPUtil.getDecoderStream((InputStream)decryptedDataStream);
        PGPObjectFactory factory = new PGPObjectFactory(decodedDataStream, keyFingerprintCalculator);
        return this.processPGPPackets(factory, ++depth);
    }

    private InputStream processPGPCompressedData(PGPCompressedData pgpCompressedData, int depth) throws PGPException, IOException {
        CompressionAlgorithm compressionAlgorithm = CompressionAlgorithm.fromId(pgpCompressedData.getAlgorithm());
        LOGGER.debug("Depth {}: Encountered PGPCompressedData: {}", (Object)depth, (Object)compressionAlgorithm);
        this.resultBuilder.setCompressionAlgorithm(compressionAlgorithm);
        InputStream inflatedDataStream = pgpCompressedData.getDataStream();
        InputStream decodedDataStream = PGPUtil.getDecoderStream((InputStream)inflatedDataStream);
        PGPObjectFactory objectFactory = new PGPObjectFactory(decodedDataStream, keyFingerprintCalculator);
        return this.processPGPPackets(objectFactory, ++depth);
    }

    private InputStream processOnePassSignatureList(@Nonnull PGPObjectFactory objectFactory, PGPOnePassSignatureList onePassSignatures, int depth) throws PGPException, IOException {
        LOGGER.debug("Depth {}: Encountered PGPOnePassSignatureList of size {}", (Object)depth, (Object)onePassSignatures.size());
        this.initOnePassSignatures(onePassSignatures);
        return this.processPGPPackets(objectFactory, ++depth);
    }

    private InputStream processPGPLiteralData(@Nonnull PGPObjectFactory objectFactory, PGPLiteralData pgpLiteralData, int depth) throws IOException {
        LOGGER.debug("Depth {}: Found PGPLiteralData", (Object)depth);
        InputStream literalDataInputStream = pgpLiteralData.getInputStream();
        this.resultBuilder.setFileName(pgpLiteralData.getFileName()).setModificationDate(pgpLiteralData.getModificationTime()).setFileEncoding(StreamEncoding.fromCode(pgpLiteralData.getFormat()));
        if (this.onePassSignatureChecks.isEmpty()) {
            LOGGER.debug("No OnePassSignatures found -> We are done");
            return literalDataInputStream;
        }
        PGPSignatureList signatures = this.parseSignatures(objectFactory);
        List<PGPSignature> signatureList = SignatureUtils.toList(signatures);
        for (int i = 0; i < this.onePassSignatureChecks.size(); ++i) {
            this.onePassSignatureChecks.get(i).setSignature(signatureList.get(this.onePassSignatureChecks.size() - i - 1));
        }
        return new SignatureInputStream.VerifySignatures(literalDataInputStream, this.onePassSignatureChecks, this.detachedSignatureChecks, this.options, this.resultBuilder){};
    }

    private PGPSignatureList parseSignatures(PGPObjectFactory objectFactory) throws IOException {
        PGPSignatureList signatureList = null;
        Object pgpObject = objectFactory.nextObject();
        while (pgpObject != null && signatureList == null) {
            if (pgpObject instanceof PGPSignatureList) {
                signatureList = (PGPSignatureList)pgpObject;
                continue;
            }
            pgpObject = objectFactory.nextObject();
        }
        if (signatureList == null || signatureList.isEmpty()) {
            throw new IOException("Verification failed - No Signatures found");
        }
        return signatureList;
    }

    private InputStream decryptSessionKey(@Nonnull PGPEncryptedDataList encryptedDataList) throws PGPException {
        Iterator encryptedDataIterator = encryptedDataList.getEncryptedDataObjects();
        if (!encryptedDataIterator.hasNext()) {
            throw new PGPException("Decryption failed - EncryptedDataList has no items");
        }
        PGPPrivateKey decryptionKey = null;
        PGPPublicKeyEncryptedData encryptedSessionKey = null;
        block4: while (encryptedDataIterator.hasNext()) {
            PGPEncryptedData encryptedData = (PGPEncryptedData)encryptedDataIterator.next();
            if (!encryptedData.isIntegrityProtected()) {
                throw new MessageNotIntegrityProtectedException();
            }
            if (encryptedData instanceof PGPPBEEncryptedData) {
                PGPPBEEncryptedData pbeEncryptedData = (PGPPBEEncryptedData)encryptedData;
                for (Passphrase passphrase : this.options.getDecryptionPassphrases()) {
                    PBEDataDecryptorFactory passphraseDecryptor = ImplementationFactory.getInstance().getPBEDataDecryptorFactory(passphrase);
                    try {
                        InputStream decryptedDataStream = pbeEncryptedData.getDataStream(passphraseDecryptor);
                        SymmetricKeyAlgorithm symmetricKeyAlgorithm = SymmetricKeyAlgorithm.fromId(pbeEncryptedData.getSymmetricAlgorithm(passphraseDecryptor));
                        this.throwIfAlgorithmIsRejected(symmetricKeyAlgorithm);
                        this.resultBuilder.setSymmetricKeyAlgorithm(symmetricKeyAlgorithm);
                        this.integrityProtectedEncryptedInputStream = new IntegrityProtectedInputStream(decryptedDataStream, (PGPEncryptedData)pbeEncryptedData);
                        return this.integrityProtectedEncryptedInputStream;
                    }
                    catch (PGPException e) {
                        LOGGER.debug("Probable passphrase mismatch, skip PBE encrypted data block", (Throwable)e);
                    }
                }
                continue;
            }
            if (!(encryptedData instanceof PGPPublicKeyEncryptedData)) continue;
            PGPPublicKeyEncryptedData publicKeyEncryptedData = (PGPPublicKeyEncryptedData)encryptedData;
            long keyId = publicKeyEncryptedData.getKeyID();
            if (this.options.getDecryptionKeys().isEmpty()) continue;
            if (keyId != 0L) {
                LOGGER.debug("PGPEncryptedData is encrypted for key {}", (Object)Long.toHexString(keyId));
                this.resultBuilder.addRecipientKeyId(keyId);
                PGPSecretKeyRing decryptionKeyRing = this.findDecryptionKeyRing(keyId);
                if (decryptionKeyRing == null) continue;
                PGPSecretKey secretKey = decryptionKeyRing.getSecretKey(keyId);
                LOGGER.debug("Found respective secret key {}", (Object)Long.toHexString(keyId));
                encryptedSessionKey = publicKeyEncryptedData;
                decryptionKey = UnlockSecretKey.unlockSecretKey(secretKey, this.options.getSecretKeyProtector(decryptionKeyRing));
                this.resultBuilder.setDecryptionKey(new SubkeyIdentifier((PGPKeyRing)decryptionKeyRing, decryptionKey.getKeyID()));
                continue;
            }
            LOGGER.debug("Hidden recipient detected. Try to decrypt with all available secret keys.");
            for (PGPSecretKeyRing ring : this.options.getDecryptionKeys()) {
                KeyRingInfo info = new KeyRingInfo((PGPKeyRing)ring);
                List<PGPPublicKey> encryptionSubkeys = info.getEncryptionSubkeys(EncryptionPurpose.STORAGE_AND_COMMUNICATIONS);
                for (PGPPublicKey pubkey : encryptionSubkeys) {
                    PGPSecretKey key = ring.getSecretKey(pubkey.getKeyID());
                    if (key == null) continue;
                    PGPPrivateKey privateKey = UnlockSecretKey.unlockSecretKey(key, this.options.getSecretKeyProtector(ring).getDecryptor(key.getKeyID()));
                    PublicKeyDataDecryptorFactory decryptorFactory = ImplementationFactory.getInstance().getPublicKeyDataDecryptorFactory(privateKey);
                    try {
                        publicKeyEncryptedData.getSymmetricAlgorithm(decryptorFactory);
                        LOGGER.debug("Found correct key {} for hidden recipient decryption.", (Object)Long.toHexString(key.getKeyID()));
                        decryptionKey = privateKey;
                        this.resultBuilder.setDecryptionKey(new SubkeyIdentifier((PGPKeyRing)ring, decryptionKey.getKeyID()));
                        encryptedSessionKey = publicKeyEncryptedData;
                        continue block4;
                    }
                    catch (ClassCastException | PGPException e) {
                        LOGGER.debug("Skipping wrong key {} for hidden recipient decryption.", (Object)Long.toHexString(key.getKeyID()), (Object)e);
                    }
                }
            }
        }
        return this.decryptWith(encryptedSessionKey, decryptionKey);
    }

    private InputStream decryptWith(PGPPublicKeyEncryptedData encryptedSessionKey, PGPPrivateKey decryptionKey) throws PGPException {
        if (decryptionKey == null) {
            throw new MissingDecryptionMethodException("Decryption failed - No suitable decryption key or passphrase found");
        }
        PublicKeyDataDecryptorFactory dataDecryptor = ImplementationFactory.getInstance().getPublicKeyDataDecryptorFactory(decryptionKey);
        SymmetricKeyAlgorithm symmetricKeyAlgorithm = SymmetricKeyAlgorithm.fromId(encryptedSessionKey.getSymmetricAlgorithm(dataDecryptor));
        if (symmetricKeyAlgorithm == SymmetricKeyAlgorithm.NULL) {
            LOGGER.debug("Message is unencrypted");
        } else {
            LOGGER.debug("Message is encrypted using {}", (Object)symmetricKeyAlgorithm);
        }
        this.throwIfAlgorithmIsRejected(symmetricKeyAlgorithm);
        this.resultBuilder.setSymmetricKeyAlgorithm(symmetricKeyAlgorithm);
        this.integrityProtectedEncryptedInputStream = new IntegrityProtectedInputStream(encryptedSessionKey.getDataStream(dataDecryptor), (PGPEncryptedData)encryptedSessionKey);
        return this.integrityProtectedEncryptedInputStream;
    }

    private void throwIfAlgorithmIsRejected(SymmetricKeyAlgorithm algorithm) throws UnacceptableAlgorithmException {
        if (!PGPainless.getPolicy().getSymmetricKeyDecryptionAlgoritmPolicy().isAcceptable(algorithm)) {
            throw new UnacceptableAlgorithmException("Data is " + (algorithm == SymmetricKeyAlgorithm.NULL ? "unencrypted" : "encrypted with symmetric algorithm " + (Object)((Object)algorithm)) + " which is not acceptable as per PGPainless' policy.\nTo mark this algorithm as acceptable, use PGPainless.getPolicy().setSymmetricKeyDecryptionAlgorithmPolicy().");
        }
    }

    private void initOnePassSignatures(@Nonnull PGPOnePassSignatureList onePassSignatureList) throws PGPException {
        Iterator iterator = onePassSignatureList.iterator();
        if (!iterator.hasNext()) {
            throw new PGPException("Verification failed - No OnePassSignatures found");
        }
        this.processOnePassSignatures(iterator);
    }

    private void processOnePassSignatures(Iterator<PGPOnePassSignature> signatures) throws PGPException {
        while (signatures.hasNext()) {
            PGPOnePassSignature signature = signatures.next();
            this.processOnePassSignature(signature);
        }
    }

    private void processOnePassSignature(PGPOnePassSignature signature) throws PGPException {
        long keyId = signature.getKeyID();
        LOGGER.debug("Encountered OnePassSignature from {}", (Object)Long.toHexString(keyId));
        PGPPublicKeyRing verificationKeyRing = this.findSignatureVerificationKeyRing(keyId);
        if (verificationKeyRing == null) {
            LOGGER.debug("Missing verification key from {}", (Object)Long.toHexString(keyId));
            return;
        }
        PGPPublicKey verificationKey = verificationKeyRing.getPublicKey(keyId);
        signature.init(verifierBuilderProvider, verificationKey);
        OnePassSignatureCheck onePassSignature = new OnePassSignatureCheck(signature, verificationKeyRing);
        this.onePassSignatureChecks.add(onePassSignature);
    }

    private PGPSecretKeyRing findDecryptionKeyRing(long keyId) {
        for (PGPSecretKeyRing key : this.options.getDecryptionKeys()) {
            if (key.getSecretKey(keyId) == null) continue;
            return key;
        }
        return null;
    }

    private PGPPublicKeyRing findSignatureVerificationKeyRing(long keyId) {
        PGPPublicKeyRing verificationKeyRing = null;
        for (PGPPublicKeyRing publicKeyRing : this.options.getCertificates()) {
            PGPPublicKey verificationKey = publicKeyRing.getPublicKey(keyId);
            if (verificationKey == null) continue;
            LOGGER.debug("Found public key {} for signature verification", (Object)Long.toHexString(keyId));
            verificationKeyRing = publicKeyRing;
            break;
        }
        if (verificationKeyRing == null && this.options.getMissingCertificateCallback() != null) {
            verificationKeyRing = this.options.getMissingCertificateCallback().onMissingPublicKeyEncountered(keyId);
        }
        return verificationKeyRing;
    }
}

