/*
 * Decompiled with CFR 0.152.
 */
package org.pgpainless.decryption_verification;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Stack;
import javax.annotation.Nonnull;
import org.bouncycastle.bcpg.ArmoredInputStream;
import org.bouncycastle.bcpg.BCPGInputStream;
import org.bouncycastle.bcpg.S2K;
import org.bouncycastle.bcpg.UnsupportedPacketVersionException;
import org.bouncycastle.openpgp.PGPCompressedData;
import org.bouncycastle.openpgp.PGPEncryptedData;
import org.bouncycastle.openpgp.PGPEncryptedDataList;
import org.bouncycastle.openpgp.PGPException;
import org.bouncycastle.openpgp.PGPKeyRing;
import org.bouncycastle.openpgp.PGPLiteralData;
import org.bouncycastle.openpgp.PGPOnePassSignature;
import org.bouncycastle.openpgp.PGPPBEEncryptedData;
import org.bouncycastle.openpgp.PGPPrivateKey;
import org.bouncycastle.openpgp.PGPPublicKey;
import org.bouncycastle.openpgp.PGPPublicKeyEncryptedData;
import org.bouncycastle.openpgp.PGPPublicKeyRing;
import org.bouncycastle.openpgp.PGPSecretKey;
import org.bouncycastle.openpgp.PGPSecretKeyRing;
import org.bouncycastle.openpgp.PGPSessionKeyEncryptedData;
import org.bouncycastle.openpgp.PGPSignature;
import org.bouncycastle.openpgp.PGPSignatureList;
import org.bouncycastle.openpgp.operator.PBEDataDecryptorFactory;
import org.bouncycastle.openpgp.operator.PGPContentVerifierBuilderProvider;
import org.bouncycastle.openpgp.operator.PublicKeyDataDecryptorFactory;
import org.bouncycastle.openpgp.operator.SessionKeyDataDecryptorFactory;
import org.bouncycastle.util.io.TeeInputStream;
import org.pgpainless.PGPainless;
import org.pgpainless.algorithm.CompressionAlgorithm;
import org.pgpainless.algorithm.EncryptionPurpose;
import org.pgpainless.algorithm.OpenPgpPacket;
import org.pgpainless.algorithm.StreamEncoding;
import org.pgpainless.algorithm.SymmetricKeyAlgorithm;
import org.pgpainless.decryption_verification.ConsumerOptions;
import org.pgpainless.decryption_verification.DecryptionStream;
import org.pgpainless.decryption_verification.IntegrityProtectedInputStream;
import org.pgpainless.decryption_verification.MessageMetadata;
import org.pgpainless.decryption_verification.MissingKeyPassphraseStrategy;
import org.pgpainless.decryption_verification.OpenPgpInputStream;
import org.pgpainless.decryption_verification.SignatureVerification;
import org.pgpainless.decryption_verification.TeeBCPGInputStream;
import org.pgpainless.decryption_verification.cleartext_signatures.ClearsignedMessageUtil;
import org.pgpainless.decryption_verification.cleartext_signatures.MultiPassStrategy;
import org.pgpainless.decryption_verification.syntax_check.InputSymbol;
import org.pgpainless.decryption_verification.syntax_check.PDA;
import org.pgpainless.decryption_verification.syntax_check.StackSymbol;
import org.pgpainless.exception.MalformedOpenPgpMessageException;
import org.pgpainless.exception.MessageNotIntegrityProtectedException;
import org.pgpainless.exception.MissingDecryptionMethodException;
import org.pgpainless.exception.MissingPassphraseException;
import org.pgpainless.exception.SignatureValidationException;
import org.pgpainless.exception.UnacceptableAlgorithmException;
import org.pgpainless.implementation.ImplementationFactory;
import org.pgpainless.key.SubkeyIdentifier;
import org.pgpainless.key.info.KeyRingInfo;
import org.pgpainless.key.protection.SecretKeyRingProtector;
import org.pgpainless.key.protection.UnlockSecretKey;
import org.pgpainless.key.util.KeyIdUtil;
import org.pgpainless.key.util.KeyRingUtils;
import org.pgpainless.policy.Policy;
import org.pgpainless.signature.SignatureUtils;
import org.pgpainless.signature.consumer.CertificateValidator;
import org.pgpainless.signature.consumer.OnePassSignatureCheck;
import org.pgpainless.signature.consumer.SignatureCheck;
import org.pgpainless.signature.consumer.SignatureValidator;
import org.pgpainless.util.ArmoredInputStreamFactory;
import org.pgpainless.util.Passphrase;
import org.pgpainless.util.SessionKey;
import org.pgpainless.util.Tuple;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class OpenPgpMessageInputStream
extends DecryptionStream {
    private static final Logger LOGGER = LoggerFactory.getLogger(OpenPgpMessageInputStream.class);
    protected final ConsumerOptions options;
    private final Policy policy;
    protected final PDA syntaxVerifier = new PDA();
    protected TeeBCPGInputStream packetInputStream;
    protected InputStream nestedInputStream;
    private boolean closed = false;
    private final Signatures signatures;
    private final MessageMetadata.Layer metadata;

    public static OpenPgpMessageInputStream create(@Nonnull InputStream inputStream, @Nonnull ConsumerOptions options) throws IOException, PGPException {
        return OpenPgpMessageInputStream.create(inputStream, options, PGPainless.getPolicy());
    }

    public static OpenPgpMessageInputStream create(@Nonnull InputStream inputStream, @Nonnull ConsumerOptions options, @Nonnull Policy policy) throws PGPException, IOException {
        return OpenPgpMessageInputStream.create(inputStream, options, new MessageMetadata.Message(), policy);
    }

    protected static OpenPgpMessageInputStream create(@Nonnull InputStream inputStream, @Nonnull ConsumerOptions options, @Nonnull MessageMetadata.Layer metadata, @Nonnull Policy policy) throws IOException, PGPException {
        OpenPgpInputStream openPgpIn = new OpenPgpInputStream(inputStream);
        openPgpIn.reset();
        if (openPgpIn.isNonOpenPgp() || options.isForceNonOpenPgpData()) {
            return new OpenPgpMessageInputStream(Type.non_openpgp, openPgpIn, options, metadata, policy);
        }
        if (openPgpIn.isBinaryOpenPgp()) {
            return new OpenPgpMessageInputStream(Type.standard, openPgpIn, options, metadata, policy);
        }
        if (openPgpIn.isAsciiArmored()) {
            ArmoredInputStream armorIn = ArmoredInputStreamFactory.get(openPgpIn);
            if (armorIn.isClearText()) {
                ((MessageMetadata.Message)metadata).cleartextSigned = true;
                return new OpenPgpMessageInputStream(Type.cleartext_signed, (InputStream)armorIn, options, metadata, policy);
            }
            return new OpenPgpMessageInputStream(Type.standard, (InputStream)armorIn, options, metadata, policy);
        }
        throw new AssertionError((Object)"Cannot deduce type of data.");
    }

    protected OpenPgpMessageInputStream(@Nonnull InputStream inputStream, @Nonnull ConsumerOptions options, @Nonnull MessageMetadata.Layer metadata, @Nonnull Policy policy) throws PGPException, IOException {
        this.policy = policy;
        this.options = options;
        this.metadata = metadata;
        this.signatures = new Signatures(options);
        if (metadata instanceof MessageMetadata.Message) {
            this.signatures.addDetachedSignatures(options.getDetachedSignatures());
        }
        this.packetInputStream = new TeeBCPGInputStream(BCPGInputStream.wrap((InputStream)inputStream), this.signatures);
        this.consumePackets();
    }

    protected OpenPgpMessageInputStream(@Nonnull Type type, @Nonnull InputStream inputStream, @Nonnull ConsumerOptions options, @Nonnull MessageMetadata.Layer metadata, @Nonnull Policy policy) throws PGPException, IOException {
        this.policy = policy;
        this.options = options;
        this.metadata = metadata;
        this.signatures = new Signatures(options);
        if (metadata instanceof MessageMetadata.Message) {
            this.signatures.addDetachedSignatures(options.getDetachedSignatures());
        }
        switch (type) {
            case standard: {
                this.packetInputStream = new TeeBCPGInputStream(BCPGInputStream.wrap((InputStream)inputStream), this.signatures);
                this.consumePackets();
                break;
            }
            case cleartext_signed: {
                MultiPassStrategy multiPassStrategy = options.getMultiPassStrategy();
                PGPSignatureList detachedSignatures = ClearsignedMessageUtil.detachSignaturesFromInbandClearsignedMessage(inputStream, multiPassStrategy.getMessageOutputStream());
                for (PGPSignature signature : detachedSignatures) {
                    this.signatures.addDetachedSignature(signature);
                }
                options.forceNonOpenPgpData();
                this.nestedInputStream = new TeeInputStream(multiPassStrategy.getMessageInputStream(), (OutputStream)this.signatures);
                break;
            }
            case non_openpgp: {
                this.packetInputStream = null;
                this.nestedInputStream = new TeeInputStream(inputStream, (OutputStream)this.signatures);
            }
        }
    }

    /*
     * Enabled aggressive block sorting
     */
    private void consumePackets() throws IOException, PGPException {
        OpenPgpPacket nextPacket;
        if (this.packetInputStream == null) {
            return;
        }
        while ((nextPacket = this.packetInputStream.nextPacketTag()) != null) {
            this.signatures.nextPacket(nextPacket);
            switch (nextPacket) {
                case LIT: {
                    this.processLiteralData();
                    return;
                }
                case COMP: {
                    this.processCompressedData();
                    return;
                }
                case OPS: {
                    this.processOnePassSignature();
                    break;
                }
                case SIG: {
                    this.processSignature();
                    break;
                }
                case PKESK: 
                case SKESK: 
                case SED: 
                case SEIPD: {
                    if (!this.processEncryptedData()) throw new MissingDecryptionMethodException("No working decryption method found.");
                    return;
                }
                case MARKER: {
                    LOGGER.debug("Skipping Marker Packet");
                    this.packetInputStream.readMarker();
                    break;
                }
                case SK: 
                case PK: 
                case SSK: 
                case PSK: 
                case TRUST: 
                case UID: 
                case UATTR: {
                    throw new MalformedOpenPgpMessageException("Illegal Packet in Stream: " + (Object)((Object)nextPacket));
                }
                case MDC: {
                    throw new MalformedOpenPgpMessageException("Unexpected Packet in Stream: " + (Object)((Object)nextPacket));
                }
                case EXP_1: 
                case EXP_2: 
                case EXP_3: 
                case EXP_4: {
                    throw new MalformedOpenPgpMessageException("Unsupported Packet in Stream: " + (Object)((Object)nextPacket));
                }
            }
        }
    }

    private void processLiteralData() throws IOException {
        LOGGER.debug("Literal Data Packet at depth " + this.metadata.depth + " encountered");
        this.syntaxVerifier.next(InputSymbol.LiteralData);
        PGPLiteralData literalData = this.packetInputStream.readLiteralData();
        this.metadata.setChild(new MessageMetadata.LiteralData(literalData.getFileName(), literalData.getModificationTime(), StreamEncoding.requireFromCode(literalData.getFormat())));
        this.nestedInputStream = literalData.getDataStream();
    }

    private void processCompressedData() throws IOException, PGPException {
        this.syntaxVerifier.next(InputSymbol.CompressedData);
        this.signatures.enterNesting();
        PGPCompressedData compressedData = this.packetInputStream.readCompressedData();
        MessageMetadata.CompressedData compressionLayer = new MessageMetadata.CompressedData(CompressionAlgorithm.fromId(compressedData.getAlgorithm()), this.metadata.depth + 1);
        LOGGER.debug("Compressed Data Packet (" + (Object)((Object)compressionLayer.algorithm) + ") at depth " + this.metadata.depth + " encountered");
        InputStream decompressed = compressedData.getDataStream();
        this.nestedInputStream = new OpenPgpMessageInputStream(decompressed, this.options, compressionLayer, this.policy);
    }

    private void processOnePassSignature() throws PGPException, IOException {
        this.syntaxVerifier.next(InputSymbol.OnePassSignature);
        PGPOnePassSignature onePassSignature = this.packetInputStream.readOnePassSignature();
        LOGGER.debug("One-Pass-Signature Packet by key " + KeyIdUtil.formatKeyId(onePassSignature.getKeyID()) + " at depth " + this.metadata.depth + " encountered");
        this.signatures.addOnePassSignature(onePassSignature);
    }

    private void processSignature() throws PGPException, IOException {
        PGPSignature signature;
        boolean isSigForOPS = this.syntaxVerifier.peekStack() == StackSymbol.ops;
        this.syntaxVerifier.next(InputSymbol.Signature);
        try {
            signature = this.packetInputStream.readSignature();
        }
        catch (UnsupportedPacketVersionException e) {
            LOGGER.debug("Unsupported Signature at depth " + this.metadata.depth + " encountered.", (Throwable)e);
            return;
        }
        long keyId = SignatureUtils.determineIssuerKeyId(signature);
        if (isSigForOPS) {
            LOGGER.debug("Signature Packet corresponding to One-Pass-Signature by key " + KeyIdUtil.formatKeyId(keyId) + " at depth " + this.metadata.depth + " encountered");
            this.signatures.leaveNesting();
            this.signatures.addCorrespondingOnePassSignature(signature, this.metadata, this.policy);
        } else {
            LOGGER.debug("Prepended Signature Packet by key " + KeyIdUtil.formatKeyId(keyId) + " at depth " + this.metadata.depth + " encountered");
            this.signatures.addPrependedSignature(signature);
        }
    }

    private boolean processEncryptedData() throws IOException, PGPException {
        PGPPrivateKey privateKey;
        SecretKeyRingProtector protector;
        SubkeyIdentifier decryptionKeyId;
        PGPSecretKey secretKey;
        PGPSecretKeyRing decryptionKeys;
        LOGGER.debug("Symmetrically Encrypted Data Packet at depth " + this.metadata.depth + " encountered");
        this.syntaxVerifier.next(InputSymbol.EncryptedData);
        PGPEncryptedDataList encDataList = this.packetInputStream.readEncryptedDataList();
        if (!encDataList.isIntegrityProtected()) {
            LOGGER.warn("Symmetrically Encrypted Data Packet is not integrity-protected.");
            if (!this.options.isIgnoreMDCErrors()) {
                throw new MessageNotIntegrityProtectedException();
            }
        }
        SortedESKs esks = new SortedESKs(encDataList);
        LOGGER.debug("Symmetrically Encrypted Integrity-Protected Data has " + esks.skesks.size() + " SKESK(s) and " + (esks.pkesks.size() + esks.anonPkesks.size()) + " PKESK(s) from which " + esks.anonPkesks.size() + " PKESK(s) have an anonymous recipient");
        for (SubkeyIdentifier subkeyIdentifier : this.options.getCustomDecryptorFactories().keySet()) {
            LOGGER.debug("Attempt decryption with custom decryptor factory with key " + subkeyIdentifier);
            PublicKeyDataDecryptorFactory decryptorFactory = this.options.getCustomDecryptorFactories().get(subkeyIdentifier);
            for (PGPPublicKeyEncryptedData pGPPublicKeyEncryptedData : esks.pkesks) {
                if (pGPPublicKeyEncryptedData.getKeyID() != subkeyIdentifier.getSubkeyId() || !this.decryptPKESKAndStream(esks, subkeyIdentifier, decryptorFactory, pGPPublicKeyEncryptedData)) continue;
                return true;
            }
        }
        if (this.options.getSessionKey() != null) {
            LOGGER.debug("Attempt decryption with provided session key");
            SessionKey sessionKey = this.options.getSessionKey();
            this.throwIfUnacceptable(sessionKey.getAlgorithm());
            SessionKeyDataDecryptorFactory sessionKeyDataDecryptorFactory = ImplementationFactory.getInstance().getSessionKeyDataDecryptorFactory(sessionKey);
            MessageMetadata.EncryptedData encryptedData = new MessageMetadata.EncryptedData(sessionKey.getAlgorithm(), this.metadata.depth + 1);
            PGPSessionKeyEncryptedData pGPSessionKeyEncryptedData = encDataList.extractSessionKeyEncryptedData();
            try {
                InputStream inputStream = pGPSessionKeyEncryptedData.getDataStream(sessionKeyDataDecryptorFactory);
                encryptedData.sessionKey = sessionKey;
                IntegrityProtectedInputStream integrityProtected = new IntegrityProtectedInputStream(inputStream, (PGPEncryptedData)pGPSessionKeyEncryptedData, this.options);
                this.nestedInputStream = new OpenPgpMessageInputStream(integrityProtected, this.options, encryptedData, this.policy);
                LOGGER.debug("Successfully decrypted data with provided session key");
                return true;
            }
            catch (PGPException pGPException) {
                LOGGER.debug("Decryption using provided session key failed. Mismatched session key and message?", (Throwable)pGPException);
            }
        }
        for (Passphrase passphrase : this.options.getDecryptionPassphrases()) {
            for (PGPPBEEncryptedData pGPPBEEncryptedData : esks.skesks) {
                LOGGER.debug("Attempt decryption with provided passphrase");
                SymmetricKeyAlgorithm symmetricKeyAlgorithm = SymmetricKeyAlgorithm.requireFromId(pGPPBEEncryptedData.getAlgorithm());
                try {
                    this.throwIfUnacceptable(symmetricKeyAlgorithm);
                }
                catch (UnacceptableAlgorithmException e) {
                    LOGGER.debug("Skipping SKESK with unacceptable encapsulation algorithm", (Throwable)((Object)e));
                    continue;
                }
                PBEDataDecryptorFactory decryptorFactory = ImplementationFactory.getInstance().getPBEDataDecryptorFactory(passphrase);
                if (!this.decryptSKESKAndStream(esks, pGPPBEEncryptedData, decryptorFactory)) continue;
                return true;
            }
        }
        ArrayList<Tuple<PGPSecretKey, Object>> postponedDueToMissingPassphrase = new ArrayList<Tuple<PGPSecretKey, Object>>();
        for (Object pkesk : esks.pkesks) {
            long l = pkesk.getKeyID();
            LOGGER.debug("Encountered PKESK for recipient " + KeyIdUtil.formatKeyId(l));
            decryptionKeys = this.getDecryptionKey(l);
            if (decryptionKeys == null) {
                LOGGER.debug("Skipping PKESK because no matching key " + KeyIdUtil.formatKeyId(l) + " was provided");
                continue;
            }
            secretKey = decryptionKeys.getSecretKey(l);
            if (OpenPgpMessageInputStream.hasUnsupportedS2KSpecifier(secretKey, decryptionKeyId = new SubkeyIdentifier((PGPKeyRing)decryptionKeys, secretKey.getKeyID()))) continue;
            LOGGER.debug("Attempt decryption using secret key " + decryptionKeyId);
            protector = this.options.getSecretKeyProtector(decryptionKeys);
            if (!protector.hasPassphraseFor(l)) {
                LOGGER.debug("Missing passphrase for key " + decryptionKeyId + ". Postponing decryption until all other keys were tried");
                postponedDueToMissingPassphrase.add(new Tuple<PGPSecretKey, Object>(secretKey, pkesk));
                continue;
            }
            privateKey = UnlockSecretKey.unlockSecretKey(secretKey, protector);
            if (!this.decryptWithPrivateKey(esks, privateKey, decryptionKeyId, (PGPPublicKeyEncryptedData)pkesk)) continue;
            return true;
        }
        for (Object pkesk : esks.anonPkesks) {
            for (Tuple<PGPSecretKeyRing, PGPSecretKey> tuple : this.findPotentialDecryptionKeys((PGPPublicKeyEncryptedData)pkesk)) {
                decryptionKeys = tuple.getA();
                secretKey = tuple.getB();
                if (OpenPgpMessageInputStream.hasUnsupportedS2KSpecifier(secretKey, decryptionKeyId = new SubkeyIdentifier((PGPKeyRing)decryptionKeys, secretKey.getKeyID()))) continue;
                LOGGER.debug("Attempt decryption of anonymous PKESK with key " + decryptionKeyId);
                protector = this.options.getSecretKeyProtector(tuple.getA());
                if (!protector.hasPassphraseFor(secretKey.getKeyID())) {
                    LOGGER.debug("Missing passphrase for key " + decryptionKeyId + ". Postponing decryption until all other keys were tried.");
                    postponedDueToMissingPassphrase.add(new Tuple<PGPSecretKey, Object>(secretKey, pkesk));
                    continue;
                }
                privateKey = UnlockSecretKey.unlockSecretKey(secretKey, protector);
                if (!this.decryptWithPrivateKey(esks, privateKey, decryptionKeyId, (PGPPublicKeyEncryptedData)pkesk)) continue;
                return true;
            }
        }
        if (this.options.getMissingKeyPassphraseStrategy() == MissingKeyPassphraseStrategy.THROW_EXCEPTION) {
            HashSet<SubkeyIdentifier> hashSet = new HashSet<SubkeyIdentifier>();
            for (Tuple tuple : postponedDueToMissingPassphrase) {
                PGPSecretKey pGPSecretKey = (PGPSecretKey)tuple.getA();
                PGPSecretKeyRing keys = this.getDecryptionKey(pGPSecretKey.getKeyID());
                hashSet.add(new SubkeyIdentifier((PGPKeyRing)keys, pGPSecretKey.getKeyID()));
            }
            if (!hashSet.isEmpty()) {
                throw new MissingPassphraseException(hashSet);
            }
        } else if (this.options.getMissingKeyPassphraseStrategy() == MissingKeyPassphraseStrategy.INTERACTIVE) {
            for (Object pkesk : esks.pkesks) {
                for (Tuple tuple : postponedDueToMissingPassphrase) {
                    long keyId;
                    PGPSecretKeyRing decryptionKey;
                    SubkeyIdentifier decryptionKeyId2;
                    PGPSecretKey secretKey2 = (PGPSecretKey)tuple.getA();
                    if (OpenPgpMessageInputStream.hasUnsupportedS2KSpecifier(secretKey2, decryptionKeyId2 = new SubkeyIdentifier((PGPKeyRing)(decryptionKey = this.getDecryptionKey(keyId = secretKey2.getKeyID())), keyId))) continue;
                    LOGGER.debug("Attempt decryption with key " + decryptionKeyId2 + " while interactively requesting its passphrase");
                    SecretKeyRingProtector protector2 = this.options.getSecretKeyProtector(decryptionKey);
                    PGPPrivateKey privateKey2 = UnlockSecretKey.unlockSecretKey(secretKey2, protector2);
                    if (!this.decryptWithPrivateKey(esks, privateKey2, decryptionKeyId2, (PGPPublicKeyEncryptedData)pkesk)) continue;
                    return true;
                }
            }
        } else {
            throw new IllegalStateException("Invalid PostponedKeysStrategy set in consumer options.");
        }
        LOGGER.debug("Failed to decrypt encrypted data packet");
        return false;
    }

    private boolean decryptWithPrivateKey(SortedESKs esks, PGPPrivateKey privateKey, SubkeyIdentifier decryptionKeyId, PGPPublicKeyEncryptedData pkesk) throws PGPException, IOException {
        PublicKeyDataDecryptorFactory decryptorFactory = ImplementationFactory.getInstance().getPublicKeyDataDecryptorFactory(privateKey);
        return this.decryptPKESKAndStream(esks, decryptionKeyId, decryptorFactory, pkesk);
    }

    private static boolean hasUnsupportedS2KSpecifier(PGPSecretKey secretKey, SubkeyIdentifier decryptionKeyId) {
        int s2kType;
        S2K s2K = secretKey.getS2K();
        if (s2K != null && (s2kType = s2K.getType()) >= 100 && s2kType <= 110) {
            LOGGER.debug("Skipping PKESK because key " + decryptionKeyId + " has unsupported private S2K specifier " + s2kType);
            return true;
        }
        return false;
    }

    private boolean decryptSKESKAndStream(SortedESKs esks, PGPPBEEncryptedData symEsk, PBEDataDecryptorFactory decryptorFactory) throws IOException, UnacceptableAlgorithmException {
        try {
            InputStream decrypted = symEsk.getDataStream(decryptorFactory);
            SessionKey sessionKey = new SessionKey(symEsk.getSessionKey(decryptorFactory));
            this.throwIfUnacceptable(sessionKey.getAlgorithm());
            MessageMetadata.EncryptedData encryptedData = new MessageMetadata.EncryptedData(sessionKey.getAlgorithm(), this.metadata.depth + 1);
            encryptedData.sessionKey = sessionKey;
            encryptedData.recipients = new ArrayList<Long>();
            for (PGPPublicKeyEncryptedData pkesk : esks.pkesks) {
                encryptedData.recipients.add(pkesk.getKeyID());
            }
            LOGGER.debug("Successfully decrypted data with passphrase");
            IntegrityProtectedInputStream integrityProtected = new IntegrityProtectedInputStream(decrypted, (PGPEncryptedData)symEsk, this.options);
            this.nestedInputStream = new OpenPgpMessageInputStream(integrityProtected, this.options, encryptedData, this.policy);
            return true;
        }
        catch (UnacceptableAlgorithmException e) {
            throw e;
        }
        catch (PGPException e) {
            LOGGER.debug("Decryption of encrypted data packet using password failed. Password mismatch?", (Throwable)e);
            return false;
        }
    }

    private boolean decryptPKESKAndStream(SortedESKs esks, SubkeyIdentifier decryptionKeyId, PublicKeyDataDecryptorFactory decryptorFactory, PGPPublicKeyEncryptedData asymEsk) throws IOException, UnacceptableAlgorithmException {
        try {
            InputStream decrypted = asymEsk.getDataStream(decryptorFactory);
            SessionKey sessionKey = new SessionKey(asymEsk.getSessionKey(decryptorFactory));
            this.throwIfUnacceptable(sessionKey.getAlgorithm());
            MessageMetadata.EncryptedData encryptedData = new MessageMetadata.EncryptedData(SymmetricKeyAlgorithm.requireFromId(asymEsk.getSymmetricAlgorithm(decryptorFactory)), this.metadata.depth + 1);
            encryptedData.decryptionKey = decryptionKeyId;
            encryptedData.sessionKey = sessionKey;
            encryptedData.recipients = new ArrayList<Long>();
            for (PGPPublicKeyEncryptedData pkesk : esks.pkesks) {
                encryptedData.recipients.add(pkesk.getKeyID());
            }
            LOGGER.debug("Successfully decrypted data with key " + decryptionKeyId);
            IntegrityProtectedInputStream integrityProtected = new IntegrityProtectedInputStream(decrypted, (PGPEncryptedData)asymEsk, this.options);
            this.nestedInputStream = new OpenPgpMessageInputStream(integrityProtected, this.options, encryptedData, this.policy);
            return true;
        }
        catch (UnacceptableAlgorithmException e) {
            throw e;
        }
        catch (PGPException e) {
            LOGGER.debug("Decryption of encrypted data packet using secret key failed.", (Throwable)e);
            return false;
        }
    }

    private void throwIfUnacceptable(SymmetricKeyAlgorithm algorithm) throws UnacceptableAlgorithmException {
        if (!this.policy.getSymmetricKeyDecryptionAlgorithmPolicy().isAcceptable(algorithm)) {
            throw new UnacceptableAlgorithmException("Symmetric-Key algorithm " + (Object)((Object)algorithm) + " is not acceptable for message decryption.");
        }
    }

    private List<Tuple<PGPSecretKeyRing, PGPSecretKey>> findPotentialDecryptionKeys(PGPPublicKeyEncryptedData pkesk) {
        int algorithm = pkesk.getAlgorithm();
        ArrayList<Tuple<PGPSecretKeyRing, PGPSecretKey>> decryptionKeyCandidates = new ArrayList<Tuple<PGPSecretKeyRing, PGPSecretKey>>();
        for (PGPSecretKeyRing secretKeys : this.options.getDecryptionKeys()) {
            KeyRingInfo info = PGPainless.inspectKeyRing((PGPKeyRing)secretKeys);
            for (PGPPublicKey publicKey : info.getEncryptionSubkeys(EncryptionPurpose.ANY)) {
                if (publicKey.getAlgorithm() != algorithm || !info.isSecretKeyAvailable(publicKey.getKeyID())) continue;
                PGPSecretKey candidate = secretKeys.getSecretKey(publicKey.getKeyID());
                decryptionKeyCandidates.add(new Tuple<PGPSecretKeyRing, PGPSecretKey>(secretKeys, candidate));
            }
        }
        return decryptionKeyCandidates;
    }

    private PGPSecretKeyRing getDecryptionKey(long keyID) {
        for (PGPSecretKeyRing secretKeys : this.options.getDecryptionKeys()) {
            PGPSecretKey decryptionKey = secretKeys.getSecretKey(keyID);
            if (decryptionKey == null) continue;
            KeyRingInfo info = new KeyRingInfo((PGPKeyRing)secretKeys, this.policy, new Date());
            List<PGPPublicKey> encryptionKeys = info.getEncryptionSubkeys(EncryptionPurpose.ANY);
            for (PGPPublicKey key : encryptionKeys) {
                if (key.getKeyID() != keyID) continue;
                return secretKeys;
            }
            LOGGER.debug("Subkey " + Long.toHexString(keyID) + " cannot be used for decryption.");
        }
        return null;
    }

    @Override
    public int read() throws IOException {
        boolean eos;
        int r;
        if (this.nestedInputStream == null) {
            if (this.packetInputStream != null) {
                this.syntaxVerifier.assertValid();
            }
            return -1;
        }
        try {
            r = this.nestedInputStream.read();
        }
        catch (IOException e) {
            r = -1;
        }
        boolean bl = eos = r == -1;
        if (!eos) {
            byte b = (byte)r;
            this.signatures.updateLiteral(b);
        } else {
            this.nestedInputStream.close();
            this.collectMetadata();
            this.nestedInputStream = null;
            if (this.packetInputStream != null) {
                try {
                    this.consumePackets();
                }
                catch (PGPException e) {
                    throw new RuntimeException(e);
                }
            }
            this.signatures.finish(this.metadata, this.policy);
        }
        return r;
    }

    @Override
    public int read(@Nonnull byte[] b, int off, int len) throws IOException {
        if (this.nestedInputStream == null) {
            if (this.packetInputStream != null) {
                this.syntaxVerifier.assertValid();
            }
            return -1;
        }
        int r = this.nestedInputStream.read(b, off, len);
        if (r != -1) {
            this.signatures.updateLiteral(b, off, r);
        } else {
            this.nestedInputStream.close();
            this.collectMetadata();
            this.nestedInputStream = null;
            if (this.packetInputStream != null) {
                try {
                    this.consumePackets();
                }
                catch (PGPException e) {
                    throw new RuntimeException(e);
                }
            }
            this.signatures.finish(this.metadata, this.policy);
        }
        return r;
    }

    @Override
    public void close() throws IOException {
        super.close();
        if (this.closed) {
            if (this.packetInputStream != null) {
                this.syntaxVerifier.assertValid();
            }
            return;
        }
        if (this.nestedInputStream != null) {
            this.nestedInputStream.close();
            this.collectMetadata();
            this.nestedInputStream = null;
        }
        try {
            this.consumePackets();
        }
        catch (PGPException e) {
            throw new RuntimeException(e);
        }
        if (this.packetInputStream != null) {
            this.syntaxVerifier.next(InputSymbol.EndOfSequence);
            this.syntaxVerifier.assertValid();
            this.packetInputStream.close();
        }
        this.closed = true;
    }

    private void collectMetadata() {
        if (this.nestedInputStream instanceof OpenPgpMessageInputStream) {
            OpenPgpMessageInputStream child = (OpenPgpMessageInputStream)this.nestedInputStream;
            this.metadata.setChild((MessageMetadata.Nested)((Object)child.metadata));
        }
    }

    @Override
    public MessageMetadata getMetadata() {
        if (!this.closed) {
            throw new IllegalStateException("Stream must be closed before access to metadata can be granted.");
        }
        return new MessageMetadata((MessageMetadata.Message)this.metadata);
    }

    private static final class Signatures
    extends OutputStream {
        final ConsumerOptions options;
        final List<SignatureCheck> detachedSignatures;
        final List<SignatureCheck> prependedSignatures;
        final List<OnePassSignatureCheck> onePassSignatures;
        final Stack<List<OnePassSignatureCheck>> opsUpdateStack;
        List<OnePassSignatureCheck> literalOPS = new ArrayList<OnePassSignatureCheck>();
        final List<PGPSignature> correspondingSignatures;
        final List<SignatureVerification.Failure> prependedSignaturesWithMissingCert = new ArrayList<SignatureVerification.Failure>();
        final List<SignatureVerification.Failure> inbandSignaturesWithMissingCert = new ArrayList<SignatureVerification.Failure>();
        final List<SignatureVerification.Failure> detachedSignaturesWithMissingCert = new ArrayList<SignatureVerification.Failure>();
        boolean isLiteral = true;

        private Signatures(ConsumerOptions options) {
            this.options = options;
            this.detachedSignatures = new ArrayList<SignatureCheck>();
            this.prependedSignatures = new ArrayList<SignatureCheck>();
            this.onePassSignatures = new ArrayList<OnePassSignatureCheck>();
            this.opsUpdateStack = new Stack();
            this.correspondingSignatures = new ArrayList<PGPSignature>();
        }

        void addDetachedSignatures(Collection<PGPSignature> signatures) {
            for (PGPSignature signature : signatures) {
                this.addDetachedSignature(signature);
            }
        }

        void addDetachedSignature(PGPSignature signature) {
            SignatureCheck check = this.initializeSignature(signature);
            long keyId = SignatureUtils.determineIssuerKeyId(signature);
            if (check != null) {
                this.detachedSignatures.add(check);
            } else {
                LOGGER.debug("No suitable certificate for verification of signature by key " + KeyIdUtil.formatKeyId(keyId) + " found.");
                this.detachedSignaturesWithMissingCert.add(new SignatureVerification.Failure(new SignatureVerification(signature, null), new SignatureValidationException("Missing verification key")));
            }
        }

        void addPrependedSignature(PGPSignature signature) {
            SignatureCheck check = this.initializeSignature(signature);
            long keyId = SignatureUtils.determineIssuerKeyId(signature);
            if (check != null) {
                this.prependedSignatures.add(check);
            } else {
                LOGGER.debug("No suitable certificate for verification of signature by key " + KeyIdUtil.formatKeyId(keyId) + " found.");
                this.prependedSignaturesWithMissingCert.add(new SignatureVerification.Failure(new SignatureVerification(signature, null), new SignatureValidationException("Missing verification key")));
            }
        }

        SignatureCheck initializeSignature(PGPSignature signature) {
            long keyId = SignatureUtils.determineIssuerKeyId(signature);
            PGPPublicKeyRing certificate = this.findCertificate(keyId);
            if (certificate == null) {
                return null;
            }
            SubkeyIdentifier verifierKey = new SubkeyIdentifier((PGPKeyRing)certificate, keyId);
            Signatures.initialize(signature, certificate, keyId);
            return new SignatureCheck(signature, (PGPKeyRing)certificate, verifierKey);
        }

        void addOnePassSignature(PGPOnePassSignature signature) {
            PGPPublicKeyRing certificate = this.findCertificate(signature.getKeyID());
            if (certificate != null) {
                OnePassSignatureCheck ops = new OnePassSignatureCheck(signature, certificate);
                Signatures.initialize(signature, certificate);
                this.onePassSignatures.add(ops);
                this.literalOPS.add(ops);
            }
            if (signature.isContaining()) {
                this.enterNesting();
            }
        }

        void addCorrespondingOnePassSignature(PGPSignature signature, MessageMetadata.Layer layer, Policy policy) {
            boolean found = false;
            long keyId = SignatureUtils.determineIssuerKeyId(signature);
            for (int i = this.onePassSignatures.size() - 1; i >= 0; --i) {
                OnePassSignatureCheck onePassSignature = this.onePassSignatures.get(i);
                if (onePassSignature.getOnePassSignature().getKeyID() != keyId) continue;
                found = true;
                if (onePassSignature.getSignature() != null) continue;
                onePassSignature.setSignature(signature);
                SignatureVerification verification = new SignatureVerification(signature, new SubkeyIdentifier((PGPKeyRing)onePassSignature.getVerificationKeys(), onePassSignature.getOnePassSignature().getKeyID()));
                try {
                    SignatureValidator.signatureWasCreatedInBounds(this.options.getVerifyNotBefore(), this.options.getVerifyNotAfter()).verify(signature);
                    CertificateValidator.validateCertificateAndVerifyOnePassSignature(onePassSignature, policy);
                    LOGGER.debug("Acceptable signature by key " + verification.getSigningKey());
                    layer.addVerifiedOnePassSignature(verification);
                }
                catch (SignatureValidationException e) {
                    LOGGER.debug("Rejected signature by key " + verification.getSigningKey(), (Throwable)((Object)e));
                    layer.addRejectedOnePassSignature(new SignatureVerification.Failure(verification, e));
                }
                break;
            }
            if (!found) {
                LOGGER.debug("No suitable certificate for verification of signature by key " + KeyIdUtil.formatKeyId(keyId) + " found.");
                this.inbandSignaturesWithMissingCert.add(new SignatureVerification.Failure(new SignatureVerification(signature, null), new SignatureValidationException("Missing verification key")));
            }
        }

        void enterNesting() {
            this.opsUpdateStack.push(this.literalOPS);
            this.literalOPS = new ArrayList<OnePassSignatureCheck>();
        }

        void leaveNesting() {
            if (this.opsUpdateStack.isEmpty()) {
                return;
            }
            this.opsUpdateStack.pop();
        }

        private static void initialize(@Nonnull PGPSignature signature, @Nonnull PGPPublicKeyRing certificate, long keyId) {
            PGPContentVerifierBuilderProvider verifierProvider = ImplementationFactory.getInstance().getPGPContentVerifierBuilderProvider();
            try {
                signature.init(verifierProvider, certificate.getPublicKey(keyId));
            }
            catch (PGPException e) {
                throw new RuntimeException(e);
            }
        }

        private static void initialize(@Nonnull PGPOnePassSignature ops, @Nonnull PGPPublicKeyRing certificate) {
            PGPContentVerifierBuilderProvider verifierProvider = ImplementationFactory.getInstance().getPGPContentVerifierBuilderProvider();
            try {
                ops.init(verifierProvider, certificate.getPublicKey(ops.getKeyID()));
            }
            catch (PGPException e) {
                throw new RuntimeException(e);
            }
        }

        private PGPPublicKeyRing findCertificate(long keyId) {
            PGPPublicKeyRing cert = this.options.getCertificateSource().getCertificate(keyId);
            if (cert != null) {
                return cert;
            }
            if (this.options.getMissingCertificateCallback() != null) {
                return this.options.getMissingCertificateCallback().onMissingPublicKeyEncountered(keyId);
            }
            return null;
        }

        public void updateLiteral(byte b) {
            for (OnePassSignatureCheck ops : this.literalOPS) {
                ops.getOnePassSignature().update(b);
            }
            for (SignatureCheck detached : this.detachedSignatures) {
                detached.getSignature().update(b);
            }
            for (SignatureCheck prepended : this.prependedSignatures) {
                prepended.getSignature().update(b);
            }
        }

        public void updateLiteral(byte[] b, int off, int len) {
            for (OnePassSignatureCheck ops : this.literalOPS) {
                ops.getOnePassSignature().update(b, off, len);
            }
            for (SignatureCheck detached : this.detachedSignatures) {
                detached.getSignature().update(b, off, len);
            }
            for (SignatureCheck prepended : this.prependedSignatures) {
                prepended.getSignature().update(b, off, len);
            }
        }

        public void updatePacket(byte b) {
            for (int i = this.opsUpdateStack.size() - 1; i >= 0; --i) {
                List nestedOPSs = (List)this.opsUpdateStack.get(i);
                for (OnePassSignatureCheck ops : nestedOPSs) {
                    ops.getOnePassSignature().update(b);
                }
            }
        }

        public void updatePacket(byte[] buf, int off, int len) {
            for (int i = this.opsUpdateStack.size() - 1; i >= 0; --i) {
                List nestedOPSs = (List)this.opsUpdateStack.get(i);
                for (OnePassSignatureCheck ops : nestedOPSs) {
                    ops.getOnePassSignature().update(buf, off, len);
                }
            }
        }

        public void finish(MessageMetadata.Layer layer, Policy policy) {
            SignatureVerification verification;
            for (SignatureCheck detached : this.detachedSignatures) {
                verification = new SignatureVerification(detached.getSignature(), detached.getSigningKeyIdentifier());
                try {
                    SignatureValidator.signatureWasCreatedInBounds(this.options.getVerifyNotBefore(), this.options.getVerifyNotAfter()).verify(detached.getSignature());
                    CertificateValidator.validateCertificateAndVerifyInitializedSignature(detached.getSignature(), KeyRingUtils.publicKeys(detached.getSigningKeyRing()), policy);
                    LOGGER.debug("Acceptable signature by key " + verification.getSigningKey());
                    layer.addVerifiedDetachedSignature(verification);
                }
                catch (SignatureValidationException e) {
                    LOGGER.debug("Rejected signature by key " + verification.getSigningKey(), (Throwable)((Object)e));
                    layer.addRejectedDetachedSignature(new SignatureVerification.Failure(verification, e));
                }
            }
            for (SignatureCheck prepended : this.prependedSignatures) {
                verification = new SignatureVerification(prepended.getSignature(), prepended.getSigningKeyIdentifier());
                try {
                    SignatureValidator.signatureWasCreatedInBounds(this.options.getVerifyNotBefore(), this.options.getVerifyNotAfter()).verify(prepended.getSignature());
                    CertificateValidator.validateCertificateAndVerifyInitializedSignature(prepended.getSignature(), KeyRingUtils.publicKeys(prepended.getSigningKeyRing()), policy);
                    LOGGER.debug("Acceptable signature by key " + verification.getSigningKey());
                    layer.addVerifiedPrependedSignature(verification);
                }
                catch (SignatureValidationException e) {
                    LOGGER.debug("Rejected signature by key " + verification.getSigningKey(), (Throwable)((Object)e));
                    layer.addRejectedPrependedSignature(new SignatureVerification.Failure(verification, e));
                }
            }
            for (SignatureVerification.Failure rejected : this.inbandSignaturesWithMissingCert) {
                layer.addRejectedOnePassSignature(rejected);
            }
            for (SignatureVerification.Failure rejected : this.prependedSignaturesWithMissingCert) {
                layer.addRejectedPrependedSignature(rejected);
            }
            for (SignatureVerification.Failure rejected : this.detachedSignaturesWithMissingCert) {
                layer.addRejectedDetachedSignature(rejected);
            }
        }

        @Override
        public void write(int b) {
            this.updatePacket((byte)b);
        }

        @Override
        public void write(@Nonnull byte[] b, int off, int len) {
            this.updatePacket(b, off, len);
        }

        public void nextPacket(OpenPgpPacket nextPacket) {
            if (nextPacket == OpenPgpPacket.LIT) {
                this.isLiteral = true;
                if (this.literalOPS.isEmpty() && !this.opsUpdateStack.isEmpty()) {
                    this.literalOPS = this.opsUpdateStack.pop();
                }
            } else {
                this.isLiteral = false;
            }
        }
    }

    private static class SortedESKs {
        private final List<PGPPBEEncryptedData> skesks = new ArrayList<PGPPBEEncryptedData>();
        private final List<PGPPublicKeyEncryptedData> pkesks = new ArrayList<PGPPublicKeyEncryptedData>();
        private final List<PGPPublicKeyEncryptedData> anonPkesks = new ArrayList<PGPPublicKeyEncryptedData>();

        SortedESKs(PGPEncryptedDataList esks) {
            for (PGPEncryptedData esk : esks) {
                if (esk instanceof PGPPBEEncryptedData) {
                    this.skesks.add((PGPPBEEncryptedData)esk);
                    continue;
                }
                if (esk instanceof PGPPublicKeyEncryptedData) {
                    PGPPublicKeyEncryptedData pkesk = (PGPPublicKeyEncryptedData)esk;
                    if (pkesk.getKeyID() != 0L) {
                        this.pkesks.add(pkesk);
                        continue;
                    }
                    this.anonPkesks.add(pkesk);
                    continue;
                }
                throw new IllegalArgumentException("Unknown ESK class type.");
            }
        }

        public List<PGPEncryptedData> all() {
            ArrayList<PGPEncryptedData> esks = new ArrayList<PGPEncryptedData>();
            esks.addAll(this.skesks);
            esks.addAll(this.pkesks);
            esks.addAll(this.anonPkesks);
            return esks;
        }
    }

    static enum Type {
        standard,
        cleartext_signed,
        non_openpgp;

    }
}

