/*
 * Decompiled with CFR 0.152.
 */
package org.pgpainless.signature;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.util.Date;
import org.bouncycastle.openpgp.PGPException;
import org.bouncycastle.openpgp.PGPPublicKeyRing;
import org.bouncycastle.openpgp.PGPSignature;
import org.bouncycastle.util.io.Streams;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.TestTemplate;
import org.junit.jupiter.api.extension.ExtendWith;
import org.pgpainless.PGPainless;
import org.pgpainless.decryption_verification.ConsumerOptions;
import org.pgpainless.decryption_verification.DecryptionStream;
import org.pgpainless.decryption_verification.MessageMetadata;
import org.pgpainless.exception.SignatureValidationException;
import org.pgpainless.policy.Policy;
import org.pgpainless.signature.SignatureUtils;
import org.pgpainless.signature.consumer.CertificateValidator;
import org.pgpainless.util.TestAllImplementations;

public class CertificateValidatorTest {
    @TestTemplate
    @ExtendWith(value={TestAllImplementations.class})
    public void testPrimaryKeySignsAndIsHardRevokedUnknown() throws IOException, PGPException {
        String key = "-----BEGIN PGP ARMORED FILE-----\nComment: ASCII Armor added by openpgp-interoperability-test-suite\n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\nY87zSryajDMFXQS0exdvhN4AXDlPlB3Rrkj7CQ==\n=MhJL\n-----END PGP ARMORED FILE-----\n";
        String sigPredatesPrimaryKey = "-----BEGIN PGP ARMORED FILE-----\nComment: ASCII Armor added by openpgp-interoperability-test-suite\n\nwsBzBAABCgAGBYJYaEaAACEJEGhPrWLcA4+7FiEE8tFQpP6Ykl1R6RU5aE+tYtwD\nj7ttqgf9Gp4T5Q19cNL9Eyz1nlw11HDHT1wxfGHU5li76y7oo4Jqim15sEPDJWmc\nIpYVrczpCI95aCuaE6yfzpjZlXMzftwex3DjM98vyZH4W9teKcOnpAVjn3dLoQJA\ni4fiq3VaLgl+1OYOwu3DmwGKJZubHM3oPia9pbuyvL5Scvx+QCG0AVnssnt2QswG\nuU6J35QgBdjG2iC043sUoyxTSk929iamdQnOGchjcaATb4E4+HvtkRy4IirKxiKK\nc535BHJRLgWQzUcDDZ5kHf3SPLTNsigFFEwUyf5voFcn/DSMWSzPaVecaafTVJW2\nu8G1R5mjuxDRup8p//X1BSk1FpSmvw==\n=3/dv\n-----END PGP ARMORED FILE-----\n";
        String sigSubkeyNotBound = "-----BEGIN PGP ARMORED FILE-----\nComment: ASCII Armor added by openpgp-interoperability-test-suite\n\nwsBzBAABCgAGBYJa564AACEJEGhPrWLcA4+7FiEE8tFQpP6Ykl1R6RU5aE+tYtwD\nj7ufRgf/QOsaJZgQaQ5daQrfBItOEcW+5acgY1TCwMVmc/nzBqC32TOvMaM3dypf\nwJbqzxHQIes+ivKDF872VWlMA2BErifpdsogbS0pdik/qU+AjMhr188xKpZKG/IY\n6BtuUPeSpsimx3UeEN3kt79fMtewBo0EXo3ujCyPpIF/9Vpd7L9jlJSvRBuM0/aR\ngbRsclEw4JZ98B3t7F3rLmx+F57Zre0ctzT4tHE6IaCYpEClr6Tepj/UxExYOt2l\nhKgVN8Wsuug7XYdOFmxqrV967m3CTnF8AspmxwgKK6NXjVLplfqij7Rp2URPWrWn\nPp3CQRGUWJdMeMY9P1MpFq6XiXtftw==\n=Ld1q\n-----END PGP ARMORED FILE-----\n";
        String sigPrimaryKeyRevoked = "-----BEGIN PGP ARMORED FILE-----\nComment: ASCII Armor added by openpgp-interoperability-test-suite\n\nwsBzBAABCgAGBYJdP4iAACEJEGhPrWLcA4+7FiEE8tFQpP6Ykl1R6RU5aE+tYtwD\nj7sYXQf8CZw6Kx4oyI8ZJ2c9RjVZmUFEirAoXH7oYA+Ye+wSAY9OtqE/x2SOYaC6\nQHiB93/wkvpqCVkLy2lenzpD7WXLbuFZ+/5jXp1o+sVHXfLSWo6pfIhOjj9FSr8x\nqqlqUfKwkbA6WYgju+qKC35SYdSptix7unaFkO41UdsM8wGQh880HSRMBMFPzg07\n3hMNYXoEJjFlIkxJSMu2WL7N0Q/4xE2iJftsQjUYAtJ/C/YK2I6dhW+CZremnv5R\n/8W+oH5Q63lYU8YL4wYnJQvkHjKs/kjLpoPmqL8kdHjndSpU+KOYr5w61XuEp2hp\nr8trtljVaVIQX2rYawSlqKkWXt0yag==\n=xVd8\n-----END PGP ARMORED FILE-----\n";
        String sigPrimaryKeyRevalidated = "-----BEGIN PGP ARMORED FILE-----\nComment: ASCII Armor added by openpgp-interoperability-test-suite\n\nwsBzBAABCgAGBYJe/cFVACEJEGhPrWLcA4+7FiEE8tFQpP6Ykl1R6RU5aE+tYtwD\nj7vmhQf/UB456IXc8ub8HTExab1d5KqOGSUWpwIznTu8Wk8YuzWKEE8ZeZvPmv8K\niJfBoOx59YrlOfpLAKcTR9Ql+IFbWsIkqPxX7U1SGldhfQm7iaK5Dn6+mmQFOz/s\nZCIavWJ7opsp11JmQAt4FFojv789YswaS7VI1zjDj7EeRiATtzna/GqCYgeCM0cc\nsIe/1j1H2oh9YvYIpPMSGDJPo7T1Ji4Ie3iEQEYNYPuw1Hb7gWYncHXZGJq1nDf/\nWAoI9gSFagpsPW0k9cfEAOVNLNYSyi0CSnQWSjq8THbHKiLPFwsP3hvT2oHycWbK\nu5SfXaTsbMeVQJNdjCNsHq2bOXPGLw==\n=2BW4\n-----END PGP ARMORED FILE-----\n";
        PGPPublicKeyRing publicKeys = PGPainless.readKeyRing().publicKeyRing(key);
        PGPSignature predatesPrimaryKey = (PGPSignature)SignatureUtils.readSignatures((String)sigPredatesPrimaryKey).get(0);
        PGPSignature unboundSubkey = (PGPSignature)SignatureUtils.readSignatures((String)sigSubkeyNotBound).get(0);
        PGPSignature primaryKeyRevoked = (PGPSignature)SignatureUtils.readSignatures((String)sigPrimaryKeyRevoked).get(0);
        PGPSignature primaryKeyRevalidated = (PGPSignature)SignatureUtils.readSignatures((String)sigPrimaryKeyRevalidated).get(0);
        Policy policy = PGPainless.getPolicy();
        Date validationDate = new Date();
        String data = "Hello, World";
        Assertions.assertThrows(SignatureValidationException.class, () -> CertificateValidator.validateCertificateAndVerifyUninitializedSignature((PGPSignature)predatesPrimaryKey, (InputStream)CertificateValidatorTest.getSignedData(data), (PGPPublicKeyRing)publicKeys, (Policy)policy, (Date)validationDate), (String)"Signature predates primary key");
        Assertions.assertThrows(SignatureValidationException.class, () -> CertificateValidator.validateCertificateAndVerifyUninitializedSignature((PGPSignature)unboundSubkey, (InputStream)CertificateValidatorTest.getSignedData(data), (PGPPublicKeyRing)publicKeys, (Policy)policy, (Date)validationDate), (String)"Primary key hard revoked");
        Assertions.assertThrows(SignatureValidationException.class, () -> CertificateValidator.validateCertificateAndVerifyUninitializedSignature((PGPSignature)primaryKeyRevoked, (InputStream)CertificateValidatorTest.getSignedData(data), (PGPPublicKeyRing)publicKeys, (Policy)policy, (Date)validationDate), (String)"Primary key hard revoked");
        Assertions.assertThrows(SignatureValidationException.class, () -> CertificateValidator.validateCertificateAndVerifyUninitializedSignature((PGPSignature)primaryKeyRevalidated, (InputStream)CertificateValidatorTest.getSignedData(data), (PGPPublicKeyRing)publicKeys, (Policy)policy, (Date)validationDate), (String)"Primary key hard revoked");
    }

    @TestTemplate
    @ExtendWith(value={TestAllImplementations.class})
    public void testSubkeySignsPrimaryKeyIsHardRevokedUnknown() throws IOException, PGPException {
        String key = "-----BEGIN PGP ARMORED FILE-----\nComment: ASCII Armor added by openpgp-interoperability-test-suite\n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\nY87zSryajDMFXQS0exdvhN4AXDlPlB3Rrkj7CQ==\n=MhJL\n-----END PGP ARMORED FILE-----\n";
        String sigPredatesPrimaryKey = "-----BEGIN PGP ARMORED FILE-----\nComment: ASCII Armor added by openpgp-interoperability-test-suite\n\nwsBzBAABCgAGBYJYaEaAACEJEEpyNKOhITplFiEEUXksDkji/alOk7kRSnI0o6Eh\nOmVtqgf/YSG+b8lY01/4oCrHoNMECDDbpI5+8WkeT0CcdlEd5TGj3AHesT6H6XmL\nZxaHHOwtkuDh0bIAiYGl36e4ult5XZQhFIwUXGde6myLE+fpCGsBJwNu+TDIrbg3\nPGqnVZNlcU+2sP5JhJfAn8VtLENuHkbIC3+kH8xBIrkPTc0rbNBgyzX5eFO20U0D\nbHCCjfjVDpZ8l7N2NlsRYvU0kTzN5GvwbS1HnMOovF9ZKkEpzxxw6IRJIapaE2L9\nadMKIRAqrIIjfj6Z9nETd1nZE79t1zSw1trfArPaJQr46krgh1ocLQoD/c+PhB9l\nsRxQBnWERgQaDJByq0kwKSnwWAsyxw==\n=SDmD\n-----END PGP ARMORED FILE-----\n";
        String sigSigningKeyUnbound = "-----BEGIN PGP ARMORED FILE-----\nComment: ASCII Armor added by openpgp-interoperability-test-suite\n\nwsBzBAABCgAGBYJa564AACEJEEpyNKOhITplFiEEUXksDkji/alOk7kRSnI0o6Eh\nOmWfRgf9ECjIviU1pN0GiMZGci3Vce2b42LIqH0JZApOeRDpcyXgxi3/CmdaewpT\nw9l18gsXhioRg4xUzMFrYSgyYZ9VajFggMjbeX5dSV3rsOSJSiEEyDbeGi0TcA/Y\nGUifX4EfKx5X5nI/wevnYjmruDp9SqaPLHIZK1soOoPzueZ8wKyJ9A4vVG4bvxVX\nFnwBf6mRE/0Z8IoHlRJdq0fSzW4RgX8KAtK8SfyGOnk7LDaevVuL6iE5v0Gsu0oh\ncHlI6Llm97EVxP93KZ1J7TQIi/a6PUJb5XCIw0K/iyuNuAzETgm8LVJyn6UwL4ym\nKcNieOK8Qcoivq0kCYuv/0Tbk13jVQ==\n=5hOz\n-----END PGP ARMORED FILE-----";
        String sigSubkeyRevoked = "-----BEGIN PGP ARMORED FILE-----\nComment: ASCII Armor added by openpgp-interoperability-test-suite\n\nwsBzBAABCgAGBYJdP4iAACEJEEpyNKOhITplFiEEUXksDkji/alOk7kRSnI0o6Eh\nOmUYXQf/dGNZay40bZEpcnxYl+Kq+gRQESeDhg/xOfGfSCLQncMH+UYPaUKANC2g\nCfMNN1wd8ZWrvgyTVo3TVfK1P1RYa9nrvKoKN3bjsFcY6V7VciPW58xVNsuxsEEC\nGEH96TQy+FsP680tRnzQ3Dbw/JT6o6Xi+HLf4JVFceapBgyth61E5gN5w3azxVFr\nGfwIfHvepOjCIq9tRZsRFEBp3XVZ/AF+zQMG5nfIVSm1kVtZjb7KXc3Bj48DVrmb\nXLxPJz7PLY0cgOsXXxROIdtFT+mbVQg2j247hxnhItwtLeQrafb5T8ibeihRlkhK\n1tfKv31EP8tAVqgTjw+qD32bH9h77w==\n=MOaJ\n-----END PGP ARMORED FILE-----\n";
        String sigSubkeyRevalidated = "-----BEGIN PGP ARMORED FILE-----\nComment: ASCII Armor added by openpgp-interoperability-test-suite\n\nwsBzBAABCgAGBYJe/cFVACEJEEpyNKOhITplFiEEUXksDkji/alOk7kRSnI0o6Eh\nOmXmhQf+LjM2vmdKYnxfVy40Vcdoy2Jov/ZD2RSMFff+fIoCXmWfEetila7v5xHj\nZXq6aevTEN1vgW3T6Q1OjFhnGpMl9wvya9mszfn5BBKukFtLkHeay0PtYuUcrfJC\nUIQCx9PFZzgRJFyGsCqrXBc1VIe2DV3d8dq74unTeCEmWdvAZKdjoUYzRohMtcZ+\n0QctCCJE1kRFJuH/TIdxxwKPtBZfOolSlpS0Z5xxa2sILqUvQ2Dq3hBctUM4g6hB\nY8uafI8qIMwWl4DQDzPpQ917d6J+GCdN0Aib6ZOsvmgR5wrBOFiDpRJ/W9W6+rgs\nI5V/t2y6h6gaHbanggc0cMOaMTtEKQ==\n=lkHs\n-----END PGP ARMORED FILE-----\n";
        PGPPublicKeyRing publicKeys = PGPainless.readKeyRing().publicKeyRing(key);
        PGPSignature predatesPrimaryKey = (PGPSignature)SignatureUtils.readSignatures((String)sigPredatesPrimaryKey).get(0);
        PGPSignature unboundSubkey = (PGPSignature)SignatureUtils.readSignatures((String)sigSigningKeyUnbound).get(0);
        PGPSignature revokedSubkey = (PGPSignature)SignatureUtils.readSignatures((String)sigSubkeyRevoked).get(0);
        PGPSignature revalidatedSubkey = (PGPSignature)SignatureUtils.readSignatures((String)sigSubkeyRevalidated).get(0);
        Policy policy = PGPainless.getPolicy();
        Date validationDate = new Date();
        String data = "Hello, World";
        Assertions.assertThrows(SignatureValidationException.class, () -> CertificateValidator.validateCertificateAndVerifyUninitializedSignature((PGPSignature)predatesPrimaryKey, (InputStream)CertificateValidatorTest.getSignedData(data), (PGPPublicKeyRing)publicKeys, (Policy)policy, (Date)validationDate), (String)"Signature predates primary key");
        Assertions.assertThrows(SignatureValidationException.class, () -> CertificateValidator.validateCertificateAndVerifyUninitializedSignature((PGPSignature)unboundSubkey, (InputStream)CertificateValidatorTest.getSignedData(data), (PGPPublicKeyRing)publicKeys, (Policy)policy, (Date)validationDate), (String)"Signing key unbound + hard revocation");
        Assertions.assertThrows(SignatureValidationException.class, () -> CertificateValidator.validateCertificateAndVerifyUninitializedSignature((PGPSignature)revokedSubkey, (InputStream)CertificateValidatorTest.getSignedData(data), (PGPPublicKeyRing)publicKeys, (Policy)policy, (Date)validationDate), (String)"Primary key is hard revoked");
        Assertions.assertThrows(SignatureValidationException.class, () -> CertificateValidator.validateCertificateAndVerifyUninitializedSignature((PGPSignature)revalidatedSubkey, (InputStream)CertificateValidatorTest.getSignedData(data), (PGPPublicKeyRing)publicKeys, (Policy)policy, (Date)validationDate), (String)"Primary key is hard revoked");
    }

    @TestTemplate
    @ExtendWith(value={TestAllImplementations.class})
    public void testSubkeySignsAndIsHardRevokedUnknown() throws IOException, PGPException {
        String keyWithHardRev = "-----BEGIN PGP ARMORED FILE-----\nComment: ASCII Armor added by openpgp-interoperability-test-suite\n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\nY87zSryajDMFXQS0exdvhN4AXDlPlB3Rrkj7CQ==\n=yTKS\n-----END PGP ARMORED FILE-----\n";
        PGPPublicKeyRing publicKeys = PGPainless.readKeyRing().publicKeyRing(keyWithHardRev);
        String sigPredatesPrimaryKey = "-----BEGIN PGP ARMORED FILE-----\nComment: ASCII Armor added by openpgp-interoperability-test-suite\n\nwsBzBAABCgAGBYJYaEaAACEJEEpyNKOhITplFiEEUXksDkji/alOk7kRSnI0o6Eh\nOmVtqgf/YSG+b8lY01/4oCrHoNMECDDbpI5+8WkeT0CcdlEd5TGj3AHesT6H6XmL\nZxaHHOwtkuDh0bIAiYGl36e4ult5XZQhFIwUXGde6myLE+fpCGsBJwNu+TDIrbg3\nPGqnVZNlcU+2sP5JhJfAn8VtLENuHkbIC3+kH8xBIrkPTc0rbNBgyzX5eFO20U0D\nbHCCjfjVDpZ8l7N2NlsRYvU0kTzN5GvwbS1HnMOovF9ZKkEpzxxw6IRJIapaE2L9\nadMKIRAqrIIjfj6Z9nETd1nZE79t1zSw1trfArPaJQr46krgh1ocLQoD/c+PhB9l\nsRxQBnWERgQaDJByq0kwKSnwWAsyxw==\n=SDmD\n-----END PGP ARMORED FILE-----\n";
        String sigUnboundBeforeHardRevocation = "-----BEGIN PGP ARMORED FILE-----\nComment: ASCII Armor added by openpgp-interoperability-test-suite\n\nwsBzBAABCgAGBYJa564AACEJEEpyNKOhITplFiEEUXksDkji/alOk7kRSnI0o6Eh\nOmWfRgf9ECjIviU1pN0GiMZGci3Vce2b42LIqH0JZApOeRDpcyXgxi3/CmdaewpT\nw9l18gsXhioRg4xUzMFrYSgyYZ9VajFggMjbeX5dSV3rsOSJSiEEyDbeGi0TcA/Y\nGUifX4EfKx5X5nI/wevnYjmruDp9SqaPLHIZK1soOoPzueZ8wKyJ9A4vVG4bvxVX\nFnwBf6mRE/0Z8IoHlRJdq0fSzW4RgX8KAtK8SfyGOnk7LDaevVuL6iE5v0Gsu0oh\ncHlI6Llm97EVxP93KZ1J7TQIi/a6PUJb5XCIw0K/iyuNuAzETgm8LVJyn6UwL4ym\nKcNieOK8Qcoivq0kCYuv/0Tbk13jVQ==\n=5hOz\n-----END PGP ARMORED FILE-----\n";
        String sigAfterHardRevocation = "-----BEGIN PGP ARMORED FILE-----\nComment: ASCII Armor added by openpgp-interoperability-test-suite\n\nwsBzBAABCgAGBYJdP4iAACEJEEpyNKOhITplFiEEUXksDkji/alOk7kRSnI0o6Eh\nOmUYXQf/dGNZay40bZEpcnxYl+Kq+gRQESeDhg/xOfGfSCLQncMH+UYPaUKANC2g\nCfMNN1wd8ZWrvgyTVo3TVfK1P1RYa9nrvKoKN3bjsFcY6V7VciPW58xVNsuxsEEC\nGEH96TQy+FsP680tRnzQ3Dbw/JT6o6Xi+HLf4JVFceapBgyth61E5gN5w3azxVFr\nGfwIfHvepOjCIq9tRZsRFEBp3XVZ/AF+zQMG5nfIVSm1kVtZjb7KXc3Bj48DVrmb\nXLxPJz7PLY0cgOsXXxROIdtFT+mbVQg2j247hxnhItwtLeQrafb5T8ibeihRlkhK\n1tfKv31EP8tAVqgTjw+qD32bH9h77w==\n=MOaJ\n-----END PGP ARMORED FILE-----\n";
        String sigAfterRevalidation = "-----BEGIN PGP ARMORED FILE-----\nComment: ASCII Armor added by openpgp-interoperability-test-suite\n\nwsBzBAABCgAGBYJe/cFVACEJEEpyNKOhITplFiEEUXksDkji/alOk7kRSnI0o6Eh\nOmXmhQf+LjM2vmdKYnxfVy40Vcdoy2Jov/ZD2RSMFff+fIoCXmWfEetila7v5xHj\nZXq6aevTEN1vgW3T6Q1OjFhnGpMl9wvya9mszfn5BBKukFtLkHeay0PtYuUcrfJC\nUIQCx9PFZzgRJFyGsCqrXBc1VIe2DV3d8dq74unTeCEmWdvAZKdjoUYzRohMtcZ+\n0QctCCJE1kRFJuH/TIdxxwKPtBZfOolSlpS0Z5xxa2sILqUvQ2Dq3hBctUM4g6hB\nY8uafI8qIMwWl4DQDzPpQ917d6J+GCdN0Aib6ZOsvmgR5wrBOFiDpRJ/W9W6+rgs\nI5V/t2y6h6gaHbanggc0cMOaMTtEKQ==\n=lkHs\n-----END PGP ARMORED FILE-----\n";
        PGPSignature predatesPrimaryKey = (PGPSignature)SignatureUtils.readSignatures((String)sigPredatesPrimaryKey).get(0);
        PGPSignature unboundKey = (PGPSignature)SignatureUtils.readSignatures((String)sigUnboundBeforeHardRevocation).get(0);
        PGPSignature afterHardRevocation = (PGPSignature)SignatureUtils.readSignatures((String)sigAfterHardRevocation).get(0);
        PGPSignature afterRevalidation = (PGPSignature)SignatureUtils.readSignatures((String)sigAfterRevalidation).get(0);
        Policy policy = PGPainless.getPolicy();
        Date validationDate = new Date();
        String data = "Hello World :)";
        Assertions.assertThrows(SignatureValidationException.class, () -> CertificateValidator.validateCertificateAndVerifyUninitializedSignature((PGPSignature)predatesPrimaryKey, (InputStream)CertificateValidatorTest.getSignedData(data), (PGPPublicKeyRing)publicKeys, (Policy)policy, (Date)validationDate), (String)"Signature predates primary key");
        Assertions.assertThrows(SignatureValidationException.class, () -> CertificateValidator.validateCertificateAndVerifyUninitializedSignature((PGPSignature)unboundKey, (InputStream)CertificateValidatorTest.getSignedData(data), (PGPPublicKeyRing)publicKeys, (Policy)policy, (Date)validationDate), (String)"Signing key unbound + hard revocation");
        Assertions.assertThrows(SignatureValidationException.class, () -> CertificateValidator.validateCertificateAndVerifyUninitializedSignature((PGPSignature)afterHardRevocation, (InputStream)CertificateValidatorTest.getSignedData(data), (PGPPublicKeyRing)publicKeys, (Policy)policy, (Date)validationDate), (String)"Hard revocation invalidates key at all times");
        Assertions.assertThrows(SignatureValidationException.class, () -> CertificateValidator.validateCertificateAndVerifyUninitializedSignature((PGPSignature)afterRevalidation, (InputStream)CertificateValidatorTest.getSignedData(data), (PGPPublicKeyRing)publicKeys, (Policy)policy, (Date)validationDate), (String)"Hard revocation invalidates key at all times");
    }

    @TestTemplate
    @ExtendWith(value={TestAllImplementations.class})
    public void testPrimaryKeySignsAndIsSoftRevokedSuperseded() throws IOException, PGPException {
        String keyWithSoftRev = "-----BEGIN PGP ARMORED FILE-----\nComment: ASCII Armor added by openpgp-interoperability-test-suite\n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\nexdvhN4AXDlPlB3Rrkj7CQ==\n=qQpG\n-----END PGP ARMORED FILE-----\n";
        String sigPredatesPrimaryKey = "-----BEGIN PGP ARMORED FILE-----\nComment: ASCII Armor added by openpgp-interoperability-test-suite\n\nwsBzBAABCgAGBYJYaEaAACEJEGhPrWLcA4+7FiEE8tFQpP6Ykl1R6RU5aE+tYtwD\nj7ttqgf9Gp4T5Q19cNL9Eyz1nlw11HDHT1wxfGHU5li76y7oo4Jqim15sEPDJWmc\nIpYVrczpCI95aCuaE6yfzpjZlXMzftwex3DjM98vyZH4W9teKcOnpAVjn3dLoQJA\ni4fiq3VaLgl+1OYOwu3DmwGKJZubHM3oPia9pbuyvL5Scvx+QCG0AVnssnt2QswG\nuU6J35QgBdjG2iC043sUoyxTSk929iamdQnOGchjcaATb4E4+HvtkRy4IirKxiKK\nc535BHJRLgWQzUcDDZ5kHf3SPLTNsigFFEwUyf5voFcn/DSMWSzPaVecaafTVJW2\nu8G1R5mjuxDRup8p//X1BSk1FpSmvw==\n=3/dv\n-----END PGP ARMORED FILE-----\n";
        String sigKeyIsValid = "-----BEGIN PGP ARMORED FILE-----\nComment: ASCII Armor added by openpgp-interoperability-test-suite\n\nwsBzBAABCgAGBYJa564AACEJEGhPrWLcA4+7FiEE8tFQpP6Ykl1R6RU5aE+tYtwD\nj7ufRgf/QOsaJZgQaQ5daQrfBItOEcW+5acgY1TCwMVmc/nzBqC32TOvMaM3dypf\nwJbqzxHQIes+ivKDF872VWlMA2BErifpdsogbS0pdik/qU+AjMhr188xKpZKG/IY\n6BtuUPeSpsimx3UeEN3kt79fMtewBo0EXo3ujCyPpIF/9Vpd7L9jlJSvRBuM0/aR\ngbRsclEw4JZ98B3t7F3rLmx+F57Zre0ctzT4tHE6IaCYpEClr6Tepj/UxExYOt2l\nhKgVN8Wsuug7XYdOFmxqrV967m3CTnF8AspmxwgKK6NXjVLplfqij7Rp2URPWrWn\nPp3CQRGUWJdMeMY9P1MpFq6XiXtftw==\n=Ld1q\n-----END PGP ARMORED FILE-----\n";
        String sigKeyIsRevoked = "-----BEGIN PGP ARMORED FILE-----\nComment: ASCII Armor added by openpgp-interoperability-test-suite\n\nwsBzBAABCgAGBYJdP4iAACEJEGhPrWLcA4+7FiEE8tFQpP6Ykl1R6RU5aE+tYtwD\nj7sYXQf8CZw6Kx4oyI8ZJ2c9RjVZmUFEirAoXH7oYA+Ye+wSAY9OtqE/x2SOYaC6\nQHiB93/wkvpqCVkLy2lenzpD7WXLbuFZ+/5jXp1o+sVHXfLSWo6pfIhOjj9FSr8x\nqqlqUfKwkbA6WYgju+qKC35SYdSptix7unaFkO41UdsM8wGQh880HSRMBMFPzg07\n3hMNYXoEJjFlIkxJSMu2WL7N0Q/4xE2iJftsQjUYAtJ/C/YK2I6dhW+CZremnv5R\n/8W+oH5Q63lYU8YL4wYnJQvkHjKs/kjLpoPmqL8kdHjndSpU+KOYr5w61XuEp2hp\nr8trtljVaVIQX2rYawSlqKkWXt0yag==\n=xVd8\n-----END PGP ARMORED FILE-----\n";
        String sigKeyIsRevalidated = "-----BEGIN PGP ARMORED FILE-----\nComment: ASCII Armor added by openpgp-interoperability-test-suite\n\nwsBzBAABCgAGBYJe/cFVACEJEGhPrWLcA4+7FiEE8tFQpP6Ykl1R6RU5aE+tYtwD\nj7vmhQf/UB456IXc8ub8HTExab1d5KqOGSUWpwIznTu8Wk8YuzWKEE8ZeZvPmv8K\niJfBoOx59YrlOfpLAKcTR9Ql+IFbWsIkqPxX7U1SGldhfQm7iaK5Dn6+mmQFOz/s\nZCIavWJ7opsp11JmQAt4FFojv789YswaS7VI1zjDj7EeRiATtzna/GqCYgeCM0cc\nsIe/1j1H2oh9YvYIpPMSGDJPo7T1Ji4Ie3iEQEYNYPuw1Hb7gWYncHXZGJq1nDf/\nWAoI9gSFagpsPW0k9cfEAOVNLNYSyi0CSnQWSjq8THbHKiLPFwsP3hvT2oHycWbK\nu5SfXaTsbMeVQJNdjCNsHq2bOXPGLw==\n=2BW4\n-----END PGP ARMORED FILE-----\n";
        PGPPublicKeyRing publicKeys = PGPainless.readKeyRing().publicKeyRing(keyWithSoftRev);
        PGPSignature predatesPrimaryKey = (PGPSignature)SignatureUtils.readSignatures((String)sigPredatesPrimaryKey).get(0);
        PGPSignature keyIsValid = (PGPSignature)SignatureUtils.readSignatures((String)sigKeyIsValid).get(0);
        PGPSignature keyIsRevoked = (PGPSignature)SignatureUtils.readSignatures((String)sigKeyIsRevoked).get(0);
        PGPSignature keyIsRevalidated = (PGPSignature)SignatureUtils.readSignatures((String)sigKeyIsRevalidated).get(0);
        Policy policy = PGPainless.getPolicy();
        String data = "Hello, World";
        Assertions.assertThrows(SignatureValidationException.class, () -> CertificateValidator.validateCertificateAndVerifyUninitializedSignature((PGPSignature)predatesPrimaryKey, (InputStream)CertificateValidatorTest.getSignedData(data), (PGPPublicKeyRing)publicKeys, (Policy)policy, (Date)predatesPrimaryKey.getCreationTime()), (String)"Signature predates primary key creation date");
        Assertions.assertDoesNotThrow(() -> CertificateValidator.validateCertificateAndVerifyUninitializedSignature((PGPSignature)keyIsValid, (InputStream)CertificateValidatorTest.getSignedData(data), (PGPPublicKeyRing)publicKeys, (Policy)policy, (Date)keyIsValid.getCreationTime()), (String)"Signature is valid");
        Assertions.assertThrows(SignatureValidationException.class, () -> CertificateValidator.validateCertificateAndVerifyUninitializedSignature((PGPSignature)keyIsRevoked, (InputStream)CertificateValidatorTest.getSignedData(data), (PGPPublicKeyRing)publicKeys, (Policy)policy, (Date)keyIsRevoked.getCreationTime()), (String)"Signing key is revoked at this point");
        Assertions.assertDoesNotThrow(() -> CertificateValidator.validateCertificateAndVerifyUninitializedSignature((PGPSignature)keyIsRevalidated, (InputStream)CertificateValidatorTest.getSignedData(data), (PGPPublicKeyRing)publicKeys, (Policy)policy, (Date)keyIsRevalidated.getCreationTime()), (String)"Signature is valid, as signing key is revalidated");
    }

    @TestTemplate
    @ExtendWith(value={TestAllImplementations.class})
    public void testSubkeySignsPrimaryKeyIsSoftRevokedSuperseded() throws IOException, PGPException {
        String key = "-----BEGIN PGP ARMORED FILE-----\nComment: ASCII Armor added by openpgp-interoperability-test-suite\n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\nexdvhN4AXDlPlB3Rrkj7CQ==\n=qQpG\n-----END PGP ARMORED FILE-----\n";
        String sigPredatesPrimaryKey = "-----BEGIN PGP ARMORED FILE-----\nComment: ASCII Armor added by openpgp-interoperability-test-suite\n\nwsBzBAABCgAGBYJYaEaAACEJEEpyNKOhITplFiEEUXksDkji/alOk7kRSnI0o6Eh\nOmVtqgf/YSG+b8lY01/4oCrHoNMECDDbpI5+8WkeT0CcdlEd5TGj3AHesT6H6XmL\nZxaHHOwtkuDh0bIAiYGl36e4ult5XZQhFIwUXGde6myLE+fpCGsBJwNu+TDIrbg3\nPGqnVZNlcU+2sP5JhJfAn8VtLENuHkbIC3+kH8xBIrkPTc0rbNBgyzX5eFO20U0D\nbHCCjfjVDpZ8l7N2NlsRYvU0kTzN5GvwbS1HnMOovF9ZKkEpzxxw6IRJIapaE2L9\nadMKIRAqrIIjfj6Z9nETd1nZE79t1zSw1trfArPaJQr46krgh1ocLQoD/c+PhB9l\nsRxQBnWERgQaDJByq0kwKSnwWAsyxw==\n=SDmD\n-----END PGP ARMORED FILE-----\n";
        String sigSubkeyNotBound = "-----BEGIN PGP ARMORED FILE-----\nComment: ASCII Armor added by openpgp-interoperability-test-suite\n\nwsBzBAABCgAGBYJa564AACEJEEpyNKOhITplFiEEUXksDkji/alOk7kRSnI0o6Eh\nOmWfRgf9ECjIviU1pN0GiMZGci3Vce2b42LIqH0JZApOeRDpcyXgxi3/CmdaewpT\nw9l18gsXhioRg4xUzMFrYSgyYZ9VajFggMjbeX5dSV3rsOSJSiEEyDbeGi0TcA/Y\nGUifX4EfKx5X5nI/wevnYjmruDp9SqaPLHIZK1soOoPzueZ8wKyJ9A4vVG4bvxVX\nFnwBf6mRE/0Z8IoHlRJdq0fSzW4RgX8KAtK8SfyGOnk7LDaevVuL6iE5v0Gsu0oh\ncHlI6Llm97EVxP93KZ1J7TQIi/a6PUJb5XCIw0K/iyuNuAzETgm8LVJyn6UwL4ym\nKcNieOK8Qcoivq0kCYuv/0Tbk13jVQ==\n=5hOz\n-----END PGP ARMORED FILE-----\n";
        String sigKeyRevoked = "-----BEGIN PGP ARMORED FILE-----\nComment: ASCII Armor added by openpgp-interoperability-test-suite\n\nwsBzBAABCgAGBYJdP4iAACEJEEpyNKOhITplFiEEUXksDkji/alOk7kRSnI0o6Eh\nOmUYXQf/dGNZay40bZEpcnxYl+Kq+gRQESeDhg/xOfGfSCLQncMH+UYPaUKANC2g\nCfMNN1wd8ZWrvgyTVo3TVfK1P1RYa9nrvKoKN3bjsFcY6V7VciPW58xVNsuxsEEC\nGEH96TQy+FsP680tRnzQ3Dbw/JT6o6Xi+HLf4JVFceapBgyth61E5gN5w3azxVFr\nGfwIfHvepOjCIq9tRZsRFEBp3XVZ/AF+zQMG5nfIVSm1kVtZjb7KXc3Bj48DVrmb\nXLxPJz7PLY0cgOsXXxROIdtFT+mbVQg2j247hxnhItwtLeQrafb5T8ibeihRlkhK\n1tfKv31EP8tAVqgTjw+qD32bH9h77w==\n=MOaJ\n-----END PGP ARMORED FILE-----\n";
        String sigKeyValid = "-----BEGIN PGP ARMORED FILE-----\nComment: ASCII Armor added by openpgp-interoperability-test-suite\n\nwsBzBAABCgAGBYJe/cFVACEJEEpyNKOhITplFiEEUXksDkji/alOk7kRSnI0o6Eh\nOmXmhQf+LjM2vmdKYnxfVy40Vcdoy2Jov/ZD2RSMFff+fIoCXmWfEetila7v5xHj\nZXq6aevTEN1vgW3T6Q1OjFhnGpMl9wvya9mszfn5BBKukFtLkHeay0PtYuUcrfJC\nUIQCx9PFZzgRJFyGsCqrXBc1VIe2DV3d8dq74unTeCEmWdvAZKdjoUYzRohMtcZ+\n0QctCCJE1kRFJuH/TIdxxwKPtBZfOolSlpS0Z5xxa2sILqUvQ2Dq3hBctUM4g6hB\nY8uafI8qIMwWl4DQDzPpQ917d6J+GCdN0Aib6ZOsvmgR5wrBOFiDpRJ/W9W6+rgs\nI5V/t2y6h6gaHbanggc0cMOaMTtEKQ==\n=lkHs\n-----END PGP ARMORED FILE-----\n";
        PGPPublicKeyRing publicKeys = PGPainless.readKeyRing().publicKeyRing(key);
        PGPSignature predatesPrimaryKey = (PGPSignature)SignatureUtils.readSignatures((String)sigPredatesPrimaryKey).get(0);
        PGPSignature keyNotBound = (PGPSignature)SignatureUtils.readSignatures((String)sigSubkeyNotBound).get(0);
        PGPSignature keyRevoked = (PGPSignature)SignatureUtils.readSignatures((String)sigKeyRevoked).get(0);
        PGPSignature valid = (PGPSignature)SignatureUtils.readSignatures((String)sigKeyValid).get(0);
        Policy policy = PGPainless.getPolicy();
        String data = "Hello, World";
        Date validationDate = new Date();
        Assertions.assertThrows(SignatureValidationException.class, () -> CertificateValidator.validateCertificateAndVerifyUninitializedSignature((PGPSignature)predatesPrimaryKey, (InputStream)CertificateValidatorTest.getSignedData(data), (PGPPublicKeyRing)publicKeys, (Policy)policy, (Date)validationDate), (String)"Signature predates primary key creation date");
        Assertions.assertThrows(SignatureValidationException.class, () -> CertificateValidator.validateCertificateAndVerifyUninitializedSignature((PGPSignature)keyNotBound, (InputStream)CertificateValidatorTest.getSignedData(data), (PGPPublicKeyRing)publicKeys, (Policy)policy, (Date)validationDate), (String)"Signing key is not bound at this point");
        Assertions.assertThrows(SignatureValidationException.class, () -> CertificateValidator.validateCertificateAndVerifyUninitializedSignature((PGPSignature)keyRevoked, (InputStream)CertificateValidatorTest.getSignedData(data), (PGPPublicKeyRing)publicKeys, (Policy)policy, (Date)validationDate), (String)"Signing key is revoked at this point");
        Assertions.assertDoesNotThrow(() -> CertificateValidator.validateCertificateAndVerifyUninitializedSignature((PGPSignature)valid, (InputStream)CertificateValidatorTest.getSignedData(data), (PGPPublicKeyRing)publicKeys, (Policy)policy, (Date)validationDate), (String)"Signing key is revalidated");
    }

    @TestTemplate
    @ExtendWith(value={TestAllImplementations.class})
    public void testPrimaryKeySignsAndIsSoftRevokedRetired() throws IOException, PGPException {
        String key = "-----BEGIN PGP ARMORED FILE-----\nComment: ASCII Armor added by openpgp-interoperability-test-suite\n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\nkLTz9GPO80q8mowzBV0EtHsXb4TeAFw5T5Qd0a5I+wk=\n=7j+m\n-----END PGP ARMORED FILE-----\n";
        String sigPredatesPrimaryKey = "-----BEGIN PGP ARMORED FILE-----\nComment: ASCII Armor added by openpgp-interoperability-test-suite\n\nwsBzBAABCgAGBYJYaEaAACEJEGhPrWLcA4+7FiEE8tFQpP6Ykl1R6RU5aE+tYtwD\nj7ttqgf9Gp4T5Q19cNL9Eyz1nlw11HDHT1wxfGHU5li76y7oo4Jqim15sEPDJWmc\nIpYVrczpCI95aCuaE6yfzpjZlXMzftwex3DjM98vyZH4W9teKcOnpAVjn3dLoQJA\ni4fiq3VaLgl+1OYOwu3DmwGKJZubHM3oPia9pbuyvL5Scvx+QCG0AVnssnt2QswG\nuU6J35QgBdjG2iC043sUoyxTSk929iamdQnOGchjcaATb4E4+HvtkRy4IirKxiKK\nc535BHJRLgWQzUcDDZ5kHf3SPLTNsigFFEwUyf5voFcn/DSMWSzPaVecaafTVJW2\nu8G1R5mjuxDRup8p//X1BSk1FpSmvw==\n=3/dv\n-----END PGP ARMORED FILE-----\n";
        String sigValid = "-----BEGIN PGP ARMORED FILE-----\nComment: ASCII Armor added by openpgp-interoperability-test-suite\n\nwsBzBAABCgAGBYJa564AACEJEGhPrWLcA4+7FiEE8tFQpP6Ykl1R6RU5aE+tYtwD\nj7ufRgf/QOsaJZgQaQ5daQrfBItOEcW+5acgY1TCwMVmc/nzBqC32TOvMaM3dypf\nwJbqzxHQIes+ivKDF872VWlMA2BErifpdsogbS0pdik/qU+AjMhr188xKpZKG/IY\n6BtuUPeSpsimx3UeEN3kt79fMtewBo0EXo3ujCyPpIF/9Vpd7L9jlJSvRBuM0/aR\ngbRsclEw4JZ98B3t7F3rLmx+F57Zre0ctzT4tHE6IaCYpEClr6Tepj/UxExYOt2l\nhKgVN8Wsuug7XYdOFmxqrV967m3CTnF8AspmxwgKK6NXjVLplfqij7Rp2URPWrWn\nPp3CQRGUWJdMeMY9P1MpFq6XiXtftw==\n=Ld1q\n-----END PGP ARMORED FILE-----\n";
        String sigRevoked = "-----BEGIN PGP ARMORED FILE-----\nComment: ASCII Armor added by openpgp-interoperability-test-suite\n\nwsBzBAABCgAGBYJdP4iAACEJEGhPrWLcA4+7FiEE8tFQpP6Ykl1R6RU5aE+tYtwD\nj7sYXQf8CZw6Kx4oyI8ZJ2c9RjVZmUFEirAoXH7oYA+Ye+wSAY9OtqE/x2SOYaC6\nQHiB93/wkvpqCVkLy2lenzpD7WXLbuFZ+/5jXp1o+sVHXfLSWo6pfIhOjj9FSr8x\nqqlqUfKwkbA6WYgju+qKC35SYdSptix7unaFkO41UdsM8wGQh880HSRMBMFPzg07\n3hMNYXoEJjFlIkxJSMu2WL7N0Q/4xE2iJftsQjUYAtJ/C/YK2I6dhW+CZremnv5R\n/8W+oH5Q63lYU8YL4wYnJQvkHjKs/kjLpoPmqL8kdHjndSpU+KOYr5w61XuEp2hp\nr8trtljVaVIQX2rYawSlqKkWXt0yag==\n=xVd8\n-----END PGP ARMORED FILE-----\n";
        String sigReLegitimized = "-----BEGIN PGP ARMORED FILE-----\nComment: ASCII Armor added by openpgp-interoperability-test-suite\n\nwsBzBAABCgAGBYJe/cFVACEJEGhPrWLcA4+7FiEE8tFQpP6Ykl1R6RU5aE+tYtwD\nj7vmhQf/UB456IXc8ub8HTExab1d5KqOGSUWpwIznTu8Wk8YuzWKEE8ZeZvPmv8K\niJfBoOx59YrlOfpLAKcTR9Ql+IFbWsIkqPxX7U1SGldhfQm7iaK5Dn6+mmQFOz/s\nZCIavWJ7opsp11JmQAt4FFojv789YswaS7VI1zjDj7EeRiATtzna/GqCYgeCM0cc\nsIe/1j1H2oh9YvYIpPMSGDJPo7T1Ji4Ie3iEQEYNYPuw1Hb7gWYncHXZGJq1nDf/\nWAoI9gSFagpsPW0k9cfEAOVNLNYSyi0CSnQWSjq8THbHKiLPFwsP3hvT2oHycWbK\nu5SfXaTsbMeVQJNdjCNsHq2bOXPGLw==\n=2BW4\n-----END PGP ARMORED FILE-----\n";
        PGPPublicKeyRing publicKeys = PGPainless.readKeyRing().publicKeyRing(key);
        PGPSignature predatesPrimaryKey = (PGPSignature)SignatureUtils.readSignatures((String)sigPredatesPrimaryKey).get(0);
        PGPSignature valid = (PGPSignature)SignatureUtils.readSignatures((String)sigValid).get(0);
        PGPSignature revoked = (PGPSignature)SignatureUtils.readSignatures((String)sigRevoked).get(0);
        PGPSignature revalidated = (PGPSignature)SignatureUtils.readSignatures((String)sigReLegitimized).get(0);
        Policy policy = PGPainless.getPolicy();
        Date validationDate = new Date();
        String data = "Hello, World";
        Assertions.assertThrows(SignatureValidationException.class, () -> CertificateValidator.validateCertificateAndVerifyUninitializedSignature((PGPSignature)predatesPrimaryKey, (InputStream)CertificateValidatorTest.getSignedData(data), (PGPPublicKeyRing)publicKeys, (Policy)policy, (Date)validationDate), (String)"Signature predates primary key creation date");
        Assertions.assertDoesNotThrow(() -> CertificateValidator.validateCertificateAndVerifyUninitializedSignature((PGPSignature)valid, (InputStream)CertificateValidatorTest.getSignedData(data), (PGPPublicKeyRing)publicKeys, (Policy)policy, (Date)validationDate), (String)"Signature is valid");
        Assertions.assertThrows(SignatureValidationException.class, () -> CertificateValidator.validateCertificateAndVerifyUninitializedSignature((PGPSignature)revoked, (InputStream)CertificateValidatorTest.getSignedData(data), (PGPPublicKeyRing)publicKeys, (Policy)policy, (Date)validationDate), (String)"Primary key is revoked");
        Assertions.assertDoesNotThrow(() -> CertificateValidator.validateCertificateAndVerifyUninitializedSignature((PGPSignature)revalidated, (InputStream)CertificateValidatorTest.getSignedData(data), (PGPPublicKeyRing)publicKeys, (Policy)policy, (Date)validationDate), (String)"Primary key is re-legitimized");
    }

    @TestTemplate
    @ExtendWith(value={TestAllImplementations.class})
    public void testTemporaryValidity() throws IOException, PGPException {
        String keyA = "-----BEGIN PGP PUBLIC KEY BLOCK-----\nComment: D1A6 6E1A 23B1 82C9 980F  788C FBFC C82A 015E 7330\nComment: Bob Babbage <bob@openpgp.example>\n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\n=UNh9\n-----END PGP PUBLIC KEY BLOCK-----\n";
        String keyB = "-----BEGIN PGP PUBLIC KEY BLOCK-----\nComment: D1A6 6E1A 23B1 82C9 980F  788C FBFC C82A 015E 7330\nComment: Bob Babbage <bob@openpgp.example>\n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\niz7OblW3I9Dt\n=qacX\n-----END PGP PUBLIC KEY BLOCK-----\n";
        String keyC = "-----BEGIN PGP PUBLIC KEY BLOCK-----\nComment: D1A6 6E1A 23B1 82C9 980F  788C FBFC C82A 015E 7330\nComment: Bob Babbage <bob@openpgp.example>\n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\n916X5bFdC0Rr\n=RTZe\n-----END PGP PUBLIC KEY BLOCK-----\n";
        String keyASigT0 = "-----BEGIN PGP SIGNATURE-----\n\nwsE7BAABCgBvBYJdkyfyCRD7/MgqAV5zMEcUAAAAAAAeACBzYWx0QG5vdGF0aW9u\ncy5zZXF1b2lhLXBncC5vcmfAtB+VtWWsZc4DiK/QK7LEDG+KAl+ORwXnUahXErz3\nGhYhBNGmbhojsYLJmA94jPv8yCoBXnMwAABlEAv/fe0No+TMQTq5xkyv2xNgzibB\nsqeOBbmZL82PfGJTthEtZ+eKi0pNnGPlFH0VIkRskNRKEZqLGJh2d/xDstiPBTfc\nTzTn66f43h+WnGRqGLYbKHfrJ+Mn/wHub9zxauDvpjcIqOLYscuZ+rvp/OyExoST\nlOmfLsrpGgS//eMDchgeFTybZy3UP7dw+BHHSlMW/s4QvgqqS8v5x2bNgMcairzN\nBMxafx6Mp4RSk46tF+INzPgV8W+ImyGXtkZcBmiv9/u36X6GESSyHl/DVf+ZlHm9\nu35dGn8S2PhyE12eQnLZJoorqcIOtwx1sAD+317Dprv8fwcHBhgyKLVsx2O5NTWn\nfZNRtdjQiUBt03D4Qdxoru/UzpGZWq4OLLZVB7u2B4tRQohkBCH96s5hvY2GHXSN\nMd3FCvms+lCEFi70Ae1KZCpD1/DHCFQcT5fYMYVNU1HFB8eQoKMM3kmv0VpJ31CB\nbJ0esmX6JDukiepzcr09w7bbkhGvKnDgUM3dNxhv\n=PfHW\n-----END PGP SIGNATURE-----\n";
        String keyASigT1_T2 = "-----BEGIN PGP SIGNATURE-----\n\nwsE7BAABCgBvBYJduBHyCRD7/MgqAV5zMEcUAAAAAAAeACBzYWx0QG5vdGF0aW9u\ncy5zZXF1b2lhLXBncC5vcmcglTWhGNT4tUfWuGHHpZUQZGDvZ4AKjIM/B97lFgac\nXBYhBNGmbhojsYLJmA94jPv8yCoBXnMwAADPmAv9EZlRAp65+JLsABQa8sk0YgRU\n28n6/Wcv2nj0iZMtbnXu2hLJqPj62JihCNbElEgJb7NIYbTPEAiyUisq8sa+E12F\nPvBsqcNAdLxlDRyPcwlDrbl0fRqLt/fiItMp6lHnfl2SfrSkzWv8t2DswPS+LXpt\nk5GBWxnJMRQ2Lf1L8rriFY1p6D9XtO8n9wGWMzCoxp3q+diQa8YJbhppPPskU5rT\nbPYH7f1FnU2OzzxgeqlAjtpjkTmzy3RNRlzGEYk6NnNbHQN0sQoiQdOaHMJ/qKMF\nwanYX2iatYCCYJGMDv3Ysw6nXLyiZ6dbqFEJwzRCiZQ2ZBYvzRyEnMojIZE3CXOv\na+h5qm8Ffc+ZPc0spBoTxFCz8+Rmp5qNT9kFOIQUF9YOs/fo8sCr32ZbLu/bBsy9\nRhswX42fnAZ4DZJKlAORbtYmM52WsPxArGLnpOpgD8aT4Mt7SsVpAJXnjn43NKTS\nQu+KulEt+Sn2/ioiQoKknxJWDkBZxHLkgRC3ejTO\n=et/I\n-----END PGP SIGNATURE-----\n";
        String keyASigT2_T3 = "-----BEGIN PGP SIGNATURE-----\n\nwsE7BAABCgBvBYJd3PvyCRD7/MgqAV5zMEcUAAAAAAAeACBzYWx0QG5vdGF0aW9u\ncy5zZXF1b2lhLXBncC5vcmexsHg/HXrdneCOLkWkDj95sRwrSkFKyzFUJtY+9xLv\n1RYhBNGmbhojsYLJmA94jPv8yCoBXnMwAAAqEQv+NdTRkL4vJA1Gst/LK7f3ytjF\nH7kkMI9aTQVxnscsNMGZYBjzaT410if/axqCyAmJX3lPWfHIsIZWmglxEPK04K2b\nQl04JEpTzEVKIKGLBPhfLT/QamtZJmGpoGw7ba8lZ8Xrarb1Lf9srRBF+WKIzuwd\nFQAGsPw/57Q4YXNwPnPJwQbOrfuZLLGGZLczGiBW8900NH0Rg82fcMdO2XmuXhMb\nNnwYoXH7vV+i2uodfdmNhts9ENeiPmPF7DPiKDQtSQ6LLspVE/RoP/lXdd7cGSQR\nJ1IaWX1rN598wBz1wNVwBpxucxxIjm9JYER9+eV3oo08b0+DF4OppDcrJvGnQcdE\nQuEOAqNeSrFGqXglfZ/Cep5ZkIVPtJA88e9FZYz7xlx49bh1os1jc7jeSA6xG/1O\n48ZQiPrLkVZp/62j0GvFEZzOj0CVb+/J2gsQ32SYkKeiZfXIdodReQMYAiXHeEj2\nmtuOFO8yVqWUwnlUwQoVAnr9zMulZB2Np29wvevU\n=kusD\n-----END PGP SIGNATURE-----\n";
        String keyASigT3_now = "-----BEGIN PGP SIGNATURE-----\n\nwsE7BAABCgBvBYJeAeXyCRD7/MgqAV5zMEcUAAAAAAAeACBzYWx0QG5vdGF0aW9u\ncy5zZXF1b2lhLXBncC5vcmcElcHwXV0gKjafJ0j2dKF+EeukF5ywMBZMMkEyyjG4\njhYhBNGmbhojsYLJmA94jPv8yCoBXnMwAAASuAwAnu1KFMNci1CEkzJb5fkVZRGz\nCvo2wC0HXhwMoGm/icuw3qTNWpj3wCA6tOTgIGT6FwfYV42mNkFYpiUSelpIdTP7\nR5wf7cvu2wS5sZo+q9a3K4T2gWu+hlLO00/q8LJYTAG3dQZd6Mhk3gPUh3qNWHuR\nzi3gXajxaQ1yJrFHjRt86DBks5vCBWeFkcNQcuIZgoKlHsGxEgfQo0Yej6v4FBrQ\nxr3iU4GhSAQOFmZQPL2AVOfE8if9CqNRNLGpkloEDAhoSf+TxRyWXFfvXZQGKgSA\noKbgQFyUgdybPFXiQa8ezZaO23risIG/7oe+rAM0vOWMA0f2F0d2W5t4UeZLLxsu\nGh+7ZYK/MDF1HgFHjYefoW7pSPoNzaSIFv6goCtTr1O2c7BnO9QxU1H1rWgkFUd0\nNWHrk3H89te7fP94GtBskR1OnT8zxWVMtFx8NEDicrSw/sKqmxkxh0xW434ZtXgi\nFT2kVzTaUKN+UQ7UIs9wgtqb+s7Dvb7b1bO8wvLg\n=DeXH\n-----END PGP SIGNATURE-----\n";
        String keyBSigT0 = "-----BEGIN PGP SIGNATURE-----\n\nwsE7BAABCgBvBYJdkyfyCRD7/MgqAV5zMEcUAAAAAAAeACBzYWx0QG5vdGF0aW9u\ncy5zZXF1b2lhLXBncC5vcmfAtB+VtWWsZc4DiK/QK7LEDG+KAl+ORwXnUahXErz3\nGhYhBNGmbhojsYLJmA94jPv8yCoBXnMwAABlEAv/fe0No+TMQTq5xkyv2xNgzibB\nsqeOBbmZL82PfGJTthEtZ+eKi0pNnGPlFH0VIkRskNRKEZqLGJh2d/xDstiPBTfc\nTzTn66f43h+WnGRqGLYbKHfrJ+Mn/wHub9zxauDvpjcIqOLYscuZ+rvp/OyExoST\nlOmfLsrpGgS//eMDchgeFTybZy3UP7dw+BHHSlMW/s4QvgqqS8v5x2bNgMcairzN\nBMxafx6Mp4RSk46tF+INzPgV8W+ImyGXtkZcBmiv9/u36X6GESSyHl/DVf+ZlHm9\nu35dGn8S2PhyE12eQnLZJoorqcIOtwx1sAD+317Dprv8fwcHBhgyKLVsx2O5NTWn\nfZNRtdjQiUBt03D4Qdxoru/UzpGZWq4OLLZVB7u2B4tRQohkBCH96s5hvY2GHXSN\nMd3FCvms+lCEFi70Ae1KZCpD1/DHCFQcT5fYMYVNU1HFB8eQoKMM3kmv0VpJ31CB\nbJ0esmX6JDukiepzcr09w7bbkhGvKnDgUM3dNxhv\n=PfHW\n-----END PGP SIGNATURE-----\n";
        String keyBSigT1_T2 = "-----BEGIN PGP SIGNATURE-----\n\nwsE7BAABCgBvBYJduBHyCRD7/MgqAV5zMEcUAAAAAAAeACBzYWx0QG5vdGF0aW9u\ncy5zZXF1b2lhLXBncC5vcmcglTWhGNT4tUfWuGHHpZUQZGDvZ4AKjIM/B97lFgac\nXBYhBNGmbhojsYLJmA94jPv8yCoBXnMwAADPmAv9EZlRAp65+JLsABQa8sk0YgRU\n28n6/Wcv2nj0iZMtbnXu2hLJqPj62JihCNbElEgJb7NIYbTPEAiyUisq8sa+E12F\nPvBsqcNAdLxlDRyPcwlDrbl0fRqLt/fiItMp6lHnfl2SfrSkzWv8t2DswPS+LXpt\nk5GBWxnJMRQ2Lf1L8rriFY1p6D9XtO8n9wGWMzCoxp3q+diQa8YJbhppPPskU5rT\nbPYH7f1FnU2OzzxgeqlAjtpjkTmzy3RNRlzGEYk6NnNbHQN0sQoiQdOaHMJ/qKMF\nwanYX2iatYCCYJGMDv3Ysw6nXLyiZ6dbqFEJwzRCiZQ2ZBYvzRyEnMojIZE3CXOv\na+h5qm8Ffc+ZPc0spBoTxFCz8+Rmp5qNT9kFOIQUF9YOs/fo8sCr32ZbLu/bBsy9\nRhswX42fnAZ4DZJKlAORbtYmM52WsPxArGLnpOpgD8aT4Mt7SsVpAJXnjn43NKTS\nQu+KulEt+Sn2/ioiQoKknxJWDkBZxHLkgRC3ejTO\n=et/I\n-----END PGP SIGNATURE-----\n";
        String keyBSigT2_T3 = "-----BEGIN PGP SIGNATURE-----\n\nwsE7BAABCgBvBYJd3PvyCRD7/MgqAV5zMEcUAAAAAAAeACBzYWx0QG5vdGF0aW9u\ncy5zZXF1b2lhLXBncC5vcmexsHg/HXrdneCOLkWkDj95sRwrSkFKyzFUJtY+9xLv\n1RYhBNGmbhojsYLJmA94jPv8yCoBXnMwAAAqEQv+NdTRkL4vJA1Gst/LK7f3ytjF\nH7kkMI9aTQVxnscsNMGZYBjzaT410if/axqCyAmJX3lPWfHIsIZWmglxEPK04K2b\nQl04JEpTzEVKIKGLBPhfLT/QamtZJmGpoGw7ba8lZ8Xrarb1Lf9srRBF+WKIzuwd\nFQAGsPw/57Q4YXNwPnPJwQbOrfuZLLGGZLczGiBW8900NH0Rg82fcMdO2XmuXhMb\nNnwYoXH7vV+i2uodfdmNhts9ENeiPmPF7DPiKDQtSQ6LLspVE/RoP/lXdd7cGSQR\nJ1IaWX1rN598wBz1wNVwBpxucxxIjm9JYER9+eV3oo08b0+DF4OppDcrJvGnQcdE\nQuEOAqNeSrFGqXglfZ/Cep5ZkIVPtJA88e9FZYz7xlx49bh1os1jc7jeSA6xG/1O\n48ZQiPrLkVZp/62j0GvFEZzOj0CVb+/J2gsQ32SYkKeiZfXIdodReQMYAiXHeEj2\nmtuOFO8yVqWUwnlUwQoVAnr9zMulZB2Np29wvevU\n=kusD\n-----END PGP SIGNATURE-----\n";
        String keyBSigT3_now = "-----BEGIN PGP SIGNATURE-----\n\nwsE7BAABCgBvBYJeAeXyCRD7/MgqAV5zMEcUAAAAAAAeACBzYWx0QG5vdGF0aW9u\ncy5zZXF1b2lhLXBncC5vcmcElcHwXV0gKjafJ0j2dKF+EeukF5ywMBZMMkEyyjG4\njhYhBNGmbhojsYLJmA94jPv8yCoBXnMwAAASuAwAnu1KFMNci1CEkzJb5fkVZRGz\nCvo2wC0HXhwMoGm/icuw3qTNWpj3wCA6tOTgIGT6FwfYV42mNkFYpiUSelpIdTP7\nR5wf7cvu2wS5sZo+q9a3K4T2gWu+hlLO00/q8LJYTAG3dQZd6Mhk3gPUh3qNWHuR\nzi3gXajxaQ1yJrFHjRt86DBks5vCBWeFkcNQcuIZgoKlHsGxEgfQo0Yej6v4FBrQ\nxr3iU4GhSAQOFmZQPL2AVOfE8if9CqNRNLGpkloEDAhoSf+TxRyWXFfvXZQGKgSA\noKbgQFyUgdybPFXiQa8ezZaO23risIG/7oe+rAM0vOWMA0f2F0d2W5t4UeZLLxsu\nGh+7ZYK/MDF1HgFHjYefoW7pSPoNzaSIFv6goCtTr1O2c7BnO9QxU1H1rWgkFUd0\nNWHrk3H89te7fP94GtBskR1OnT8zxWVMtFx8NEDicrSw/sKqmxkxh0xW434ZtXgi\nFT2kVzTaUKN+UQ7UIs9wgtqb+s7Dvb7b1bO8wvLg\n=DeXH\n-----END PGP SIGNATURE-----\n";
        String keyCSigT0 = "-----BEGIN PGP SIGNATURE-----\n\nwsE7BAABCgBvBYJdkyfyCRD7/MgqAV5zMEcUAAAAAAAeACBzYWx0QG5vdGF0aW9u\ncy5zZXF1b2lhLXBncC5vcmfAtB+VtWWsZc4DiK/QK7LEDG+KAl+ORwXnUahXErz3\nGhYhBNGmbhojsYLJmA94jPv8yCoBXnMwAABlEAv/fe0No+TMQTq5xkyv2xNgzibB\nsqeOBbmZL82PfGJTthEtZ+eKi0pNnGPlFH0VIkRskNRKEZqLGJh2d/xDstiPBTfc\nTzTn66f43h+WnGRqGLYbKHfrJ+Mn/wHub9zxauDvpjcIqOLYscuZ+rvp/OyExoST\nlOmfLsrpGgS//eMDchgeFTybZy3UP7dw+BHHSlMW/s4QvgqqS8v5x2bNgMcairzN\nBMxafx6Mp4RSk46tF+INzPgV8W+ImyGXtkZcBmiv9/u36X6GESSyHl/DVf+ZlHm9\nu35dGn8S2PhyE12eQnLZJoorqcIOtwx1sAD+317Dprv8fwcHBhgyKLVsx2O5NTWn\nfZNRtdjQiUBt03D4Qdxoru/UzpGZWq4OLLZVB7u2B4tRQohkBCH96s5hvY2GHXSN\nMd3FCvms+lCEFi70Ae1KZCpD1/DHCFQcT5fYMYVNU1HFB8eQoKMM3kmv0VpJ31CB\nbJ0esmX6JDukiepzcr09w7bbkhGvKnDgUM3dNxhv\n=PfHW\n-----END PGP SIGNATURE-----\n";
        String keyCSigT1_T2 = "-----BEGIN PGP SIGNATURE-----\n\nwsE7BAABCgBvBYJduBHyCRD7/MgqAV5zMEcUAAAAAAAeACBzYWx0QG5vdGF0aW9u\ncy5zZXF1b2lhLXBncC5vcmcglTWhGNT4tUfWuGHHpZUQZGDvZ4AKjIM/B97lFgac\nXBYhBNGmbhojsYLJmA94jPv8yCoBXnMwAADPmAv9EZlRAp65+JLsABQa8sk0YgRU\n28n6/Wcv2nj0iZMtbnXu2hLJqPj62JihCNbElEgJb7NIYbTPEAiyUisq8sa+E12F\nPvBsqcNAdLxlDRyPcwlDrbl0fRqLt/fiItMp6lHnfl2SfrSkzWv8t2DswPS+LXpt\nk5GBWxnJMRQ2Lf1L8rriFY1p6D9XtO8n9wGWMzCoxp3q+diQa8YJbhppPPskU5rT\nbPYH7f1FnU2OzzxgeqlAjtpjkTmzy3RNRlzGEYk6NnNbHQN0sQoiQdOaHMJ/qKMF\nwanYX2iatYCCYJGMDv3Ysw6nXLyiZ6dbqFEJwzRCiZQ2ZBYvzRyEnMojIZE3CXOv\na+h5qm8Ffc+ZPc0spBoTxFCz8+Rmp5qNT9kFOIQUF9YOs/fo8sCr32ZbLu/bBsy9\nRhswX42fnAZ4DZJKlAORbtYmM52WsPxArGLnpOpgD8aT4Mt7SsVpAJXnjn43NKTS\nQu+KulEt+Sn2/ioiQoKknxJWDkBZxHLkgRC3ejTO\n=et/I\n-----END PGP SIGNATURE-----\n";
        String keyCSigT2_T3 = "-----BEGIN PGP SIGNATURE-----\n\nwsE7BAABCgBvBYJd3PvyCRD7/MgqAV5zMEcUAAAAAAAeACBzYWx0QG5vdGF0aW9u\ncy5zZXF1b2lhLXBncC5vcmexsHg/HXrdneCOLkWkDj95sRwrSkFKyzFUJtY+9xLv\n1RYhBNGmbhojsYLJmA94jPv8yCoBXnMwAAAqEQv+NdTRkL4vJA1Gst/LK7f3ytjF\nH7kkMI9aTQVxnscsNMGZYBjzaT410if/axqCyAmJX3lPWfHIsIZWmglxEPK04K2b\nQl04JEpTzEVKIKGLBPhfLT/QamtZJmGpoGw7ba8lZ8Xrarb1Lf9srRBF+WKIzuwd\nFQAGsPw/57Q4YXNwPnPJwQbOrfuZLLGGZLczGiBW8900NH0Rg82fcMdO2XmuXhMb\nNnwYoXH7vV+i2uodfdmNhts9ENeiPmPF7DPiKDQtSQ6LLspVE/RoP/lXdd7cGSQR\nJ1IaWX1rN598wBz1wNVwBpxucxxIjm9JYER9+eV3oo08b0+DF4OppDcrJvGnQcdE\nQuEOAqNeSrFGqXglfZ/Cep5ZkIVPtJA88e9FZYz7xlx49bh1os1jc7jeSA6xG/1O\n48ZQiPrLkVZp/62j0GvFEZzOj0CVb+/J2gsQ32SYkKeiZfXIdodReQMYAiXHeEj2\nmtuOFO8yVqWUwnlUwQoVAnr9zMulZB2Np29wvevU\n=kusD\n-----END PGP SIGNATURE-----\n";
        String keyCSigT3_now = "-----BEGIN PGP SIGNATURE-----\n\nwsE7BAABCgBvBYJeAeXyCRD7/MgqAV5zMEcUAAAAAAAeACBzYWx0QG5vdGF0aW9u\ncy5zZXF1b2lhLXBncC5vcmcElcHwXV0gKjafJ0j2dKF+EeukF5ywMBZMMkEyyjG4\njhYhBNGmbhojsYLJmA94jPv8yCoBXnMwAAASuAwAnu1KFMNci1CEkzJb5fkVZRGz\nCvo2wC0HXhwMoGm/icuw3qTNWpj3wCA6tOTgIGT6FwfYV42mNkFYpiUSelpIdTP7\nR5wf7cvu2wS5sZo+q9a3K4T2gWu+hlLO00/q8LJYTAG3dQZd6Mhk3gPUh3qNWHuR\nzi3gXajxaQ1yJrFHjRt86DBks5vCBWeFkcNQcuIZgoKlHsGxEgfQo0Yej6v4FBrQ\nxr3iU4GhSAQOFmZQPL2AVOfE8if9CqNRNLGpkloEDAhoSf+TxRyWXFfvXZQGKgSA\noKbgQFyUgdybPFXiQa8ezZaO23risIG/7oe+rAM0vOWMA0f2F0d2W5t4UeZLLxsu\nGh+7ZYK/MDF1HgFHjYefoW7pSPoNzaSIFv6goCtTr1O2c7BnO9QxU1H1rWgkFUd0\nNWHrk3H89te7fP94GtBskR1OnT8zxWVMtFx8NEDicrSw/sKqmxkxh0xW434ZtXgi\nFT2kVzTaUKN+UQ7UIs9wgtqb+s7Dvb7b1bO8wvLg\n=DeXH\n-----END PGP SIGNATURE-----\n";
        PGPPublicKeyRing keysA = PGPainless.readKeyRing().publicKeyRing(keyA);
        PGPPublicKeyRing keysB = PGPainless.readKeyRing().publicKeyRing(keyB);
        PGPPublicKeyRing keysC = PGPainless.readKeyRing().publicKeyRing(keyC);
        PGPSignature sigAT0 = (PGPSignature)SignatureUtils.readSignatures((String)keyASigT0).get(0);
        PGPSignature sigAT1_T2 = (PGPSignature)SignatureUtils.readSignatures((String)keyASigT1_T2).get(0);
        PGPSignature sigAT2_T3 = (PGPSignature)SignatureUtils.readSignatures((String)keyASigT2_T3).get(0);
        PGPSignature sigAT3_now = (PGPSignature)SignatureUtils.readSignatures((String)keyASigT3_now).get(0);
        PGPSignature sigBT0 = (PGPSignature)SignatureUtils.readSignatures((String)keyBSigT0).get(0);
        PGPSignature sigBT1_T2 = (PGPSignature)SignatureUtils.readSignatures((String)keyBSigT1_T2).get(0);
        PGPSignature sigBT2_T3 = (PGPSignature)SignatureUtils.readSignatures((String)keyBSigT2_T3).get(0);
        PGPSignature sigBT3_now = (PGPSignature)SignatureUtils.readSignatures((String)keyBSigT3_now).get(0);
        PGPSignature sigCT0 = (PGPSignature)SignatureUtils.readSignatures((String)keyCSigT0).get(0);
        PGPSignature sigCT1_T2 = (PGPSignature)SignatureUtils.readSignatures((String)keyCSigT1_T2).get(0);
        PGPSignature sigCT2_T3 = (PGPSignature)SignatureUtils.readSignatures((String)keyCSigT2_T3).get(0);
        PGPSignature sigCT3_now = (PGPSignature)SignatureUtils.readSignatures((String)keyCSigT3_now).get(0);
        Policy policy = PGPainless.getPolicy();
        Date validationDate = new Date();
        String data = "Hello World :)";
        Assertions.assertThrows(SignatureValidationException.class, () -> CertificateValidator.validateCertificateAndVerifyUninitializedSignature((PGPSignature)sigAT0, (InputStream)CertificateValidatorTest.getSignedData(data), (PGPPublicKeyRing)keysA, (Policy)policy, (Date)validationDate), (String)"Signature predates key creation time");
        Assertions.assertDoesNotThrow(() -> CertificateValidator.validateCertificateAndVerifyUninitializedSignature((PGPSignature)sigAT1_T2, (InputStream)CertificateValidatorTest.getSignedData(data), (PGPPublicKeyRing)keysA, (Policy)policy, (Date)validationDate), (String)"Key valid");
        Assertions.assertThrows(SignatureValidationException.class, () -> CertificateValidator.validateCertificateAndVerifyUninitializedSignature((PGPSignature)sigAT2_T3, (InputStream)CertificateValidatorTest.getSignedData(data), (PGPPublicKeyRing)keysA, (Policy)policy, (Date)validationDate), (String)"Key is not valid, as subkey binding expired");
        Assertions.assertDoesNotThrow(() -> CertificateValidator.validateCertificateAndVerifyUninitializedSignature((PGPSignature)sigAT3_now, (InputStream)CertificateValidatorTest.getSignedData(data), (PGPPublicKeyRing)keysA, (Policy)policy, (Date)validationDate), (String)"Key is valid again");
        Assertions.assertThrows(SignatureValidationException.class, () -> CertificateValidator.validateCertificateAndVerifyUninitializedSignature((PGPSignature)sigBT0, (InputStream)CertificateValidatorTest.getSignedData(data), (PGPPublicKeyRing)keysB, (Policy)policy, (Date)validationDate), (String)"Signature predates key creation time");
        Assertions.assertDoesNotThrow(() -> CertificateValidator.validateCertificateAndVerifyUninitializedSignature((PGPSignature)sigBT1_T2, (InputStream)CertificateValidatorTest.getSignedData(data), (PGPPublicKeyRing)keysB, (Policy)policy, (Date)validationDate), (String)"Key is valid");
        Assertions.assertThrows(SignatureValidationException.class, () -> CertificateValidator.validateCertificateAndVerifyUninitializedSignature((PGPSignature)sigBT2_T3, (InputStream)CertificateValidatorTest.getSignedData(data), (PGPPublicKeyRing)keysB, (Policy)policy, (Date)validationDate), (String)"Primary key is not signing-capable");
        Assertions.assertDoesNotThrow(() -> CertificateValidator.validateCertificateAndVerifyUninitializedSignature((PGPSignature)sigBT3_now, (InputStream)CertificateValidatorTest.getSignedData(data), (PGPPublicKeyRing)keysB, (Policy)policy, (Date)validationDate), (String)"Key is valid again");
        Assertions.assertThrows(SignatureValidationException.class, () -> CertificateValidator.validateCertificateAndVerifyUninitializedSignature((PGPSignature)sigCT0, (InputStream)CertificateValidatorTest.getSignedData(data), (PGPPublicKeyRing)keysC, (Policy)policy, (Date)validationDate), (String)"Signature predates key creation time");
        Assertions.assertDoesNotThrow(() -> CertificateValidator.validateCertificateAndVerifyUninitializedSignature((PGPSignature)sigCT1_T2, (InputStream)CertificateValidatorTest.getSignedData(data), (PGPPublicKeyRing)keysC, (Policy)policy, (Date)validationDate), (String)"Key is valid");
        Assertions.assertThrows(SignatureValidationException.class, () -> CertificateValidator.validateCertificateAndVerifyUninitializedSignature((PGPSignature)sigCT2_T3, (InputStream)CertificateValidatorTest.getSignedData(data), (PGPPublicKeyRing)keysC, (Policy)policy, (Date)validationDate), (String)"Key is revoked");
        Assertions.assertDoesNotThrow(() -> CertificateValidator.validateCertificateAndVerifyUninitializedSignature((PGPSignature)sigCT3_now, (InputStream)CertificateValidatorTest.getSignedData(data), (PGPPublicKeyRing)keysC, (Policy)policy, (Date)validationDate), (String)"Key is valid again");
    }

    private static InputStream getSignedData(String data) {
        return new ByteArrayInputStream(data.getBytes(StandardCharsets.UTF_8));
    }

    @Test
    public void testNoIssuer() throws IOException, PGPException {
        String SIG = "-----BEGIN PGP SIGNATURE-----\n\nwsExBAABCABlBYJhBZl3RxQAAAAAAB4AIHNhbHRAbm90YXRpb25zLnNlcXVvaWEt\ncGdwLm9yZ6u++H3Om9Z3+mn7nyt3TSxs4o/D4fHi+e+X+/7jEj3dFiEE0aZuGiOx\ngsmYD3iM+/zIKgFeczAAAOQKDACD4krXC/9GQQcl+waBglgmQaUWd+YEpXlPDQc3\ndPF7kkc/pRmmGVur8kk8rWmuQwNvQt7vGD2YJzcWBFvSSZlbYa89Gd3L5H2keTYC\n0Q/VbZqBp9ONtK3D9Yrti2jMYdcb30LZQnsSUsB0mwX41AxjPLx0FAcVMAEADjq/\nzLkJ1y+vXCko653qS/um6VQGK8JeNHRz8ghrc4k0E2yYadWWO07wxuoTAlwskf32\nMdSVeIUQIEZqEgWMbqtqglqL3eNWk56IonCePtulkQSN6qi4JpqTeudlOm+Zp/IN\nTk+vrdS8PyQ0CiM794t/iHZDK5Jyz4ccFhTLPbF58xnwkwqdWPdf8WjCzaLfZcuN\nzajVdDXOBjNC1s10voENZhChsZYc9BfHqVMrOe8ngLKCCzqlJ/E930OJiZX5pc+H\nmHaHdn0m6lrhM5JX6RT9TNS/yrjdpkScb68Rcf+rXdeSoCE/aDOCVXhIhVTMQ9di\nCh80fhwtYYjAM9G0/AEs7uBCGQQ=\n=NDO/\n-----END PGP SIGNATURE-----\n";
        String CERT = "-----BEGIN PGP PUBLIC KEY BLOCK-----\nComment: Bob's OpenPGP certificate\n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\nNEJd3XZRzaXZE2aAMQ==\n=NXei\n-----END PGP PUBLIC KEY BLOCK-----\n";
        String DATA = "Hello World :)";
        DecryptionStream decryptionStream = PGPainless.decryptAndOrVerify().onInputStream((InputStream)new ByteArrayInputStream(DATA.getBytes(StandardCharsets.UTF_8))).withOptions(new ConsumerOptions().addVerificationCert(PGPainless.readKeyRing().publicKeyRing(CERT)).addVerificationOfDetachedSignatures((InputStream)new ByteArrayInputStream(SIG.getBytes(StandardCharsets.UTF_8))));
        Streams.drain((InputStream)decryptionStream);
        decryptionStream.close();
        MessageMetadata metadata = decryptionStream.getMetadata();
        Assertions.assertFalse((boolean)metadata.getVerifiedSignatures().isEmpty());
    }
}

