/*
 * Decompiled with CFR 0.152.
 */
package org.pgpainless.signature.consumer;

import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.concurrent.ConcurrentHashMap;
import org.bouncycastle.bcpg.sig.KeyFlags;
import org.bouncycastle.bcpg.sig.NotationData;
import org.bouncycastle.bcpg.sig.SignatureCreationTime;
import org.bouncycastle.openpgp.PGPException;
import org.bouncycastle.openpgp.PGPPublicKey;
import org.bouncycastle.openpgp.PGPSignature;
import org.bouncycastle.openpgp.PGPSignatureList;
import org.bouncycastle.openpgp.PGPSignatureSubpacketVector;
import org.bouncycastle.openpgp.PGPUserAttributeSubpacketVector;
import org.pgpainless.algorithm.HashAlgorithm;
import org.pgpainless.algorithm.KeyFlag;
import org.pgpainless.algorithm.PublicKeyAlgorithm;
import org.pgpainless.algorithm.SignatureSubpacket;
import org.pgpainless.algorithm.SignatureType;
import org.pgpainless.exception.SignatureValidationException;
import org.pgpainless.implementation.ImplementationFactory;
import org.pgpainless.key.OpenPgpFingerprint;
import org.pgpainless.policy.Policy;
import org.pgpainless.signature.SignatureUtils;
import org.pgpainless.signature.subpackets.SignatureSubpacketsUtil;
import org.pgpainless.util.DateUtil;
import org.pgpainless.util.NotationRegistry;

public abstract class SignatureValidator {
    public abstract void verify(PGPSignature var1) throws SignatureValidationException;

    public static SignatureValidator wasPossiblyMadeByKey(final PGPPublicKey signingKey) {
        return new SignatureValidator(){

            @Override
            public void verify(PGPSignature signature) throws SignatureValidationException {
                OpenPgpFingerprint signingKeyFingerprint = OpenPgpFingerprint.of(signingKey);
                Long issuer = SignatureSubpacketsUtil.getIssuerKeyIdAsLong(signature);
                if (issuer != null && issuer.longValue() != signingKey.getKeyID()) {
                    throw new SignatureValidationException("Signature was not created by " + signingKeyFingerprint + " (signature issuer: " + Long.toHexString(issuer) + ")");
                }
                OpenPgpFingerprint fingerprint = SignatureSubpacketsUtil.getIssuerFingerprintAsOpenPgpFingerprint(signature);
                if (fingerprint != null && !fingerprint.equals(signingKeyFingerprint)) {
                    throw new SignatureValidationException("Signature was not created by " + signingKeyFingerprint + " (signature fingerprint: " + fingerprint + ")");
                }
            }
        };
    }

    public static SignatureValidator hasValidPrimaryKeyBindingSignatureIfRequired(final PGPPublicKey primaryKey, final PGPPublicKey subkey, final Policy policy, final Date referenceDate) {
        return new SignatureValidator(){

            @Override
            public void verify(PGPSignature signature) throws SignatureValidationException {
                if (!PublicKeyAlgorithm.requireFromId(signature.getKeyAlgorithm()).isSigningCapable()) {
                    return;
                }
                KeyFlags keyFlags = SignatureSubpacketsUtil.getKeyFlags(signature);
                if (keyFlags == null) {
                    return;
                }
                if (!KeyFlag.hasKeyFlag(keyFlags.getFlags(), KeyFlag.SIGN_DATA) && !KeyFlag.hasKeyFlag(keyFlags.getFlags(), KeyFlag.CERTIFY_OTHER)) {
                    return;
                }
                try {
                    PGPSignatureList embeddedSignatures = SignatureSubpacketsUtil.getEmbeddedSignature(signature);
                    if (embeddedSignatures == null) {
                        throw new SignatureValidationException("Missing primary key binding signature on signing capable subkey " + Long.toHexString(subkey.getKeyID()), Collections.emptyMap());
                    }
                    boolean hasValidPrimaryKeyBinding = false;
                    ConcurrentHashMap<PGPSignature, Exception> rejectedEmbeddedSigs = new ConcurrentHashMap<PGPSignature, Exception>();
                    for (PGPSignature embedded : embeddedSignatures) {
                        if (SignatureType.valueOf(embedded.getSignatureType()) != SignatureType.PRIMARYKEY_BINDING) continue;
                        try {
                            2.signatureStructureIsAcceptable(subkey, policy).verify(embedded);
                            2.signatureIsEffective(referenceDate).verify(embedded);
                            2.correctPrimaryKeyBindingSignature(primaryKey, subkey).verify(embedded);
                            hasValidPrimaryKeyBinding = true;
                            break;
                        }
                        catch (SignatureValidationException e) {
                            rejectedEmbeddedSigs.put(embedded, e);
                        }
                    }
                    if (!hasValidPrimaryKeyBinding) {
                        throw new SignatureValidationException("Missing primary key binding signature on signing capable subkey " + Long.toHexString(subkey.getKeyID()), rejectedEmbeddedSigs);
                    }
                }
                catch (PGPException e) {
                    throw new SignatureValidationException("Cannot process list of embedded signatures.", e);
                }
            }
        };
    }

    public static SignatureValidator signatureStructureIsAcceptable(final PGPPublicKey signingKey, final Policy policy) {
        return new SignatureValidator(){

            @Override
            public void verify(PGPSignature signature) throws SignatureValidationException {
                3.signatureIsNotMalformed(signingKey).verify(signature);
                if (signature.getVersion() >= 4) {
                    3.signatureDoesNotHaveCriticalUnknownNotations(policy.getNotationRegistry()).verify(signature);
                    3.signatureDoesNotHaveCriticalUnknownSubpackets().verify(signature);
                }
                3.signatureUsesAcceptableHashAlgorithm(policy).verify(signature);
                3.signatureUsesAcceptablePublicKeyAlgorithm(policy, signingKey).verify(signature);
            }
        };
    }

    public static SignatureValidator signatureUsesAcceptablePublicKeyAlgorithm(final Policy policy, final PGPPublicKey signingKey) {
        return new SignatureValidator(){

            @Override
            public void verify(PGPSignature signature) throws SignatureValidationException {
                PublicKeyAlgorithm algorithm = PublicKeyAlgorithm.requireFromId(signingKey.getAlgorithm());
                int bitStrength = signingKey.getBitStrength();
                if (bitStrength == -1) {
                    throw new SignatureValidationException("Cannot determine bit strength of signing key.");
                }
                if (!policy.getPublicKeyAlgorithmPolicy().isAcceptable(algorithm, bitStrength)) {
                    throw new SignatureValidationException("Signature was made using unacceptable key. " + (Object)((Object)algorithm) + " (" + bitStrength + " bits) is not acceptable according to the public key algorithm policy.");
                }
            }
        };
    }

    public static SignatureValidator signatureUsesAcceptableHashAlgorithm(final Policy policy) {
        return new SignatureValidator(){

            @Override
            public void verify(PGPSignature signature) throws SignatureValidationException {
                try {
                    HashAlgorithm hashAlgorithm = HashAlgorithm.requireFromId(signature.getHashAlgorithm());
                    Policy.HashAlgorithmPolicy hashAlgorithmPolicy = SignatureValidator.getHashAlgorithmPolicyForSignature(signature, policy);
                    if (!hashAlgorithmPolicy.isAcceptable(signature.getHashAlgorithm(), signature.getCreationTime())) {
                        throw new SignatureValidationException("Signature uses unacceptable hash algorithm " + (Object)((Object)hashAlgorithm) + " (Signature creation time: " + DateUtil.formatUTCDate(signature.getCreationTime()) + ")");
                    }
                }
                catch (NoSuchElementException e) {
                    throw new SignatureValidationException("Signature uses unknown hash algorithm " + signature.getHashAlgorithm());
                }
            }
        };
    }

    private static Policy.HashAlgorithmPolicy getHashAlgorithmPolicyForSignature(PGPSignature signature, Policy policy) {
        SignatureType type = SignatureType.valueOf(signature.getSignatureType());
        Policy.HashAlgorithmPolicy hashAlgorithmPolicy = type == SignatureType.CERTIFICATION_REVOCATION || type == SignatureType.KEY_REVOCATION || type == SignatureType.SUBKEY_REVOCATION ? policy.getRevocationSignatureHashAlgorithmPolicy() : policy.getSignatureHashAlgorithmPolicy();
        return hashAlgorithmPolicy;
    }

    public static SignatureValidator signatureDoesNotHaveCriticalUnknownNotations(final NotationRegistry registry) {
        return new SignatureValidator(){

            @Override
            public void verify(PGPSignature signature) throws SignatureValidationException {
                List<NotationData> hashedNotations = SignatureSubpacketsUtil.getHashedNotationData(signature);
                for (NotationData notation : hashedNotations) {
                    if (!notation.isCritical() || registry.isKnownNotation(notation.getNotationName())) continue;
                    throw new SignatureValidationException("Signature contains unknown critical notation '" + notation.getNotationName() + "' in its hashed area.");
                }
            }
        };
    }

    public static SignatureValidator signatureDoesNotHaveCriticalUnknownSubpackets() {
        return new SignatureValidator(){

            @Override
            public void verify(PGPSignature signature) throws SignatureValidationException {
                PGPSignatureSubpacketVector hashedSubpackets = signature.getHashedSubPackets();
                for (int criticalTag : hashedSubpackets.getCriticalTags()) {
                    try {
                        SignatureSubpacket.requireFromCode(criticalTag);
                    }
                    catch (NoSuchElementException e) {
                        throw new SignatureValidationException("Signature contains unknown critical subpacket of type " + Long.toHexString(criticalTag));
                    }
                }
            }
        };
    }

    public static SignatureValidator signatureIsEffective() {
        return SignatureValidator.signatureIsEffective(new Date());
    }

    public static SignatureValidator signatureIsEffective(final Date referenceDate) {
        return new SignatureValidator(){

            @Override
            public void verify(PGPSignature signature) throws SignatureValidationException {
                8.signatureIsAlreadyEffective(referenceDate).verify(signature);
                8.signatureIsNotYetExpired(referenceDate).verify(signature);
            }
        };
    }

    public static SignatureValidator signatureIsAlreadyEffective(final Date referenceDate) {
        return new SignatureValidator(){

            @Override
            public void verify(PGPSignature signature) throws SignatureValidationException {
                Date signatureCreationTime = SignatureSubpacketsUtil.getSignatureCreationTime(signature).getTime();
                if (SignatureUtils.isHardRevocation(signature)) {
                    return;
                }
                if (signatureCreationTime.after(referenceDate)) {
                    throw new SignatureValidationException("Signature was created at " + signatureCreationTime + " and is therefore not yet valid at " + referenceDate);
                }
            }
        };
    }

    public static SignatureValidator signatureIsNotYetExpired(final Date referenceDate) {
        return new SignatureValidator(){

            @Override
            public void verify(PGPSignature signature) throws SignatureValidationException {
                if (SignatureUtils.isHardRevocation(signature)) {
                    return;
                }
                Date signatureExpirationTime = SignatureSubpacketsUtil.getSignatureExpirationTimeAsDate(signature);
                if (signatureExpirationTime != null && signatureExpirationTime.before(referenceDate)) {
                    throw new SignatureValidationException("Signature is already expired (expiration: " + signatureExpirationTime + ", validation: " + referenceDate + ")");
                }
            }
        };
    }

    public static SignatureValidator signatureIsNotMalformed(final PGPPublicKey creator) {
        return new SignatureValidator(){

            @Override
            public void verify(PGPSignature signature) throws SignatureValidationException {
                if (signature.getVersion() >= 4) {
                    11.signatureHasHashedCreationTime().verify(signature);
                }
                11.signatureDoesNotPredateSigningKey(creator).verify(signature);
                if (signature.getSignatureType() != SignatureType.PRIMARYKEY_BINDING.getCode()) {
                    11.signatureDoesNotPredateSigningKeyBindingDate(creator).verify(signature);
                }
            }
        };
    }

    public static SignatureValidator signatureDoesNotPredateSignee(PGPPublicKey signee) {
        return SignatureValidator.signatureDoesNotPredateKeyCreation(signee);
    }

    public static SignatureValidator signatureHasHashedCreationTime() {
        return new SignatureValidator(){

            @Override
            public void verify(PGPSignature signature) throws SignatureValidationException {
                SignatureCreationTime creationTime = SignatureSubpacketsUtil.getSignatureCreationTime(signature);
                if (creationTime == null) {
                    throw new SignatureValidationException("Malformed signature. Signature has no signature creation time subpacket in its hashed area.");
                }
            }
        };
    }

    public static SignatureValidator signatureDoesNotPredateSigningKey(PGPPublicKey key) {
        return SignatureValidator.signatureDoesNotPredateKeyCreation(key);
    }

    public static SignatureValidator signatureDoesNotPredateKeyCreation(final PGPPublicKey key) {
        return new SignatureValidator(){

            @Override
            public void verify(PGPSignature signature) throws SignatureValidationException {
                Date signatureCreationTime;
                Date keyCreationTime = key.getCreationTime();
                if (keyCreationTime.after(signatureCreationTime = signature.getCreationTime())) {
                    throw new SignatureValidationException("Signature predates key (key creation: " + keyCreationTime + ", signature creation: " + signatureCreationTime + ")");
                }
            }
        };
    }

    public static SignatureValidator signatureDoesNotPredateSigningKeyBindingDate(final PGPPublicKey signingKey) {
        return new SignatureValidator(){

            @Override
            public void verify(PGPSignature signature) throws SignatureValidationException {
                if (signingKey.isMasterKey()) {
                    return;
                }
                boolean predatesBindingSig = true;
                Iterator<PGPSignature> bindingSignatures = signingKey.getSignaturesOfType(SignatureType.SUBKEY_BINDING.getCode());
                if (!bindingSignatures.hasNext()) {
                    throw new SignatureValidationException("Signing subkey does not have a subkey binding signature.");
                }
                while (bindingSignatures.hasNext()) {
                    PGPSignature bindingSig = bindingSignatures.next();
                    if (bindingSig.getCreationTime().after(signature.getCreationTime())) continue;
                    predatesBindingSig = false;
                }
                if (predatesBindingSig) {
                    throw new SignatureValidationException("Signature was created before the signing key was bound to the key ring.");
                }
            }
        };
    }

    public static SignatureValidator correctSubkeyBindingSignature(final PGPPublicKey primaryKey, final PGPPublicKey subkey) {
        return new SignatureValidator(){

            @Override
            public void verify(PGPSignature signature) throws SignatureValidationException {
                if (primaryKey.getKeyID() == subkey.getKeyID()) {
                    throw new SignatureValidationException("Primary key cannot be its own subkey.");
                }
                try {
                    signature.init(ImplementationFactory.getInstance().getPGPContentVerifierBuilderProvider(), primaryKey);
                    boolean valid = signature.verifyCertification(primaryKey, subkey);
                    if (!valid) {
                        throw new SignatureValidationException("Signature is not correct.");
                    }
                }
                catch (ClassCastException | PGPException e) {
                    throw new SignatureValidationException("Cannot verify subkey binding signature correctness", e);
                }
            }
        };
    }

    public static SignatureValidator correctPrimaryKeyBindingSignature(final PGPPublicKey primaryKey, final PGPPublicKey subkey) {
        return new SignatureValidator(){

            @Override
            public void verify(PGPSignature signature) throws SignatureValidationException {
                try {
                    signature.init(ImplementationFactory.getInstance().getPGPContentVerifierBuilderProvider(), subkey);
                    boolean valid = signature.verifyCertification(primaryKey, subkey);
                    if (!valid) {
                        throw new SignatureValidationException("Primary Key Binding Signature is not correct.");
                    }
                }
                catch (ClassCastException | PGPException e) {
                    throw new SignatureValidationException("Cannot verify primary key binding signature correctness", e);
                }
            }
        };
    }

    public static SignatureValidator correctSignatureOverKey(final PGPPublicKey signer, final PGPPublicKey signee) {
        return new SignatureValidator(){

            @Override
            public void verify(PGPSignature signature) throws SignatureValidationException {
                try {
                    signature.init(ImplementationFactory.getInstance().getPGPContentVerifierBuilderProvider(), signer);
                    boolean valid = signer.getKeyID() == signee.getKeyID() || signature.getSignatureType() == 31 ? signature.verifyCertification(signee) : signature.verifyCertification(signer, signee);
                    if (!valid) {
                        throw new SignatureValidationException("Signature is not correct.");
                    }
                }
                catch (ClassCastException | PGPException e) {
                    throw new SignatureValidationException("Cannot verify direct-key signature correctness", e);
                }
            }
        };
    }

    public static SignatureValidator signatureIsCertification() {
        return SignatureValidator.signatureIsOfType(SignatureType.POSITIVE_CERTIFICATION, SignatureType.CASUAL_CERTIFICATION, SignatureType.GENERIC_CERTIFICATION, SignatureType.NO_CERTIFICATION);
    }

    public static SignatureValidator signatureIsOfType(final SignatureType ... signatureTypes) {
        return new SignatureValidator(){

            @Override
            public void verify(PGPSignature signature) throws SignatureValidationException {
                SignatureType type = SignatureType.valueOf(signature.getSignatureType());
                boolean valid = false;
                for (SignatureType allowed : signatureTypes) {
                    if (type != allowed) continue;
                    valid = true;
                    break;
                }
                if (!valid) {
                    throw new SignatureValidationException("Signature is of type " + (Object)((Object)type) + " while only " + Arrays.toString((Object[])signatureTypes) + " are allowed here.");
                }
            }
        };
    }

    public static SignatureValidator correctSignatureOverUserId(final String userId, final PGPPublicKey certifiedKey, final PGPPublicKey certifyingKey) {
        return new SignatureValidator(){

            @Override
            public void verify(PGPSignature signature) throws SignatureValidationException {
                try {
                    signature.init(ImplementationFactory.getInstance().getPGPContentVerifierBuilderProvider(), certifyingKey);
                    boolean valid = signature.verifyCertification(userId, certifiedKey);
                    if (!valid) {
                        throw new SignatureValidationException("Signature over user-id '" + userId + "' is not correct.");
                    }
                }
                catch (ClassCastException | PGPException e) {
                    throw new SignatureValidationException("Cannot verify signature over user-id '" + userId + "'.", e);
                }
            }
        };
    }

    public static SignatureValidator correctSignatureOverUserAttributes(final PGPUserAttributeSubpacketVector userAttributes, final PGPPublicKey certifiedKey, final PGPPublicKey certifyingKey) {
        return new SignatureValidator(){

            @Override
            public void verify(PGPSignature signature) throws SignatureValidationException {
                try {
                    signature.init(ImplementationFactory.getInstance().getPGPContentVerifierBuilderProvider(), certifyingKey);
                    boolean valid = signature.verifyCertification(userAttributes, certifiedKey);
                    if (!valid) {
                        throw new SignatureValidationException("Signature over user-attribute vector is not correct.");
                    }
                }
                catch (ClassCastException | PGPException e) {
                    throw new SignatureValidationException("Cannot verify signature over user-attribute vector.", e);
                }
            }
        };
    }

    public static SignatureValidator signatureWasCreatedInBounds(final Date notBefore, final Date notAfter) {
        return new SignatureValidator(){

            @Override
            public void verify(PGPSignature signature) throws SignatureValidationException {
                Date timestamp2 = signature.getCreationTime();
                if (notBefore != null && timestamp2.before(notBefore)) {
                    throw new SignatureValidationException("Signature was made before the earliest allowed signature creation time. Created: " + DateUtil.formatUTCDate(timestamp2) + " Earliest allowed: " + DateUtil.formatUTCDate(notBefore));
                }
                if (notAfter != null && timestamp2.after(notAfter)) {
                    throw new SignatureValidationException("Signature was made after the latest allowed signature creation time. Created: " + DateUtil.formatUTCDate(timestamp2) + " Latest allowed: " + DateUtil.formatUTCDate(notAfter));
                }
            }
        };
    }
}

