/*
 * Decompiled with CFR 0.152.
 */
package org.pharmgkb.parser.vcf;

import java.io.BufferedWriter;
import java.io.Closeable;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.Writer;
import java.lang.invoke.MethodHandles;
import java.nio.file.Path;
import java.util.Arrays;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import org.apache.commons.io.IOUtils;
import org.pharmgkb.parser.vcf.VcfUtils;
import org.pharmgkb.parser.vcf.model.BaseMetadata;
import org.pharmgkb.parser.vcf.model.FormatMetadata;
import org.pharmgkb.parser.vcf.model.InfoMetadata;
import org.pharmgkb.parser.vcf.model.VcfMetadata;
import org.pharmgkb.parser.vcf.model.VcfPosition;
import org.pharmgkb.parser.vcf.model.VcfSample;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class VcfWriter
implements Closeable {
    private static final Logger sf_logger = LoggerFactory.getLogger(MethodHandles.lookup().lookupClass());
    private final Path m_file;
    private final PrintWriter m_writer;
    private int m_lineNumber;

    private VcfWriter(@Nullable Path file, @Nonnull PrintWriter writer) {
        this.m_file = file;
        this.m_writer = writer;
    }

    public void writeHeader(@Nonnull VcfMetadata metadata) {
        this.printLine("##fileformat=" + metadata.getFileFormat());
        this.printLines("INFO", metadata.getInfo().values());
        this.printLines("FILTER", metadata.getFilters().values());
        this.printLines("FORMAT", metadata.getFormats().values());
        this.printLines("ALT", metadata.getAlts().values());
        this.printLines("contig", metadata.getContigs().values());
        this.printLines("SAMPLE", metadata.getSamples().values());
        this.printLines("PEDIGREE", metadata.getPedigrees());
        for (String key : metadata.getRawPropertyKeys()) {
            this.printPropertyLines(key, metadata.getRawValuesOfProperty(key));
        }
        StringBuilder sb = new StringBuilder("#CHROM\tPOS\tID\tREF\tALT\tQUAL\tFILTER\tINFO");
        if (metadata.getNumSamples() > 0) {
            sb.append("\tFORMAT");
        }
        for (int i = 0; i < metadata.getNumSamples(); ++i) {
            sb.append("\t").append(metadata.getSampleName(i));
        }
        this.printLine(sb);
        this.m_writer.flush();
        sf_logger.info("Wrote {} lines of header{}", (Object)this.m_lineNumber, (Object)(this.m_file == null ? "" : " to " + this.m_file));
    }

    public void writeLine(@Nonnull VcfMetadata metadata, @Nonnull VcfPosition position, @Nonnull List<VcfSample> samples) {
        StringBuilder sb = new StringBuilder();
        sb.append(position.getChromosome()).append("\t");
        sb.append(position.getPosition()).append("\t");
        this.addListOrElse(position.getIds(), ";", ".", sb);
        if (position.getRef().isEmpty()) {
            sf_logger.warn("No REF bases, but the column is required (on line {})", (Object)this.m_lineNumber);
        }
        this.addListOrElse(Arrays.asList(position.getRef()), ",", ".", sb);
        this.addListOrElse(position.getAltBases(), ",", ".", sb);
        this.addStringOrElse(position.getQuality(), ".", sb);
        this.addListOrElse(position.getFilters(), ";", "PASS", sb);
        this.addInfoOrDot(metadata, position, sb);
        position.getFilters().stream().filter(key -> !metadata.getFilters().containsKey(key)).forEach(key -> {
            if (key.equals(".")) {
                sf_logger.warn("Position {}:{} has FILTER {}; the absence of a filter should instead be marked with PASS (on line {})", new Object[]{position.getChromosome(), position.getPosition(), key, this.m_lineNumber});
            } else {
                sf_logger.warn("Position {}:{} has FILTER {}, but there is no FILTER metadata with that name (on line {})", new Object[]{position.getChromosome(), position.getPosition(), key, this.m_lineNumber});
            }
        });
        this.addFormatConditionally(position, sb);
        int sampleIndex = 0;
        for (VcfSample sample : samples) {
            this.addSampleConditionally(metadata, sampleIndex, position, sample, sb);
            ++sampleIndex;
        }
        String line = sb.toString();
        if (line.endsWith("\t")) {
            line = line.substring(0, line.length() - 1);
        }
        this.printLine(line);
        this.m_writer.flush();
    }

    @Override
    public void close() {
        IOUtils.closeQuietly((Writer)this.m_writer);
    }

    private void addFormatConditionally(@Nonnull VcfPosition position, @Nonnull StringBuilder sb) {
        Iterator<String> formats = position.getFormat().iterator();
        if (!formats.hasNext()) {
            return;
        }
        while (formats.hasNext()) {
            sb.append(formats.next());
            if (!formats.hasNext()) continue;
            sb.append(":");
        }
        sb.append("\t");
    }

    private void addSampleConditionally(@Nonnull VcfMetadata metadata, int sampleIndex, @Nonnull VcfPosition position, @Nonnull VcfSample sample, @Nonnull StringBuilder sb) {
        Iterator<String> keys = sample.getPropertyKeys().iterator();
        if (!keys.hasNext() && position.getFormat().isEmpty()) {
            return;
        }
        for (String key2 : position.getFormat()) {
            keys.next();
            if (!metadata.getFormats().containsKey(key2)) {
                sf_logger.warn("Sample #{} for {}:{} contains FORMAT {}, but there is no FORMAT metadata with that name (on line {})", new Object[]{sampleIndex, position.getChromosome(), position.getPosition(), key2, this.m_lineNumber});
            }
            if (!sample.containsProperty(key2)) {
                sf_logger.warn("Sample #{} is missing property {} (on line {})", new Object[]{sampleIndex, key2, this.m_lineNumber});
            }
            String value = sample.getProperty(key2);
            FormatMetadata format = metadata.getFormats().get(key2);
            Integer number = null;
            try {
                number = Integer.parseInt(format.getNumber());
            }
            catch (NumberFormatException numberFormatException) {
                // empty catch block
            }
            if (number != null && number == 1) {
                try {
                    VcfUtils.convertProperty(format.getType(), value);
                }
                catch (IllegalArgumentException e) {
                    sf_logger.warn("Property {} for sample #{} is not of type {} (on line {})", new Object[]{key2, sampleIndex, format.getType(), this.m_lineNumber});
                }
            }
            sb.append(value);
            if (!keys.hasNext()) continue;
            sb.append(":");
        }
        sample.getPropertyKeys().stream().filter(key -> !position.getFormat().contains(key)).forEach(key -> sf_logger.warn("Sample #{} contains extra property {} (on line {})", new Object[]{sampleIndex, key, this.m_lineNumber}));
        sb.append("\t");
    }

    private void addInfoOrDot(@Nonnull VcfMetadata metadata, @Nonnull VcfPosition position, @Nonnull StringBuilder sb) {
        Iterator<String> keys = position.getInfoKeys().iterator();
        if (!keys.hasNext()) {
            sb.append(".");
        }
        while (keys.hasNext()) {
            String key = keys.next();
            List<String> values = position.getInfo(key);
            assert (values != null);
            if (!metadata.getInfo().containsKey(key)) {
                sf_logger.warn("Position {}:{} contains INFO {}, but there is no INFO metadata with that name (on line {})", new Object[]{position.getChromosome(), position.getPosition(), key, this.m_lineNumber});
            } else {
                InfoMetadata info = metadata.getInfo().get(key);
                for (String value : values) {
                    Integer number = null;
                    try {
                        number = Integer.parseInt(info.getNumber());
                    }
                    catch (NumberFormatException numberFormatException) {
                        // empty catch block
                    }
                    if (number == null || number != 1) continue;
                    try {
                        VcfUtils.convertProperty(info.getType(), value);
                    }
                    catch (IllegalArgumentException e) {
                        sf_logger.warn("Property {} is not of type {} (on line {})", new Object[]{key, info.getType(), this.m_lineNumber});
                    }
                }
            }
            sb.append(key);
            if (!(values.isEmpty() || values.size() == 1 && values.get(0).isEmpty())) {
                sb.append("=").append(values.get(0));
                for (int i = 1; i < values.size(); ++i) {
                    sb.append(",").append(values.get(i));
                }
            }
            if (!keys.hasNext()) continue;
            sb.append(";");
        }
        sb.append("\t");
    }

    private void addStringOrElse(@Nullable Object object, @Nonnull String missingValue, @Nonnull StringBuilder sb) {
        if (object == null || object.toString().isEmpty()) {
            sb.append(missingValue);
        } else {
            sb.append(object.toString());
        }
        sb.append("\t");
    }

    private void addListOrElse(@Nonnull List<String> list, @Nonnull String delimiter, @Nonnull String missingValue, @Nonnull StringBuilder sb) {
        if (list.isEmpty()) {
            sb.append(missingValue);
        } else {
            sb.append(list.get(0));
            for (int i = 1; i < list.size(); ++i) {
                sb.append(delimiter).append(list.get(i));
            }
        }
        sb.append("\t");
    }

    private void printPropertyLines(@Nonnull String name, @Nonnull Collection<String> list) {
        for (String string : list) {
            this.printLine("##" + name + "=" + string);
        }
    }

    private void printLines(@Nonnull String name, @Nonnull Collection<? extends BaseMetadata> list) {
        for (BaseMetadata baseMetadata : list) {
            this.printLine(this.getAllProperties(name, baseMetadata));
        }
    }

    private String getAllProperties(@Nonnull String name, @Nonnull BaseMetadata metadata) {
        StringBuilder sb = new StringBuilder("##");
        sb.append(name).append("=<");
        int i = 0;
        for (Map.Entry<String, String> entry : metadata.getPropertiesRaw().entrySet()) {
            if (i > 0) {
                sb.append(",");
            }
            sb.append(entry.getKey()).append("=").append(entry.getValue());
            ++i;
        }
        sb.append(">");
        return sb.toString();
    }

    private void printLine(@Nonnull Object line) {
        String string = line.toString();
        if (string.contains("\n")) {
            throw new RuntimeException("Something went wrong writing line #" + this.m_lineNumber + ": [[[" + string + "]]] contains more than one line");
        }
        this.m_writer.println(line);
        ++this.m_lineNumber;
        if (this.m_lineNumber % 1000 == 0) {
            sf_logger.info("Wrote {} lines{}", (Object)this.m_lineNumber, (Object)(this.m_file == null ? "" : " to " + this.m_file));
        }
    }

    public static class Builder {
        private Path m_file;
        private PrintWriter m_writer;

        public Builder toFile(Path file) {
            this.m_file = file;
            return this;
        }

        public Builder toWriter(PrintWriter writer) {
            this.m_writer = writer;
            return this;
        }

        public VcfWriter build() throws IOException {
            if (this.m_file != null) {
                this.m_writer = new PrintWriter(new BufferedWriter(new FileWriter(this.m_file.toFile()), 65536));
            }
            if (this.m_writer == null) {
                throw new IllegalStateException("Must specify either file or writer");
            }
            return new VcfWriter(this.m_file, this.m_writer);
        }
    }
}

