/*
 * Decompiled with CFR 0.152.
 */
package org.pharmgkb.parser.vcf;

import com.google.common.base.Preconditions;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.ListMultimap;
import java.io.BufferedReader;
import java.io.Closeable;
import java.io.IOException;
import java.lang.invoke.MethodHandles;
import java.math.BigDecimal;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import org.apache.commons.lang3.StringUtils;
import org.pharmgkb.parser.vcf.VcfFormatException;
import org.pharmgkb.parser.vcf.VcfLineParser;
import org.pharmgkb.parser.vcf.VcfUtils;
import org.pharmgkb.parser.vcf.model.BaseMetadata;
import org.pharmgkb.parser.vcf.model.ContigMetadata;
import org.pharmgkb.parser.vcf.model.FormatMetadata;
import org.pharmgkb.parser.vcf.model.IdDescriptionMetadata;
import org.pharmgkb.parser.vcf.model.InfoMetadata;
import org.pharmgkb.parser.vcf.model.VcfMetadata;
import org.pharmgkb.parser.vcf.model.VcfPosition;
import org.pharmgkb.parser.vcf.model.VcfSample;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class VcfParser
implements Closeable {
    private static final Logger sf_logger = LoggerFactory.getLogger(MethodHandles.lookup().lookupClass());
    private static final Pattern sf_tabSplitter = Pattern.compile("\t");
    private static final Pattern sf_commaSplitter = Pattern.compile(",");
    private static final Pattern sf_colonSplitter = Pattern.compile(":");
    private static final Pattern sf_semicolonSplitter = Pattern.compile(";");
    private final boolean m_rsidsOnly;
    private final BufferedReader m_reader;
    private VcfMetadata m_vcfMetadata;
    private final VcfLineParser m_vcfLineParser;
    private int m_lineNumber;
    private boolean m_alreadyFinished;

    private VcfParser(@Nonnull BufferedReader reader, boolean rsidsOnly, @Nonnull VcfLineParser lineParser) {
        this.m_reader = reader;
        this.m_rsidsOnly = rsidsOnly;
        this.m_vcfLineParser = lineParser;
    }

    @Nonnull
    public VcfMetadata parseMetadata() throws IOException {
        String line;
        if (this.m_vcfMetadata != null) {
            throw new IllegalStateException("Metadata has already been parsed.");
        }
        VcfMetadata.Builder mdBuilder = new VcfMetadata.Builder();
        while ((line = this.m_reader.readLine()) != null) {
            ++this.m_lineNumber;
            if (line.startsWith("##")) {
                try {
                    this.parseMetadata(mdBuilder, line);
                    continue;
                }
                catch (VcfFormatException ex) {
                    ex.addMetadata(this.m_lineNumber, "metadata");
                    throw ex;
                }
                catch (RuntimeException e) {
                    throw new VcfFormatException(this.m_lineNumber, "metadata", e);
                }
            }
            if (!line.startsWith("#")) continue;
            try {
                this.parseColumnInfo(mdBuilder, line);
                break;
            }
            catch (VcfFormatException ex) {
                ex.addMetadata(this.m_lineNumber, "column (# header)");
                throw ex;
            }
            catch (RuntimeException e) {
                throw new VcfFormatException(this.m_lineNumber, "column (# header)", e);
            }
        }
        this.m_vcfMetadata = mdBuilder.build();
        if (this.m_vcfMetadata.getNumSamples() == this.m_vcfMetadata.getSamples().size()) {
            for (int i = 0; i < this.m_vcfMetadata.getNumSamples(); ++i) {
                String sampleName = this.m_vcfMetadata.getSampleName(i);
                if (this.m_vcfMetadata.getSamples().containsKey(sampleName)) continue;
                sf_logger.warn("Sample {} is missing in the metadata", (Object)sampleName);
            }
        } else {
            sf_logger.warn("There are {} samples in the header but {} in the metadata", (Object)this.m_vcfMetadata.getNumSamples(), (Object)this.m_vcfMetadata.getSamples().size());
        }
        return this.m_vcfMetadata;
    }

    @Nullable
    public VcfMetadata getMetadata() {
        return this.m_vcfMetadata;
    }

    public void parse() throws IOException {
        boolean hasNext = true;
        while (hasNext) {
            hasNext = this.parseNextLine();
        }
    }

    public boolean parseNextLine() throws IOException, VcfFormatException {
        String line;
        if (this.m_alreadyFinished) {
            throw new IllegalStateException("Already finished reading the stream");
        }
        if (this.m_vcfMetadata == null) {
            this.parseMetadata();
        }
        if ((line = this.m_reader.readLine()) == null) {
            this.m_alreadyFinished = true;
            return false;
        }
        ++this.m_lineNumber;
        if (line.startsWith("#")) {
            return true;
        }
        try {
            long position;
            if (StringUtils.stripToNull((String)line) == null) {
                throw new VcfFormatException("Empty line", this.m_lineNumber);
            }
            List<String> data = this.toList(sf_tabSplitter, line);
            if (data.size() != this.m_vcfMetadata.getNumColumns()) {
                throw new VcfFormatException("Data line does not have expected number of columns (got " + data.size() + " vs. " + this.m_vcfMetadata.getNumColumns() + ")", this.m_lineNumber);
            }
            String chromosome = data.get(0);
            try {
                position = Long.parseLong(data.get(1));
            }
            catch (NumberFormatException e) {
                throw new VcfFormatException("POS '" + data.get(1) + "' is not a number");
            }
            List<String> ids = null;
            if (!data.get(2).equals(".")) {
                if (this.m_rsidsOnly && !VcfUtils.RSID_PATTERN.matcher(data.get(2)).find()) {
                    return true;
                }
                ids = this.toList(sf_semicolonSplitter, data.get(2));
            } else if (this.m_rsidsOnly) {
                return true;
            }
            String ref = data.get(3);
            List<String> alt = null;
            if (!data.get(7).isEmpty() && !data.get(4).equals(".")) {
                alt = this.toList(sf_commaSplitter, data.get(4));
            }
            BigDecimal quality = null;
            if (!data.get(5).isEmpty() && !data.get(5).equals(".")) {
                try {
                    quality = new BigDecimal(data.get(5));
                }
                catch (NumberFormatException e) {
                    throw new VcfFormatException("QUAL '" + data.get(5) + "' is not a number");
                }
            }
            List<String> filters = null;
            if (!data.get(6).equals("PASS")) {
                filters = this.toList(sf_semicolonSplitter, data.get(6));
            }
            ArrayListMultimap info = null;
            if (!data.get(7).equals("") && !data.get(7).equals(".")) {
                info = ArrayListMultimap.create();
                List<String> props = this.toList(sf_semicolonSplitter, data.get(7));
                for (String prop : props) {
                    int idx = prop.indexOf(61);
                    if (idx == -1) {
                        info.put((Object)prop, (Object)"");
                        continue;
                    }
                    String key = prop.substring(0, idx);
                    String value = prop.substring(idx + 1);
                    info.putAll((Object)key, this.toList(sf_commaSplitter, value));
                }
            }
            List<String> format = null;
            if (data.size() >= 9 && data.get(8) != null) {
                format = this.toList(sf_colonSplitter, data.get(8));
            }
            VcfPosition pos = new VcfPosition(chromosome, position, ids, ref, alt, quality, filters, (ListMultimap<String, String>)info, format);
            ArrayList<VcfSample> samples = new ArrayList<VcfSample>();
            for (int x = 9; x < data.size(); ++x) {
                samples.add(new VcfSample(format, this.toList(sf_colonSplitter, data.get(x))));
            }
            this.m_vcfLineParser.parseLine(this.m_vcfMetadata, pos, samples);
            return true;
        }
        catch (VcfFormatException ex) {
            ex.addMetadata(this.m_lineNumber, "data");
            throw ex;
        }
        catch (RuntimeException e) {
            throw new VcfFormatException(this.m_lineNumber, "data", e);
        }
    }

    @Override
    public void close() {
        try {
            this.m_reader.close();
        }
        catch (Exception ex) {
            sf_logger.info("Error closing reader", (Throwable)ex);
        }
    }

    private void parseMetadata(@Nonnull VcfMetadata.Builder mdBuilder, @Nonnull String line) {
        int idx = line.indexOf("=");
        String propName = line.substring(2, idx).trim();
        String propValue = line.substring(idx + 1).trim();
        sf_logger.debug("{} : {}", (Object)propName, (Object)propValue);
        switch (propName) {
            case "fileformat": {
                mdBuilder.setFileFormat(propValue);
                break;
            }
            case "ALT": 
            case "FILTER": 
            case "INFO": 
            case "FORMAT": 
            case "contig": 
            case "SAMPLE": 
            case "PEDIGREE": {
                this.parseMetadataProperty(mdBuilder, propName, VcfParser.removeAngleBrackets(propValue));
                break;
            }
            default: {
                mdBuilder.addRawProperty(propName, propValue);
            }
        }
    }

    @Nonnull
    private static String removeAngleBrackets(@Nonnull String string) throws VcfFormatException {
        if (string.startsWith("<") && string.endsWith(">")) {
            return string.substring(1, string.length() - 1);
        }
        throw new VcfFormatException("Angle brackets not present for: '" + string + "'");
    }

    private void parseMetadataProperty(@Nonnull VcfMetadata.Builder mdBuilder, @Nonnull String propName, @Nonnull String value) {
        Map<String, String> props = VcfUtils.extractPropertiesFromLine(value);
        switch (propName.toLowerCase()) {
            case "alt": {
                mdBuilder.addAlt(new IdDescriptionMetadata(props, true));
                break;
            }
            case "filter": {
                mdBuilder.addFilter(new IdDescriptionMetadata(props, true));
                break;
            }
            case "info": {
                mdBuilder.addInfo(new InfoMetadata(props));
                break;
            }
            case "format": {
                mdBuilder.addFormat(new FormatMetadata(props));
                break;
            }
            case "contig": {
                mdBuilder.addContig(new ContigMetadata(props));
                break;
            }
            case "sample": {
                mdBuilder.addSample(new IdDescriptionMetadata(props, true));
                break;
            }
            case "pedigree": {
                mdBuilder.addPedigree(new BaseMetadata(props));
            }
        }
    }

    @Nonnull
    private List<String> toList(@Nonnull Pattern pattern, @Nullable String string) {
        String[] array = pattern.split(string);
        ArrayList<String> list = new ArrayList<String>(array.length);
        Collections.addAll(list, array);
        return list;
    }

    public int getLineNumber() {
        return this.m_lineNumber;
    }

    private void parseColumnInfo(@Nonnull VcfMetadata.Builder mdBuilder, @Nonnull String line) {
        List<String> cols = Arrays.asList(sf_tabSplitter.split(line));
        if (cols.size() < 8) {
            throw new VcfFormatException("Header line does not have mandatory (tab-delimited) columns", this.m_lineNumber);
        }
        mdBuilder.setColumns(cols);
    }

    public static class Builder {
        private BufferedReader m_reader;
        private Path m_vcfFile;
        private boolean m_rsidsOnly;
        private VcfLineParser m_vcfLineParser;

        public Builder fromFile(@Nonnull Path dataFile) {
            Preconditions.checkNotNull((Object)dataFile);
            if (this.m_reader != null) {
                throw new IllegalStateException("Already loading from reader");
            }
            if (!dataFile.toString().endsWith(".vcf")) {
                throw new IllegalArgumentException("Not a VCF file (doesn't end with .vcf extension");
            }
            this.m_vcfFile = dataFile;
            return this;
        }

        public Builder fromReader(@Nonnull BufferedReader reader) {
            Preconditions.checkNotNull((Object)reader);
            if (this.m_vcfFile != null) {
                throw new IllegalStateException("Already loading from file");
            }
            this.m_reader = reader;
            return this;
        }

        public Builder rsidsOnly() {
            this.m_rsidsOnly = true;
            return this;
        }

        public Builder parseWith(@Nonnull VcfLineParser lineParser) {
            Preconditions.checkNotNull((Object)lineParser);
            this.m_vcfLineParser = lineParser;
            return this;
        }

        public VcfParser build() throws IOException {
            if (this.m_vcfLineParser == null) {
                throw new IllegalStateException("Missing VcfLineParser");
            }
            if (this.m_vcfFile != null) {
                this.m_reader = Files.newBufferedReader(this.m_vcfFile);
            }
            if (this.m_reader == null) {
                throw new IllegalStateException("Must specify either file or reader to parse");
            }
            return new VcfParser(this.m_reader, this.m_rsidsOnly, this.m_vcfLineParser);
        }
    }
}

