/*
 * Decompiled with CFR 0.152.
 */
package org.pipecraft.infra.storage.amazon_s3;

import com.amazonaws.AmazonClientException;
import com.amazonaws.AmazonServiceException;
import com.amazonaws.services.kms.model.KeyUnavailableException;
import com.amazonaws.services.s3.AmazonS3;
import com.amazonaws.services.s3.model.CannedAccessControlList;
import com.amazonaws.services.s3.model.CopyObjectRequest;
import com.amazonaws.services.s3.model.DeleteObjectRequest;
import com.amazonaws.services.s3.model.GetObjectRequest;
import com.amazonaws.services.s3.model.ListObjectsRequest;
import com.amazonaws.services.s3.model.ObjectListing;
import com.amazonaws.services.s3.model.ObjectMetadata;
import com.amazonaws.services.s3.model.PutObjectRequest;
import com.amazonaws.services.s3.model.S3Object;
import com.amazonaws.services.s3.model.S3ObjectSummary;
import com.google.common.collect.Iterators;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.UncheckedIOException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.NoSuchElementException;
import org.pipecraft.infra.io.SizedInputStream;
import org.pipecraft.infra.storage.Bucket;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class S3Bucket
extends Bucket<S3ObjectSummary> {
    private final AmazonS3 s3;
    private static final Logger logger = LoggerFactory.getLogger(S3Bucket.class);

    S3Bucket(AmazonS3 connection, String bucketName) {
        super(bucketName);
        this.s3 = connection;
    }

    public void put(String key, InputStream input, long length, String contentType, boolean isPublic, boolean allowOverride) throws IOException {
        if (!allowOverride) {
            throw new UnsupportedOperationException();
        }
        this.validateNotFolderPath(key);
        ObjectMetadata meta = new ObjectMetadata();
        meta.setContentLength(length);
        meta.setContentType(contentType);
        PutObjectRequest putRequest = new PutObjectRequest(this.getBucketName(), key, input, meta);
        if (isPublic) {
            putRequest.setCannedAcl(CannedAccessControlList.PublicRead);
        }
        try {
            this.s3.putObject(putRequest);
        }
        catch (AmazonClientException e) {
            throw S3Bucket.mapToIOException(S3Bucket.extractAmazonServiceException(e), "Failed uploading '" + this.getBucketName() + "/" + key + "'");
        }
    }

    public OutputStream getOutputStream(String key, int chunkSize) throws IOException {
        throw new UnsupportedOperationException();
    }

    public void copyToAnotherBucket(String fromKey, String toBucket, String toKey) throws IOException {
        try {
            CopyObjectRequest copyObjRequest = new CopyObjectRequest(this.getBucketName(), fromKey, toBucket, toKey);
            this.s3.copyObject(copyObjRequest);
        }
        catch (AmazonClientException e) {
            throw S3Bucket.mapToIOException(S3Bucket.extractAmazonServiceException(e), "Failed copying '" + this.getBucketName() + "/" + fromKey + "' to '" + toBucket + "/" + toKey + "'");
        }
    }

    public void delete(S3ObjectSummary obj) throws IOException {
        try {
            DeleteObjectRequest deleteObjectRequest = new DeleteObjectRequest(this.getBucketName(), obj.getKey());
            this.s3.deleteObject(deleteObjectRequest);
        }
        catch (AmazonClientException e) {
            throw S3Bucket.mapToIOException(S3Bucket.extractAmazonServiceException(e), "Failed deleting '" + this.getBucketName() + "/" + obj.getKey() + "'");
        }
    }

    public void get(S3ObjectSummary meta, File output) throws IOException {
        try {
            GetObjectRequest req = new GetObjectRequest(this.getBucketName(), meta.getKey());
            this.s3.getObject(req, output);
        }
        catch (KeyUnavailableException e) {
            throw new FileNotFoundException("Key " + meta.getKey() + " not found in bucket " + this.getBucketName());
        }
        catch (AmazonClientException e) {
            if (!output.delete()) {
                logger.error("Cannot delete the malformed file from a failed download.");
            }
            throw S3Bucket.mapToIOException(S3Bucket.extractAmazonServiceException(e), "Failed downloading '" + this.getBucketName() + "/" + meta.getKey() + "'");
        }
    }

    public SizedInputStream getAsStream(S3ObjectSummary meta, int chunkSize) throws IOException {
        S3Object obj;
        GetObjectRequest req = new GetObjectRequest(this.getBucketName(), meta.getKey());
        try {
            obj = this.s3.getObject(req);
        }
        catch (AmazonClientException ace) {
            throw S3Bucket.mapToIOException(S3Bucket.extractAmazonServiceException(ace), "Failed getting input stream for '" + this.getBucketName() + "/" + meta.getKey() + "'");
        }
        return new SizedInputStream((InputStream)obj.getObjectContent(), Long.valueOf(obj.getObjectMetadata().getContentLength()));
    }

    public Iterator<S3ObjectSummary> listObjects(String folderPath, boolean recursive) throws IOException {
        String normalizedPath = this.normalizeFolderPath(folderPath);
        return Iterators.filter((Iterator)new S3ObjectIterator(normalizedPath, recursive), v -> !v.getKey().equals(normalizedPath));
    }

    public URL generateSignedUrl(String key, String contentType, int expirationSeconds, boolean isPublicRead) {
        throw new UnsupportedOperationException();
    }

    public URL generateReadOnlyUrl(String key, int expirationSeconds) {
        throw new UnsupportedOperationException();
    }

    public URL generateResumableSignedUrlForUpload(String key, String contentType, int expirationSeconds, Long maxContentLengthInBytes, boolean isPublic) {
        throw new UnsupportedOperationException();
    }

    public S3ObjectSummary getObjectMetadata(String key) throws IOException {
        if (this.isFolderPath(key)) {
            throw new FileNotFoundException("File not found: '" + this.getBucketName() + "/" + key + "'");
        }
        try {
            ObjectMetadata meta = this.s3.getObjectMetadata(this.getBucketName(), key);
            S3ObjectSummary res = new S3ObjectSummary();
            res.setBucketName(this.getBucketName());
            res.setKey(key);
            res.setETag(meta.getETag());
            res.setLastModified(meta.getLastModified());
            res.setSize(meta.getContentLength());
            res.setStorageClass(meta.getStorageClass());
            return res;
        }
        catch (AmazonClientException ace) {
            throw S3Bucket.mapToIOException(S3Bucket.extractAmazonServiceException(ace), "Failed getting metadata of '" + this.getBucketName() + "/" + key + "'");
        }
    }

    public String getPath(S3ObjectSummary keyMetadata) {
        if (!keyMetadata.getBucketName().equals(this.getBucketName())) {
            throw new IllegalArgumentException("The given meta object belongs to a different bucket ('" + keyMetadata.getBucketName() + "')");
        }
        return keyMetadata.getKey();
    }

    public long getLength(S3ObjectSummary keyMetadata) {
        return keyMetadata.getSize();
    }

    public Long getLastUpdated(S3ObjectSummary objMetadata) {
        Date lastModified = objMetadata.getLastModified();
        return lastModified == null ? null : Long.valueOf(lastModified.getTime());
    }

    public boolean exists(String key) throws IOException {
        try {
            return this.s3.doesObjectExist(this.getBucketName(), key) && this.isFilePath(key);
        }
        catch (AmazonClientException ace) {
            throw S3Bucket.mapToIOException(S3Bucket.extractAmazonServiceException(ace), "Failed checking existence of '" + this.getBucketName() + "/" + key + "'");
        }
    }

    public S3ObjectSummary compose(List<String> gsPaths, String composedFilePath, boolean removeComprisingFiles) throws IOException {
        throw new UnsupportedOperationException();
    }

    private static AmazonServiceException extractAmazonServiceException(Throwable e) {
        while (!(e instanceof AmazonServiceException)) {
            if ((e = e.getCause()) != null) continue;
            return null;
        }
        return (AmazonServiceException)e;
    }

    private static IOException mapToIOException(AmazonServiceException e, String msg) {
        if (e == null) {
            return new IOException(msg);
        }
        switch (e.getErrorCode()) {
            case "404 Not Found": {
                return new FileNotFoundException(msg + ". Remote file not found.");
            }
        }
        return new IOException(msg, (Throwable)e);
    }

    private class S3ObjectIterator
    implements Iterator<S3ObjectSummary> {
        private ObjectListing page;
        private ListObjectsRequest req;
        private List<S3ObjectSummary> currList;
        private int currIndexInList;

        public S3ObjectIterator(String prefix, boolean recursive) throws IOException {
            try {
                this.req = new ListObjectsRequest().withBucketName(S3Bucket.this.getBucketName()).withPrefix(prefix);
                if (!recursive) {
                    this.req.withDelimiter("/");
                }
                this.page = S3Bucket.this.s3.listObjects(this.req);
                this.currList = this.buildList(this.page);
            }
            catch (AmazonClientException ace) {
                throw S3Bucket.mapToIOException(S3Bucket.extractAmazonServiceException(ace), "");
            }
        }

        @Override
        public boolean hasNext() {
            while (this.page.getNextMarker() != null || this.currIndexInList < this.currList.size()) {
                if (this.currIndexInList < this.currList.size()) {
                    return true;
                }
                this.fetchNextPage();
            }
            return false;
        }

        @Override
        public S3ObjectSummary next() {
            if (!this.hasNext()) {
                throw new NoSuchElementException();
            }
            if (this.currList.size() == this.currIndexInList) {
                this.fetchNextPage();
            }
            return this.currList.get(this.currIndexInList++);
        }

        private void fetchNextPage() {
            try {
                this.req.setMarker(this.page.getNextMarker());
                this.page = S3Bucket.this.s3.listObjects(this.req);
                this.currList = this.buildList(this.page);
                this.currIndexInList = 0;
            }
            catch (AmazonClientException e) {
                throw new UncheckedIOException(S3Bucket.mapToIOException(S3Bucket.extractAmazonServiceException(e), "IO failure during remote files iteration"));
            }
        }

        private List<S3ObjectSummary> buildList(ObjectListing page) {
            List fileSummaries = page.getObjectSummaries();
            ArrayList<S3ObjectSummary> res = new ArrayList<S3ObjectSummary>(fileSummaries.size() + page.getCommonPrefixes().size());
            res.addAll(fileSummaries);
            for (String subFolderPath : page.getCommonPrefixes()) {
                S3ObjectSummary subFolderSummary = new S3ObjectSummary();
                subFolderSummary.setBucketName(S3Bucket.this.getBucketName());
                subFolderSummary.setKey(subFolderPath);
                res.add(subFolderSummary);
            }
            return res;
        }
    }
}

