/*
 * Decompiled with CFR 0.152.
 */
package org.pipecraft.infra.storage.google_cs;

import com.google.auth.ServiceAccountSigner;
import com.google.cloud.BatchResult;
import com.google.cloud.ReadChannel;
import com.google.cloud.RetryHelper;
import com.google.cloud.WriteChannel;
import com.google.cloud.storage.Acl;
import com.google.cloud.storage.Blob;
import com.google.cloud.storage.BlobId;
import com.google.cloud.storage.BlobInfo;
import com.google.cloud.storage.HttpMethod;
import com.google.cloud.storage.Storage;
import com.google.cloud.storage.StorageBatch;
import com.google.cloud.storage.StorageBatchResult;
import com.google.cloud.storage.StorageException;
import com.google.common.collect.Iterators;
import com.google.common.collect.UnmodifiableIterator;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.io.Closeable;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.InterruptedIOException;
import java.io.OutputStream;
import java.net.URL;
import java.nio.ByteBuffer;
import java.nio.channels.Channels;
import java.nio.channels.FileChannel;
import java.nio.channels.ReadableByteChannel;
import java.nio.channels.WritableByteChannel;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Vector;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import org.apache.commons.lang3.mutable.MutableObject;
import org.pipecraft.infra.concurrent.FailableInterruptibleConsumer;
import org.pipecraft.infra.concurrent.ParallelTaskProcessor;
import org.pipecraft.infra.io.FileUtils;
import org.pipecraft.infra.io.Retrier;
import org.pipecraft.infra.io.SizedInputStream;
import org.pipecraft.infra.storage.Bucket;
import org.pipecraft.infra.storage.IllegalJsonException;
import org.pipecraft.infra.storage.google_cs.GSInputStream;
import org.pipecraft.infra.storage.google_cs.GSOutputStream;
import org.pipecraft.infra.storage.google_cs.SliceTransferJobDetails;
import org.pipecraft.infra.storage.google_cs.SlicedTransferFileHandler;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class GoogleStorageBucket
extends Bucket<Blob> {
    private static final Logger LOGGER = LoggerFactory.getLogger(GoogleStorageBucket.class);
    public static final String X_GOOG_ACL_HEADER = "x-goog-acl";
    public static final String ACL_PUBLIC_READ = "public-read";
    public static final String X_GOOG_RESUMABLE_HEADER = "x-goog-resumable";
    public static final String X_GOOG_CONTENT_LENGTH_RANGE = "x-goog-content-length-range";
    private static final Storage.BlobWriteOption[] BLOB_WRITE_OPTION_DOES_NOT_EXIST = new Storage.BlobWriteOption[]{Storage.BlobWriteOption.doesNotExist()};
    private static final Storage.BlobWriteOption[] BLOB_WRITE_OPTIONS_EMPTY = new Storage.BlobWriteOption[0];
    private static final long MAX_EXPIRATION_SECONDS = TimeUnit.DAYS.toSeconds(7L);
    private static final int COMPOSE_FILE_COUNT_LIMIT = 32;
    private static final int COMPOSE_MAX_PARALLELISM = 16;
    private static final int MEGABYTE = 0x100000;
    private static final int SLICED_DOWNLOAD_DEFAULT_CHUNK_SIZE = 0x200000;
    private static final int SLICED_DOWNLOAD_DEFAULT_PARALLELISM = 20;
    private static final int SLICED_DOWNLOAD_SLICE_SIZE = 0x1400000;
    private static final int BUFFER_SIZE = 65536;
    private static final List<Acl> PUBLIC_READ_ACL = Collections.singletonList(Acl.of((Acl.Entity)Acl.User.ofAllUsers(), (Acl.Role)Acl.Role.READER));
    private final Storage storage;

    GoogleStorageBucket(Storage storage, String bucketName) {
        super(bucketName);
        this.storage = storage;
    }

    public void put(String key, InputStream input, long length, String contentType, boolean isPublic, boolean allowOverride) throws IOException {
        try (InputStream is = input;){
            this.validateNotFolderPath(key);
            BlobInfo.Builder blobInfo = BlobInfo.newBuilder((String)this.getBucketName(), (String)key);
            if (contentType != null) {
                blobInfo.setContentType(contentType);
            }
            if (isPublic) {
                blobInfo.setAcl(PUBLIC_READ_ACL);
            }
            BlobInfo blob = blobInfo.build();
            Storage.BlobWriteOption[] blobTargetOptions = allowOverride ? BLOB_WRITE_OPTIONS_EMPTY : BLOB_WRITE_OPTION_DOES_NOT_EXIST;
            try (WriteChannel writer = this.storage.writer(blob, blobTargetOptions);){
                int read;
                byte[] buffer = new byte[65536];
                while ((read = is.read(buffer)) >= 0) {
                    writer.write(ByteBuffer.wrap(buffer, 0, read));
                }
            }
        }
        catch (StorageException se) {
            throw GoogleStorageBucket.mapToIOException(se, "Failed uploading to " + this.getBucketName() + "/" + key);
        }
    }

    public OutputStream getOutputStream(String key, int chunkSize) throws IOException {
        WriteChannel writer;
        this.validateNotFolderPath(key);
        try {
            BlobInfo blobInfo = BlobInfo.newBuilder((String)this.getBucketName(), (String)key).build();
            writer = this.storage.writer(blobInfo, new Storage.BlobWriteOption[0]);
            if (chunkSize > 0) {
                writer.setChunkSize(chunkSize);
            }
        }
        catch (StorageException se) {
            throw GoogleStorageBucket.mapToIOException(se, "Failed getting output stream for '" + this.getBucketName() + "/" + key + "'");
        }
        return new GSOutputStream(Channels.newOutputStream((WritableByteChannel)writer));
    }

    public void get(Blob meta, File output) throws IOException {
        Path path = output.toPath();
        try {
            meta.downloadTo(path);
        }
        catch (RetryHelper.RetryHelperException | StorageException e) {
            throw GoogleStorageBucket.mapToIOException(GoogleStorageBucket.extractStorageException(e), "Failed getting " + this.getBucketName() + "/" + meta.getName());
        }
    }

    public void get(ExecutorService ex, String key, File output, int chunkSize) throws IOException, InterruptedException {
        ArrayList<SliceTransferJobDetails> sliceJobs = new ArrayList<SliceTransferJobDetails>();
        try {
            Blob blob = this.getObjectMetadata(key);
            this.createSlicedJobs(blob, output, chunkSize, sliceJobs);
            ParallelTaskProcessor.runFailable((ExecutorService)ex, sliceJobs, (FailableInterruptibleConsumer)new SliceReaderTask(DEFAULT_RETRIER));
        }
        catch (FileNotFoundException e) {
            throw e;
        }
        catch (StorageException | IOException e) {
            throw GoogleStorageBucket.mapToIOException(GoogleStorageBucket.extractStorageException(e), "Failed getting " + this.getBucketName() + "/" + key);
        }
        finally {
            FileUtils.close(sliceJobs);
        }
    }

    public void get(ExecutorService ex, String key, File output) throws IOException, InterruptedException {
        this.get(ex, key, output, 0);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void getSliced(String key, File output, int chunkSize) throws IOException, InterruptedException {
        ThreadFactory tf = new ThreadFactoryBuilder().setNameFormat("Sliced-download-%d").build();
        ThreadPoolExecutor ex = new ThreadPoolExecutor(20, 20, 0L, TimeUnit.MINUTES, new LinkedBlockingQueue<Runnable>(), tf);
        try {
            this.get(ex, key, output, chunkSize);
        }
        finally {
            ex.shutdownNow();
        }
    }

    @Deprecated
    public void getSliced(String key, File output) throws IOException {
        try {
            this.getSliced(key, output, 0);
        }
        catch (InterruptedException e) {
            throw new InterruptedIOException("Interrupted while doing a sliced download of '" + this.getBucketName() + "/" + key + "'");
        }
    }

    public <C> C getFromJson(String key, Class<C> clazz) throws IOException, IllegalJsonException {
        try {
            return (C)super.getFromJson(key, clazz);
        }
        catch (StorageException se) {
            throw GoogleStorageBucket.mapToIOException(se, "Failed getting JSON from '" + this.getBucketName() + "/" + key + "'");
        }
    }

    public SizedInputStream getAsStream(Blob meta, int chunkSize) throws IOException {
        try {
            ReadChannel reader = meta.reader(new Blob.BlobSourceOption[0]);
            if (chunkSize > 0) {
                reader.setChunkSize(chunkSize);
            }
            return new SizedInputStream((InputStream)new GSInputStream(Channels.newInputStream((ReadableByteChannel)reader)), meta.getSize());
        }
        catch (StorageException se) {
            throw GoogleStorageBucket.mapToIOException(se, "Failed getting input stream for '" + this.getBucketName() + "/" + meta.getName() + "'");
        }
    }

    public Set<File> getAllRegularFilesByMetaInterruptibly(Collection<Blob> metaObjects, File targetFolder, Function<String, String> fileNameResolver, int parallelism, int maxRetries, int initialRetrySleepSec, double waitTimeFactor) throws IOException, InterruptedException {
        targetFolder.mkdirs();
        if (!targetFolder.isDirectory()) {
            throw new IOException("Not a folder: " + targetFolder);
        }
        HashSet<File> res = new HashSet<File>();
        ArrayList<SliceTransferJobDetails> sliceJobs = new ArrayList<SliceTransferJobDetails>();
        for (Blob blob : metaObjects) {
            File localFile = new File(targetFolder, fileNameResolver.apply(blob.getName()));
            res.add(localFile);
            this.createSlicedJobs(blob, localFile, 0, sliceJobs);
        }
        ThreadFactory tf = new ThreadFactoryBuilder().setNameFormat("Sliced-download-multi-%d").build();
        ThreadPoolExecutor ex = new ThreadPoolExecutor(parallelism, parallelism, 0L, TimeUnit.MILLISECONDS, new LinkedBlockingQueue<Runnable>(), tf);
        try {
            ParallelTaskProcessor.runFailable((ExecutorService)ex, sliceJobs, (FailableInterruptibleConsumer)new SliceReaderTask(new Retrier(initialRetrySleepSec * 1000, waitTimeFactor, maxRetries + 1)));
        }
        catch (FileNotFoundException e) {
            throw e;
        }
        catch (StorageException | IOException e) {
            throw GoogleStorageBucket.mapToIOException(GoogleStorageBucket.extractStorageException(e), "Failed downloading all " + metaObjects.size() + " files from bucket " + this.getBucketName());
        }
        finally {
            FileUtils.close(sliceJobs);
            ex.shutdownNow();
        }
        return res;
    }

    public void copyToAnotherBucket(String fromKey, String toBucket, String toKey) throws IOException {
        BlobId targetBlob = BlobId.of((String)toBucket, (String)toKey);
        Path fromPath = Paths.get(this.getBucketName(), fromKey);
        Path toPath = Paths.get(toBucket, toKey);
        LOGGER.debug("Copying from " + fromPath.toString() + " to " + toPath.toString());
        Storage.CopyRequest copyRequest = Storage.CopyRequest.of((String)this.getBucketName(), (String)fromKey, (BlobId)targetBlob);
        try {
            this.storage.copy(copyRequest).getResult();
        }
        catch (StorageException se) {
            throw GoogleStorageBucket.mapToIOException(se, "Failed copying from '" + this.getBucketName() + "/" + fromKey + "' to '" + toBucket + "/" + toKey + "'");
        }
    }

    public void delete(Blob obj) throws IOException {
        try {
            this.storage.delete(obj.getBlobId());
        }
        catch (StorageException se) {
            throw GoogleStorageBucket.mapToIOException(se, "Failed deleting '" + this.getBucketName() + "/" + obj.getName() + "'");
        }
    }

    public void deleteAllByMetaInterruptibly(Collection<Blob> fileRefs, int parallelism, int maxRetries, int initialRetrySleepSec, double waitTimeFactor) throws IOException, InterruptedException {
        try {
            final AtomicReference exception = new AtomicReference();
            final CountDownLatch terminationLatch = new CountDownLatch(fileRefs.size());
            StorageBatch batch = this.storage.batch();
            for (Blob meta : fileRefs) {
                StorageBatchResult r = batch.delete(meta.getBlobId(), new Storage.BlobSourceOption[0]);
                r.notify((BatchResult.Callback)new BatchResult.Callback<Boolean, StorageException>(){

                    public void success(Boolean resultBlob) {
                        terminationLatch.countDown();
                    }

                    public void error(StorageException storageException) {
                        exception.compareAndSet(null, storageException);
                        long nonCompleted = Math.max(1L, terminationLatch.getCount());
                        int i = 0;
                        while ((long)i < nonCompleted) {
                            terminationLatch.countDown();
                            ++i;
                        }
                    }
                });
            }
            batch.submit();
            terminationLatch.await();
            StorageException thrownExc = (StorageException)((Object)exception.get());
            if (thrownExc != null) {
                throw thrownExc;
            }
        }
        catch (StorageException se) {
            throw GoogleStorageBucket.mapToIOException(se, "Failed submitting delete batch request");
        }
    }

    public boolean exists(String key) throws IOException {
        BlobId blobId = BlobId.of((String)this.getBucketName(), (String)key);
        try {
            return this.storage.get(blobId) != null && this.isFilePath(key);
        }
        catch (StorageException se) {
            throw GoogleStorageBucket.mapToIOException(se, "Failed checking existence of '" + this.getBucketName() + "/" + key + "'");
        }
    }

    public Iterator<Blob> listObjects(String folderPath, boolean recursive) throws IOException {
        UnmodifiableIterator it;
        try {
            Storage.BlobListOption[] blobListOptionArray;
            String normalizedPath = this.normalizeFolderPath(folderPath);
            com.google.cloud.storage.Bucket bucket = this.storage.get(this.getBucketName(), new Storage.BucketGetOption[0]);
            Storage.BlobListOption prefixOption = Storage.BlobListOption.prefix((String)normalizedPath);
            if (recursive) {
                Storage.BlobListOption[] blobListOptionArray2 = new Storage.BlobListOption[1];
                blobListOptionArray = blobListOptionArray2;
                blobListOptionArray2[0] = prefixOption;
            } else {
                Storage.BlobListOption[] blobListOptionArray3 = new Storage.BlobListOption[2];
                blobListOptionArray3[0] = prefixOption;
                blobListOptionArray = blobListOptionArray3;
                blobListOptionArray3[1] = Storage.BlobListOption.currentDirectory();
            }
            Storage.BlobListOption[] options = blobListOptionArray;
            it = bucket.list(options).iterateAll().iterator();
            it = Iterators.filter(it, b -> !b.getName().equals(normalizedPath));
        }
        catch (StorageException se) {
            throw GoogleStorageBucket.mapToIOException(se, "Failed listing '" + this.getBucketName() + "/" + folderPath + "'");
        }
        return it;
    }

    public URL generateSignedUrl(String key, String contentType, int expirationSeconds, boolean isPublicRead) {
        BlobInfo.Builder builder = BlobInfo.newBuilder((String)this.getBucketName(), (String)key).setContentType(contentType);
        if (isPublicRead) {
            builder.setAcl(PUBLIC_READ_ACL);
        }
        BlobInfo blobInfo = builder.build();
        Blob blob = this.storage.create(blobInfo, new Storage.BlobTargetOption[0]);
        return blob.signUrl((long)expirationSeconds, TimeUnit.SECONDS, new Storage.SignUrlOption[]{Storage.SignUrlOption.withContentType(), Storage.SignUrlOption.httpMethod((HttpMethod)HttpMethod.PUT)});
    }

    public URL generateReadOnlyUrl(String key, int expirationSeconds) {
        BlobId blobId = BlobId.of((String)this.getBucketName(), (String)key);
        Blob blob = this.storage.get(blobId);
        return blob.signUrl((long)expirationSeconds, TimeUnit.SECONDS, new Storage.SignUrlOption[]{Storage.SignUrlOption.httpMethod((HttpMethod)HttpMethod.GET)});
    }

    public URL generateResumableSignedUrlForUpload(String key, String contentType, int expirationSeconds, Long maxContentLengthInBytes, boolean isPublic) throws IOException {
        if ((long)expirationSeconds > MAX_EXPIRATION_SECONDS) {
            throw new IllegalArgumentException("Expiration Time can't be longer than " + MAX_EXPIRATION_SECONDS + " seconds");
        }
        BlobId blobId = BlobId.of((String)this.getBucketName(), (String)key);
        BlobInfo blobinfo = BlobInfo.newBuilder((BlobId)blobId).build();
        HashMap<String, Object> extensionHeaders = new HashMap<String, Object>();
        extensionHeaders.put("Content-Type", contentType);
        extensionHeaders.put(X_GOOG_RESUMABLE_HEADER, "start");
        if (maxContentLengthInBytes != null) {
            extensionHeaders.put(X_GOOG_CONTENT_LENGTH_RANGE, "0," + maxContentLengthInBytes);
        }
        if (isPublic) {
            extensionHeaders.put(X_GOOG_ACL_HEADER, ACL_PUBLIC_READ);
        }
        try {
            return this.storage.signUrl(blobinfo, (long)expirationSeconds, TimeUnit.SECONDS, new Storage.SignUrlOption[]{Storage.SignUrlOption.httpMethod((HttpMethod)HttpMethod.POST), Storage.SignUrlOption.withExtHeaders(extensionHeaders), Storage.SignUrlOption.withV4Signature()});
        }
        catch (ServiceAccountSigner.SigningException e) {
            throw new IOException("Could not sign url", e);
        }
    }

    public Blob getObjectMetadata(String key) throws IOException {
        BlobId blobId = BlobId.of((String)this.getBucketName(), (String)key);
        try {
            Blob res = this.storage.get(blobId);
            if (res == null || res.isDirectory()) {
                throw new FileNotFoundException("File not found: '" + this.getBucketName() + "/" + key + "'");
            }
            return res;
        }
        catch (StorageException se) {
            throw GoogleStorageBucket.mapToIOException(se, "Failed getting metadata of file '" + this.getBucketName() + "/" + key + "'");
        }
    }

    public Map<String, Blob> getObjectMetadata(Collection<String> filePaths) throws IOException, InterruptedException {
        final Map<String, Blob> res = Collections.synchronizedMap(new HashMap());
        try {
            final AtomicReference exception = new AtomicReference();
            final CountDownLatch terminationLatch = new CountDownLatch(filePaths.size());
            StorageBatch batch = this.storage.batch();
            for (final String path : filePaths) {
                StorageBatchResult r = batch.get(this.getBucketName(), path, new Storage.BlobGetOption[0]);
                r.notify((BatchResult.Callback)new BatchResult.Callback<Blob, StorageException>(){

                    public void success(Blob resultBlob) {
                        res.put(path, resultBlob);
                        terminationLatch.countDown();
                    }

                    public void error(StorageException storageException) {
                        exception.compareAndSet(null, storageException);
                        long nonCompleted = Math.max(1L, terminationLatch.getCount());
                        int i = 0;
                        while ((long)i < nonCompleted) {
                            terminationLatch.countDown();
                            ++i;
                        }
                    }
                });
            }
            batch.submit();
            terminationLatch.await();
            StorageException thrownExc = (StorageException)((Object)exception.get());
            if (thrownExc != null) {
                throw thrownExc;
            }
            return res;
        }
        catch (StorageException se) {
            throw GoogleStorageBucket.mapToIOException(se, "Failed submitting batch request for metadata");
        }
    }

    public String getPath(Blob keyMetadata) {
        if (!keyMetadata.getBucket().equals(this.getBucketName())) {
            throw new IllegalArgumentException("The given Blob belongs to a different bucket ('" + keyMetadata.getBucket() + "')");
        }
        return keyMetadata.getName();
    }

    public long getLength(Blob keyMetadata) {
        return keyMetadata.getSize();
    }

    public Long getLastUpdated(Blob keyMetadata) {
        if (keyMetadata.isDirectory()) {
            return null;
        }
        return keyMetadata.getUpdateTime();
    }

    private Blob composeChunk(List<String> gsPaths, String composedFilePath) throws IOException {
        Storage.ComposeRequest composeRequest = Storage.ComposeRequest.newBuilder().addSource(gsPaths).setTarget(BlobInfo.newBuilder((String)this.getBucketName(), (String)composedFilePath).setContentType("text/plain").build()).build();
        try {
            return this.storage.compose(composeRequest);
        }
        catch (StorageException e) {
            throw GoogleStorageBucket.mapToIOException(e, "Failed composing files");
        }
    }

    public Blob compose(List<String> gsPaths, String composedFilePath, boolean removeComprisingFiles) throws IOException {
        ExecutorService executor = null;
        try {
            this.validateNotFolderPath(composedFilePath);
            List<String> composeQueue = gsPaths;
            Vector tmpFiles = new Vector(gsPaths.size() * 2);
            executor = Executors.newFixedThreadPool(Math.min(gsPaths.size(), 16));
            while (composeQueue.size() > 32) {
                List<String> inputPaths = composeQueue;
                List indices = IntStream.iterate(0, i -> i < inputPaths.size(), i -> i + 32).boxed().collect(Collectors.toList());
                Vector<Object> outputPaths = new Vector<Object>(Collections.nCopies(indices.size(), null));
                ParallelTaskProcessor.runFailable((ExecutorService)executor, indices, i -> {
                    if (i == inputPaths.size() - 1) {
                        outputPaths.set(outputPaths.size() - 1, ((String)inputPaths.get((int)i)));
                        return;
                    }
                    String tmpFile = composedFilePath + "." + Integer.toHexString(ThreadLocalRandom.current().nextInt()) + ".tmp";
                    this.composeChunk(inputPaths.subList((int)i, Math.min(i + 32, inputPaths.size())), tmpFile);
                    outputPaths.set(i / 32, tmpFile);
                    tmpFiles.add(tmpFile);
                });
                composeQueue = outputPaths;
            }
            Blob res = this.composeChunk(composeQueue, composedFilePath);
            if (removeComprisingFiles) {
                gsPaths.stream().filter(path -> !path.equals(composedFilePath)).forEach(tmpFiles::add);
            }
            this.deleteAllInterruptibly(tmpFiles, Runtime.getRuntime().availableProcessors());
            Blob blob = res;
            return blob;
        }
        catch (InterruptedException e) {
            throw new InterruptedIOException("Interrupted while operating on remote files");
        }
        finally {
            if (executor != null) {
                executor.shutdown();
            }
        }
    }

    private static StorageException extractStorageException(Throwable e) {
        while (!(e instanceof StorageException)) {
            if ((e = e.getCause()) != null) continue;
            return null;
        }
        return (StorageException)e;
    }

    private static IOException mapToIOException(StorageException e, String msg) {
        if (e == null) {
            return new IOException(msg);
        }
        switch (e.getCode()) {
            case 404: {
                return new FileNotFoundException(msg + ". Remote file not found.");
            }
            case 412: {
                return new FileAlreadyExistsException(msg + ". File already exists.");
            }
        }
        return new IOException(msg, e);
    }

    private void createSlicedJobs(Blob blob, File output, int chunkSize, List<SliceTransferJobDetails> sliceJobs) {
        long fileSize;
        if (chunkSize < 0) {
            throw new IllegalArgumentException("Invalid chunkSize: " + chunkSize);
        }
        if (chunkSize == 0) {
            chunkSize = 0x200000;
        }
        if ((fileSize = blob.getSize().longValue()) == 0L) {
            sliceJobs.add(new SliceTransferJobDetails(blob, chunkSize, new SlicedTransferFileHandler(output, 1), 0L, 0L));
        } else {
            long sliceSize = Math.min(fileSize, (long)Math.max(chunkSize, 0x1400000));
            int numberOfSlices = (int)(fileSize / sliceSize);
            long lastSliceSize = fileSize % sliceSize;
            if (lastSliceSize != 0L) {
                ++numberOfSlices;
            }
            SlicedTransferFileHandler fileHandler = new SlicedTransferFileHandler(output, numberOfSlices);
            for (int i = 0; i < numberOfSlices; ++i) {
                long length = i == numberOfSlices - 1 && lastSliceSize != 0L ? lastSliceSize : sliceSize;
                sliceJobs.add(new SliceTransferJobDetails(blob, chunkSize, fileHandler, (long)i * sliceSize, length));
            }
        }
    }

    private static class SliceReaderTask
    implements FailableInterruptibleConsumer<SliceTransferJobDetails, IOException> {
        private final Retrier retrier;

        public SliceReaderTask(Retrier retrier) {
            this.retrier = retrier;
        }

        public void accept(SliceTransferJobDetails slice) throws IOException {
            MutableObject readerRef = new MutableObject();
            try {
                this.retrier.run(() -> {
                    try {
                        ByteBuffer buffer = ByteBuffer.allocate(65536);
                        ReadChannel reader = slice.getReadChannel();
                        readerRef.setValue((Object)reader);
                        FileChannel writer = slice.getTargetFileHandler().getWriter();
                        long sliceSize = slice.getLength();
                        long currentPosition = slice.getPosition();
                        reader.seek(currentPosition);
                        int totalBytesRead = 0;
                        SliceReaderTask.setBufferLimit(buffer, totalBytesRead, sliceSize);
                        while ((long)totalBytesRead < sliceSize) {
                            int readBytes = reader.read(buffer);
                            if (readBytes == -1) {
                                throw new IOException("Premature end of file. Read " + totalBytesRead + ", but expected " + (sliceSize - (long)totalBytesRead) + " more bytes.");
                            }
                            buffer.flip();
                            writer.write(buffer, currentPosition);
                            currentPosition += (long)readBytes;
                            buffer.clear();
                            SliceReaderTask.setBufferLimit(buffer, totalBytesRead += readBytes, sliceSize);
                        }
                        slice.getTargetFileHandler().doneSliceProcessing();
                    }
                    catch (StorageException e) {
                        throw GoogleStorageBucket.mapToIOException(e, "Slice download failed");
                    }
                });
            }
            catch (InterruptedException e) {
                throw new InterruptedIOException();
            }
            finally {
                FileUtils.close((Closeable)((Closeable)readerRef.getValue()));
            }
        }

        private static void setBufferLimit(ByteBuffer buffer, long totalBytesRead, long sliceSize) {
            if (totalBytesRead + 65536L > sliceSize) {
                int limit = (int)(sliceSize - totalBytesRead);
                buffer.limit(limit);
            }
        }
    }
}

