#!/usr/bin/env pwsh

param
(
    [Parameter(Mandatory = $false, Position = 0)]
    [string] $Param1 = ""
)

Set-StrictMode -Version latest
$ErrorActionPreference = "Stop"

# Load support functions
. "$PSScriptRoot/../../common/include.ps1"

# Get component metadata
$component = Read-Component -Path "$PSScriptRoot/../../../component.json"

try {
    $startedAt = Get-Date
    # Publish docker images
    . "$PSScriptRoot/publish_docker.ps1"
    
    # Publish by language
    switch ($component.language) {
        "dart" {
            . "$PSScriptRoot/publish_dart.ps1"
        }
        "dotnet" {
            if ($Param1 -eq "") {
                . "$PSScriptRoot/publish_dotnet.ps1"
            } else {
                . "$PSScriptRoot/publish_dotnet.ps1" $Param1
            }
        }
        "java" {
            . "$PSScriptRoot/publish_java.ps1"
        }
        "node" {
            . "$PSScriptRoot/publish_node.ps1"
        }
        "python" {
            . "$PSScriptRoot/publish_python.ps1"
        }
        {($_ -eq "angular") -or ($_ -eq "vue") -or ($_ -eq "react")} {
            if ($component.type -eq "lib") {
                . "$PSScriptRoot/publish_node.ps1"
            } else {
                . "$PSScriptRoot/publish_s3.ps1"
            }
        }
        Default {
            Write-Host "Publishing for $($component.language) is not supported"
        }
    }
}
catch {
    Write-Output $_
    $LASTEXITCODE = 1
}
finally {
    # Verify script exit code
    if (Test-Path -Path "variable:LASTEXITCODE") {
        # Save script exit code
        $scriptExitCode = $LASTEXITCODE
    } else {
        # Set exit code if script was empty
        $scriptExitCode = 0
    }
    $completedAt = Get-Date

    # Save script result for bitbucket measure
    if ($null -ne $env:BITBUCKET_BUILD_NUMBER) {
        # Read measure file with pipeline steps results
        $measureFileName = "$PSScriptRoot/../../../../measure.json"
        if (Test-Path -Path $measureFileName) {
            $measure = Get-Content -Path $measureFileName | ConvertFrom-Json
            # Fix single item array issue on ubuntu powershell
            if ($measure.Length -eq 1) {
                $measure = @($measure)
            }
        } else {
            $measure = @()
        }
        # Define is script finished successfully
        if ($scriptExitCode -eq 0) {
            $successfull = $True
        } else {
            $successfull = $False
        }
        # Create result object
        $result = @{
            "name" = "publish";
            "successfull" = $successfull;
            "duration" = [Math]::Ceiling($(New-Timespan -Start $startedAt -End $completedAt).TotalSeconds);
        }
        $measure += $result
        # Save script result to file
        ConvertTo-Json $measure | Set-Content -Path $measureFileName
    }

    # Retain original exit code
    exit $scriptExitCode
}
