#!/usr/bin/env pwsh

Set-StrictMode -Version latest
$ErrorActionPreference = "Stop"

# Load support functions
. "$PSScriptRoot/../../common/include.ps1"

# Get component metadata
$component = Read-Component -Path "$PSScriptRoot/../../../component.json"
# Create tag
$tag = "v$($component.version)-$($component.build)"
if ($env:IS_MONOREPO) {
    $tag = "$($component.name)/$tag"
}

# Verify build server and set environment variable
Set-EnvBuildServer

# Change git remote so that git uses ssh on push
switch ($env:BUILD_SERVER) {
    "gitlab" {
        Write-Host "Updating git origin to push tags from GitLab pipeline"
        git remote set-url origin "https://gitlab-ci-token:$($env:GITLAB_TOKEN)@gitlab.com/$($env:CI_PROJECT_NAMESPACE)/$($env:CI_PROJECT_NAME).git"
    } 
    "jenkins" {
        Write-Host "Updating git origin to push tags from Jenkins pipeline"
        $remote = git remote get-url origin
        $newRemote = $remote.Replace("https://","https://$($env:GIT_USERNAME):$($env:GIT_TOKEN)@")
        git remote set-url origin $newRemote
    } 
    "azure" {
        Write-Host "Updating git origin to push tags from Azure DevOps pipeline"
        $remote = git remote get-url origin
        $newRemote = "https://$($env:GIT_USERNAME):$($env:GIT_TOKEN)@$($remote.Split("@")[1])"
        git remote set-url origin $newRemote
    }
}

# Verify existing tag and cleanup if needed
$remoteTags = git ls-remote --tags origin
foreach($remoteTag in $remoteTags) {
    if ($remoteTag -match $tag) {
        # Remove existing tag
        $null = git push --delete origin $tag
    }
}
$localTags = git tag -l
foreach($localTag in $localTags) {
    if ($localTag -match $tag) {
        # Remove existing tag
        $null = git tag -d $tag
    }
}

# Tag and push
git tag $tag
git push origin $tag

# Check for successful tagging
if ($LastExitCode -ne 0) {
    Write-Error "Error on git push."
}
