#!/usr/bin/env pwsh

param
(
    [Parameter(Mandatory = $true, Position = 0)]
    [string] $Cluster,
    [Parameter(Mandatory = $true, Position = 1)]
    [string] $Namespace,
    [Parameter(Mandatory = $true, Position = 2)]
    [string] $Product,
    [Parameter(Mandatory = $true, Position = 3)]
    [string] $Baseline
)

Set-StrictMode -Version latest
$ErrorActionPreference = "Stop"

# Switch context to CI environment
kubectl config use-context $Cluster

# Get product components images
Write-Host "Getting list of deployed components..."
$podsImages = $(kubectl get pods -n $Namespace -l product=$Product -o jsonpath="{.items[*].spec.containers[0].image}").Split(" ")

# Define registry server name
$server = ""
$registry = $podsImages[0].SubString(0,$podsImages[0].LastIndexOf("/"))
$pos = $registry.IndexOf("/")
if ($pos -gt 0) {
    $server = $registry.Substring(0, $pos)
}

# Automatically login to server
if ($null -ne $env:DOCKER_USER -and $null -ne $env:DOCKER_PASS) {
    Write-Host "Logging in to $server"
    docker login $server -u $env:DOCKER_USER -p $env:DOCKER_PASS
}

# Process throught each image
foreach ($podImage in $podsImages) {
    # Save docker pod image and baseline image
    $dockerImage = $podImage
    $dockerImageWithTag = $dockerImage
    # Remove tag from image if it exists
    if ($dockerImageWithTag.IndexOf(":") -gt 0) {
        $dockerImage = $dockerImage.Split(":")[0]
    }
    $baselineImage = $dockerImage + ":" + $Baseline

    # Pull image before tag
    Write-Host "Pulling '$dockerImageWithTag' image..."
    docker pull $dockerImageWithTag

    # Tag image with baseline tag
    docker tag $dockerImageWithTag $baselineImage

    # Push image to registry
    Write-Host "Pushing '$baselineImage' image..."
    docker push $baselineImage

    # Cleanup
    docker rmi $dockerImageWithTag --force
    docker rmi $baselineImage --force
}

Write-Host "All images of pods with label 'product=$Product' tagged '$Baseline' successfully" -ForegroundColor Green
