#!/usr/bin/env pwsh

param
(
    [Parameter(Mandatory = $false, Position = 0)]
    [string] $Param1 = ""
)

Set-StrictMode -Version latest
$ErrorActionPreference = "Stop"

# Load support functions
. "$PSScriptRoot/../../common/include.ps1"

# Get component metadata
$component = Read-Component -Path "$PSScriptRoot/../../../component.json"

try {
    $startedAt = Get-Date
    # Publish docker images
    Write-Host "`n--- Executing docker publish ---`n"
    . "$PSScriptRoot/publish_docker.ps1"
    
    # Publish by language
    switch ($component.language) {
        "dart" {
            Write-Host "`n--- Executing dart publish ---`n"
            . "$PSScriptRoot/publish_dart.ps1"
        }
        "dotnet" {
            Write-Host "`n--- Executing ditnet publish ---`n"
            if ($Param1 -eq "") {
                . "$PSScriptRoot/publish_dotnet.ps1"
            } else {
                . "$PSScriptRoot/publish_dotnet.ps1" $Param1
            }
        }
        "java" {
            Write-Host "`n--- Executing java publish ---`n"
            . "$PSScriptRoot/publish_java.ps1"
        }
        "node" {
            Write-Host "`n--- Executing node publish ---`n"
            . "$PSScriptRoot/publish_node.ps1"
            $s3Artifact = $component.artifacts | Where-Object {$_.type -eq "s3"}
            if ($null -ne $s3Artifact) {
                Write-Host "`n--- Executing s3 publish ---`n"
                . "$PSScriptRoot/publish_s3.ps1"
            }
        }
        "python" {
            Write-Host "`n--- Executing python publish ---`n"
            . "$PSScriptRoot/publish_python.ps1"
        }
        {"angular", "vue", "react" -eq $_} {
            if ($component.type -eq "lib") {
                Write-Host "`n--- Executing node publish ---`n"
                . "$PSScriptRoot/publish_node.ps1"
            } else {
                Write-Host "`n--- Executing s3 publish ---`n"
                . "$PSScriptRoot/publish_s3.ps1"
            }
        }
        Default {
            Write-Host "Publishing for $($component.language) is not supported"
        }
    }
}
catch {
    Write-Output $_
    $LASTEXITCODE = 1
}
finally {
    # Verify script exit code
    if (Test-Path -Path "variable:LASTEXITCODE") {
        # Save script exit code
        $scriptExitCode = $LASTEXITCODE
    } else {
        # Set exit code if script was empty
        $scriptExitCode = 0
    }
    $completedAt = Get-Date

    # Save script result for bitbucket measure
    if ($null -ne $env:BITBUCKET_BUILD_NUMBER) {
        # Read measure file with pipeline steps results
        $measureFileName = "$PSScriptRoot/../../../../measure.json"
        if (Test-Path -Path $measureFileName) {
            $measure = Get-Content -Path $measureFileName | ConvertFrom-Json
            # Fix single item array issue on ubuntu powershell
            if ($measure.Length -eq 1) {
                $measure = @($measure)
            }
        } else {
            $measure = @()
        }
        # Define is script finished successfully
        if ($scriptExitCode -eq 0) {
            $successfull = $True
        } else {
            $successfull = $False
        }
        # Create result object
        $result = @{
            "name" = "publish";
            "successfull" = $successfull;
            "duration" = [Math]::Ceiling($(New-Timespan -Start $startedAt -End $completedAt).TotalSeconds);
        }
        $measure += $result
        # Save script result to file
        ConvertTo-Json $measure | Set-Content -Path $measureFileName
    }

    # Retain original exit code
    exit $scriptExitCode
}
