#!/usr/bin/env pwsh

Set-StrictMode -Version latest
$ErrorActionPreference = "Stop"

# Load support functions
. "$PSScriptRoot/../../common/include.ps1"

# Get component metadata
$component = Read-Component -Path "$PSScriptRoot/../../../component.json"
$s3Artifact = $component.artifacts | Where-Object {$_.type -eq "s3"}
try {
    $baseHref = $component.base_href
} catch {
    $baseHref = $null
}

# Skip publish if artifact not set
if ($s3Artifact -eq $null) {
    Write-Host "Didn't find any artifacts of the 's3' type in component.json. Publish to s3 skipped."
    exit 0
}

# check for build folder
switch ($component.language) {
    "react" {
        if (-not (Test-Path -Path "$PSScriptRoot/../../../build")) {
            Write-Error "Error! Build folder not found in $(Get-Location)"
        }
    }
    "vue" {
        if (-not (Test-Path -Path "$PSScriptRoot/../../../build")) {
            Write-Error "Error! Build folder not found in $(Get-Location)"
        }
    }
    {"angular", "node" -eq $_} {
        if (-not (Test-Path -Path "$PSScriptRoot/../../../dist")) {
            Write-Error "Dist folder not found in $(Get-Location)"
        }
    }
    Default {
        Write-Error "S3 publish for $($component.language) is not supported."
    }
}

# get s3 bucket name
if ($null -ne $s3Artifact) {
    $s3Bucket = $s3Artifact.bucket
}
if ($null -ne $env:S3_BUCKET) {
    $s3Bucket = $env:S3_BUCKET
}
if ([string]::IsNullOrEmpty($s3Bucket)) {
    Write-Error "Set s3 bucket name via the 'artifacts' section of the component.json file (using the 'bucket' property) or via the S3_BUCKET environment variable"
}

# upload to s3
switch ($component.language) {
    "react" {
        aws s3 sync "$PSScriptRoot/../../../build" "s3://$s3Bucket/$baseHref" --delete
    }
    "vue" {
        aws s3 sync "$PSScriptRoot/../../../build" "s3://$s3Bucket/$baseHref" --delete
    }
    "angular" {
        aws s3 sync "$PSScriptRoot/../../../dist/$($component.name)" "s3://$s3Bucket/$baseHref" --delete
    }
    "node" {
        $zipFileName = "$($component.name)-lambda-v$($component.version).zip"
        aws s3 cp "$PSScriptRoot/../../../dist/$zipFileName" "s3://$s3Bucket/$zipFileName"
    }
}

if ($LastExitCode -ne 0) {
    Write-Error "Publish to s3 failed. See logs above for more information and check that valid credentials are set in the environment variables."
}
