#!/usr/bin/env pwsh

param
(
    [Parameter(Mandatory = $true, Position = 0)]
    [string] $Org,

    [Parameter(Mandatory = $true, Position = 1)]
    [string] $Name,

    [Parameter(Mandatory = $true, Position = 2)]
    [string] $AWSSecretKeyId,

    [Parameter(Mandatory = $true, Position = 3)]
    [string] $AWSSecretAccessKey,

    [Parameter(Mandatory = $true, Position = 4)]
    [string] $AWSS3Bucket,

    [Parameter(Mandatory = $true, Position = 5)]
    [string] $AWSS3BucketFolder,

    [Parameter(Mandatory = $true, Position = 6)]
    [string] $Token
)

Set-StrictMode -Version latest
$ErrorActionPreference = "Stop"

# Set base url for web requests
$baseUrl = "https://gitlab.com/api/v4/projects/$Org%2F$Name/pipelines"
$headers = @{
    "PRIVATE-TOKEN" = $Token
}

# Get list of latest pipelines
$r = Invoke-WebRequest -Method 'Get' -Uri $baseUrl -Headers $headers
$pipelinesListData = $r.Content | ConvertFrom-Json

# Verify for pipeline existence and get latetest pipeline
if ($r.statusCode -eq 404) {
    Write-Error "Missing pipeline for $Org/$Name"
}

# Get latest pipeline id (ignore skipped pipelines)
$latestPipeline = $pipelinesListData | Where-Object {$_.status -ne "skipped"} | Select-Object -First 1

# Get latest pipeline info
$latestPipelineData = Invoke-WebRequest -Method 'Get' -Uri "$baseUrl/$($latestPipeline.id)" -Headers $headers | ConvertFrom-Json

# Get latest pipeline jobs info
$latestPipelineJobsData = Invoke-WebRequest -Method 'Get' -Uri "$baseUrl/$($latestPipeline.id)/jobs?per_page=100" -Headers $headers | ConvertFrom-Json

# Initialize resulting object
$latestPipelineInfo = @{
    "name" = $name
    "full_name" = $name # gitlab doesn't have separate human readable name for pipeline
    "date" = $latestPipelineData.created_at
    "pipeline_url" = $latestPipelineData.web_url
    "triggered_by" = "$($latestPipelineData.user.username) ($($latestPipelineData.user.name))"
    "commit_url" = $latestPipelineJobsData[0].commit.web_url
}

# Get pipeline status
$latestPipelineInfo.status = $latestPipelineData.status

# Get status of pipeline jobs
$latestPipelineInfo.builded = $null
$latestPipelineInfo.tested = $null
$latestPipelineInfo.packaged = $null
$latestPipelineInfo.published = $null

$jobsDuration = @()

foreach ($job in $latestPipelineJobsData | Where-Object {$_.stage -eq "authoring"}) {
    if ($job.name -eq "build") {
        if ($job.status -eq "success") {
            $latestPipelineInfo.builded = $True
        } else {
            $latestPipelineInfo.builded = $False
        }
    }
    elseif ($job.name -eq "test") {
        if ($job.status -eq "success") {
            $latestPipelineInfo.tested = $True
        } else {
            $latestPipelineInfo.tested = $False
        }
    }
    elseif ($job.name -eq "package") {
        if ($job.status -eq "success") {
            $latestPipelineInfo.packaged = $True
        } else {
            $latestPipelineInfo.packaged = $False
        }
    }
    elseif ($job.name -eq "publish") {
        if ($job.status -eq "success") {
            $latestPipelineInfo.published = $True
        } else {
            $latestPipelineInfo.published = $False
        }
    }

    # Get job duration
    if (($job.status -eq "success") -or ($job.status -eq "failed")) { # ignore manual jobs
        $startedAt = [datetime]$job.started_at
        $finishedAt = [datetime]$job.finished_at
        $td = New-Timespan -Start $startedAt -End $finishedAt
        $duration = [Math]::Ceiling($td.TotalSeconds)

        $jobsDuration += @{
            "name" = $job.name
            "duration" = $duration
        }

        if ($latestPipelineInfo.status -ne "failed") {
            $latestPipelineInfo.status = $job.status
        }
    }
}
[array]::Reverse($jobsDuration)

# Get pipeline duration
$duration = 0
foreach($job in $jobsDuration) {
    $duration += $job.duration
}
$latestPipelineInfo.duration = $duration

# Use reusable code to update metrics on s3
Update-S3Metrics
