# Basic Microservice in Java

This basic microservice executes simple requests.

Supported functionality:
* Deployment platforms: Standalone Process, Docker
* External APIs: Commandable HTTP, REST/HTTP, gRPC, Commandable gRPC
* Health checks: Heartbeat, Status
* Swagger: http://localhost:8080/swagger/index.html

Dependencies on infrastructure services:
* (optional) ElasticSearch for logging
* (optional) Prometheus Push Gateway for performance metrics

There are no dependencies on other microservices.

<a name="links"></a> Quick links:

* Client Libraries:
    - [Node.js SDK](https://bitbucket.org/entinco/eic-templates-services-node/src/master/client-basic-pipservices/)
    - [.NET SDK](https://bitbucket.org/entinco/eic-templates-services-dotnet/src/master/client-basic-pipservices/)
    - [Golang SDK](https://bitbucket.org/entinco/eic-templates-services-go/src/master/client-basic-pipservices/)
    - [Dart SDK](https://bitbucket.org/entinco/eic-templates-services-dart/src/master/client-basic-pipservices/)
    - [Python SDK](https://bitbucket.org/entinco/eic-templates-services-python/src/master/client-basic-pipservices/)
* [API Reference](docs)
* [Change Log](CHANGELOG.md)
* [License](LICENSE)
* [ToDo List](TODO.md)
* [Usage Notes](USAGE.md)

##  Contract

```java
public record RequestV1(String value) {
}

public record ResponseV1(String value) {
}

public interface IBasicController {
    ResponseV1 doSomething(IContext context, RequestV1 request);
}
```


## Get

Get the microservice source from Bitbucket:
```bash
git clone git@bitbucket.org:entinco/eic-templates-services-java.git
```

## Use


Go to the pom.xml file in Maven project and add dependencies:
```xml
<dependency>
    <groupId>com.service.basic</groupId>
    <artifactId>service-basic-pipservices</artifactId>
    <version>1.0.0</version>
</dependency>
```

## Run

The microservice can be configured using the environment variables:

* ELASTICSEARCH_LOGGING_ENABLED - turn on Elasticsearch logs and metrics
* ELASTICSEARCH_PROTOCOL - connection protocol: http or https
* ELASTICSEARCH_SERVICE_URI - resource URI or connection string with all parameters in it
* ELASTICSEARCH_SERVICE_HOST - host name or IP address
* ELASTICSEARCH_SERVICE_PORT - port number
* DEFAULT_RESPONSE - default response from service if empty request
* GRPC_ENABLED - turn on GRPC endpoint
* GRPC_PORT - GRPC port number (default: 8090)
* HTTP_ENABLED - turn on HTTP endpoint
* HTTP_PORT - HTTP port number (default: 8080)
* PUSHGATEWAY_METRICS_ENABLED - turn on pushgetway for prometheus
* PUSHGATEWAY_PROTOCOL - connection protocol: http or https
* PUSHGATEWAY_METRICS_SERVICE_URI - resource URI or connection string with all parameters in it
* PUSHGATEWAY_METRICS_SERVICE_HOST - host name or IP address
* PUSHGATEWAY_METRICS_SERVICE_PORT - port number
* SWAGGER_ROUTE - the path where the swagger service will be available
* SWAGGER_NAME - the header name of swagger service
* SWAGGER_DESCRIPTION - the text description of swagger service

Inside your code get the reference to the client library
```java
import com.client.basic.BasicCommandableHttpClientV1;
```

Instantiate the client
```java
// Create the client instance
var client = new BasicCommandableHttpClientV1();
```

Define client configuration parameters.
```java
// Client configuration
var httpConfig = ConfigParams.fromTuples(
    "connection.protocol", "http",
    "connection.host", "localhost",
    "connection.port", 3000
);
client.configure(httpConfig);
```

Connect to the microservice
```java
// Connect to the microservice
client.open(Context.fromTraceId("123"));
```

Create a new entity
```java
var req = new RequestV1('Bla bla bla');

res = client.doSomething(Context.fromTraceId("123"), req);
console.log("Response: " + res.value);
```


## Develop

For development you shall install the following prerequisites:
* Java 16+
* Maven 3+
* Visual Studio Code or another IDE of your choice
* Docker

Compile the microservice:
```bash
mvn compile
```

Run automated tests:
```bash
mvn test
```

Run performance benchmark:
```bash
mvn exec:exec@benchmark
```
Run linter:
```bash
mvn checkstyle:check
```

Generate API documentation:
```bash
./docgen.ps1
```

Before committing changes run dockerized build and test as:
```bash
./build.ps1
./test.ps1
./clean.ps1
```

## Contacts

This microservice was created by and is currently maintained by *Responsible Person Name*.