package com.service.basic.logic;

import com.service.basic.data.version1.RequestV1;
import com.service.basic.data.version1.ResponseV1;
import org.pipservices4.commons.errors.ConfigException;
import org.pipservices4.components.config.ConfigParams;
import org.pipservices4.components.config.IConfigurable;
import org.pipservices4.components.context.IContext;
import org.pipservices4.components.refer.IReferenceable;
import org.pipservices4.components.refer.IReferences;
import org.pipservices4.components.refer.ReferenceException;
import org.pipservices4.observability.count.CompositeCounters;
import org.pipservices4.observability.log.CompositeLogger;
import org.pipservices4.rpc.commands.CommandSet;
import org.pipservices4.rpc.commands.ICommandable;

/**
 * Implements the business logic operations of the controller.
 */
public class BasicService implements IBasicService, IConfigurable, IReferenceable, ICommandable {
    private BasicCommandSet _commandSet;
    private final CompositeLogger _logger = new CompositeLogger();
    private final CompositeCounters _counters = new CompositeCounters();
    private final Object _lock = new Object();
    private String _defaultResponse = "";

    public BasicService() {}

        /**
     * Configures component by passing configuration parameters.
     *
     * @param config    configuration parameters to be set.
     */
    @Override
    public void configure(ConfigParams config) throws ConfigException {
        this._logger.configure(config);
        this._defaultResponse = config.getAsStringWithDefault("configuration.response", this._defaultResponse);
    }

            /**
     * Sets references to dependent components.
     *
     * @param references 	references to locate the component dependencies.
     */
    public void setReferences(IReferences references) throws ReferenceException, ConfigException {
        this._logger.setReferences(references);
        this._counters.setReferences(references);
    }

	@Override
	public CommandSet getCommandSet() {
		if (_commandSet == null) {
			_commandSet = new BasicCommandSet(this);
		}

		return _commandSet;
	}

        /**
     * Some API function of module
     * @param request a request string
     * @return a response string
     */

     @Override
     public ResponseV1 doSomething(IContext context, RequestV1 request) {
        var response = new ResponseV1();
         synchronized (_lock) {            
            response.setValue(this._defaultResponse);
    
            if (request != null) {
                response.setValue((request.getValue() != null && !request.getValue().isEmpty()) ? request.getValue() : this._defaultResponse);
                this._logger.info(context, "Processed request: " + response.getValue());
            } else {
                var err = new NullPointerException("NullPointerException");
                this._logger.error(context, err, "Processed null request");
                throw err;
            }
    
            this._counters.incrementOne("basic.did_something");    
         }
         return response;
     }
}
