package benchmark;

import com.service.basic.controllers.version1.BasicCommandableGrpcControllerV1;
import com.service.basic.controllers.version1.BasicGrpcControllerV1;
import com.service.basic.data.version1.RequestV1;
import com.service.basic.logic.BasicService;
import com.service.basic.protos.BasicControllerGrpc;
import io.grpc.ManagedChannel;
import io.grpc.ManagedChannelBuilder;
import org.pipservices4.commons.errors.ApplicationException;
import org.pipservices4.components.config.ConfigParams;
import org.pipservices4.components.context.Context;
import org.pipservices4.components.exec.Parameters;
import org.pipservices4.components.refer.Descriptor;
import org.pipservices4.components.refer.References;
import org.pipservices4.grpc.test.TestCommandableGrpcClient;

import java.util.concurrent.TimeUnit;

public class GrpcBenchmark {
    private static final System.Logger logger = System.getLogger("GrpcBenchmark");
    private static final ConfigParams grpcConfig = ConfigParams.fromTuples(
            "connection.protocol", "http",
            "connection.host", "localhost",
            "connection.port", 3000
    );

    private static BasicCommandableGrpcControllerV1 commandableController;

    private static TestCommandableGrpcClient commandableClient;

    private static BasicGrpcControllerV1 grpcController;

    private static ManagedChannel _channel;
    private static BasicControllerGrpc.BasicControllerBlockingStub grpcClient;

    public static void setupCommandableGrpc() throws ApplicationException {
        var service = new BasicService();
        commandableController = new BasicCommandableGrpcControllerV1();
        commandableController.configure(grpcConfig);

        commandableClient = new TestCommandableGrpcClient("basic_v1.BasicController");
        commandableClient.configure(grpcConfig);

        References references = References.fromTuples(
                new Descriptor("service-basic", "service", "default", "default", "1.0"), service,
                new Descriptor("service-basic", "controller", "commandable-grpc", "default", "1.0"), commandableController
        );
        service.setReferences(references);
        commandableController.setReferences(references);

        commandableController.open(null);
        commandableClient.open(null);
    }

    public static void teardownCommandableGrpc() throws ApplicationException {
        commandableClient.close(null);
        commandableController.close(null);
    }

    public static void setupGrpc() throws ApplicationException {
        var service = new BasicService();
        grpcController = new BasicGrpcControllerV1();
        grpcController.configure(grpcConfig);

        _channel = ManagedChannelBuilder.forTarget("localhost:3000")
                // Channels are secure by default (via SSL/TLS). For the example we disable TLS to avoid
                // needing certificates.
                .usePlaintext()
                .build();
        grpcClient = BasicControllerGrpc.newBlockingStub(_channel);

        References references = References.fromTuples(
                new Descriptor("service-basic", "service", "default", "default", "1.0"), service,
                new Descriptor("service-basic", "controller", "grpc", "default", "1.0"), grpcController
        );
        service.setReferences(references);
        grpcController.setReferences(references);

        grpcController.open(null);
    }

    public static void teardownGrpc() throws ApplicationException, InterruptedException {
        _channel.shutdownNow().awaitTermination(5, TimeUnit.SECONDS);
        grpcController.close(null);
    }

    private static void benchmarkCommandableGrpc() throws ApplicationException {
        setupCommandableGrpc();
        try {
            var random = new RandomRequest();
            var iterations = System.getenv("BENCHMARK_ITERATIONS") != null ?
                    Integer.parseInt(System.getenv("BENCHMARK_ITERATIONS")) : 100;

            var start = System.nanoTime();
            logger.log(System.Logger.Level.INFO, "CommandableGrpc | Starting benchmark with " + iterations + " iterations...");

            for (var i = 0; i < iterations; i++) {
                var request = new RequestV1(random.nextRequest(10, 25));
                commandableClient.callCommand(RequestV1.class, "do_something", Context.fromTraceId("benchmark"),
                        Parameters.fromTuples("request", request));
            }

            var elapsedTime = System.nanoTime() - start;
            var ms = ((float) elapsedTime) / 1000000;
            logger.log(System.Logger.Level.INFO, String.format("CommandableGrpc | Benchmark completed in %.3f ms", ms));
        } finally {
            teardownCommandableGrpc();
        }
    }

    private static void benchmarkGrpc() throws ApplicationException, InterruptedException {
        setupGrpc();
        try {
            var random = new RandomRequest();
            var iterations = System.getenv("BENCHMARK_ITERATIONS") != null ?
                    Integer.parseInt(System.getenv("BENCHMARK_ITERATIONS")) : 100;

            var start = System.nanoTime();
            logger.log(System.Logger.Level.INFO, "Grpc | Starting benchmark with " + iterations + " iterations...");

            for (var i = 0; i < iterations; i++) {
                var request = new RequestV1(random.nextRequest(10, 25));
                var res = grpcClient.doSomething(
                        com.service.basic.protos.Query.newBuilder()
                                .setRequest(com.service.basic.protos.RequestV1.newBuilder().setValue(request.getValue()).build())
                                .build());
            }

            var elapsedTime = System.nanoTime() - start;
            var ms = ((float) elapsedTime) / 1000000;
            logger.log(System.Logger.Level.INFO, String.format("Grpc | Benchmark completed in %.3f ms", ms));
        } finally {
            teardownGrpc();
        }
    }

    public static void main(String[] args) throws ApplicationException, InterruptedException {
        benchmarkCommandableGrpc();
        benchmarkGrpc();

        System.exit(0);
    }
}
