package benchmark;

import com.service.basic.controllers.version1.BasicCommandableHttpControllerV1;
import com.service.basic.controllers.version1.BasicRestControllerV1;
import com.service.basic.data.version1.RequestV1;
import com.service.basic.logic.BasicService;
import jakarta.ws.rs.HttpMethod;
import org.pipservices4.commons.errors.ApplicationException;
import org.pipservices4.components.config.ConfigParams;
import org.pipservices4.components.context.Context;
import org.pipservices4.components.exec.Parameters;
import org.pipservices4.components.refer.Descriptor;
import org.pipservices4.components.refer.References;
import org.pipservices4.http.test.TestCommandableHttpClient;
import org.pipservices4.http.test.TestRestClient;

public class HttpBenchmark {
    private static final System.Logger logger = System.getLogger("HttpBenchmark");
    private static final ConfigParams httpConfig = ConfigParams.fromTuples(
            "connection.protocol", "http",
            "connection.host", "localhost",
            "connection.port", 3001
    );

    private static BasicCommandableHttpControllerV1 commandableController;

    private static TestCommandableHttpClient commandableClient;

    private static BasicRestControllerV1 restController;

    private static TestRestClient restClient;

    public static void setupCommandableHttp() throws ApplicationException {
        var service = new BasicService();
        commandableController = new BasicCommandableHttpControllerV1();
        commandableController.configure(httpConfig);

        commandableClient = new TestCommandableHttpClient("commandable_basic/v1");
        commandableClient.configure(httpConfig);

        References references = References.fromTuples(
                new Descriptor("service-basic", "service", "default", "default", "1.0"), service,
                new Descriptor("service-basic", "controller", "commandable-http", "default", "1.0"), commandableController
        );
        service.setReferences(references);
        commandableController.setReferences(references);

        commandableController.open(null);
        commandableClient.open(null);
    }

    public static void teardownCommandableHttp() throws ApplicationException {
        commandableClient.close(null);
        commandableController.close(null);
    }

    public static void setupRestHttp() throws ApplicationException {
        var service = new BasicService();
        restController = new BasicRestControllerV1();
        restController.configure(httpConfig);

        restClient = new TestRestClient("basic/v1");
        restClient.configure(httpConfig);

        References references = References.fromTuples(
                new Descriptor("service-basic", "service", "default", "default", "1.0"), service,
                new Descriptor("service-basic", "controller", "rest-http", "default", "1.0"), restController
        );
        service.setReferences(references);
        restController.setReferences(references);

        restController.open(null);
        restClient.open(null);
    }

    public static void teardownRestHttp() throws ApplicationException {
        restClient.close(null);
        restController.close(null);
    }

    private static void benchmarkCommandableHttp() throws ApplicationException {
        setupCommandableHttp();
        try {
            var random = new RandomRequest();
            var iterations = System.getenv("BENCHMARK_ITERATIONS") != null ?
                    Integer.parseInt(System.getenv("BENCHMARK_ITERATIONS")) : 100;

            var start = System.nanoTime();
            logger.log(System.Logger.Level.INFO, "CommandableHttp | Starting benchmark with " + iterations + " iterations...");

            for (var i = 0; i < iterations; i++) {
                var request = new RequestV1(random.nextRequest(10, 25));
                commandableClient.callCommand(RequestV1.class, "do_something", Context.fromTraceId("benchmark"),
                        Parameters.fromTuples("request", request));
            }

            var elapsedTime = System.nanoTime() - start;
            var ms = ((float) elapsedTime) / 1000000;
            logger.log(System.Logger.Level.INFO, String.format("CommandableHttp | Benchmark completed in %.3f ms", ms));
        } finally {
            teardownCommandableHttp();
        }
    }

    private static void benchmarkRestHttp() throws ApplicationException {
        setupRestHttp();
        try {
            var random = new RandomRequest();
            var iterations = System.getenv("BENCHMARK_ITERATIONS") != null ?
                    Integer.parseInt(System.getenv("BENCHMARK_ITERATIONS")) : 100;

            var start = System.nanoTime();
            logger.log(System.Logger.Level.INFO, "RestHttp | Starting benchmark with " + iterations + " iterations...");

            for (var i = 0; i < iterations; i++) {
                var request = new RequestV1(random.nextRequest(10, 25));
                restClient.call(RequestV1.class, Context.fromTraceId("benchmark"), HttpMethod.POST, "do_something",
                        request);
            }

            var elapsedTime = System.nanoTime() - start;
            var ms = ((float) elapsedTime) / 1000000;
            logger.log(System.Logger.Level.INFO, String.format("RestHttp | Benchmark completed in %.3f ms", ms));
        } finally {
            teardownRestHttp();
        }
    }

    public static void main(String[] args) throws ApplicationException {
        benchmarkCommandableHttp();
        benchmarkRestHttp();

        System.exit(0);
    }
}
