package com.service.basic.controllers.version1;

import com.service.basic.data.version1.RequestV1;
import com.service.basic.data.version1.RequestV1Schema;
import com.service.basic.logic.IBasicService;
import jakarta.ws.rs.HttpMethod;
import jakarta.ws.rs.container.ContainerRequestContext;
import jakarta.ws.rs.core.Response;
import org.pipservices4.commons.convert.JsonConverter;
import org.pipservices4.commons.errors.ConfigException;
import org.pipservices4.components.context.Context;
import org.pipservices4.components.refer.Descriptor;
import org.pipservices4.components.refer.IReferences;
import org.pipservices4.components.refer.ReferenceException;
import org.pipservices4.data.validate.ObjectSchema;
import org.pipservices4.http.controllers.RestController;

import java.io.IOException;
import java.nio.charset.StandardCharsets;

/**
 * Controller that receives remove calls via HTTP protocol
 * to operations automatically generated for commands defined in <a href="https://pip-services4-java.github.io/pip-services4-rpc-java/org/pipservices4/rpc/commands/ICommandable.html">ICommandable</a> components.
 * Each command is exposed as POST operation that receives all parameters in body object.
 */
public class BasicRestControllerV1 extends RestController {
    private IBasicService _service;

        /**
     * Creates an instance of this service.
     */
    public BasicRestControllerV1() {
        super();
        this._baseRoute = "basic/v1";
        this._dependencyResolver.put(
            "service",
            new Descriptor("service-basic", "service", "default", "*", "*")
        );
    }

        /**
     * Sets references to dependent components.
     *
     * @param references 	references to locate the component dependencies.
     */
    public void setReferences(IReferences references) throws ReferenceException, ConfigException {
        super.setReferences(references);
        this._service = this._dependencyResolver.getOneRequired(IBasicService.class, "service");
    }

        /**
     * Service command method that define the operation.
     * 
     * @param req HTTP request call object.
     * @return response object defined in proto file.
     */
    private Response doSomething(ContainerRequestContext req) {
        var context = Context.fromTraceId(this.getTraceId(req));
        var timing = this.instrument(context, "basic.v1.do_something");
        try {
            var res = this._service.doSomething(
                context,
                    JsonConverter.fromJson(
                            RequestV1.class,
                            new String(req.getEntityStream().readAllBytes(), StandardCharsets.UTF_8)
                    )
            );
            timing.endSuccess();
            return this.sendResult(res);
        } catch (IOException err) {
            timing.endFailure(err);
            return sendError(err);
        }
    }
    
    /**
     * Registers all service routes in Lambda Function.
     */
    @Override
    public void register() {
        this.registerRoute(HttpMethod.POST, "/do_something", new ObjectSchema()
            .withRequiredProperty("body", new RequestV1Schema()), this::doSomething);
        this.registerOpenApiSpecFromFile("./src/com/service/basic/swagger/requests_v1.yaml");
    }
}
