# Usage Notes

## Overview

This template is used as a starting point to develop basic microservices. It demonstrates a few selected patterns and practices to be followed in the organization to bring consistency and quality into the development process.

## Patterns and Practices

The following and the patterns and practices implemented in this template.
* Basic Project Structure
* Independent Component Lifecycle
* Minimal Component Documentation
* Autogenerated API documentation
* Dockerized Build, Test and Packaging
* Dockerized Development Environment
* Automated Testing
* Performance Benchmarks
* Deployment Time and Test Configurations
* Zero Onboarding Process
* Loosely Coupled Components
* Deployment-time Composition
* Microservice Packaging
* Synchronous Invocations
* Correlation (Trace) IDs
* Error Propagation
* Logging and Monitoring
* Commandable Interfaces
* Self-documenting External APIs
* Interface Versioning
* Health Checks

### Basic Project Structure

Any component shall have a standard high-level structure that allows
any person without previous training to understand the content of the component
and easily navigate through it's structure. It is followed traditional 
conversions for Java components with a few minor additions.

The project structure has the following (see below).

Source code:

* `/benchmark` - a folder with performance benchmarks for non-functional tests
* `/docker` - a folder with dockerized build scripts
* `/src` - a folder with production code
* `/test` - a folder with functional tests

Java project configuration:

* `pom.xml` - Maven build file

Minimalistic documentation:

* `/docs` - a folder with autogenerated documentation by TypeDoc
* `README.md` - component description
* `CHANGELOG.md` - a list of changes
* `TODO.md` - a list of planned ToDo items
* `LICENSE` - a license for this component

Scripts for dockerized build and test process:

* `component.json` - component descriptor used by automated delivery scripts
* `build.p1` - a script to build (compile) source code
* `test.ps1` - a script to run functional and non-functional tests
* `package.ps1` - a script to package component into a releasable artifact
* `clean.ps1` - a script to clear temporary artifacts left by the build and test process

### Independent Component Life-Cycle

To ensure productive development and long-term sustainability of source code, each component
shall have a totally decoupled independent lifecycle. That means it can be transferred over to someone,
changed, tested and released without relying on other components and their source code.

Most common mistakes that shall be avoided:

* Link code in the same mono-repo outside the component
* Rely on scripts located outside the component
* Use shared test datasets with attempt to reuse them across multiple components

Instead, component owners shall ensure:

* No code or scripts exist outside of the component folder
* Any code dependencies shall be done via formally released references from officiall repositories
* All scripts required for development shall come with the component
* All test data shall be included into or generated by component tests and benchmarks

### Minimal Component Documentation

It is known that most developers are good writers and prefer to skip the documentation step.
However, without documentation new developers and users of the component will have hard time
to understand what the component does and how to use it. 

The path to success is to require an absolute minimum amount of documentation
and provide clear samples and templates that developers can follow without much effort.

The recommended minimal set of documents that shall be written and maintained by developers include:

* `README.md` - a description of the component that descripts it's purpose, key features, steps to develop and use it as well as contact info of responsible person(s)=
* `CHANGELOG.md` - a list of implemented changes broken down by releases with dates and persons who did the changes
* `TODO.md` - a list of planned / recommended changes for the component. This file can also be used by reviewers to document required corrective actions
* `LICENSE` - a text with a license as it's required by law

### Autogenerated API Documentation

In addition to a minimum set of hand-written documentation, it is necessary to have a good descriptions of public APIs for component consumers.
The well written code shall be self-explanatory by itself. But in some cases it is easier to have documentation that explains the API.
This template uses JavaDoc that automatically generates API documentation using the source code and [specially formatted comments](https://typedoc.org/guides/doccomments/). 

To generate API documentation developer shall run the following command manually after he changes the API:

```bash
> docgen.ps1
```

The generated documentation will be placed into the `/docs` folder.

Developers do not need to install any additional tools except docker
as the document generation process is fully dockerized.

In most source control systems the `/docs` folder has to be configured to expose a static website
in order to access the documentation via a web browser. Refer to your responsible DevOps engineer for details.

### Dockerized Build, Test and Packaging

Each component has to be build and tested twice: by developers on their components and by CICD pipeline on build servers.
That process often is far from simple and may contain a few coviats. Developers may have different ways to build and test. 
Most often they use their IDE to do that. However, when someone needs to automate that process on the build server,
he/she has to learn from the component developer all the details of the component. Moreover, when developer changes
something in the way the component  is built or tested, that immediately breaks CICD pipeline and someone has
to spend even more time to go a fix it.

A better way to address that issue it to make developer fully responsible for automation of the build a test process. 
He writes scripts, uses docker to run the required tool chain without installing and configuring them on every computer.
Then those scripts are executed by CICD pipeline and it gets the final release artifact in a predefined location 
according to the component type.

Scripts that automate the build and test process and expected by CICD pipelines are listed below and executed in that order:

* `build.ps1` - a script to build (compile) the component source code
* `test.ps1` - a script to run functional and non-functional tests for the component
* `package.ps1` - a script that packages the final release artifact (if necessary)
* `clean.ps1` - a script that deleted temporary artifacts left by the build and test scripts (sametimes due to unexpected exit)

In addition to the script, the developer and DevOps engineer define metadata for the component in `component.json` file. 
Information there helps to identify the component and drive the automated delivery process correctly.

Example of the `component.json`:
```bash
{
    "name":  "service-basic-pipservices",
    "type": "microservice",
    "language": "java",
    "version": "1.0.0",
    "build": 0,
    "registry": "entinco",
    "artifacts": [
    ]
}
```

To make the automation of the build and test process easier, this template contains all the necessary pieced.

Automated build process:

* `build.ps1` - a script to build (compile) the source code. It shall return 0 exit code for success and non-zero code in case of any errors.
* `/docker/Dockerfile.build` - a dockerfile to generate a docker image that does the build inside and then copy the result of the build into a local file system.

Automated test process:

* `test.ps1` - a script to run functional and non-functional tests. It shall return 0 exit code for success and non-zero in case of any errors.
* `/docker/Dockerfile.test` - a dockerfile to generate a docker image that takes the compiled code and runs the tests
* `/docker/docker-compose.test.yml` - a docker-compose configuration that instantiates all external dependencies required by tests and then executes the tests.

Automated package process:

* `package.ps1` - a script to package the final release artifact. For this type of component is it not used and added here just to provide a complete set of scripts for CICD pipelines.

Automated clean up process:

* `clean.ps1` - a script to perform cleanup after the build and test

Automated API document generation:

* `docgen.ps1` - a script that generates API documentation. It is usually ignored by CICD pipelines and shall be run manually by the component developer.
* `/docker/Dockerfile.docs` - a dockerfile to generate API documentation. Then the created documentation shall be copied over into the local file system into the `/docs` folder.

### Dockerized Development Environment

Some components implement functionality that connects to and uses other components such as databases, 
message brokers, cache services or other application services. Installation and configuration of those services
and connection of the developed component to them represent a serious challenge and may take a good amount of time.

To simplify the process, this template contains a `docker-compose.dev.yml` script in the `/docker` folder that
runs all external dependencies for this component and exposes them to the localhost. That way developers don't need
to spend a minute of the valuable time on their environment and only need an IDE and docker installed
on their development machines. 

For more details look at the **Zero Onboarding Process** chapter below.

### Automated Testing

Automated testing is absolutely necessary to implement stable and high-quality components.
They include functional testing that validate "what" component is doing, and non-functional tests that validate "how" it is doing it.

Functional tests usually include unit and integration tests. Unit tests validate each piece of code (class or component) in isolation.
While integration tests validate how component interacts with external services (they are started by docker-compose files).
Developers shall write tests to check "happy path", boundary and negative conditions and follow other testing best practices.

Non-functional tests validate things like performance, concurrency, reliability, scalability, capacity, and other non-functional characteristics.
Developers may implement non-functional tests in form of benchmarks that execute a critical transaction (operation), checks response
and measures elapsed time and number of runs. Then benchmarks can be configured to test various scenarios:

* For **performance** run a benchmark as many times as possible and check number of runs or/and response time
* For **concurrency** run a benchmarks in severall parallel threads and check for invalid response
* For **reliability** run a benchmark for a long time or/and simulate failure states. Check for # of invalid responses and recovery time
* For **capacity** run a benchmark on a specific system configuration with realistic volume of data and check # of runs.
* For **scalability** run capacity tests several times on different configurations to check how capacity changes when configuration is scaled up or down

Functional tests located in the `/test` folder, and non-functional tests are in the `/benchmark` folder.

In addition to hand-written tests, the component can be checked automatically by linters to perform static code analysis.
The linter may also be included into the `test.ps1` script and shall return non-zero exit code is the static code analysis detects critical
violation of coding standards. That way they will be treated as errors and developers will be forced to fix them as soon as possible.

### Deployment-time and Test Configurations

Since docker is the primary mechanism to execute tests and often used to package and run backend, it is recommended to use environment variables
as the standard mechanism to pass deployment time configurations such as connection parameters, modes of operations, user credentials and so forth.
Other deployment platforms such as AWS Lambda, GCP Cloud Functions or Azure Functions support environment variables as well making them ubiquites.

So, when a component has a deployment-time configuration, or a test needs a configuration to connect to external services, then use environment variables
to pass them over.

Another trick is to set defaults to configuration parameters to point tests and logic to dockerized development environment instantiated by `/docker/docker-compose.dev.yml`. 
That allows to have a shortcut for new developers who are starting to work the component and haven't learned all necessary steps to configure it.
More about that read in the next chapter.

## Zero Onboarding Process

Onboarding time may become a critical metric for medium or large development teams. When new developers are introduced into a team,
they may spend hours or even days to learn a component they assigned to, install and configure dependencies, fix builds until they are able to make changes.
It is very sad to see, when developer waste several hours to make the component work, before they can make a quick fix that takes just a few minutes. 
Such a waste!

To address that important issue, it is recommemded to create a component such a way that new developers will spend virtually no time for onboarding
After pulling the code from source code repository, they shall be able to learn it and make work in less than 5 minutes!

Here is how that process may look like:

1. Clone repository with the component code
2. Read README.md to learn how to work with the component (that's the longest step)
3. Run `> build.ps1` to compile the code and `> test.ps1` to execute tests
4. Run `> docker-compose -f ./docker/docker-compose.dev/yml` to instantiate external dependencies. Then open the code in IDE and start working with the code.

When a new developer is able to run the component and its tests, he can start pocking the code and make changes.
Rerunning the tests allows to ensure that nothing is broken.

To allow that quick and easy onboarding process a few things need to be done:

1. The code repository has to have a clear plain structure that make all the components highly visible. That way developer can quickly locate the require component and clone it locally. (See *Recommendations for Mono- and Multi-repos*)
2. The component shall have a clear standard structure. It has to follow industry best practices, and be intuitively clear to locate necessary parts. (See *Basic Project Structure* chapter)
3. The component shall have a README.md file that describes steps to run and test the component (See *Minimal Component Documentation* chapter)
4. Build, test and packaging processes shall be dockerized. And automation scripts shall be placed into the root component folder where they can be quickly found (See *Dockerized Build, Test and Packaging* chapter)
5. The component shall have automated tests to validate it is working as expected (See *Automated Testing* chapter)
6. External infrastructure and application services, required for the component and its tests shall be dockerized and launched by a single docker-compose script (See *Dockerized Development Environment* chapter)
7. Component deployment-time configurations and test configurations shall be passed via environment variables when possible. And the variables shall be defaulted to the dockerized development environment (See *Deployment-time and Test Configurations* chapter)

## How to Use This Template

This template is created to turn it into a real microservice using simple Search & Replace:

* Replace **sample** with **<yourlibraryname>** in files, Java namespaces and text notes (Lower case).
* Replace **Sample** with **<YourLibraryName>** in files, class names and text notes (Mixed case).
* Refactor the code and implement the microservices following your design. 
* Update the component documentation and license information.

## Additional Information

This template was developed by Enterprise Innovation Consulting.
To learn more about our services visit [http://www.entinco.com](http://www.entinco.com)