/*
 * Decompiled with CFR 0.152.
 */
package org.powertac.factoredcustomer;

import java.util.HashMap;
import java.util.Map;
import org.apache.log4j.Logger;
import org.joda.time.DateTime;
import org.powertac.common.TariffSubscription;
import org.powertac.common.TimeService;
import org.powertac.common.Timeslot;
import org.powertac.common.WeatherReport;
import org.powertac.common.repo.WeatherReportRepo;
import org.powertac.common.spring.SpringApplicationContext;
import org.powertac.factoredcustomer.CapacityBundle;
import org.powertac.factoredcustomer.CapacityProfile;
import org.powertac.factoredcustomer.CustomerProfile;
import org.powertac.factoredcustomer.ParserFunctions;
import org.powertac.factoredcustomer.TimeseriesGenerator;
import org.w3c.dom.Element;

final class CapacityManager {
    private static Logger log = Logger.getLogger((String)CapacityManager.class.getName());
    private TimeService timeService;
    private WeatherReportRepo weatherReportRepo;
    private final double SMOOTHING_WEIGHT = 0.4;
    private final TimeseriesGenerator tsGenerator;
    private final CustomerProfile customerProfile;
    private final CapacityProfile capacityProfile;
    private final Map<Integer, Double> baseCapacities = new HashMap<Integer, Double>();
    private final Map<Integer, Double> adjCapacities = new HashMap<Integer, Double>();

    CapacityManager(CustomerProfile customer, CapacityBundle bundle, Element xml) {
        this.customerProfile = customer;
        this.capacityProfile = new CapacityProfile(xml, bundle);
        this.timeService = (TimeService)SpringApplicationContext.getBean((String)"timeService");
        this.weatherReportRepo = (WeatherReportRepo)SpringApplicationContext.getBean((String)"weatherReportRepo");
        this.tsGenerator = this.capacityProfile.baseCapacityType == CapacityProfile.BaseCapacityType.TIMESERIES ? new TimeseriesGenerator(this.capacityProfile.baseTimeseriesProfile) : null;
    }

    private double getPopulationRatio(int customerCount, int population) {
        return (double)customerCount / (double)population;
    }

    public double getBaseCapacity(Timeslot timeslot) {
        Double ret = this.baseCapacities.get(timeslot.getSerialNumber());
        if (ret == null) {
            ret = this.drawBaseCapacitySample(timeslot);
        }
        return ret;
    }

    public double drawBaseCapacitySample(Timeslot timeslot) {
        double baseCapacity = 0.0;
        switch (this.capacityProfile.baseCapacityType) {
            case POPULATION: {
                baseCapacity = this.capacityProfile.basePopulationCapacity.drawSample();
                break;
            }
            case INDIVIDUAL: {
                for (int i = 0; i < this.customerProfile.customerInfo.getPopulation(); ++i) {
                    double draw = this.capacityProfile.baseIndividualCapacity.drawSample();
                    baseCapacity += draw;
                }
                break;
            }
            case TIMESERIES: {
                baseCapacity = this.getBaseCapacityFromTimeseries(timeslot);
                break;
            }
            default: {
                throw new Error(this.getName() + ": Unexpected base capacity type: " + (Object)((Object)this.capacityProfile.baseCapacityType));
            }
        }
        Double prevCapacity = this.baseCapacities.get(timeslot.getSerialNumber() - 1);
        if (prevCapacity != null) {
            baseCapacity = 0.4 * prevCapacity + 0.6 * baseCapacity;
        }
        baseCapacity = this.truncateTo2Decimals(baseCapacity);
        this.baseCapacities.put(timeslot.getSerialNumber(), baseCapacity);
        return baseCapacity;
    }

    private double getBaseCapacityFromTimeseries(Timeslot timeslot) {
        try {
            return this.tsGenerator.generateNext(timeslot);
        }
        catch (ArrayIndexOutOfBoundsException e) {
            log.error((Object)(this.getName() + ": Tried to get base capacity from time series at index beyond maximum!"));
            throw e;
        }
    }

    public double useCapacity(Timeslot timeslot, TariffSubscription subscription) {
        double baseCapacity = this.getBaseCapacity(timeslot);
        if (Double.isNaN(baseCapacity)) {
            throw new Error("Base capacity is NaN!");
        }
        this.logCapacityDetails(this.getName() + ": Base capacity for timeslot " + timeslot.getSerialNumber() + " = " + baseCapacity);
        double adjustedCapacity = baseCapacity;
        adjustedCapacity = this.adjustCapacityForPopulationRatio(adjustedCapacity, subscription);
        adjustedCapacity = this.adjustCapacityForPeriodicSkew(adjustedCapacity);
        adjustedCapacity = this.adjustCapacityForWeather(timeslot, adjustedCapacity);
        adjustedCapacity = this.adjustCapacityForTariffRates(timeslot, subscription, adjustedCapacity);
        if (Double.isNaN(adjustedCapacity)) {
            throw new Error("Adjusted capacity is NaN for base capacity = " + baseCapacity);
        }
        adjustedCapacity = this.truncateTo2Decimals(adjustedCapacity);
        this.adjCapacities.put(timeslot.getSerialNumber(), adjustedCapacity);
        log.info((Object)(this.getName() + ": Adjusted capacity for tariff " + subscription.getTariff().getId() + " = " + adjustedCapacity));
        return adjustedCapacity;
    }

    private double adjustCapacityForPopulationRatio(double capacity, TariffSubscription subscription) {
        double popRatio = this.getPopulationRatio(subscription.getCustomersCommitted(), this.customerProfile.customerInfo.getPopulation());
        this.logCapacityDetails(this.getName() + ": population ratio = " + popRatio);
        return capacity * popRatio;
    }

    private double adjustCapacityForPeriodicSkew(double capacity) {
        DateTime now = this.timeService.getCurrentDateTime();
        int day = now.getDayOfWeek();
        int hour = now.getHourOfDay();
        double periodicSkew = this.capacityProfile.dailySkew[day - 1] * this.capacityProfile.hourlySkew[hour];
        this.logCapacityDetails(this.getName() + ": periodic skew = " + periodicSkew);
        return capacity * periodicSkew;
    }

    private double adjustCapacityForWeather(Timeslot timeslot, double capacity) {
        WeatherReport weather = this.weatherReportRepo.currentWeatherReport();
        this.logCapacityDetails(this.getName() + ": weather = (" + weather.getTemperature() + ", " + weather.getWindSpeed() + ", " + weather.getWindDirection() + ", " + weather.getCloudCover() + ")");
        double weatherFactor = 1.0;
        if (this.capacityProfile.temperatureInfluence == CapacityProfile.InfluenceKind.DIRECT) {
            int temperature = (int)Math.round(weather.getTemperature());
            weatherFactor *= this.capacityProfile.temperatureMap.get(temperature).doubleValue();
        } else if (this.capacityProfile.temperatureInfluence == CapacityProfile.InfluenceKind.DEVIATION) {
            int curr = (int)Math.round(weather.getTemperature());
            int ref = (int)Math.round(this.capacityProfile.temperatureReference);
            double deviationFactor = 1.0;
            if (curr > ref) {
                for (int t = ref + 1; t <= curr; ++t) {
                    deviationFactor += this.capacityProfile.temperatureMap.get(t).doubleValue();
                }
            } else if (curr < ref) {
                for (int t = curr; t < ref; ++t) {
                    deviationFactor += this.capacityProfile.temperatureMap.get(t).doubleValue();
                }
            }
            weatherFactor *= deviationFactor;
        }
        if (this.capacityProfile.windSpeedInfluence == CapacityProfile.InfluenceKind.DIRECT) {
            int windSpeed = (int)Math.round(weather.getWindSpeed());
            weatherFactor *= this.capacityProfile.windSpeedMap.get(windSpeed).doubleValue();
            if ((double)windSpeed > 0.0 && this.capacityProfile.windDirectionInfluence == CapacityProfile.InfluenceKind.DIRECT) {
                int windDirection = (int)Math.round(weather.getWindDirection());
                weatherFactor *= this.capacityProfile.windDirectionMap.get(windDirection).doubleValue();
            }
        }
        if (this.capacityProfile.cloudCoverInfluence == CapacityProfile.InfluenceKind.DIRECT) {
            int cloudCover = (int)Math.round(weather.getCloudCover());
            weatherFactor *= this.capacityProfile.cloudCoverMap.get(cloudCover).doubleValue();
        }
        this.logCapacityDetails(this.getName() + ": weather factor = " + weatherFactor);
        return capacity * weatherFactor;
    }

    private double adjustCapacityForTariffRates(Timeslot timeslot, TariffSubscription subscription, double baseCapacity) {
        if (baseCapacity - 0.0 < 0.01) {
            return baseCapacity;
        }
        double chargeForBase = subscription.getTariff().getUsageCharge(timeslot.getStartInstant(), baseCapacity, subscription.getTotalUsage());
        double rateForBase = chargeForBase / baseCapacity;
        double benchmarkRate = this.capacityProfile.benchmarkRates.get(this.timeService.getHourOfDay());
        double rateRatio = rateForBase / benchmarkRate;
        double tariffRatesFactor = this.determineTariffRatesFactor(rateRatio);
        this.logCapacityDetails(this.getName() + ": tariff rates factor = " + tariffRatesFactor);
        return baseCapacity * tariffRatesFactor;
    }

    private double determineTariffRatesFactor(double rateRatio) {
        switch (this.capacityProfile.elasticityModelType) {
            case CONTINUOUS: {
                return this.determineContinuousElasticityFactor(rateRatio);
            }
            case STEPWISE: {
                return this.determineStepwiseElasticityFactor(rateRatio);
            }
        }
        throw new Error("Unexpected elasticity model type: " + (Object)((Object)this.capacityProfile.elasticityModelType));
    }

    private double determineContinuousElasticityFactor(double rateRatio) {
        double percentChange = (rateRatio - 1.0) / 0.01;
        double elasticityRatio = Double.parseDouble(this.capacityProfile.elasticityModelXml.getAttribute("ratio"));
        String range = this.capacityProfile.elasticityModelXml.getAttribute("range");
        String[] minmax = range.split("~");
        double low = Double.parseDouble(minmax[0]);
        double high = Double.parseDouble(minmax[1]);
        return Math.max(low, Math.min(high, 1.0 + percentChange * elasticityRatio));
    }

    private double determineStepwiseElasticityFactor(double rateRatio) {
        double[][] elasticity = null;
        if (elasticity == null) {
            elasticity = ParserFunctions.parseMapToDoubleArray(this.capacityProfile.elasticityModelXml.getAttribute("map"));
        }
        if (Math.abs(rateRatio - 1.0) < 0.01 || elasticity.length == 0) {
            return 1.0;
        }
        if (this.capacityProfile.parentBundle.getCapacityType() == CapacityProfile.CapacityType.CONSUMPTION && rateRatio < 1.0) {
            return 1.0;
        }
        if (this.capacityProfile.parentBundle.getCapacityType() == CapacityProfile.CapacityType.PRODUCTION && rateRatio > 1.0) {
            return 1.0;
        }
        boolean RATE_RATIO_INDEX = false;
        boolean CAPACITY_FACTOR_INDEX = true;
        double rateLowerBound = Double.NEGATIVE_INFINITY;
        double rateUpperBound = Double.POSITIVE_INFINITY;
        double lowerBoundCapacityFactor = 1.0;
        double upperBoundCapacityFactor = 1.0;
        for (int i = 0; i < elasticity.length; ++i) {
            double r = elasticity[i][0];
            if (r <= rateRatio && r > rateLowerBound) {
                rateLowerBound = r;
                lowerBoundCapacityFactor = elasticity[i][1];
            }
            if (!(r >= rateRatio) || !(r < rateUpperBound)) continue;
            rateUpperBound = r;
            upperBoundCapacityFactor = elasticity[i][1];
        }
        return rateRatio < 1.0 ? upperBoundCapacityFactor : lowerBoundCapacityFactor;
    }

    private double truncateTo2Decimals(double x) {
        double fract;
        double whole;
        if (x > 0.0) {
            whole = Math.floor(x);
            fract = Math.floor((x - whole) * 100.0) / 100.0;
        } else {
            whole = Math.ceil(x);
            fract = Math.ceil((x - whole) * 100.0) / 100.0;
        }
        return whole + fract;
    }

    private void logCapacityDetails(String msg) {
        log.debug((Object)msg);
    }

    private String getName() {
        return this.customerProfile.name;
    }

    public String toString() {
        return "CapacityManager:" + this.getName();
    }
}

