/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 * 
 * Copyright 2024 Neil C Smith.
 * 
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3 only, as
 * published by the Free Software Foundation.
 * 
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 for more details.
 * 
 * You should have received a copy of the GNU General Public License version 3
 * along with this work; if not, see http://www.gnu.org/licenses/
 * 
 * 
 * Please visit https://www.praxislive.org if you need additional information or
 * have any questions.
 */
package org.praxislive.ide.audio.options;

import java.util.List;
import java.util.stream.Stream;
import javax.sound.sampled.AudioFormat;
import javax.sound.sampled.AudioSystem;
import javax.sound.sampled.DataLine;
import javax.sound.sampled.Line;
import javax.sound.sampled.Mixer;
import org.praxislive.core.Settings;

final class GeneralAudioPanel extends javax.swing.JPanel {

    private static final String KEY_LIBRARY = "audio.library";
    private static final String KEY_DEVICE = "audio.device";
    private static final String KEY_BUFFERSIZE = "audio.buffersize";
    private static final String KEY_INPUT_DEVICE = "audio.input-device";

    private static final String JAVASOUND = "JavaSound";
    private static final String JACK = "JACK";
    private static final DeviceInfo DEFAULT_DEVICE_INFO = new DeviceInfo(null, 32, 32);

    private static final List<Library> LIBRARIES = List.of(
            new Library(JAVASOUND, "JavaSound"),
            new Library(JACK, "Jack Audio Connection Kit")
    );

    private static final List<Integer> BUFFERSIZES = List.of(
            32, 64, 128, 256, 512, 1024, 2048, 4096
    );

    private final GeneralAudioOptionsPanelController controller;

    GeneralAudioPanel(GeneralAudioOptionsPanelController controller) {
        this.controller = controller;
        initComponents();
        initChoosers();
    }

    /**
     * This method is called from within the constructor to initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is always
     * regenerated by the Form Editor.
     */
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {

        defaultConfigPanel = new javax.swing.JPanel();
        defaultConfigDescription = new javax.swing.JLabel();
        libraryLbl = new javax.swing.JLabel();
        libraryChooser = new javax.swing.JComboBox();
        deviceLbl = new javax.swing.JLabel();
        deviceChooser = new javax.swing.JComboBox();
        buffersizeLbl = new javax.swing.JLabel();
        buffersizeChooser = new javax.swing.JComboBox();
        inputDeviceLbl = new javax.swing.JLabel();
        inputDeviceChooser = new javax.swing.JComboBox();

        defaultConfigPanel.setBorder(javax.swing.BorderFactory.createTitledBorder(org.openide.util.NbBundle.getMessage(GeneralAudioPanel.class, "GeneralAudioPanel.defaultConfigPanel.border.title"))); // NOI18N

        defaultConfigDescription.setForeground(javax.swing.UIManager.getDefaults().getColor("textInactiveText"));
        defaultConfigDescription.setHorizontalAlignment(javax.swing.SwingConstants.RIGHT);
        org.openide.awt.Mnemonics.setLocalizedText(defaultConfigDescription, org.openide.util.NbBundle.getMessage(GeneralAudioPanel.class, "GeneralAudioPanel.defaultConfigDescription.text")); // NOI18N

        org.openide.awt.Mnemonics.setLocalizedText(libraryLbl, org.openide.util.NbBundle.getMessage(GeneralAudioPanel.class, "GeneralAudioPanel.libraryLbl.text")); // NOI18N

        libraryChooser.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                libraryChooserActionPerformed(evt);
            }
        });

        org.openide.awt.Mnemonics.setLocalizedText(deviceLbl, org.openide.util.NbBundle.getMessage(GeneralAudioPanel.class, "GeneralAudioPanel.deviceLbl.text")); // NOI18N

        deviceChooser.setEnabled(false);
        deviceChooser.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                deviceChooserActionPerformed(evt);
            }
        });

        org.openide.awt.Mnemonics.setLocalizedText(buffersizeLbl, org.openide.util.NbBundle.getMessage(GeneralAudioPanel.class, "GeneralAudioPanel.buffersizeLbl.text")); // NOI18N

        buffersizeChooser.setEnabled(false);

        org.openide.awt.Mnemonics.setLocalizedText(inputDeviceLbl, org.openide.util.NbBundle.getMessage(GeneralAudioPanel.class, "GeneralAudioPanel.inputDeviceLbl.text")); // NOI18N

        inputDeviceChooser.setEnabled(false);

        javax.swing.GroupLayout defaultConfigPanelLayout = new javax.swing.GroupLayout(defaultConfigPanel);
        defaultConfigPanel.setLayout(defaultConfigPanelLayout);
        defaultConfigPanelLayout.setHorizontalGroup(
            defaultConfigPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(defaultConfigPanelLayout.createSequentialGroup()
                .addGroup(defaultConfigPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                    .addGroup(defaultConfigPanelLayout.createSequentialGroup()
                        .addGap(42, 42, 42)
                        .addComponent(defaultConfigDescription, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE))
                    .addGroup(defaultConfigPanelLayout.createSequentialGroup()
                        .addContainerGap()
                        .addGroup(defaultConfigPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.TRAILING)
                            .addComponent(buffersizeLbl)
                            .addComponent(libraryLbl)
                            .addComponent(deviceLbl)
                            .addComponent(inputDeviceLbl))
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                        .addGroup(defaultConfigPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                            .addComponent(inputDeviceChooser, 0, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                            .addComponent(buffersizeChooser, 0, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                            .addComponent(libraryChooser, 0, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                            .addComponent(deviceChooser, javax.swing.GroupLayout.Alignment.TRAILING, 0, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE))))
                .addContainerGap())
        );
        defaultConfigPanelLayout.setVerticalGroup(
            defaultConfigPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(defaultConfigPanelLayout.createSequentialGroup()
                .addContainerGap()
                .addGroup(defaultConfigPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(libraryChooser, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                    .addComponent(libraryLbl))
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.UNRELATED)
                .addGroup(defaultConfigPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(deviceChooser, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                    .addComponent(deviceLbl))
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addGroup(defaultConfigPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(inputDeviceChooser, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                    .addComponent(inputDeviceLbl))
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.UNRELATED)
                .addGroup(defaultConfigPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(buffersizeLbl)
                    .addComponent(buffersizeChooser, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED, 205, Short.MAX_VALUE)
                .addComponent(defaultConfigDescription)
                .addContainerGap())
        );

        javax.swing.GroupLayout layout = new javax.swing.GroupLayout(this);
        this.setLayout(layout);
        layout.setHorizontalGroup(
            layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(layout.createSequentialGroup()
                .addContainerGap()
                .addComponent(defaultConfigPanel, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                .addContainerGap())
        );
        layout.setVerticalGroup(
            layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(layout.createSequentialGroup()
                .addContainerGap()
                .addComponent(defaultConfigPanel, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                .addContainerGap())
        );
    }// </editor-fold>//GEN-END:initComponents

    private void libraryChooserActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_libraryChooserActionPerformed
        Object o = libraryChooser.getSelectedItem();
        if (o instanceof Library) {
            initLibrary((Library) o);
        }
    }//GEN-LAST:event_libraryChooserActionPerformed

    private void deviceChooserActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_deviceChooserActionPerformed
        Object o = deviceChooser.getSelectedItem();
        if (o instanceof DeviceInfo info) {
            if (info.name() != null && info.inputs() > 0) {
                inputDeviceChooser.setSelectedItem(DEFAULT_DEVICE_INFO);
                inputDeviceChooser.setEnabled(false);
            } else {
                inputDeviceChooser.setEnabled(inputDeviceChooser.getItemCount() > 1);
            }
        }
    }//GEN-LAST:event_deviceChooserActionPerformed

    private void initChoosers() {
        LIBRARIES.forEach(libraryChooser::addItem);
        BUFFERSIZES.forEach(buffersizeChooser::addItem);
    }

    void load() {
        String libName = Settings.get(KEY_LIBRARY, JAVASOUND);
        Library active = LIBRARIES.stream()
                .filter(l -> l.name().equals(libName))
                .findFirst().orElse(LIBRARIES.getFirst());
        libraryChooser.setSelectedItem(active);
    }

    void store() {
        Object o = libraryChooser.getSelectedItem();
        if (o instanceof Library library) {
            Settings.put(KEY_LIBRARY, library.name());
        }

        o = deviceChooser.getSelectedItem();
        if (o instanceof DeviceInfo info) {
            Settings.put(KEY_DEVICE, info.name());
        }

        o = inputDeviceChooser.getSelectedItem();
        if (o instanceof DeviceInfo info) {
            Settings.put(KEY_INPUT_DEVICE, info.name());
        }

        if (buffersizeChooser.isEnabled()) {
            o = buffersizeChooser.getSelectedItem();
            if (o instanceof Integer buffersize) {
                Settings.putInt(KEY_BUFFERSIZE, buffersize);
            }
        }

    }

    private void initLibrary(Library lib) {
        deviceChooser.removeAllItems();
        inputDeviceChooser.removeAllItems();
        deviceChooser.addItem(DEFAULT_DEVICE_INFO);
        deviceChooser.setSelectedItem(DEFAULT_DEVICE_INFO);
        inputDeviceChooser.addItem(DEFAULT_DEVICE_INFO);
        inputDeviceChooser.setSelectedItem(DEFAULT_DEVICE_INFO);

        if (JAVASOUND.equals(lib.name())) {
            String devName = Settings.get(KEY_DEVICE, "");
            String inDevName = Settings.get(KEY_INPUT_DEVICE, "");

            for (DeviceInfo info : findJavaSoundDevices()) {
                if (info.outputs > 0) {
                    deviceChooser.addItem(info);
                    if (devName.equals(info.name())) {
                        deviceChooser.setSelectedItem(info);
                    }
                } else if (info.inputs > 0) {
                    inputDeviceChooser.addItem(info);
                    if (inDevName.equals(info.name())) {
                        inputDeviceChooser.setSelectedItem(info);
                    }
                }
            }

            deviceChooser.setEnabled(true);
            inputDeviceChooser.setEnabled(inputDeviceChooser.getItemCount() > 1);

            int bsize = Settings.getInt(KEY_BUFFERSIZE, 2048);
            buffersizeChooser.setSelectedItem(bsize);
            buffersizeChooser.setEnabled(true);

        } else {
            deviceChooser.setEnabled(false);
            inputDeviceChooser.setEnabled(false);
            buffersizeChooser.setSelectedItem(2048);
            buffersizeChooser.setEnabled(false);
        }

    }

    private List<DeviceInfo> findJavaSoundDevices() {
        return Stream.of(AudioSystem.getMixerInfo())
                .map(AudioSystem::getMixer)
                .map(mixer -> {
                    return new DeviceInfo(mixer.getMixerInfo().getName(),
                            maxChannels(mixer, true),
                            maxChannels(mixer, false));
                }).toList();
    }

    private int maxChannels(Mixer mixer, boolean input) {
        int max = 0;
        Line.Info[] lines = input ? mixer.getTargetLineInfo() : mixer.getSourceLineInfo();
        for (Line.Info line : lines) {
            if (line instanceof DataLine.Info info) {
                AudioFormat[] formats = info.getFormats();
                for (AudioFormat format : formats) {
                    int channels = format.getChannels();
                    if (channels == AudioSystem.NOT_SPECIFIED) {
                        max = 32;
                    } else if (channels > max) {
                        max = channels;
                    }
                }
            }
        }
        return max;
    }

    boolean valid() {
        // check whether form is consistent and complete
        return true;
    }
    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JComboBox buffersizeChooser;
    private javax.swing.JLabel buffersizeLbl;
    private javax.swing.JLabel defaultConfigDescription;
    private javax.swing.JPanel defaultConfigPanel;
    private javax.swing.JComboBox deviceChooser;
    private javax.swing.JLabel deviceLbl;
    private javax.swing.JComboBox inputDeviceChooser;
    private javax.swing.JLabel inputDeviceLbl;
    private javax.swing.JComboBox libraryChooser;
    private javax.swing.JLabel libraryLbl;
    // End of variables declaration//GEN-END:variables

    private static record Library(String name, String displayName) {

        @Override
        public String toString() {
            return displayName();
        }

    }

    private static record DeviceInfo(String name, int inputs, int outputs) {

        @Override
        public String toString() {
            return name() == null ? "Default Device" : name();
        }

    }

}
