/*
 * Decompiled with CFR 0.152.
 */
package org.praxislive.purl;

import java.io.UnsupportedEncodingException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.Collections;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.TreeMap;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

public final class PackageURL {
    private static final String UTF8 = StandardCharsets.UTF_8.name();
    private static final Pattern PATH_SPLITTER = Pattern.compile("/");
    private final String scheme = this.validateScheme("pkg");
    private final String type;
    private final String namespace;
    private final String name;
    private final String version;
    private final Map<String, String> qualifiers;
    private final String subpath;
    private String canonicalizedForm;

    private PackageURL(String type, String namespace, String name, String version, Map<String, String> qualifiers, String subpath) throws IllegalArgumentException {
        this.type = this.validateType(type);
        this.namespace = this.validateNamespace(namespace);
        this.name = this.validateName(name);
        this.version = this.validateVersion(version);
        this.qualifiers = qualifiers == null ? null : Collections.unmodifiableMap(new TreeMap<String, String>(PackageURL.validateQualifiers(qualifiers)));
        this.subpath = PackageURL.validatePath(subpath, true);
        this.verifyTypeConstraints(this.type, this.namespace, this.name);
    }

    String scheme() {
        return this.scheme;
    }

    public String type() {
        return this.type;
    }

    public Optional<String> namespace() {
        return Optional.ofNullable(this.namespace);
    }

    public String name() {
        return this.name;
    }

    public Optional<String> version() {
        return Optional.ofNullable(this.version);
    }

    public Optional<Map<String, String>> qualifiers() {
        return Optional.ofNullable(this.qualifiers);
    }

    public Optional<String> subpath() {
        return Optional.ofNullable(this.subpath);
    }

    private String validateScheme(String value) throws IllegalArgumentException {
        if ("pkg".equals(value)) {
            return "pkg";
        }
        throw new IllegalArgumentException("The PackageURL scheme is invalid");
    }

    private String validateType(String value) throws IllegalArgumentException {
        if (value == null || value.isEmpty()) {
            throw new IllegalArgumentException("The PackageURL type cannot be null or empty");
        }
        if (value.charAt(0) >= '0' && value.charAt(0) <= '9') {
            throw new IllegalArgumentException("The PackageURL type cannot start with a number");
        }
        String retVal = value.toLowerCase(Locale.ENGLISH);
        if (retVal.chars().anyMatch(c -> !(c == 46 || c == 43 || c == 45 || c >= 97 && c <= 122 || c >= 48 && c <= 57))) {
            throw new IllegalArgumentException("The PackageURL type contains invalid characters");
        }
        return retVal;
    }

    private String validateNamespace(String value) throws IllegalArgumentException {
        if (value == null || value.isEmpty()) {
            return null;
        }
        return this.validateNamespace(value.split("/"));
    }

    private String validateNamespace(String[] values) throws IllegalArgumentException {
        if (values == null || values.length == 0) {
            return null;
        }
        String tempNamespace = PackageURL.validatePath(values, false);
        return switch (this.type) {
            case "bitbucket", "deb", "github", "golang", "rpm" -> tempNamespace.toLowerCase(Locale.ENGLISH);
            default -> tempNamespace;
        };
    }

    private String validateName(String value) throws IllegalArgumentException {
        if (value == null || value.isEmpty()) {
            throw new IllegalArgumentException("The PackageURL name specified is invalid");
        }
        return switch (this.type) {
            case "bitbucket", "deb", "github", "golang" -> value.toLowerCase(Locale.ENGLISH);
            case "pypi" -> value.replaceAll("_", "-").toLowerCase(Locale.ENGLISH);
            default -> value;
        };
    }

    private String validateVersion(String value) {
        if (value == null) {
            return null;
        }
        return value;
    }

    private static Map<String, String> validateQualifiers(Map<String, String> values) throws IllegalArgumentException {
        if (values == null) {
            return null;
        }
        for (Map.Entry<String, String> entry : values.entrySet()) {
            PackageURL.validateKey(entry.getKey());
            String value = entry.getValue();
            if (value != null && !value.isEmpty()) continue;
            throw new IllegalArgumentException("The PackageURL specified contains a qualifier key with an empty or null value");
        }
        return values;
    }

    private static String validateKey(String value) throws IllegalArgumentException {
        if (value == null || value.isEmpty()) {
            throw new IllegalArgumentException("Qualifier key is invalid: " + value);
        }
        String retValue = value.toLowerCase(Locale.ENGLISH);
        if (value.charAt(0) >= '0' && value.charAt(0) <= '9' || !value.chars().allMatch(c -> c >= 97 && c <= 122 || c >= 48 && c <= 57 || c == 46 || c == 45 || c == 95)) {
            throw new IllegalArgumentException("Qualifier key is invalid: " + value);
        }
        return retValue;
    }

    private static String validatePath(String value, boolean isSubpath) throws IllegalArgumentException {
        if (value == null || value.isEmpty()) {
            return null;
        }
        return PackageURL.validatePath(value.split("/"), isSubpath);
    }

    private static String validatePath(String[] segments, boolean isSubpath) throws IllegalArgumentException {
        if (segments == null || segments.length == 0) {
            return null;
        }
        return Arrays.stream(segments).map(segment -> {
            if (isSubpath && ("..".equals(segment) || ".".equals(segment))) {
                throw new IllegalArgumentException("Segments in the subpath may not be a period ('.') or repeated period ('..')");
            }
            if (segment.contains("/")) {
                throw new IllegalArgumentException("Segments in the namespace and subpath may not contain a forward slash ('/')");
            }
            if (segment.isEmpty()) {
                throw new IllegalArgumentException("Segments in the namespace and subpath may not be empty");
            }
            return segment;
        }).collect(Collectors.joining("/"));
    }

    public String toString() {
        return this.canonicalize();
    }

    public String canonicalize() {
        if (this.canonicalizedForm != null) {
            return this.canonicalizedForm;
        }
        StringBuilder purl = new StringBuilder();
        purl.append(this.scheme).append(":");
        if (this.type != null) {
            purl.append(this.type);
        }
        purl.append("/");
        if (this.namespace != null) {
            purl.append(this.encodePath(this.namespace));
            purl.append("/");
        }
        if (this.name != null) {
            purl.append(PackageURL.percentEncode(this.name));
        }
        if (this.version != null) {
            purl.append("@").append(PackageURL.percentEncode(this.version));
        }
        if (this.qualifiers != null && this.qualifiers.size() > 0) {
            purl.append("?");
            this.qualifiers.entrySet().stream().forEachOrdered(entry -> {
                purl.append(((String)entry.getKey()).toLowerCase(Locale.ENGLISH));
                purl.append("=");
                purl.append(PackageURL.percentEncode((String)entry.getValue()));
                purl.append("&");
            });
            purl.setLength(purl.length() - 1);
        }
        if (this.subpath != null) {
            purl.append("#").append(this.encodePath(this.subpath));
        }
        this.canonicalizedForm = purl.toString();
        return this.canonicalizedForm;
    }

    public URI toURI() {
        return URI.create(this.canonicalize());
    }

    private static String percentEncode(String input) {
        try {
            return URLEncoder.encode(input, UTF8).replace("+", "%20").replace("*", "%2A").replace("%7E", "~");
        }
        catch (UnsupportedEncodingException e) {
            return input;
        }
    }

    private static String percentDecode(String input) {
        if (input == null) {
            return null;
        }
        try {
            String decoded = URLDecoder.decode(input, UTF8);
            if (!decoded.equals(input)) {
                return decoded;
            }
        }
        catch (UnsupportedEncodingException e) {
            return input;
        }
        return input;
    }

    private void verifyTypeConstraints(String type, String namespace, String name) throws IllegalArgumentException {
        if ("maven".equals(type) && (namespace == null || namespace.isEmpty() || name == null || name.isEmpty())) {
            throw new IllegalArgumentException("The PackageURL specified is invalid. Maven requires both a namespace and name.");
        }
    }

    private static Map<String, String> parseQualifiers(String encodedString) throws IllegalArgumentException {
        Map results = Arrays.stream(encodedString.split("&")).collect(TreeMap::new, (map, value) -> {
            String[] entry = value.split("=", 2);
            if (entry.length == 2 && !entry[1].isEmpty() && map.put(entry[0].toLowerCase(Locale.ENGLISH), PackageURL.percentDecode(entry[1])) != null) {
                throw new IllegalArgumentException("Duplicate package qualifier encountere - more then one value was specified for " + entry[0].toLowerCase(Locale.ENGLISH));
            }
        }, Map::putAll);
        return PackageURL.validateQualifiers(results);
    }

    private static String[] parsePath(String value, boolean isSubpath) throws IllegalArgumentException {
        if (value == null || value.isEmpty()) {
            return null;
        }
        return (String[])PATH_SPLITTER.splitAsStream(value).filter(segment -> !segment.isEmpty() && (!isSubpath || !".".equals(segment) && !"..".equals(segment))).map(segment -> PackageURL.percentDecode(segment)).toArray(String[]::new);
    }

    private String encodePath(String path) {
        return Arrays.stream(path.split("/")).map(segment -> PackageURL.percentEncode(segment)).collect(Collectors.joining("/"));
    }

    public boolean isCoordinatesEquals(PackageURL purl) {
        return Objects.equals(this.scheme, purl.scheme) && Objects.equals(this.type, purl.type) && Objects.equals(this.namespace, purl.namespace) && Objects.equals(this.name, purl.name) && Objects.equals(this.version, purl.version);
    }

    public boolean isCanonicalEquals(PackageURL purl) {
        return this.canonicalize().equals(purl.canonicalize());
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        PackageURL other = (PackageURL)o;
        return Objects.equals(this.scheme, other.scheme) && Objects.equals(this.type, other.type) && Objects.equals(this.namespace, other.namespace) && Objects.equals(this.name, other.name) && Objects.equals(this.version, other.version) && Objects.equals(this.qualifiers, other.qualifiers) && Objects.equals(this.subpath, other.subpath);
    }

    public int hashCode() {
        return Objects.hash(this.scheme, this.type, this.namespace, this.name, this.version, this.qualifiers, this.subpath);
    }

    public static PackageURL parse(String purl) {
        if (purl == null || purl.trim().isEmpty()) {
            throw new IllegalArgumentException("Invalid purl: Contains an empty or null value");
        }
        try {
            URI uri = new URI(purl);
            return PackageURL.from(uri);
        }
        catch (URISyntaxException e) {
            throw new IllegalArgumentException("Invalid purl: " + e.getMessage());
        }
    }

    public static PackageURL from(URI uri) throws IllegalArgumentException {
        String namespace;
        String name;
        String version;
        int start;
        int end;
        Map<String, String> qualifiers;
        StringBuilder remainder;
        int index;
        if (uri.getUserInfo() != null || uri.getPort() != -1) {
            throw new IllegalArgumentException("Invalid purl: Contains parts not supported by the purl spec");
        }
        String scheme = uri.getScheme();
        String subpath = null;
        if (uri.getRawFragment() != null && !uri.getRawFragment().isEmpty()) {
            subpath = PackageURL.validatePath(PackageURL.parsePath(uri.getRawFragment(), true), true);
        }
        if ((index = (remainder = new StringBuilder(uri.getRawSchemeSpecificPart())).lastIndexOf("?")) >= 0) {
            qualifiers = PackageURL.parseQualifiers(remainder.substring(index + 1));
            remainder.setLength(index);
        } else {
            qualifiers = null;
        }
        for (end = remainder.length() - 1; end > 0 && '/' == remainder.charAt(end); --end) {
        }
        if (end < remainder.length() - 1) {
            remainder.setLength(end + 1);
        }
        for (start = 0; start < remainder.length() && '/' == remainder.charAt(start); ++start) {
        }
        index = remainder.indexOf("/", start);
        if (index <= start) {
            throw new IllegalArgumentException("Invalid purl: does not contain both a type and name");
        }
        String type = remainder.substring(start, index);
        start = index + 1;
        index = remainder.lastIndexOf("@");
        if (index >= start) {
            version = PackageURL.percentDecode(remainder.substring(index + 1));
            remainder.setLength(index);
        } else {
            version = null;
        }
        index = remainder.lastIndexOf("/");
        if (index <= start) {
            name = PackageURL.percentDecode(remainder.substring(start));
            namespace = null;
        } else {
            name = PackageURL.percentDecode(remainder.substring(index + 1));
            remainder.setLength(index);
            namespace = remainder.substring(start);
        }
        return new PackageURL(type, namespace, name, version, qualifiers, subpath);
    }

    public static Builder builder() {
        return new Builder();
    }

    static class StandardTypes {
        public static final String BITBUCKET = "bitbucket";
        public static final String CARGO = "cargo";
        public static final String COMPOSER = "composer";
        public static final String DEBIAN = "deb";
        public static final String DOCKER = "docker";
        public static final String GEM = "gem";
        public static final String GENERIC = "generic";
        public static final String GITHUB = "github";
        public static final String GOLANG = "golang";
        public static final String HEX = "hex";
        public static final String MAVEN = "maven";
        public static final String NPM = "npm";
        public static final String NUGET = "nuget";
        public static final String PYPI = "pypi";
        public static final String RPM = "rpm";

        StandardTypes() {
        }
    }

    public static final class Builder {
        private String type = null;
        private String namespace = null;
        private String name = null;
        private String version = null;
        private String subpath = null;
        private Map<String, String> qualifiers = null;

        private Builder() {
        }

        public Builder withType(String type) {
            this.type = type;
            return this;
        }

        public Builder withNamespace(String namespace) {
            this.namespace = namespace;
            return this;
        }

        public Builder withName(String name) {
            this.name = name;
            return this;
        }

        public Builder withVersion(String version) {
            this.version = version;
            return this;
        }

        public Builder withSubpath(String subpath) {
            this.subpath = subpath;
            return this;
        }

        public Builder withQualifier(String key, String value) {
            if (this.qualifiers == null) {
                this.qualifiers = new TreeMap<String, String>();
            }
            this.qualifiers.put(key, value);
            return this;
        }

        public PackageURL build() {
            return new PackageURL(this.type, this.namespace, this.name, this.version, this.qualifiers, this.subpath);
        }
    }
}

