/*
 * Decompiled with CFR 0.152.
 */
package org.projectnessie.gc.identify;

import com.google.common.base.Preconditions;
import com.google.errorprone.annotations.CanIgnoreReturnValue;
import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.util.List;
import java.util.Optional;
import java.util.Spliterator;
import java.util.UUID;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.ForkJoinTask;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.stream.Stream;
import javax.annotation.Nullable;
import org.immutables.value.Value;
import org.projectnessie.error.NessieNotFoundException;
import org.projectnessie.gc.contents.AddContents;
import org.projectnessie.gc.contents.LiveContentSetsRepository;
import org.projectnessie.gc.identify.ContentToContentReference;
import org.projectnessie.gc.identify.ContentTypeFilter;
import org.projectnessie.gc.identify.CutoffPolicy;
import org.projectnessie.gc.identify.ImmutableIdentifyLiveContents;
import org.projectnessie.gc.identify.PerRefCutoffPolicySupplier;
import org.projectnessie.gc.identify.ReferenceComparator;
import org.projectnessie.gc.identify.VisitedDeduplicator;
import org.projectnessie.gc.repository.RepositoryConnector;
import org.projectnessie.model.CommitMeta;
import org.projectnessie.model.Content;
import org.projectnessie.model.ContentKey;
import org.projectnessie.model.Detached;
import org.projectnessie.model.LogResponse;
import org.projectnessie.model.Operation;
import org.projectnessie.model.Reference;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Value.Immutable
public abstract class IdentifyLiveContents {
    private static final Logger LOGGER = LoggerFactory.getLogger(IdentifyLiveContents.class);
    public static final int DEFAULT_PARALLELISM = 4;
    private final AtomicBoolean executed = new AtomicBoolean();

    public static Builder builder() {
        return ImmutableIdentifyLiveContents.builder();
    }

    public UUID identifyLiveContents() {
        if (!this.executed.compareAndSet(false, true)) {
            throw new IllegalStateException("identifyLiveContents() has already been called.");
        }
        ForkJoinPool forkJoinPool = new ForkJoinPool(this.parallelism());
        try {
            UUID uUID = forkJoinPool.invoke(ForkJoinTask.adapt(this::walkAllReferences));
            return uUID;
        }
        finally {
            forkJoinPool.shutdown();
        }
    }

    private UUID walkAllReferences() {
        AddContents addContents = this.liveContentSetsRepository().newAddContents();
        try {
            Stream<Reference> refs = this.repositoryConnector().allReferences();
            ReferenceComparator refsCmp = this.referenceComparator();
            if (refsCmp != null) {
                refs = refs.sorted(refsCmp);
            }
            Optional<ReferencesWalkResult> result = ((Stream)refs.parallel()).map(ref -> this.identifyContentsForReference(addContents, (Reference)ref)).reduce(ReferencesWalkResult::add);
            LOGGER.info("live-set#{}: Finished walking all named references, took {}: {}.", new Object[]{addContents.id(), Duration.between(addContents.created(), this.clock().instant()), result.isPresent() ? result.get() : "<no result>"});
            addContents.finished();
            UUID uUID = addContents.id();
            return uUID;
        }
        catch (NessieNotFoundException e) {
            LOGGER.error("Failed to walk all references.", (Throwable)e);
            addContents.finishedExceptionally(e);
            throw new RuntimeException(e);
        }
        catch (RuntimeException e) {
            LOGGER.error("Failed to walk all references.", (Throwable)e);
            addContents.finishedExceptionally(e);
            throw e;
        }
        finally {
            if (addContents != null) {
                try {
                    addContents.close();
                }
                catch (Throwable throwable) {
                    Throwable throwable2;
                    throwable2.addSuppressed(throwable);
                }
            }
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private ReferencesWalkResult identifyContentsForReference(AddContents addContents, Reference namedReference) {
        CutoffPolicy cutoffPolicy = this.cutOffPolicySupplier().get(namedReference);
        if (this.visitedDeduplicator().alreadyVisited(cutoffPolicy.timestamp(), namedReference.getHash())) {
            LOGGER.debug("live-set#{}: Not submitting task to walk {}, it has already already visited using acompatible cut-off timestamp.", (Object)addContents.id(), (Object)namedReference);
            return ReferencesWalkResult.singleShortCircuit(0, 0L);
        }
        LOGGER.info("live-set#{}: Start walking the commit log of {} using {}.", new Object[]{addContents.id(), namedReference, cutoffPolicy});
        int numCommits = 0;
        long numContents = 0L;
        try (Stream<LogResponse.LogEntry> commits = this.repositoryConnector().commitLog(namedReference);){
            LogEntryHolder holder = new LogEntryHolder();
            String lastCommitId = null;
            Spliterator<LogResponse.LogEntry> spliterator = commits.spliterator();
            while (spliterator.tryAdvance(holder::set)) {
                ++numCommits;
                LogResponse.LogEntry logEntry = holder.logEntry;
                CommitMeta commitMeta = logEntry.getCommitMeta();
                Instant commitTime = commitMeta.getCommitTime();
                String commitHash = commitMeta.getHash();
                List operations = logEntry.getOperations();
                if (operations == null) continue;
                if (commitTime == null) throw new IllegalStateException("Mandatory information is null in log entry " + logEntry);
                if (commitHash == null) {
                    throw new IllegalStateException("Mandatory information is null in log entry " + logEntry);
                }
                if (this.visitedDeduplicator().alreadyVisited(cutoffPolicy.timestamp(), commitHash)) {
                    LOGGER.info("live-set#{}: Finished walking the commit log of {} using {} after {} commits, commit {} has already been checked using a compatible cut-off timestamp.", new Object[]{addContents.id(), namedReference, cutoffPolicy, numCommits, commitHash});
                    ReferencesWalkResult referencesWalkResult = ReferencesWalkResult.singleShortCircuit(numCommits, numContents);
                    return referencesWalkResult;
                }
                if (lastCommitId == null || !cutoffPolicy.isCutoff(commitTime, numCommits)) {
                    LOGGER.debug("live-set#{}: Checking commit {} with {} operations via {}.", new Object[]{addContents.id(), commitHash, operations.size(), namedReference});
                    lastCommitId = commitHash;
                    numContents += addContents.addLiveContent(operations.stream().filter(operation -> operation instanceof Operation.Put).filter(operation -> this.contentTypeFilter().test(((Operation.Put)operation).getContent().getType())).map(operation -> {
                        Operation.Put put = (Operation.Put)operation;
                        Content content = put.getContent();
                        LOGGER.debug("live-set#{}: Adding content reference for {} from commit {}.", new Object[]{addContents.id(), put, commitHash});
                        return this.contentToContentReference().contentToReference(content, commitHash, put.getKey());
                    }));
                    continue;
                }
                try {
                }
                catch (NessieNotFoundException e) {
                    throw new RuntimeException(e);
                }
                LOGGER.info("live-set#{}: Finished walking the commit log of {} using {} after {} commits, commit {} is the first non-live commit.", new Object[]{addContents.id(), namedReference, cutoffPolicy, numCommits, commitHash});
                ReferencesWalkResult referencesWalkResult = ReferencesWalkResult.single(numCommits, numContents += this.collectAllKeys(addContents, Detached.of(lastCommitId)));
                return referencesWalkResult;
            }
        }
        catch (NessieNotFoundException e) {
            throw new RuntimeException("GC-run#" + addContents.id() + ": Could not find reference " + namedReference, e);
        }
        LOGGER.info("live-set#{}: Finished walking the commit log of {} using {} after {} commits, no more commits.", new Object[]{addContents.id(), namedReference, cutoffPolicy, numCommits});
        return ReferencesWalkResult.single(numCommits, numContents);
    }

    private long collectAllKeys(AddContents addContents, Detached ref) throws NessieNotFoundException {
        return addContents.addLiveContent(this.repositoryConnector().allContents(ref, this.contentTypeFilter().validTypes()).map(e -> this.contentToContentReference().contentToReference((Content)e.getValue(), ref.getHash(), (ContentKey)e.getKey())));
    }

    abstract PerRefCutoffPolicySupplier cutOffPolicySupplier();

    abstract ContentTypeFilter contentTypeFilter();

    abstract LiveContentSetsRepository liveContentSetsRepository();

    abstract ContentToContentReference contentToContentReference();

    abstract RepositoryConnector repositoryConnector();

    @Value.Default
    Clock clock() {
        return Clock.systemUTC();
    }

    @Value.Default
    VisitedDeduplicator visitedDeduplicator() {
        return VisitedDeduplicator.NOOP;
    }

    @Nullable
    abstract ReferenceComparator referenceComparator();

    @Value.Default
    int parallelism() {
        return 4;
    }

    @Value.Check
    void verify() {
        Preconditions.checkArgument((this.parallelism() >= 1 ? 1 : 0) != 0, (Object)"Parallelism must be greater than 0");
    }

    private static final class LogEntryHolder {
        LogResponse.LogEntry logEntry;

        private LogEntryHolder() {
        }

        void set(LogResponse.LogEntry logEntry) {
            this.logEntry = logEntry;
        }
    }

    private static final class ReferencesWalkResult {
        final int numReferences;
        final int numCommits;
        final int shortCircuits;
        final long numContents;

        private ReferencesWalkResult(int numReferences, int numCommits, int shortCircuits, long numContents) {
            this.numReferences = numReferences;
            this.numCommits = numCommits;
            this.shortCircuits = shortCircuits;
            this.numContents = numContents;
        }

        static ReferencesWalkResult singleShortCircuit(int numCommits, long numContents) {
            return new ReferencesWalkResult(1, numCommits, 1, numContents);
        }

        static ReferencesWalkResult single(int numCommits, long numContents) {
            return new ReferencesWalkResult(1, numCommits, 0, numContents);
        }

        ReferencesWalkResult add(ReferencesWalkResult other) {
            return new ReferencesWalkResult(this.numReferences + other.numReferences, this.numCommits + other.numCommits, this.shortCircuits + other.shortCircuits, this.numContents + other.numContents);
        }

        public String toString() {
            return "numReferences=" + this.numReferences + ", numCommits=" + this.numCommits + ", numContents=" + this.numContents + ", shortCircuits=" + this.shortCircuits;
        }
    }

    public static interface Builder {
        @CanIgnoreReturnValue
        public Builder cutOffPolicySupplier(PerRefCutoffPolicySupplier var1);

        @CanIgnoreReturnValue
        public Builder contentTypeFilter(ContentTypeFilter var1);

        @CanIgnoreReturnValue
        public Builder liveContentSetsRepository(LiveContentSetsRepository var1);

        @CanIgnoreReturnValue
        public Builder contentToContentReference(ContentToContentReference var1);

        @CanIgnoreReturnValue
        public Builder repositoryConnector(RepositoryConnector var1);

        @CanIgnoreReturnValue
        public Builder visitedDeduplicator(VisitedDeduplicator var1);

        @CanIgnoreReturnValue
        public Builder referenceComparator(ReferenceComparator var1);

        @CanIgnoreReturnValue
        public Builder parallelism(int var1);

        public IdentifyLiveContents build();
    }
}

