/*
 * Decompiled with CFR 0.152.
 */
package org.projectnessie.gc.iceberg;

import com.google.common.base.Preconditions;
import com.google.errorprone.annotations.CanIgnoreReturnValue;
import com.google.errorprone.annotations.MustBeClosed;
import jakarta.annotation.Nonnull;
import java.io.IOException;
import java.net.URI;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;
import org.apache.iceberg.ManifestFile;
import org.apache.iceberg.ManifestReaderUtil;
import org.apache.iceberg.PartitionSpec;
import org.apache.iceberg.PartitionStatisticsFile;
import org.apache.iceberg.Snapshot;
import org.apache.iceberg.StatisticsFile;
import org.apache.iceberg.TableMetadata;
import org.apache.iceberg.TableMetadataParser;
import org.apache.iceberg.io.CloseableIterable;
import org.apache.iceberg.io.FileIO;
import org.apache.iceberg.io.InputFile;
import org.apache.iceberg.view.ViewMetadata;
import org.apache.iceberg.view.ViewMetadataParser;
import org.immutables.value.Value;
import org.projectnessie.gc.contents.ContentReference;
import org.projectnessie.gc.expire.ContentToFiles;
import org.projectnessie.gc.files.FileReference;
import org.projectnessie.gc.iceberg.ImmutableIcebergContentToFiles;
import org.projectnessie.model.Content;
import org.projectnessie.storage.uri.StorageUri;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Value.Immutable
public abstract class IcebergContentToFiles
implements ContentToFiles {
    private static final Logger LOGGER = LoggerFactory.getLogger(IcebergContentToFiles.class);
    static final String ICEBERG_NOT_FOUND_EXCEPTION = "org.apache.iceberg.exceptions.NotFoundException";
    static final String S3_KEY_NOT_FOUND_EXCEPTION = "software.amazon.awssdk.services.s3.model.NoSuchKeyException";
    static final String GCS_STORAGE_EXCEPTION = "com.google.cloud.storage.StorageException";
    static final String ADLS_BLOB_STORAGE_EXCEPTION = "com.azure.storage.blob.models.BlobStorageException";
    static final String GCS_NOT_FOUND_START = "404 Not Found";
    static final String ADLS_PATH_NOT_FOUND_CODE = "PathNotFound";
    static final String ADLS_BLOB_NOT_FOUND_CODE = "BlobNotFound";

    public static Builder builder() {
        return ImmutableIcebergContentToFiles.builder();
    }

    abstract FileIO io();

    @MustBeClosed
    public Stream<FileReference> extractFiles(ContentReference contentReference) {
        Content.Type contentType = contentReference.contentType();
        if (contentType.equals((Object)Content.Type.ICEBERG_TABLE)) {
            return this.extractTableFiles(contentReference);
        }
        if (contentType.equals((Object)Content.Type.ICEBERG_VIEW)) {
            return this.extractViewFiles(contentReference);
        }
        throw new IllegalArgumentException("Expect ICEBERG_TABLE/ICEBERG_VIEW, but got " + String.valueOf(contentType));
    }

    private Stream<FileReference> extractViewFiles(ContentReference contentReference) {
        ViewMetadata viewMetadata;
        FileIO io = this.io();
        try {
            viewMetadata = ViewMetadataParser.read((InputFile)io.newInputFile(contentReference.metadataLocation()));
        }
        catch (Exception notFoundCandidate) {
            return IcebergContentToFiles.handleNotFound(contentReference, notFoundCandidate, "View");
        }
        Stream<StorageUri> allFiles = Stream.of(IcebergContentToFiles.metadataStorageUri(contentReference));
        StorageUri baseUri = IcebergContentToFiles.baseUri(viewMetadata, contentReference);
        return IcebergContentToFiles.extractFilesRelativize(allFiles, baseUri);
    }

    private Stream<FileReference> extractTableFiles(ContentReference contentReference) {
        TableMetadata tableMetadata;
        FileIO io = this.io();
        try {
            tableMetadata = TableMetadataParser.read((FileIO)io, (String)contentReference.metadataLocation());
        }
        catch (Exception notFoundCandidate) {
            return IcebergContentToFiles.handleNotFound(contentReference, notFoundCandidate, "Table");
        }
        long snapshotId = Objects.requireNonNull(contentReference.snapshotId(), "Iceberg content is expected to have a non-null snapshot-ID");
        Snapshot snapshot = snapshotId < 0L ? tableMetadata.currentSnapshot() : tableMetadata.snapshot(snapshotId);
        Stream<StorageUri> allFiles = IcebergContentToFiles.elementaryUrisFromSnapshot(snapshot, contentReference);
        if (snapshot != null) {
            long effectiveSnapshotId = snapshot.snapshotId();
            allFiles = Stream.concat(allFiles, tableMetadata.statisticsFiles().stream().filter(s -> s.snapshotId() == effectiveSnapshotId).map(StatisticsFile::path).map(StorageUri::of));
            allFiles = Stream.concat(allFiles, tableMetadata.partitionStatisticsFiles().stream().filter(s -> s.snapshotId() == effectiveSnapshotId).map(PartitionStatisticsFile::path).map(StorageUri::of));
            Map specsById = tableMetadata.specsById();
            allFiles = Stream.concat(allFiles, Stream.of("").flatMap(x -> {
                try {
                    Stream<StorageUri> r = IcebergContentToFiles.allManifestsAndDataFiles(io, snapshot, specsById, contentReference);
                    return r;
                }
                catch (Exception e) {
                    String msg = "Failed to get manifest files for " + String.valueOf(contentReference.contentType()) + " " + String.valueOf(contentReference.contentKey()) + ", content-ID " + contentReference.contentId() + " at commit " + contentReference.commitId() + " via " + contentReference.metadataLocation();
                    LOGGER.error("{}", (Object)msg, (Object)e);
                    throw new RuntimeException(msg, e);
                }
            }));
        }
        StorageUri baseUri = IcebergContentToFiles.baseUri(tableMetadata, contentReference);
        return IcebergContentToFiles.extractFilesRelativize(allFiles, baseUri);
    }

    private static Stream<FileReference> extractFilesRelativize(Stream<StorageUri> allFiles, StorageUri baseUri) {
        return allFiles.map(arg_0 -> ((StorageUri)baseUri).relativize(arg_0)).map(u -> FileReference.of((StorageUri)u, (StorageUri)baseUri, (long)-1L));
    }

    private static Stream<FileReference> handleNotFound(ContentReference contentReference, Exception notFoundCandidate, String kind) {
        boolean notFound = false;
        for (Throwable c = notFoundCandidate; c != null; c = c.getCause()) {
            switch (c.getClass().getName()) {
                case "org.apache.iceberg.exceptions.NotFoundException": 
                case "software.amazon.awssdk.services.s3.model.NoSuchKeyException": {
                    notFound = true;
                    break;
                }
                case "com.google.cloud.storage.StorageException": {
                    if (!c.getMessage().startsWith(GCS_NOT_FOUND_START)) break;
                    notFound = true;
                    break;
                }
                case "com.azure.storage.blob.models.BlobStorageException": {
                    String message = c.getMessage();
                    if (!message.contains(ADLS_PATH_NOT_FOUND_CODE) && !message.contains(ADLS_BLOB_NOT_FOUND_CODE)) break;
                    notFound = true;
                    break;
                }
            }
            if (notFound || c == c.getCause()) break;
        }
        if (notFound) {
            LOGGER.info("{} metadata {} for snapshot ID {} for content-key {} at Nessie commit {} does not exist, probably already deleted, assuming no files", new Object[]{kind, contentReference.metadataLocation(), contentReference.snapshotId(), contentReference.contentKey(), contentReference.commitId()});
            return Stream.empty();
        }
        String msg = "Failed to extract content of " + String.valueOf(contentReference.contentType()) + " " + String.valueOf(contentReference.contentKey()) + ", content-ID " + contentReference.contentId() + " at commit " + contentReference.commitId() + " via " + contentReference.metadataLocation();
        LOGGER.error("{}", (Object)msg, (Object)notFoundCandidate);
        throw new RuntimeException(msg, notFoundCandidate);
    }

    @MustBeClosed
    static Stream<StorageUri> allManifestsAndDataFiles(FileIO io, Snapshot snapshot, Map<Integer, PartitionSpec> specsById, ContentReference contentReference) {
        return IcebergContentToFiles.allManifests(io, specsById, snapshot).flatMap(mf -> {
            StorageUri manifestFileLoc = IcebergContentToFiles.manifestFileUri(mf, contentReference);
            Stream<StorageUri> allDataAndDeleteFiles = IcebergContentToFiles.allDataAndDeleteFiles(io, specsById, mf, contentReference);
            return Stream.concat(Stream.of(manifestFileLoc), allDataAndDeleteFiles);
        });
    }

    static Stream<ManifestFile> allManifests(FileIO io, Map<Integer, PartitionSpec> specsById, Snapshot snapshot) {
        return snapshot.allManifests(io).stream();
    }

    @MustBeClosed
    static Stream<StorageUri> allDataAndDeleteFiles(FileIO io, Map<Integer, PartitionSpec> specsById, ManifestFile manifestFile, ContentReference contentReference) {
        CloseableIterable<String> iter;
        try {
            iter = ManifestReaderUtil.readPathsFromManifest(manifestFile, specsById, io);
        }
        catch (Exception e) {
            throw new RuntimeException("Failed to get paths from manifest file " + manifestFile.path(), e);
        }
        return (Stream)StreamSupport.stream(iter.spliterator(), false).map(dataFilePath -> IcebergContentToFiles.dataFileUri(dataFilePath, contentReference)).onClose(() -> {
            try {
                iter.close();
            }
            catch (IOException e) {
                throw new RuntimeException(e);
            }
        });
    }

    static StorageUri checkUri(String type, String location, ContentReference contentReference) {
        StorageUri loc = StorageUri.of((String)location);
        if (loc.scheme() == null) {
            Preconditions.checkArgument((boolean)location.startsWith("/"), (String)"Iceberg content reference points to the %s URI '%s' as content-key %s on commit %s without a scheme and with a relative path, which is not supported.", (Object)type, (Object)location, (Object)contentReference.contentKey(), (Object)contentReference.commitId());
            return StorageUri.of((String)("file://" + location));
        }
        if ("file".equals(loc.scheme())) {
            URI uri = URI.create(location);
            Preconditions.checkArgument((boolean)uri.getSchemeSpecificPart().startsWith("/"), (String)"Iceberg content reference points to the %s URI '%s' as content-key %s on commit %s with a non-absolute scheme-specific-part %s, which is not supported.", (Object[])new Object[]{type, uri, contentReference.contentKey(), contentReference.commitId(), uri.getSchemeSpecificPart()});
            Preconditions.checkArgument((uri.getHost() == null ? 1 : 0) != 0, (String)"Iceberg content reference points to the host-specific %s URI '%s' as content-key %s on commit %s without a scheme, which is not supported.", (Object)type, (Object)location, (Object)contentReference.contentKey(), (Object)contentReference.commitId());
        }
        return loc;
    }

    static Stream<StorageUri> elementaryUrisFromSnapshot(Snapshot snapshot, ContentReference contentReference) {
        StorageUri metadataLoc = IcebergContentToFiles.metadataStorageUri(contentReference);
        if (snapshot == null) {
            return Stream.of(metadataLoc);
        }
        String manifestListLocation = snapshot.manifestListLocation();
        if (manifestListLocation == null) {
            return Stream.of(metadataLoc);
        }
        StorageUri manifestListLoc = IcebergContentToFiles.checkUri("manifest list", snapshot.manifestListLocation(), contentReference);
        return Stream.of(metadataLoc, manifestListLoc);
    }

    private static StorageUri metadataStorageUri(ContentReference contentReference) {
        String metadataLocation = (String)Preconditions.checkNotNull((Object)contentReference.metadataLocation(), (String)"Iceberg content is expected to have a non-null metadata-location for content-key %s on commit %s", (Object)contentReference.contentKey(), (Object)contentReference.commitId());
        StorageUri metadataLoc = IcebergContentToFiles.checkUri("metadata", metadataLocation, contentReference);
        return metadataLoc;
    }

    static StorageUri baseUri(@Nonnull TableMetadata tableMetadata, @Nonnull ContentReference contentReference) {
        return IcebergContentToFiles.baseUri(tableMetadata.location(), contentReference);
    }

    static StorageUri baseUri(@Nonnull ViewMetadata viewMetadata, @Nonnull ContentReference contentReference) {
        return IcebergContentToFiles.baseUri(viewMetadata.location(), contentReference);
    }

    static StorageUri baseUri(@Nonnull String location, @Nonnull ContentReference contentReference) {
        Object loc = location.endsWith("/") ? location : location + "/";
        return IcebergContentToFiles.checkUri("location", (String)loc, contentReference);
    }

    static StorageUri manifestFileUri(@Nonnull ManifestFile mf, @Nonnull ContentReference contentReference) {
        String manifestFilePath = (String)Preconditions.checkNotNull((Object)mf.path(), (String)"Iceberg manifest file is expected to have a non-null path for content-key %s on commit %s", (Object)contentReference.contentKey(), (Object)contentReference.commitId());
        return IcebergContentToFiles.checkUri("manifest file", manifestFilePath, contentReference);
    }

    static StorageUri dataFileUri(@Nonnull String dataFilePath, @Nonnull ContentReference contentReference) {
        return IcebergContentToFiles.checkUri("data file", dataFilePath, contentReference);
    }

    public static interface Builder {
        @CanIgnoreReturnValue
        public Builder io(FileIO var1);

        public IcebergContentToFiles build();
    }
}

