/*
 * Decompiled with CFR 0.152.
 */
package org.projectnessie.gc.iceberg;

import com.google.common.base.Preconditions;
import com.google.errorprone.annotations.CanIgnoreReturnValue;
import com.google.errorprone.annotations.MustBeClosed;
import jakarta.annotation.Nonnull;
import java.io.IOException;
import java.net.URI;
import java.util.Objects;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;
import org.apache.iceberg.ManifestFile;
import org.apache.iceberg.ManifestReaderUtil;
import org.apache.iceberg.Snapshot;
import org.apache.iceberg.TableMetadata;
import org.apache.iceberg.TableMetadataParser;
import org.apache.iceberg.exceptions.NotFoundException;
import org.apache.iceberg.io.CloseableIterable;
import org.apache.iceberg.io.FileIO;
import org.immutables.value.Value;
import org.projectnessie.gc.contents.ContentReference;
import org.projectnessie.gc.expire.ContentToFiles;
import org.projectnessie.gc.files.FileReference;
import org.projectnessie.gc.iceberg.ImmutableIcebergContentToFiles;
import org.projectnessie.storage.uri.StorageUri;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Value.Immutable
public abstract class IcebergContentToFiles
implements ContentToFiles {
    private static final Logger LOGGER = LoggerFactory.getLogger(IcebergContentToFiles.class);
    public static final String S3_KEY_NOT_FOUND = "software.amazon.awssdk.services.s3.model.NoSuchKeyException";
    public static final String GCS_STORAGE_EXCEPTION = "com.google.cloud.storage.StorageException";
    public static final String ADLS_STORAGE_EXCEPTION = "com.azure.storage.blob.models.BlobStorageException";
    public static final String GCS_NOT_FOUND_START = "404 Not Found";
    public static final String ADLS_NOT_FOUND_CODE = "PathNotFound";

    public static Builder builder() {
        return ImmutableIcebergContentToFiles.builder();
    }

    abstract FileIO io();

    @MustBeClosed
    public Stream<FileReference> extractFiles(ContentReference contentReference) {
        TableMetadata tableMetadata;
        FileIO io = this.io();
        try {
            tableMetadata = TableMetadataParser.read((FileIO)io, (String)contentReference.metadataLocation());
        }
        catch (Exception notFoundCandidate) {
            boolean notFound = false;
            if (notFoundCandidate instanceof NotFoundException || S3_KEY_NOT_FOUND.equals(notFoundCandidate.getClass().getName())) {
                notFound = true;
            } else {
                for (Throwable c = notFoundCandidate; c != null; c = c.getCause()) {
                    String exceptionClass = c.getClass().getName();
                    String message = c.getMessage();
                    if (GCS_STORAGE_EXCEPTION.equals(exceptionClass) && message.startsWith(GCS_NOT_FOUND_START)) {
                        notFound = true;
                        break;
                    }
                    if (ADLS_STORAGE_EXCEPTION.equals(exceptionClass) && message.contains(ADLS_NOT_FOUND_CODE)) {
                        notFound = true;
                        break;
                    }
                    if (c == c.getCause()) break;
                }
            }
            if (notFound) {
                LOGGER.info("Table metadata {} for snapshot ID {} for content-key {} at Nessie commit {} does not exist, probably already deleted, assuming no files", new Object[]{contentReference.metadataLocation(), contentReference.snapshotId(), contentReference.contentKey(), contentReference.commitId()});
                return Stream.empty();
            }
            String msg = "Failed to extract content of " + contentReference.contentType() + " " + contentReference.contentKey() + ", content-ID " + contentReference.contentId() + " at commit " + contentReference.commitId() + " via " + contentReference.metadataLocation();
            LOGGER.error("{}", (Object)msg, (Object)notFoundCandidate);
            throw new RuntimeException(msg, notFoundCandidate);
        }
        long snapshotId = Objects.requireNonNull(contentReference.snapshotId(), "Iceberg content is expected to have a non-null snapshot-ID");
        Snapshot snapshot = snapshotId < 0L ? tableMetadata.currentSnapshot() : tableMetadata.snapshot(snapshotId);
        Stream<StorageUri> allFiles = IcebergContentToFiles.elementaryUrisFromSnapshot(snapshot, contentReference);
        if (snapshot != null) {
            allFiles = Stream.concat(allFiles, Stream.of("").flatMap(x -> {
                try {
                    Stream<StorageUri> r = IcebergContentToFiles.allManifestsAndDataFiles(io, snapshot, contentReference);
                    return r;
                }
                catch (Exception e) {
                    String msg = "Failed to get manifest files for " + contentReference.contentType() + " " + contentReference.contentKey() + ", content-ID " + contentReference.contentId() + " at commit " + contentReference.commitId() + " via " + contentReference.metadataLocation();
                    LOGGER.error("{}", (Object)msg, (Object)e);
                    throw new RuntimeException(msg, e);
                }
            }));
        }
        StorageUri baseUri = IcebergContentToFiles.baseUri(tableMetadata, contentReference);
        return allFiles.map(arg_0 -> ((StorageUri)baseUri).relativize(arg_0)).map(u -> FileReference.of((StorageUri)u, (StorageUri)baseUri, (long)-1L));
    }

    @MustBeClosed
    static Stream<StorageUri> allManifestsAndDataFiles(FileIO io, Snapshot snapshot, ContentReference contentReference) {
        return IcebergContentToFiles.allManifests(io, snapshot).flatMap(mf -> {
            StorageUri manifestFileLoc = IcebergContentToFiles.manifestFileUri(mf, contentReference);
            Stream<StorageUri> allDataAndDeleteFiles = IcebergContentToFiles.allDataAndDeleteFiles(io, mf, contentReference);
            return Stream.concat(Stream.of(manifestFileLoc), allDataAndDeleteFiles);
        });
    }

    static Stream<ManifestFile> allManifests(FileIO io, Snapshot snapshot) {
        return snapshot.allManifests(io).stream();
    }

    @MustBeClosed
    static Stream<StorageUri> allDataAndDeleteFiles(FileIO io, ManifestFile manifestFile, ContentReference contentReference) {
        CloseableIterable<String> iter;
        try {
            iter = ManifestReaderUtil.readPathsFromManifest(manifestFile, io);
        }
        catch (Exception e) {
            throw new RuntimeException("Failed to get paths from manifest file " + manifestFile.path(), e);
        }
        return (Stream)StreamSupport.stream(iter.spliterator(), false).map(dataFilePath -> IcebergContentToFiles.dataFileUri(dataFilePath, contentReference)).onClose(() -> {
            try {
                iter.close();
            }
            catch (IOException e) {
                throw new RuntimeException(e);
            }
        });
    }

    static StorageUri checkUri(String type, String location, ContentReference contentReference) {
        StorageUri loc = StorageUri.of((String)location);
        if (loc.scheme() == null) {
            Preconditions.checkArgument((boolean)location.startsWith("/"), (String)"Iceberg content reference points to the %s URI '%s' as content-key %s on commit %s without a scheme and with a relative path, which is not supported.", (Object)type, (Object)location, (Object)contentReference.contentKey(), (Object)contentReference.commitId());
            return StorageUri.of((String)("file://" + location));
        }
        if ("file".equals(loc.scheme())) {
            URI uri = URI.create(location);
            Preconditions.checkArgument((boolean)uri.getSchemeSpecificPart().startsWith("/"), (String)"Iceberg content reference points to the %s URI '%s' as content-key %s on commit %s with a non-absolute scheme-specific-part %s, which is not supported.", (Object[])new Object[]{type, uri, contentReference.contentKey(), contentReference.commitId(), uri.getSchemeSpecificPart()});
            Preconditions.checkArgument((uri.getHost() == null ? 1 : 0) != 0, (String)"Iceberg content reference points to the host-specific %s URI '%s' as content-key %s on commit %s without a scheme, which is not supported.", (Object)type, (Object)location, (Object)contentReference.contentKey(), (Object)contentReference.commitId());
        }
        return loc;
    }

    static Stream<StorageUri> elementaryUrisFromSnapshot(Snapshot snapshot, ContentReference contentReference) {
        String metadataLocation = (String)Preconditions.checkNotNull((Object)contentReference.metadataLocation(), (String)"Iceberg content is expected to have a non-null metadata-location for content-key %s on commit %s", (Object)contentReference.contentKey(), (Object)contentReference.commitId());
        StorageUri metadataLoc = IcebergContentToFiles.checkUri("metadata", metadataLocation, contentReference);
        if (snapshot == null) {
            return Stream.of(metadataLoc);
        }
        String manifestListLocation = snapshot.manifestListLocation();
        if (manifestListLocation == null) {
            return Stream.of(metadataLoc);
        }
        StorageUri manifestListLoc = IcebergContentToFiles.checkUri("manifest list", snapshot.manifestListLocation(), contentReference);
        return Stream.of(metadataLoc, manifestListLoc);
    }

    static StorageUri baseUri(@Nonnull TableMetadata tableMetadata, @Nonnull ContentReference contentReference) {
        String location = tableMetadata.location();
        String loc = location.endsWith("/") ? location : location + "/";
        return IcebergContentToFiles.checkUri("location", loc, contentReference);
    }

    static StorageUri manifestFileUri(@Nonnull ManifestFile mf, @Nonnull ContentReference contentReference) {
        String manifestFilePath = (String)Preconditions.checkNotNull((Object)mf.path(), (String)"Iceberg manifest file is expected to have a non-null path for content-key %s on commit %s", (Object)contentReference.contentKey(), (Object)contentReference.commitId());
        return IcebergContentToFiles.checkUri("manifest file", manifestFilePath, contentReference);
    }

    static StorageUri dataFileUri(@Nonnull String dataFilePath, @Nonnull ContentReference contentReference) {
        return IcebergContentToFiles.checkUri("data file", dataFilePath, contentReference);
    }

    public static interface Builder {
        @CanIgnoreReturnValue
        public Builder io(FileIO var1);

        public IcebergContentToFiles build();
    }
}

