/*
 * Copyright (C) 2022 Dremio
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.projectnessie.jaxrs.tests;

import static java.util.Collections.singletonList;
import static org.projectnessie.model.Validation.HASH_RULE;
import static org.projectnessie.model.Validation.REF_NAME_MESSAGE;

import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.CsvSource;
import org.projectnessie.error.NessieBadRequestException;
import org.projectnessie.model.CommitMeta;
import org.projectnessie.model.ContentKey;
import org.projectnessie.model.Tag;

/** See {@link AbstractTestRest} for details about and reason for the inheritance model. */
public abstract class AbstractRestInvalid extends AbstractRestInvalidRefs {
  public static final String COMMA_VALID_HASH_1 =
      ",1234567890123456789012345678901234567890123456789012345678901234";
  public static final String COMMA_VALID_HASH_2 = ",1234567890123456789012345678901234567890";
  public static final String COMMA_VALID_HASH_3 = ",1234567890123456";

  @ParameterizedTest
  @CsvSource({
    "x/" + COMMA_VALID_HASH_1,
    "abc'" + COMMA_VALID_HASH_1,
    ".foo" + COMMA_VALID_HASH_2,
    "abc'def'..'blah" + COMMA_VALID_HASH_2,
    "abc'de..blah" + COMMA_VALID_HASH_3,
    "abc'de@{blah" + COMMA_VALID_HASH_3
  })
  public void invalidBranchNames(String invalidBranchName, String validHash) {
    ContentKey key = ContentKey.of("x");

    String opsCountMsg = ".operations.operations: size must be between 1 and 2147483647";

    soft.assertThatThrownBy(
            () ->
                getApi()
                    .commitMultipleOperations()
                    .branchName(invalidBranchName)
                    .hash(validHash)
                    .commitMeta(CommitMeta.fromMessage(""))
                    .commit())
        .isInstanceOf(NessieBadRequestException.class)
        .hasMessageContaining("Bad Request (HTTP/400):")
        .hasMessageContaining(REF_NAME_MESSAGE)
        .hasMessageContaining(opsCountMsg);

    soft.assertThatThrownBy(
            () -> getApi().deleteBranch().branchName(invalidBranchName).hash(validHash).delete())
        .isInstanceOf(NessieBadRequestException.class)
        .hasMessageContaining("Bad Request (HTTP/400):")
        .hasMessageContaining(REF_NAME_MESSAGE);

    soft.assertThatThrownBy(
            () -> getApi().getCommitLog().refName(invalidBranchName).untilHash(validHash).get())
        .isInstanceOf(NessieBadRequestException.class)
        .hasMessageContaining("Bad Request (HTTP/400):")
        .hasMessageContaining(REF_NAME_MESSAGE);

    soft.assertThatThrownBy(
            () -> getApi().getEntries().refName(invalidBranchName).hashOnRef(validHash).get())
        .isInstanceOf(NessieBadRequestException.class)
        .hasMessageContaining("Bad Request (HTTP/400):")
        .hasMessageContaining(REF_NAME_MESSAGE);

    soft.assertThatThrownBy(() -> getApi().getReference().refName(invalidBranchName).get())
        .isInstanceOf(NessieBadRequestException.class)
        .hasMessageContaining("Bad Request (HTTP/400):")
        .hasMessageContaining(REF_NAME_MESSAGE);

    soft.assertThatThrownBy(
            () ->
                getApi()
                    .mergeRefIntoBranch()
                    .branchName(invalidBranchName)
                    .hash(validHash)
                    .fromRef(getApi().getDefaultBranch())
                    .merge())
        .isInstanceOf(NessieBadRequestException.class)
        .hasMessageContaining("Bad Request (HTTP/400):")
        .hasMessageContaining(REF_NAME_MESSAGE);

    soft.assertThatThrownBy(
            () -> getApi().deleteTag().tagName(invalidBranchName).hash(validHash).delete())
        .isInstanceOf(NessieBadRequestException.class)
        .hasMessageContaining("Bad Request (HTTP/400):")
        .hasMessageContaining(REF_NAME_MESSAGE);

    soft.assertThatThrownBy(
            () ->
                getApi()
                    .transplantCommitsIntoBranch()
                    .branchName(invalidBranchName)
                    .hash(validHash)
                    .fromRefName("main")
                    .hashesToTransplant(
                        singletonList(getApi().getReference().refName("main").get().getHash()))
                    .transplant())
        .isInstanceOf(NessieBadRequestException.class)
        .hasMessageContaining("Bad Request (HTTP/400):")
        .hasMessageContaining(REF_NAME_MESSAGE);

    soft.assertThatThrownBy(
            () ->
                getApi()
                    .getContent()
                    .key(key)
                    .refName(invalidBranchName)
                    .hashOnRef(validHash)
                    .get())
        .isInstanceOf(NessieBadRequestException.class)
        .hasMessageContaining("Bad Request (HTTP/400):")
        .hasMessageContaining(REF_NAME_MESSAGE);

    soft.assertThatThrownBy(
            () ->
                getApi()
                    .getContent()
                    .key(key)
                    .refName(invalidBranchName)
                    .hashOnRef(validHash)
                    .get())
        .isInstanceOf(NessieBadRequestException.class)
        .hasMessageContaining("Bad Request (HTTP/400):")
        .hasMessageContaining(REF_NAME_MESSAGE);

    soft.assertThatThrownBy(
            () -> getApi().getDiff().fromRefName(invalidBranchName).toRefName("main").get())
        .isInstanceOf(NessieBadRequestException.class)
        .hasMessageContaining("Bad Request (HTTP/400):")
        .hasMessageContaining(REF_NAME_MESSAGE);
  }

  @ParameterizedTest
  @CsvSource({
    "abc'" + COMMA_VALID_HASH_1,
    ".foo" + COMMA_VALID_HASH_2,
    "abc'def'..'blah" + COMMA_VALID_HASH_2,
    "abc'de..blah" + COMMA_VALID_HASH_3,
    "abc'de@{blah" + COMMA_VALID_HASH_3
  })
  public void invalidHashes(String invalidHashIn, String validHash) {
    // CsvSource maps an empty string as null
    String invalidHash = invalidHashIn != null ? invalidHashIn : "";

    String validBranchName = "hello";
    ContentKey key = ContentKey.of("x");
    Tag tag = Tag.of("valid", validHash);

    String opsCountMsg = ".operations.operations: size must be between 1 and 2147483647";

    soft.assertThatThrownBy(
            () ->
                getApi()
                    .commitMultipleOperations()
                    .branchName(validBranchName)
                    .hash(invalidHash)
                    .commitMeta(CommitMeta.fromMessage(""))
                    .commit())
        .isInstanceOf(NessieBadRequestException.class)
        .hasMessageContaining("Bad Request (HTTP/400):")
        .hasMessageContaining(HASH_RULE)
        .hasMessageContaining(opsCountMsg);
    soft.assertThatThrownBy(
            () -> getApi().deleteBranch().branchName(validBranchName).hash(invalidHash).delete())
        .isInstanceOf(NessieBadRequestException.class)
        .hasMessageContaining("Bad Request (HTTP/400):")
        .hasMessageContaining(HASH_RULE);
    soft.assertThatThrownBy(
            () ->
                getApi()
                    .assignTag()
                    .tagName(validBranchName)
                    .hash(invalidHash)
                    .assignTo(tag)
                    .assign())
        .isInstanceOf(NessieBadRequestException.class)
        .hasMessageContaining("Bad Request (HTTP/400):")
        .hasMessageContaining(HASH_RULE);
    soft.assertThatThrownBy(
            () ->
                getApi()
                    .mergeRefIntoBranch()
                    .branchName(validBranchName)
                    .hash(invalidHash)
                    .fromRef(getApi().getDefaultBranch())
                    .merge())
        .isInstanceOf(NessieBadRequestException.class)
        .hasMessageContaining("Bad Request (HTTP/400):")
        .hasMessageContaining(HASH_RULE);
    soft.assertThatThrownBy(
            () -> getApi().deleteTag().tagName(validBranchName).hash(invalidHash).delete())
        .isInstanceOf(NessieBadRequestException.class)
        .hasMessageContaining("Bad Request (HTTP/400):")
        .hasMessageContaining(HASH_RULE);
    soft.assertThatThrownBy(
            () ->
                getApi()
                    .transplantCommitsIntoBranch()
                    .branchName(validBranchName)
                    .hash(invalidHash)
                    .fromRefName("main")
                    .hashesToTransplant(
                        singletonList(getApi().getReference().refName("main").get().getHash()))
                    .transplant())
        .isInstanceOf(NessieBadRequestException.class)
        .hasMessageContaining("Bad Request (HTTP/400):")
        .hasMessageContaining(HASH_RULE);
    soft.assertThatThrownBy(() -> getApi().getContent().refName(invalidHash).get())
        .isInstanceOf(NessieBadRequestException.class)
        .hasMessageContaining("Bad Request (HTTP/400):")
        .hasMessageContaining("size must be between 1 and 2147483647")
        .hasMessageContaining(REF_NAME_MESSAGE);
    soft.assertThatThrownBy(
            () -> getApi().getContent().refName(validBranchName).hashOnRef(invalidHash).get())
        .isInstanceOf(NessieBadRequestException.class)
        .hasMessageContaining("Bad Request (HTTP/400):")
        .hasMessageContaining("size must be between 1 and 2147483647")
        .hasMessageContaining(HASH_RULE);
    soft.assertThatThrownBy(
            () ->
                getApi()
                    .getContent()
                    .key(key)
                    .refName(validBranchName)
                    .hashOnRef(invalidHash)
                    .get())
        .isInstanceOf(NessieBadRequestException.class)
        .hasMessageContaining("Bad Request (HTTP/400):")
        .hasMessageContaining(HASH_RULE);
    soft.assertThatThrownBy(
            () -> getApi().getCommitLog().refName(validBranchName).untilHash(invalidHash).get())
        .isInstanceOf(NessieBadRequestException.class)
        .hasMessageContaining("Bad Request (HTTP/400):")
        .hasMessageContaining(HASH_RULE);
    soft.assertThatThrownBy(
            () -> getApi().getCommitLog().refName(validBranchName).hashOnRef(invalidHash).get())
        .isInstanceOf(NessieBadRequestException.class)
        .hasMessageContaining("Bad Request (HTTP/400):")
        .hasMessageContaining(HASH_RULE);
    soft.assertThatThrownBy(
            () -> getApi().getEntries().refName(validBranchName).hashOnRef(invalidHash).get())
        .isInstanceOf(NessieBadRequestException.class)
        .hasMessageContaining("Bad Request (HTTP/400):")
        .hasMessageContaining(HASH_RULE);
  }

  @ParameterizedTest
  @CsvSource({
    "abc'" + COMMA_VALID_HASH_1,
    ".foo" + COMMA_VALID_HASH_2,
    "abc'def'..'blah" + COMMA_VALID_HASH_2,
    "abc'de..blah" + COMMA_VALID_HASH_3,
    "abc'de@{blah" + COMMA_VALID_HASH_3
  })
  public void invalidTags(String invalidTagName, String validHash) {
    soft.assertThatThrownBy(
            () ->
                getApi()
                    .assignTag()
                    .tagName(invalidTagName)
                    .hash(validHash)
                    .assignTo(Tag.of("validTag", validHash))
                    .assign())
        .isInstanceOf(NessieBadRequestException.class)
        .hasMessageContaining("Bad Request (HTTP/400):")
        .hasMessageContaining(REF_NAME_MESSAGE);

    soft.assertThatThrownBy(
            () -> getApi().deleteTag().tagName(invalidTagName).hash(validHash).delete())
        .isInstanceOf(NessieBadRequestException.class)
        .hasMessageContaining("Bad Request (HTTP/400):")
        .hasMessageContaining(REF_NAME_MESSAGE);
  }
}
