/*
 * Decompiled with CFR 0.152.
 */
package org.projectnessie.events.service;

import jakarta.annotation.Nullable;
import java.security.Principal;
import java.time.Instant;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import org.projectnessie.events.api.Event;
import org.projectnessie.events.api.EventType;
import org.projectnessie.events.service.EventConfig;
import org.projectnessie.events.service.EventFactory;
import org.projectnessie.events.service.EventSubscribers;
import org.projectnessie.events.service.VersionStoreEvent;
import org.projectnessie.events.spi.EventSubscriber;
import org.projectnessie.events.spi.EventSubscription;
import org.projectnessie.events.spi.ImmutableEventSubscription;
import org.projectnessie.model.Content;
import org.projectnessie.model.ContentKey;
import org.projectnessie.model.Operation;
import org.projectnessie.versioned.BranchName;
import org.projectnessie.versioned.Commit;
import org.projectnessie.versioned.CommitResult;
import org.projectnessie.versioned.Hash;
import org.projectnessie.versioned.MergeResult;
import org.projectnessie.versioned.ReferenceAssignedResult;
import org.projectnessie.versioned.ReferenceCreatedResult;
import org.projectnessie.versioned.ReferenceDeletedResult;
import org.projectnessie.versioned.Result;
import org.projectnessie.versioned.TransplantResult;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.MDC;

public class EventService
implements AutoCloseable {
    public static final String SUBSCRIPTION_ID_MDC_KEY = "nessie.events.subscription.id";
    public static final String EVENT_ID_MDC_KEY = "nessie.events.event.id";
    private static final Logger LOGGER = LoggerFactory.getLogger(EventService.class);
    protected final EventConfig config;
    protected final EventFactory factory;
    protected final EventSubscribers subscribers;
    private volatile boolean started;
    private boolean hasContentSubscribers;
    private boolean hasCommitSubscribers;

    public EventService(EventConfig config, EventFactory factory, EventSubscribers subscribers) {
        this.config = config;
        this.factory = factory;
        this.subscribers = subscribers;
    }

    public synchronized void start() {
        if (!this.started) {
            this.subscribers.start(eventSubscriber -> ImmutableEventSubscription.builder().id(this.config.getIdGenerator().get()).systemConfiguration(this.config.getSystemConfiguration()).build());
            this.hasContentSubscribers = this.subscribers.hasSubscribersFor(EventType.CONTENT_STORED) || this.subscribers.hasSubscribersFor(EventType.CONTENT_REMOVED);
            this.hasCommitSubscribers = this.hasContentSubscribers || this.subscribers.hasSubscribersFor(EventType.COMMIT);
            this.started = true;
        }
    }

    @Override
    public synchronized void close() {
        this.subscribers.close();
    }

    public void onVersionStoreEvent(VersionStoreEvent event) {
        if (!this.started) {
            return;
        }
        Result result = event.getResult();
        Principal user = event.getUser().orElse(null);
        String repositoryId = event.getRepositoryId();
        switch (result.getResultType()) {
            case COMMIT: {
                this.onCommitResult((CommitResult)result, repositoryId, user);
                break;
            }
            case MERGE: {
                this.onMergeResult((MergeResult)result, repositoryId, user);
                break;
            }
            case TRANSPLANT: {
                this.onTransplantResult((TransplantResult)result, repositoryId, user);
                break;
            }
            case REFERENCE_CREATED: {
                this.onReferenceCreatedResult((ReferenceCreatedResult)result, repositoryId, user);
                break;
            }
            case REFERENCE_ASSIGNED: {
                this.onReferenceAssignedResult((ReferenceAssignedResult)result, repositoryId, user);
                break;
            }
            case REFERENCE_DELETED: {
                this.onReferenceDeletedResult((ReferenceDeletedResult)result, repositoryId, user);
                break;
            }
            default: {
                throw new IllegalArgumentException("Unknown result type: " + String.valueOf(result.getResultType()));
            }
        }
    }

    private void onCommitResult(CommitResult result, String repositoryId, @Nullable Principal user) {
        LOGGER.debug("Received commit result: {}", (Object)result);
        this.fireCommitEvent(result.getCommit(), result.getTargetBranch(), repositoryId, user);
    }

    private void onMergeResult(MergeResult result, String repositoryId, @Nullable Principal user) {
        LOGGER.debug("Received merge result: {}", (Object)result);
        this.fireMergeEvent(result, repositoryId, user);
    }

    private void onTransplantResult(TransplantResult result, String repositoryId, @Nullable Principal user) {
        LOGGER.debug("Received transplant result: {}", (Object)result);
        this.fireTransplantEvent(result, repositoryId, user);
    }

    private void onReferenceCreatedResult(ReferenceCreatedResult result, String repositoryId, @Nullable Principal user) {
        LOGGER.debug("Received branch created result: {}", (Object)result);
        this.fireEvent(this.factory.newReferenceCreatedEvent(result, repositoryId, user));
    }

    private void onReferenceAssignedResult(ReferenceAssignedResult result, String repositoryId, @Nullable Principal user) {
        LOGGER.debug("Received reference assigned result: {}", (Object)result);
        this.fireEvent(this.factory.newReferenceUpdatedEvent(result, repositoryId, user));
    }

    private void onReferenceDeletedResult(ReferenceDeletedResult result, String repositoryId, @Nullable Principal user) {
        LOGGER.debug("Received reference deleted result: {}", (Object)result);
        this.fireEvent(this.factory.newReferenceDeletedEvent(result, repositoryId, user));
    }

    private void fireCommitEvent(Commit commit, BranchName targetBranch, String repositoryId, @Nullable Principal user) {
        this.fireEvent(this.factory.newCommitEvent(commit, targetBranch, repositoryId, user));
        if (this.hasContentSubscribers) {
            this.fireContentEvents(commit, targetBranch, repositoryId, user);
        }
    }

    private void fireMergeEvent(MergeResult result, String repositoryId, @Nullable Principal user) {
        this.fireEvent(this.factory.newMergeEvent(result, repositoryId, user));
        if (this.hasCommitSubscribers) {
            for (Commit commit : result.getCreatedCommits()) {
                this.fireCommitEvent(commit, result.getTargetBranch(), repositoryId, user);
            }
        }
    }

    private void fireTransplantEvent(TransplantResult result, String repositoryId, @Nullable Principal user) {
        this.fireEvent(this.factory.newTransplantEvent(result, repositoryId, user));
        if (this.hasCommitSubscribers) {
            for (Commit commit : result.getCreatedCommits()) {
                this.fireCommitEvent(commit, result.getTargetBranch(), repositoryId, user);
            }
        }
    }

    private void fireContentEvents(Commit commit, BranchName targetBranch, String repositoryId, @Nullable Principal user) {
        List operations = commit.getOperations();
        if (operations != null && !operations.isEmpty()) {
            Hash hash = Objects.requireNonNull(commit.getHash());
            Instant commitTime = Objects.requireNonNull(commit.getCommitMeta().getCommitTime());
            for (Operation operation : operations) {
                ContentKey contentKey = operation.getKey();
                if (operation instanceof Operation.Put) {
                    Content content = ((Operation.Put)operation).getContent();
                    this.fireEvent(this.factory.newContentStoredEvent(targetBranch, hash, commitTime, contentKey, content, repositoryId, user));
                    continue;
                }
                if (!(operation instanceof Operation.Delete)) continue;
                this.fireEvent(this.factory.newContentRemovedEvent(targetBranch, hash, commitTime, contentKey, repositoryId, user));
            }
        }
    }

    protected void fireEvent(Event event) {
        LOGGER.debug("Firing {} event: {}", (Object)event.getType(), (Object)event);
        for (Map.Entry<EventSubscription, EventSubscriber> entry : this.subscribers.getSubscriptions().entrySet()) {
            EventSubscription subscription = entry.getKey();
            EventSubscriber subscriber = entry.getValue();
            this.deliverEvent(event, subscriber, subscription);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected void deliverEvent(Event event, EventSubscriber subscriber, EventSubscription subscription) {
        MDC.put((String)SUBSCRIPTION_ID_MDC_KEY, (String)subscription.getIdAsText());
        MDC.put((String)EVENT_ID_MDC_KEY, (String)event.getIdAsText());
        try {
            if (subscriber.accepts(event)) {
                LOGGER.debug("Delivering event to subscriber {}: {}", (Object)subscriber, (Object)event);
                subscriber.onEvent(event);
                LOGGER.debug("Event successfully delivered: {}", (Object)event);
            } else {
                LOGGER.debug("Subscriber rejected event: {}", (Object)event);
            }
        }
        catch (Exception e) {
            LOGGER.error("Event could not be delivered: {}", (Object)event, (Object)e);
        }
        finally {
            MDC.remove((String)SUBSCRIPTION_ID_MDC_KEY);
            MDC.remove((String)EVENT_ID_MDC_KEY);
        }
    }
}

