/*
 * Decompiled with CFR 0.152.
 */
package org.projectnessie.versioned.storage.common.persist;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import jakarta.annotation.Nonnull;
import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.Objects;
import java.util.concurrent.ThreadLocalRandom;
import org.projectnessie.nessie.relocated.protobuf.ByteString;
import org.projectnessie.nessie.relocated.protobuf.UnsafeByteOperations;
import org.projectnessie.versioned.storage.common.persist.ObjIdHasher;
import org.projectnessie.versioned.storage.common.util.Hex;
import org.projectnessie.versioned.storage.common.util.Ser;

public abstract class ObjId {
    public static final ObjId EMPTY_OBJ_ID = ObjIdHasher.objIdHasher("empty").generate();

    public static ObjId zeroLengthObjId() {
        return ObjIdEmpty.INSTANCE;
    }

    public abstract ByteBuffer asByteBuffer();

    public abstract byte[] asByteArray();

    public ByteString asBytes() {
        return UnsafeByteOperations.unsafeWrap((byte[])this.asByteArray());
    }

    public abstract ByteBuffer serializeTo(ByteBuffer var1);

    public abstract int nibbleAt(int var1);

    public abstract int size();

    public abstract int heapSize();

    public abstract int serializedSize();

    public static ObjId objIdFromString(@Nonnull String hash) {
        Objects.requireNonNull(hash);
        int len = hash.length();
        Preconditions.checkArgument((len % 2 == 0 ? 1 : 0) != 0, (String)"hash length needs to be a multiple of two, was %s", (int)len);
        switch (len >> 1) {
            case 0: {
                return ObjIdEmpty.INSTANCE;
            }
            case 32: {
                return new ObjId256(hash);
            }
        }
        return new ObjIdGeneric(hash);
    }

    public static ObjId objIdFromBytes(ByteString bytes) {
        int len = bytes.size();
        switch (len) {
            case 0: {
                return ObjIdEmpty.INSTANCE;
            }
            case 32: {
                return new ObjId256(bytes);
            }
        }
        return new ObjIdGeneric(bytes);
    }

    public static ObjId objIdFromByteArray(byte[] bytes) {
        return ObjId.fromBytes(bytes.length, ByteBuffer.wrap(bytes));
    }

    public static ObjId objIdFromByteBuffer(@Nonnull ByteBuffer bytes) {
        int len = bytes.remaining();
        return ObjId.fromBytes(len, bytes);
    }

    public static ObjId randomObjId() {
        return ObjId256.random();
    }

    public static ObjId deserializeObjId(@Nonnull ByteBuffer bytes) {
        int len = Ser.readVarInt(bytes);
        return ObjId.fromBytes(len, bytes);
    }

    public static void skipObjId(@Nonnull ByteBuffer bytes) {
        int len = Ser.readVarInt(bytes);
        bytes.position(bytes.position() + len);
    }

    private static ObjId fromBytes(int len, ByteBuffer bytes) {
        switch (len) {
            case 0: {
                return ObjIdEmpty.INSTANCE;
            }
            case 32: {
                return new ObjId256(bytes);
            }
        }
        ByteBuffer gen = bytes.duplicate();
        int lim = gen.position() + len;
        gen.limit(lim);
        bytes.position(lim);
        return new ObjIdGeneric(gen);
    }

    public String toString() {
        throw new UnsupportedOperationException("MUST BE IMPLEMENTED");
    }

    static final class ObjIdEmpty
    extends ObjId {
        private static final ByteBuffer BB_EMPTY = ByteBuffer.allocate(0);
        static final ObjId INSTANCE = new ObjIdEmpty();

        private ObjIdEmpty() {
        }

        @Override
        public String toString() {
            return "";
        }

        @Override
        public ByteBuffer asByteBuffer() {
            return BB_EMPTY;
        }

        @Override
        public byte[] asByteArray() {
            return new byte[0];
        }

        @Override
        public ByteBuffer serializeTo(ByteBuffer target) {
            return target.put((byte)0);
        }

        @Override
        public int nibbleAt(int nibbleIndex) {
            throw new IllegalArgumentException("Invalid nibble index " + nibbleIndex);
        }

        @Override
        public int size() {
            return 0;
        }

        @Override
        public int serializedSize() {
            return 1;
        }

        @Override
        public int heapSize() {
            return 16;
        }

        public int hashCode() {
            return 0;
        }

        public boolean equals(Object obj) {
            return obj instanceof ObjIdEmpty;
        }
    }

    @VisibleForTesting
    static final class ObjId256
    extends ObjId {
        private final long l0;
        private final long l1;
        private final long l2;
        private final long l3;

        private ObjId256(String hash) {
            this.l0 = Hex.stringToLong(hash, 0);
            this.l1 = Hex.stringToLong(hash, 16);
            this.l2 = Hex.stringToLong(hash, 32);
            this.l3 = Hex.stringToLong(hash, 48);
        }

        private ObjId256(ByteBuffer bytes) {
            this.l0 = bytes.getLong();
            this.l1 = bytes.getLong();
            this.l2 = bytes.getLong();
            this.l3 = bytes.getLong();
        }

        private ObjId256(long l0, long l1, long l2, long l3) {
            this.l0 = l0;
            this.l1 = l1;
            this.l2 = l2;
            this.l3 = l3;
        }

        private ObjId256(ByteString bytes) {
            this(bytes.asReadOnlyByteBuffer());
        }

        public static ObjId random() {
            ThreadLocalRandom tlr = ThreadLocalRandom.current();
            return new ObjId256(tlr.nextLong(), tlr.nextLong(), tlr.nextLong(), tlr.nextLong());
        }

        @Override
        public int nibbleAt(int nibbleIndex) {
            if (nibbleIndex >= 0) {
                if (nibbleIndex < 16) {
                    return Hex.nibbleFromLong(this.l0, nibbleIndex);
                }
                if (nibbleIndex < 32) {
                    return Hex.nibbleFromLong(this.l1, nibbleIndex - 16);
                }
                if (nibbleIndex < 48) {
                    return Hex.nibbleFromLong(this.l2, nibbleIndex - 32);
                }
                if (nibbleIndex < 64) {
                    return Hex.nibbleFromLong(this.l3, nibbleIndex - 48);
                }
            }
            throw new IllegalArgumentException("Invalid nibble index " + nibbleIndex);
        }

        @Override
        public int size() {
            return 32;
        }

        @Override
        public int serializedSize() {
            return 33;
        }

        @Override
        public int heapSize() {
            return 48;
        }

        @Override
        public String toString() {
            StringBuilder sb = new StringBuilder(64);
            ObjId256.longToString(sb, this.l0);
            ObjId256.longToString(sb, this.l1);
            ObjId256.longToString(sb, this.l2);
            ObjId256.longToString(sb, this.l3);
            return sb.toString();
        }

        private static void longToString(StringBuilder sb, long v) {
            sb.append(Hex.hexChar((byte)(v >> 60)));
            sb.append(Hex.hexChar((byte)(v >> 56)));
            sb.append(Hex.hexChar((byte)(v >> 52)));
            sb.append(Hex.hexChar((byte)(v >> 48)));
            sb.append(Hex.hexChar((byte)(v >> 44)));
            sb.append(Hex.hexChar((byte)(v >> 40)));
            sb.append(Hex.hexChar((byte)(v >> 36)));
            sb.append(Hex.hexChar((byte)(v >> 32)));
            sb.append(Hex.hexChar((byte)(v >> 28)));
            sb.append(Hex.hexChar((byte)(v >> 24)));
            sb.append(Hex.hexChar((byte)(v >> 20)));
            sb.append(Hex.hexChar((byte)(v >> 16)));
            sb.append(Hex.hexChar((byte)(v >> 12)));
            sb.append(Hex.hexChar((byte)(v >> 8)));
            sb.append(Hex.hexChar((byte)(v >> 4)));
            sb.append(Hex.hexChar((byte)v));
        }

        @Override
        public ByteBuffer asByteBuffer() {
            return this.serializeToNoLen(ByteBuffer.allocate(32)).flip();
        }

        @Override
        public byte[] asByteArray() {
            byte[] r = new byte[32];
            this.serializeToNoLen(ByteBuffer.wrap(r));
            return r;
        }

        @Override
        public ByteBuffer serializeTo(ByteBuffer target) {
            return this.serializeToNoLen(target.put((byte)32));
        }

        private ByteBuffer serializeToNoLen(ByteBuffer target) {
            return target.putLong(this.l0).putLong(this.l1).putLong(this.l2).putLong(this.l3);
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (!(o instanceof ObjId256)) {
                return false;
            }
            ObjId256 objectId256 = (ObjId256)o;
            return this.l0 == objectId256.l0 && this.l1 == objectId256.l1 && this.l2 == objectId256.l2 && this.l3 == objectId256.l3;
        }

        public int hashCode() {
            return (int)(this.l0 >> 32);
        }
    }

    @VisibleForTesting
    static final class ObjIdGeneric
    extends ObjId {
        private final byte[] bytes;

        private ObjIdGeneric(String hash) {
            int len = hash.length();
            byte[] bytes = new byte[len >> 1];
            int i = 0;
            int c = 0;
            while (c < len) {
                byte value = (byte)(Hex.nibble(hash.charAt(c++)) << 4);
                bytes[i] = value = (byte)(value | Hex.nibble(hash.charAt(c++)));
                ++i;
            }
            Preconditions.checkArgument((bytes.length <= 256 ? 1 : 0) != 0, (Object)"Hashes longer than 256 bytes are not supported");
            this.bytes = bytes;
        }

        private ObjIdGeneric(ByteBuffer bytes) {
            int length = bytes.remaining();
            this.bytes = new byte[length];
            bytes.duplicate().get(this.bytes, 0, length);
        }

        private ObjIdGeneric(ByteString bytes) {
            this(bytes.asReadOnlyByteBuffer());
        }

        @Override
        public int nibbleAt(int nibbleIndex) {
            byte b = this.bytes[nibbleIndex >> 1];
            if ((nibbleIndex & 1) == 0) {
                b = (byte)(b >> 4);
            }
            return b & 0xF;
        }

        @Override
        public int size() {
            return this.bytes.length;
        }

        @Override
        public int serializedSize() {
            int sz = this.size();
            return sz + Ser.varIntLen(sz);
        }

        @Override
        public int heapSize() {
            return 32 + this.size();
        }

        @Override
        public String toString() {
            int len = this.bytes.length;
            StringBuilder sb = new StringBuilder(2 * len);
            int p = 0;
            int i = 0;
            while (i < len) {
                byte b = this.bytes[p];
                sb.append(Hex.hexChar((byte)(b >> 4)));
                sb.append(Hex.hexChar(b));
                ++i;
                ++p;
            }
            return sb.toString();
        }

        @Override
        public ByteBuffer asByteBuffer() {
            return ByteBuffer.wrap(this.bytes);
        }

        @Override
        public byte[] asByteArray() {
            return Arrays.copyOf(this.bytes, this.bytes.length);
        }

        @Override
        public ByteBuffer serializeTo(ByteBuffer target) {
            return Ser.putVarInt(target, this.bytes.length).put(this.bytes, 0, this.bytes.length);
        }

        public int hashCode() {
            int r = this.bytes.length;
            int p = 0;
            int h = 0;
            if (r > 0) {
                h |= (this.bytes[p++] & 0xFF) << 24;
                --r;
            }
            if (r > 0) {
                h |= (this.bytes[p++] & 0xFF) << 16;
                --r;
            }
            if (r > 0) {
                h |= (this.bytes[p++] & 0xFF) << 8;
                --r;
            }
            if (r > 0) {
                h |= this.bytes[p] & 0xFF;
            }
            return h;
        }

        public boolean equals(Object obj) {
            if (!(obj instanceof ObjIdGeneric)) {
                return false;
            }
            ObjIdGeneric that = (ObjIdGeneric)obj;
            return Arrays.equals(this.bytes, that.bytes);
        }
    }
}

