/*
 * Copyright 2017 Jorge Campins y David Uzcategui
 *
 * Este archivo forma parte de Adalid.
 *
 * Adalid es software libre; usted puede redistribuirlo y/o modificarlo bajo los terminos de la
 * licencia "GNU General Public License" publicada por la Fundacion "Free Software Foundation".
 * Adalid se distribuye con la esperanza de que pueda ser util, pero SIN NINGUNA GARANTIA; sin
 * siquiera las garantias implicitas de COMERCIALIZACION e IDONEIDAD PARA UN PROPOSITO PARTICULAR.
 *
 * Para mas detalles vea la licencia "GNU General Public License" en http://www.gnu.org/licenses
 */
package adalid.core;

import adalid.commons.TLB;
import adalid.commons.bundles.Bundle;
import adalid.commons.interfaces.Wrappable;
import adalid.commons.interfaces.Wrapper;
import adalid.commons.util.KVP;
import adalid.commons.util.ObjUtils;
import adalid.commons.util.StrUtils;
import adalid.commons.util.ThrowableUtils;
import adalid.core.expressions.VariantX;
import adalid.core.interfaces.Artifact;
import adalid.core.interfaces.DataArtifact;
import adalid.core.interfaces.Entity;
import adalid.core.interfaces.Expression;
import adalid.core.interfaces.Parameter;
import adalid.core.interfaces.PersistentEntity;
import adalid.core.interfaces.Property;
import adalid.core.wrappers.ArtifactWrapper;
import java.lang.annotation.Annotation;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Modifier;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Level;
import org.apache.log4j.Logger;

/**
 * @author Jorge Campins
 */
public abstract class AbstractArtifact implements Artifact, Wrappable {

    public static final Locale ENGLISH = Bundle.ENGLISH;

    public static final Locale SPANISH = Bundle.SPANISH;

    public static final Locale PORTUGUESE = Bundle.PORTUGUESE;

    /**
     *
     */
    private static final Logger logger = Logger.getLogger(Artifact.class);

    private static final String EOL = "\n";

    // <editor-fold defaultstate="collapsed" desc="field declarations">
    /**
     *
     */
    private boolean _declared;

    /**
     *
     */
    private Boolean _inherited;

    /**
     *
     */
    private Boolean _inheritedFromAbstract;

    /**
     *
     */
    private String _name;

    /**
     *
     */
    private String _alias;

    /**
     *
     */
    private String _sqlName;

    /**
     *
     */
    private Locale _defaultLocale;

    /**
     *
     */
//  private String _defaultLabel;
    private final Map<Locale, String> _localizedLabel = new LinkedHashMap<>();

    /**
     *
     */
//  private String _defaultShortLabel;
    private final Map<Locale, String> _localizedShortLabel = new LinkedHashMap<>();

    /**
     *
     */
//  private String _defaultCollectionLabel;
    private final Map<Locale, String> _localizedCollectionLabel = new LinkedHashMap<>();

    /**
     *
     */
//  private String _defaultCollectionShortLabel;
    private final Map<Locale, String> _localizedCollectionShortLabel = new LinkedHashMap<>();

    /**
     *
     */
//  private String _defaultDescription;
    private final Map<Locale, String> _localizedDescription = new LinkedHashMap<>();

    /**
     *
     */
//  private String _defaultShortDescription;
    private final Map<Locale, String> _localizedShortDescription = new LinkedHashMap<>();

    /**
     *
     */
//  private String _defaultTooltip;
    private final Map<Locale, String> _localizedTooltip = new LinkedHashMap<>();

    /**
     *
     */
    private Artifact _declaringArtifact;

    /**
     *
     */
    private Field _declaringField;

    /**
     *
     */
    private int _declaringFieldIndex;

    /**
     *
     */
    private int _depth;

    /**
     *
     */
    private int _round;

    /**
     *
     */
    private final Map<Class<? extends Annotation>, Field> _annotations = new LinkedHashMap<>();

    /**
     * platform-specific attributes
     */
    private final Map<String, Object> _attributes = new LinkedHashMap<>();
    // </editor-fold>

    // <editor-fold defaultstate="collapsed" desc="field getters and setters">
    /**
     * @return true if the artifact is declared
     */
    @Override
    public boolean isDeclared() {
        return _declared;
    }

    /**
     * @return true if the artifact is not declared
     */
    @Override
    public boolean isNotDeclared() {
        return !_declared;
    }

    /**
     *
     */
    void setDeclared(String name) {
        setDeclared(name, null);
    }

    /**
     *
     */
    void setDeclared(String name, Artifact declaringArtifact) {
        setDeclared(name, declaringArtifact, null);
    }

    /**
     *
     */
    void setDeclared(String name, Artifact declaringArtifact, Field declaringField) {
        setDeclared(name, declaringArtifact, declaringField, 0);
    }

    /**
     *
     */
    void setDeclared(String name, Artifact declaringArtifact, Field declaringField, int declaringFieldIndex) {
        if (_declared) {
            return;
        }
        setName(name);
        setDeclaringArtifact(declaringArtifact);
        setDeclaringField(declaringField);
        setDeclaringFieldIndex(declaringFieldIndex);
        initializeAnnotations();
        annotate(getNamedClass());
        annotate(getDeclaringField());
        _declared = true;
    }

    /**
     * @return true if the artifact is inherited
     */
    @Override
    public boolean isInherited() {
        return _declared && inherited();
    }

    /**
     * @return true if the artifact is not inherited
     */
    @Override
    public boolean isNotInherited() {
        return _declared && !inherited();
    }

    private boolean inherited() {
        if (_inherited == null) {
            Field declaringField = getDeclaringField();
            Artifact declaringArtifact = getDeclaringArtifact();
            Class<?> declaringFieldClass = declaringField.getDeclaringClass();
            Class<?> declaringFieldNamedClass = XS1.getNamedClass(declaringFieldClass);
            Class<?> declaringArtifactNamedClass = XS1.getNamedClass(declaringArtifact);
            String declaringFieldNamedClassSimpleName = declaringFieldNamedClass.getSimpleName();
            String declaringArtifactNamedClassSimpleName = declaringArtifactNamedClass.getSimpleName();
            _inherited = !declaringFieldNamedClassSimpleName.equals(declaringArtifactNamedClassSimpleName);
        }
        return _inherited;
    }

    /**
     * @return true if the artifact is inherited from an abstract class
     */
    @Override
    public boolean isInheritedFromAbstract() {
        return _declared && inheritedFromAbstract();
    }

    /**
     * @return true if the artifact is not inherited from an abstract class
     */
    @Override
    public boolean isNotInheritedFromAbstract() {
        return _declared && !inheritedFromAbstract();
    }

    private boolean inheritedFromAbstract() {
        if (_inheritedFromAbstract == null) {
            Field declaringField = getDeclaringField();
            Artifact declaringArtifact = getDeclaringArtifact();
            Class<?> declaringFieldClass = declaringField.getDeclaringClass();
            Class<?> declaringFieldNamedClass = XS1.getNamedClass(declaringFieldClass);
            Class<?> declaringArtifactNamedClass = XS1.getNamedClass(declaringArtifact);
            if (Modifier.isAbstract(declaringFieldNamedClass.getModifiers())) {
                String declaringFieldNamedClassSimpleName = declaringFieldNamedClass.getSimpleName();
                String declaringArtifactNamedClassSimpleName = declaringArtifactNamedClass.getSimpleName();
                _inheritedFromAbstract = !declaringFieldNamedClassSimpleName.equals(declaringArtifactNamedClassSimpleName);
            } else {
                _inheritedFromAbstract = false;
            }
        }
        return _inheritedFromAbstract;
    }

    /**
     * @return the name
     */
    @Override
    public String getName() {
        return _name;
    }

    /**
     * @param name the name to set
     */
    void setName(String name) {
        if (_declared) {
            return;
        }
        _name = name;
    }

    /**
     * @return the alias
     */
    @Override
    public String getAlias() {
        return _alias == null ? _name : _alias;
    }

    /**
     * El método setAlias se utiliza para establecer el alias (código alterno) del proyecto. El alias solo puede contener letras minúsculas y números,
     * debe comenzar por una letra, y no puede ser jee2ap101, meta o workspace. Se recomienda utilizar un alias que tenga el nombre de su proyecto
     * como prefijo.
     *
     * Si utiliza la plataforma jee2, el alias del proyecto maestro es el nombre del directorio raíz de los archivos generados; por lo tanto, se debe
     * establecer un alias diferente antes de cada ejecución del método generate.
     *
     * @param alias código alterno del proyecto
     */
    @Override
    public void setAlias(String alias) {
        _alias = alias;
    }

    /**
     * @return the SQL name
     */
    @Override
    public String getSqlName() {
        return _sqlName;
    }

    /**
     * El método setSqlName se utiliza para establecer el nombre SQL del artefacto. Si este método no es ejecutado, el nombre SQL se determina a
     * partir del nombre del artefacto, sustituyendo cada letra mayúscula por un guion bajo (underscore) seguido de la letra convertida en minúscula.
     *
     * @param sqlName nombre SQL del artefacto
     */
    @Override
    public void setSqlName(String sqlName) {
        _sqlName = sqlName;
    }

    /**
     * @return the default locale
     */
    @Override
    public Locale getDefaultLocale() {
        if (_defaultLocale != null) {
            return _defaultLocale;
        }
        if (_declaringArtifact != null) {
            return _declaringArtifact.getDefaultLocale();
        }
        return Bundle.getLocale();
    }

    /**
     * @return the default label
     */
    @Override
    public String getDefaultLabel() {
        return getLocalizedLabel(null);
    }

    /**
     * El método setDefaultLabel se utiliza para establecer la etiqueta del artefacto que se almacena en el archivo de recursos por defecto. En caso
     * de que el archivo de recursos para el idioma seleccionado por el usuario no esté disponible, la interfaz de la aplicación utiliza el archivo de
     * recursos por defecto para obtener el valor de la etiqueta.
     *
     * @param defaultLabel sustantivo singular que se usa como etiqueta del artefacto
     */
    @Override
    public void setDefaultLabel(String defaultLabel) {
        setLocalizedLabel(null, defaultLabel);
    }

    /**
     * @return the default short label
     */
    @Override
    public String getDefaultShortLabel() {
        return getLocalizedShortLabel(null);
    }

    /**
     * El método setDefaultShortLabel se utiliza para establecer la etiqueta corta del artefacto que se almacena en el archivo de recursos por
     * defecto. En caso de que el archivo de recursos para el idioma seleccionado por el usuario no esté disponible, la interfaz de la aplicación
     * utiliza el archivo de recursos por defecto para obtener el valor de la etiqueta.
     *
     * @param defaultShortLabel sustantivo singular, preferiblemente sin complementos, que se usa como etiqueta corta del artefacto
     */
    @Override
    public void setDefaultShortLabel(String defaultShortLabel) {
        setLocalizedShortLabel(null, defaultShortLabel);
    }

    /**
     * @return the default collection label
     */
    @Override
    public String getDefaultCollectionLabel() {
        return getLocalizedCollectionLabel(null);
    }

    /**
     * El método setDefaultCollectionLabel se utiliza para establecer la etiqueta de colección del artefacto que se almacena en el archivo de recursos
     * por defecto. En caso de que el archivo de recursos para el idioma seleccionado por el usuario no esté disponible, la interfaz de la aplicación
     * utiliza el archivo de recursos por defecto para obtener el valor de la etiqueta.
     *
     * @param defaultCollectionLabel sustantivo plural que se usa como etiqueta de colección del artefacto
     */
    @Override
    public void setDefaultCollectionLabel(String defaultCollectionLabel) {
        setLocalizedCollectionLabel(null, defaultCollectionLabel);
    }

    /**
     * @return the default collection short label
     */
    @Override
    public String getDefaultCollectionShortLabel() {
        return getLocalizedCollectionShortLabel(null);
    }

    /**
     * El método setDefaultCollectionShortLabel se utiliza para establecer la etiqueta corta de colección del artefacto que se almacena en el archivo
     * de recursos por defecto. En caso de que el archivo de recursos para el idioma seleccionado por el usuario no esté disponible, la interfaz de la
     * aplicación utiliza el archivo de recursos por defecto para obtener el valor de la etiqueta.
     *
     * @param defaultCollectionShortLabel sustantivo plural, preferiblemente sin complementos, que se usa como etiqueta corta de colección del
     * artefacto
     */
    @Override
    public void setDefaultCollectionShortLabel(String defaultCollectionShortLabel) {
        setLocalizedCollectionShortLabel(null, defaultCollectionShortLabel);
    }

    /**
     * @return the default description
     */
    @Override
    public String getDefaultDescription() {
        return getLocalizedDescription(null);
    }

    /**
     * El método setDefaultDescription se utiliza para establecer la descripción del artefacto que se almacena en el archivo de recursos por defecto.
     * En caso de que el archivo de recursos para el idioma seleccionado por el usuario no esté disponible, la interfaz de la aplicación utiliza el
     * archivo de recursos por defecto para obtener el valor de la descripción.
     *
     * @param defaultDescription una o más oraciones que describen el artefacto
     */
    @Override
    public void setDefaultDescription(String defaultDescription) {
        setLocalizedDescription(null, defaultDescription);
    }

    /**
     * @return the default short description
     */
    @Override
    public String getDefaultShortDescription() {
        return getLocalizedShortDescription(null);
    }

    /**
     * El método setDefaultShortDescription se utiliza para establecer la descripción corta del artefacto que se almacena en el archivo de recursos
     * por defecto. En caso de que el archivo de recursos para el idioma seleccionado por el usuario no esté disponible, la interfaz de la aplicación
     * utiliza el archivo de recursos por defecto para obtener el valor de la descripción.
     *
     * @param defaultShortDescription una o más oraciones que describen brevemente el artefacto
     */
    @Override
    public void setDefaultShortDescription(String defaultShortDescription) {
        setLocalizedShortDescription(null, defaultShortDescription);
    }

    /**
     * @return the default tooltip
     */
    @Override
    public String getDefaultTooltip() {
        return getLocalizedTooltip(null);
    }

    /**
     * El método setDefaultTooltip se utiliza para establecer la descripción emergente (tooltip) del artefacto que se almacena en el archivo de
     * recursos por defecto. En caso de que el archivo de recursos para el idioma seleccionado por el usuario no esté disponible, la interfaz de la
     * aplicación utiliza el archivo de recursos por defecto para obtener el valor de la descripción.
     *
     * @param defaultTooltip una o más oraciones que describen muy brevemente el artefacto
     */
    @Override
    public void setDefaultTooltip(String defaultTooltip) {
        setLocalizedTooltip(null, defaultTooltip);
    }

    /**
     * @param locale the locale for the label
     * @return the localized label
     */
    @Override
    public String getLocalizedLabel(Locale locale) {
        Locale l = localeReadingKey(locale);
        return _localizedLabel.get(l);
    }

    /**
     * El método setLocalizedLabel se utiliza para establecer la etiqueta del artefacto que se almacena en el archivo de recursos de configuración
     * regional. En caso de que el archivo de recursos para el idioma seleccionado por el usuario no esté disponible, la interfaz de la aplicación
     * utiliza el archivo de recursos por defecto para obtener el valor de la etiqueta.
     *
     * @param locale configuración regional
     * @param localizedLabel sustantivo singular que se usa como etiqueta del artefacto
     */
    @Override
    public void setLocalizedLabel(Locale locale, String localizedLabel) {
        Locale l = localeWritingKey(locale);
        if (localizedLabel == null) {
            _localizedLabel.remove(l);
        } else {
            _localizedLabel.put(l, localizedLabel);
        }
        setLocalizedShortLabel(locale, localizedLabel);
    }

    /**
     * @param locale the locale for the short label
     * @return the localized short label
     */
    @Override
    public String getLocalizedShortLabel(Locale locale) {
        Locale l = localeReadingKey(locale);
        return _localizedShortLabel.get(l);
    }

    /**
     * El método setLocalizedShortLabel se utiliza para establecer la etiqueta corta del artefacto que se almacena en el archivo de recursos de
     * configuración regional. En caso de que el archivo de recursos para el idioma seleccionado por el usuario no esté disponible, la interfaz de la
     * aplicación utiliza el archivo de recursos por defecto para obtener el valor de la etiqueta.
     *
     * @param locale configuración regional
     * @param localizedShortLabel sustantivo singular, preferiblemente sin complementos, que se usa como etiqueta corta del artefacto
     */
    @Override
    public void setLocalizedShortLabel(Locale locale, String localizedShortLabel) {
        Locale l = localeWritingKey(locale);
        if (localizedShortLabel == null) {
            _localizedShortLabel.remove(l);
        } else {
            _localizedShortLabel.put(l, localizedShortLabel);
        }
    }

    /**
     * @param locale the locale for the collection label
     * @return the localized collection label
     */
    @Override
    public String getLocalizedCollectionLabel(Locale locale) {
        Locale l = localeReadingKey(locale);
        return _localizedCollectionLabel.get(l);
    }

    /**
     * El método setLocalizedCollectionLabel se utiliza para establecer la etiqueta de colección del artefacto que se almacena en el archivo de
     * recursos de configuración regional. En caso de que el archivo de recursos para el idioma seleccionado por el usuario no esté disponible, la
     * interfaz de la aplicación utiliza el archivo de recursos por defecto para obtener el valor de la etiqueta.
     *
     * @param locale configuración regional
     * @param localizedCollectionLabel sustantivo plural que se usa como etiqueta de colección del artefacto
     */
    @Override
    public void setLocalizedCollectionLabel(Locale locale, String localizedCollectionLabel) {
        Locale l = localeWritingKey(locale);
        if (localizedCollectionLabel == null) {
            _localizedCollectionLabel.remove(l);
        } else {
            _localizedCollectionLabel.put(l, localizedCollectionLabel);
        }
        setLocalizedCollectionShortLabel(locale, localizedCollectionLabel);
    }

    /**
     * @param locale the locale for the collection short label
     * @return the localized collection short label
     */
    @Override
    public String getLocalizedCollectionShortLabel(Locale locale) {
        Locale l = localeReadingKey(locale);
        return _localizedCollectionShortLabel.get(l);
    }

    /**
     * El método setLocalizedCollectionShortLabel se utiliza para establecer la etiqueta corta de colección del artefacto que se almacena en el
     * archivo de recursos de configuración regional. En caso de que el archivo de recursos para el idioma seleccionado por el usuario no esté
     * disponible, la interfaz de la aplicación utiliza el archivo de recursos por defecto para obtener el valor de la etiqueta.
     *
     * @param locale configuración regional
     * @param localizedCollectionShortLabel sustantivo plural, preferiblemente sin complementos, que se usa como etiqueta corta de colección del
     * artefacto
     */
    @Override
    public void setLocalizedCollectionShortLabel(Locale locale, String localizedCollectionShortLabel) {
        Locale l = localeWritingKey(locale);
        if (localizedCollectionShortLabel == null) {
            _localizedCollectionShortLabel.remove(l);
        } else {
            _localizedCollectionShortLabel.put(l, localizedCollectionShortLabel);
        }
    }

    /**
     * @param locale the locale for the description
     * @return the localized description
     */
    @Override
    public String getLocalizedDescription(Locale locale) {
        Locale l = localeReadingKey(locale);
        return _localizedDescription.get(l);
    }

    /**
     * El método setLocalizedDescription se utiliza para establecer la descripción del artefacto que se almacena en el archivo de recursos de
     * configuración regional. En caso de que el archivo de recursos para el idioma seleccionado por el usuario no esté disponible, la interfaz de la
     * aplicación utiliza el archivo de recursos por defecto para obtener el valor de la descripción.
     *
     * @param locale configuración regional
     * @param localizedDescription una o más oraciones que describen el artefacto
     */
    @Override
    public void setLocalizedDescription(Locale locale, String localizedDescription) {
        Locale l = localeWritingKey(locale);
        if (localizedDescription == null) {
            _localizedDescription.remove(l);
        } else {
            _localizedDescription.put(l, localizedDescription);
        }
//      setLocalizedShortDescription(locale, localizedDescription); do not set short description here to prevent excessive inline help
    }

    /**
     * @param locale the locale for the short description
     * @return the localized short description
     */
    @Override
    public String getLocalizedShortDescription(Locale locale) {
        Locale l = localeReadingKey(locale);
        return _localizedShortDescription.get(l);
    }

    /**
     * El método setLocalizedShortDescription se utiliza para establecer la descripción corta del artefacto que se almacena en el archivo de recursos
     * de configuración regional. En caso de que el archivo de recursos para el idioma seleccionado por el usuario no esté disponible, la interfaz de
     * la aplicación utiliza el archivo de recursos por defecto para obtener el valor de la descripción.
     *
     * @param locale configuración regional
     * @param localizedShortDescription una o más oraciones que describen brevemente el artefacto
     */
    @Override
    public void setLocalizedShortDescription(Locale locale, String localizedShortDescription) {
        Locale l = localeWritingKey(locale);
        if (localizedShortDescription == null) {
            _localizedShortDescription.remove(l);
        } else {
            _localizedShortDescription.put(l, localizedShortDescription);
        }
    }

    /**
     * @param locale the locale for the tooltip
     * @return the localized tooltip
     */
    @Override
    public String getLocalizedTooltip(Locale locale) {
        Locale l = localeReadingKey(locale);
        return _localizedTooltip.get(l);
    }

    /**
     * El método setLocalizedTooltip se utiliza para establecer la descripción emergente (tooltip) del artefacto que se almacena en el archivo de
     * recursos de configuración regional. En caso de que el archivo de recursos para el idioma seleccionado por el usuario no esté disponible, la
     * interfaz de la aplicación utiliza el archivo de recursos por defecto para obtener el valor de la descripción.
     *
     * @param locale configuración regional
     * @param localizedTooltip una o más oraciones que describen muy brevemente el artefacto
     */
    @Override
    public void setLocalizedTooltip(Locale locale, String localizedTooltip) {
        Locale l = localeWritingKey(locale);
        if (localizedTooltip == null) {
            _localizedTooltip.remove(l);
        } else {
            _localizedTooltip.put(l, localizedTooltip);
        }
    }

    protected Locale localeReadingKey(Locale locale) {
        return locale == null ? Bundle.getLocale() : locale;
    }

    protected Locale localeWritingKey(Locale locale) {
        /*
        Locale l = locale == null ? Bundle.getLocale() : Bundle.isSupportedLocale(locale) ? locale : null;
        if (l == null) {
            throw new IllegalArgumentException("Locale " + locale + " not supported yet.");
        }
        return l;
        **/
        return localeReadingKey(locale);
    }

    protected char settler() {
        return '?';
    }

    /**
     * @return the declaring artifact
     */
    @Override
    public Artifact getDeclaringArtifact() {
        return _declaringArtifact;
    }

    String getDeclaringArtifactClassName() {
        return _declaringArtifact == null ? null : _declaringArtifact.getClass().getName();
    }

    String getDeclaringArtifactClassSimpleName() {
        return _declaringArtifact == null ? null : _declaringArtifact.getClass().getSimpleName();
    }

    /**
     * @param declaringArtifact the declaring artifact to set
     */
    void initDeclaringArtifact(Artifact declaringArtifact) {
        _declaringArtifact = declaringArtifact;
    }

    /**
     * @param declaringArtifact the declaring artifact to set
     */
    void setDeclaringArtifact(Artifact declaringArtifact) {
        if (_declared) {
            return;
        }
        resetDeclaringArtifact(declaringArtifact);
    }

    void resetDeclaringArtifact(Artifact declaringArtifact) {
        _declaringArtifact = declaringArtifact;
        if (declaringArtifact != null) {
            if (this instanceof Entity) {
                _depth = declaringArtifact.depth() + 1;
                _round = XS1.round(getNamedClass(), declaringArtifact);
            } else {
                _depth = declaringArtifact.depth();
            }
        }
    }

    /**
     * @return the declaring field
     */
    @Override
    public Field getDeclaringField() {
        return _declaringField;
    }

    /**
     * @param declaringField the declaring field to set
     */
    void setDeclaringField(Field declaringField) {
        if (_declared) {
            return;
        }
        resetDeclaringField(declaringField);
    }

    void resetDeclaringField(Field declaringField) {
        _declaringField = declaringField;
    }

    /**
     * @return the declaring field index
     */
    @Override
    public int getDeclaringFieldIndex() {
        return _declaringFieldIndex;
    }

    /**
     * @param declaringFieldIndex the declaring field index to set
     */
    void setDeclaringFieldIndex(int declaringFieldIndex) {
        if (_declared) {
            return;
        }
        _declaringFieldIndex = declaringFieldIndex;
    }

    /**
     * @return the declaring entity if the artifact directly declared by one, null otherwise
     */
    @Override
    public Entity getDeclaringEntity() {
        return _declaringArtifact instanceof Entity ? (Entity) _declaringArtifact : null;
    }

    /**
     * @return the declaring entity if the artifact directly declared by one, null otherwise
     */
    @Override
    public PersistentEntity getDeclaringPersistentEntity() {
        Entity declaringEntity = getDeclaringEntity();
        return declaringEntity instanceof PersistentEntity ? (PersistentEntity) declaringEntity : null;
    }

    /**
     * @return the declaring entity if the artifact directly declared by one, null otherwise
     */
    @Override
    public Entity getDeclaringEntityRoot() {
        Entity declaringEntity = getDeclaringEntity();
        return declaringEntity == null ? null : declaringEntity.getRoot();
    }

    /**
     * @return the declaring entity if the artifact directly declared by one, null otherwise
     */
    @Override
    public PersistentEntity getDeclaringPersistentEntityRoot() {
        Entity declaringEntityRoot = getDeclaringEntityRoot();
        return declaringEntityRoot instanceof PersistentEntity ? (PersistentEntity) declaringEntityRoot : null;
    }

    /**
     * @return the declaring entity if the artifact directly declared by one, null otherwise
     */
    @Override
    public Entity getDeclaringFieldEntityRoot() {
        Class<?> declaringClass = _declaringField == null ? null : _declaringField.getDeclaringClass();
        Entity declaringEntity = declaringClass == null ? null : getDeclaringEntity();
        return declaringEntity == null ? null : declaringEntity.getDeclaringProject().getEntity(declaringClass);
    }

    /**
     * @return the declaring entity if the artifact directly declared by one, null otherwise
     */
    @Override
    public PersistentEntity getDeclaringFieldPersistentEntityRoot() {
        Entity declaringFieldEntityRoot = getDeclaringFieldEntityRoot();
        return declaringFieldEntityRoot instanceof PersistentEntity ? (PersistentEntity) declaringFieldEntityRoot : null;
    }

    /**
     * @return the declaring operation if the artifact directly declared by one, null otherwise
     */
    @Override
    public Operation getDeclaringOperation() {
        return _declaringArtifact instanceof Operation ? (Operation) _declaringArtifact : null;
    }

    /**
     * @return the depth
     */
    @Override
    public int depth() {
        return _depth;
    }

    /**
     * @return the round
     */
    @Override
    public int round() {
        return _round;
    }

    /**
     * @return the annotations map
     */
    Map<Class<? extends Annotation>, Field> getAnnotations() {
        return _annotations;
    }

    @Override
    public void clearAttributes() {
        _attributes.clear();
    }

    @Override
    public void addAttributes() {
    }

    /**
     * @return the attributes map
     */
    public Map<String, Object> getAttributes() {
        return _attributes;
    }

    public Set<String> getAttributesKeySetByRegex(String regex) {
        Set<String> newSet = new TreeSet<>();
        Set<String> keySet = _attributes.keySet();
        for (String string : keySet) {
            if (string.matches(regex)) {
                newSet.add(string);
            }
        }
        return newSet;
    }
    // </editor-fold>

    // <editor-fold defaultstate="collapsed" desc="annotate">
    void initializeAnnotations() {
    }

    void annotate(Class<?> type) {
        if (type == null) {
            return;
        }
        Annotation[] annotations = type.getAnnotations();
        List<Class<? extends Annotation>> valid = getValidTypeAnnotations();
        checkAnnotations(type, null, annotations, valid);
    }

    void annotate(Field field) {
        if (field == null) {
            return;
        }
        boolean check = depth() == 0 || !isProperty();
        if (check) {
            Object clazz = null;
            if (depth() == 0 && isParameter()) {
                Parameter p = (Parameter) this;
                Entity e = p.getDeclaringArtifact().getDeclaringEntity();
                if (e != null) {
                    clazz = e.getClass();
                }
            }
            Annotation[] annotations = field.getAnnotations();
            List<Class<? extends Annotation>> valid = getValidFieldAnnotations();
            checkAnnotations(field, clazz, annotations, valid);
        }
    }

    private void checkAnnotations(Object obj1, Object obj2, Annotation[] annotations, List<Class<? extends Annotation>> valid) {
        Project project = TLC.getProject();
//      if (project == null) {
//          return;
//      }
        String pattern = "@{0} is not valid for {1}" + (obj2 == null ? "" : " referenced by {2}") + "; annotation ignored";
        String message;
        Class<? extends Annotation> annotationType;
        for (Annotation annotation : annotations) {
            annotationType = annotation.annotationType();
            if (valid.contains(annotationType)) {
                continue;
            }
            if (annotationType.getPackage().getName().startsWith("javax.xml.bind.annotation")) {
                continue;
            }
            message = MessageFormat.format(pattern, annotationType.getSimpleName(), obj1, obj2);
            project.getParser().log(Level.ERROR, message);
            project.getParser().increaseErrorCount();
        }
    }

    protected List<Class<? extends Annotation>> getValidTypeAnnotations() {
        return new ArrayList<>();
    }

    protected List<Class<? extends Annotation>> getValidFieldAnnotations() {
        return new ArrayList<>();
    }

    @Override
    public Field put(Class<? extends Annotation> annotation, Field field) {
        XS1.checkAccess();
        return _annotations.put(annotation, field);
    }
    // </editor-fold>

    public AbstractArtifact() {
        add();
    }

    private void add() {
        _defaultLocale = defaultLocale();
        Project project = TLC.getProject();
        if (project != null) {
            project.addArtifact(this);
        }
    }

    protected Locale defaultLocale() {
        return null;
    }

    /**
     * El método addAttribute permite agregar un atributo a la lista de atributos extraordinarios del artefacto. Los atributos extraordinarios son
     * parejas clave/valor, de modo que si se agregan varios atributos con la misma clave a un artefacto, el valor de tal atributo será el último
     * valor agregado.
     *
     * @param clazz clase a la que corresponde el atributo
     * @param name clave del atributo
     * @param value valor del atributo
     * @return el valor anterior asociado con la clave, o nulo si no había una asignación para la clave, o si la implementación admite valores nulos.
     */
    @Override
    public Object addAttribute(Class<?> clazz, String name, Object value) {
        return addAttribute(attributeName(clazz, name), value);
    }

    /**
     * El método addAttribute permite agregar un atributo a la lista de atributos extraordinarios del artefacto. Los atributos extraordinarios son
     * parejas clave/valor, de modo que si se agregan varios atributos con la misma clave a un artefacto, el valor de tal atributo será el último
     * valor agregado.
     *
     * @param name clave del atributo
     * @param value valor del atributo
     * @return el valor anterior asociado con la clave, o nulo si no había una asignación para la clave, o si la implementación admite valores nulos.
     */
    @Override
    public Object addAttribute(String name, Object value) {
        return value == null ? _attributes.remove(name) : _attributes.put(name, value);
    }

    /**
     * El método addAttribute permite agregar un atributo a la lista de atributos extraordinarios del artefacto. Los atributos extraordinarios son
     * parejas clave/valor, de modo que si se agregan varios atributos con la misma clave a un artefacto, el valor de tal atributo será el último
     * valor agregado.
     *
     * @param clazz clase a la que corresponde el atributo
     * @param name clave del atributo
     * @param value valor del atributo
     * @return el valor anterior asociado con la clave, o nulo si no había una asignación para la clave, o si la implementación admite valores nulos.
     */
    @Override
    public Object addAttribute(Class<?> clazz, String name, Object... value) {
        return addAttribute(attributeName(clazz, name), value);
    }

    /**
     * El método addAttribute permite agregar un atributo a la lista de atributos extraordinarios del artefacto. Los atributos extraordinarios son
     * parejas clave/valor, de modo que si se agregan varios atributos con la misma clave a un artefacto, el valor de tal atributo será el último
     * valor agregado.
     *
     * @param name clave del atributo
     * @param value valor del atributo
     * @return el valor anterior asociado con la clave, o nulo si no había una asignación para la clave, o si la implementación admite valores nulos.
     */
    @Override
    public Object addAttribute(String name, Object... value) {
        return value == null ? _attributes.remove(name) : _attributes.put(name, value);
    }

    @Override
    public Object getAttribute(Class<?> clazz, String name) {
        return getAttribute(attributeName(clazz, name));
    }

    @Override
    public Object getAttribute(String name) {
        return _attributes.get(name);
    }

    public Boolean getBooleanAttribute(Class<?> clazz, String name) {
        return getBooleanAttribute(attributeName(clazz, name));
    }

    public Boolean getBooleanAttribute(String name) {
        Object attribute = _attributes.get(name);
        return ObjUtils.toBoolean(attribute);
    }

    public Integer getIntegerAttribute(Class<?> clazz, String name) {
        return getIntegerAttribute(attributeName(clazz, name));
    }

    public Integer getIntegerAttribute(String name) {
        Object attribute = _attributes.get(name);
        return ObjUtils.toInteger(attribute);
    }

    public Integer getIntegerAttribute(Class<?> clazz, String name, Integer min, Integer max) {
        return getIntegerAttribute(attributeName(clazz, name), min, max);
    }

    public Integer getIntegerAttribute(String name, Integer min, Integer max) {
        Object attribute = _attributes.get(name);
        Integer integer = ObjUtils.toInteger(attribute);
        return ObjUtils.between(integer, min, max) ? integer : null;
    }

    public String getStringAttribute(Class<?> clazz, String name) {
        return getStringAttribute(attributeName(clazz, name));
    }

    public String getStringAttribute(String name) {
        return getStringAttribute(name, KVP.EQUALS, KVP.SEPARATOR, KVP.OPEN, KVP.CLOSE);
    }

    public String getStringAttribute(Class<?> clazz, String name, String equals, String separator, String open, String close) {
        return getStringAttribute(attributeName(clazz, name), equals, separator, open, close);
    }

    public String getStringAttribute(String name, String equals, String separator, String open, String close) {
        return StrUtils.getString(equals, separator, open, close, _attributes.get(name));
    }

    public Object getAttributesArray(Class<?> clazz, String name) {
        return getAttributesArray(attributeName(clazz, name));
    }

    public Object getAttributesArray(String name) {
        Object value = _attributes.get(name);
        return value == null ? null : value.getClass().isArray() ? value : new Object[]{value};
    }

    private String attributeName(Class<?> clazz, String name) {
        return clazz.getSimpleName() + ":" + name;
    }

    /**
     * @return the class path
     */
    @Override
    public String getClassPath() {
        return classPath(this);
    }

    private String classPath(Artifact a) {
        Artifact da = a.getDeclaringArtifact();
        String str1 = a.getName();
        String str2 = XS1.getNamedClass(a).getSimpleName();
        String str3 = str1 == null || str1.equals(str2) ? str2 : str2 + "[" + str1 + "]";
        return da == null ? str3 : classPath(da) + "." + str3;
    }

    /**
     * @param type
     * @return true if type is present in the class path
     */
    @Override
    public boolean isClassInPath(Class<?> type) {
        return isClassInPath(type, this);
    }

    private boolean isClassInPath(Class<?> clazz, Artifact artifact) {
        if (clazz != null) {
            if (artifact != null) {
                if (clazz.isAssignableFrom(XS1.getNamedClass(artifact))) {
                    return true;
                }
                Artifact declaringArtifact = artifact.getDeclaringArtifact();
                if (declaringArtifact != null) {
                    return isClassInPath(clazz, declaringArtifact);
                }
            }
        }
        return false;
    }

    /**
     * @return the path
     */
    @Override
    @SuppressWarnings("unchecked") // unchecked conversion
    public List<Artifact> getPathList() {
        List list = new ArrayList<>();
        addToPathList(list, this);
        return list;
    }

    private void addToPathList(List<Artifact> list, Artifact artifact) {
        Artifact declaringArtifact = artifact.getDeclaringArtifact();
        if (declaringArtifact != null) {
            addToPathList(list, declaringArtifact);
        }
        list.add(artifact);
    }

    /**
     * @return the path string
     */
    @Override
    public String getPathString() {
        return path(this);
    }

    private String path(Artifact a) {
        String p;
        if (a.getDeclaringField() == null || a.getDeclaringArtifact() == null) {
//          p = "_" + a.getAlias();
            p = XS1.getNamedClass(a).getSimpleName() + "." + "this";
        } else {
            p = path(a.getDeclaringArtifact()) + "." + a.getDeclaringField().getName();
            if (a.getDeclaringField().getType().isArray()) {
                p += "[" + a.getDeclaringFieldIndex() + "]";
            }
            p = StringUtils.removeStart(p, ".");
        }
        return p;
    }

    /**
     * @return the full name
     */
    @Override
    public String getFullName() {
        return StringUtils.remove(getPathString(), "." + "this");
    }

    /**
     * @return the partial name
     */
    @Override
    public String getPartialName() {
        return StringUtils.substringAfter(getPathString(), "." + "this" + ".");
    }

    protected String getValueString(Object value) {
        if (value == null) {
            return null;
        } else if (value instanceof Artifact) {
            Artifact artifact = (Artifact) value;
            return getValueString(value, artifact.getPathString());
        } else {
//          return getValueString(value, value);
            return value.toString();
        }
    }

    protected String getValueString(Object object, Object value) {
        return XS1.getNamedClass(object).getSimpleName() + "(" + value.toString() + ")";
    }

    /**
     * @return the named class
     */
    Class<?> getNamedClass() {
        return XS1.getNamedClass(this);
    }

    /**
     * @return true if this artifact is an Operation; otherwise false
     */
    @Override
    public boolean isOperation() {
        return this instanceof Operation;
    }

    private boolean isParameter() {
        return this instanceof DataArtifact && ((DataArtifact) this).isParameter();
    }

    private boolean isProperty() {
        return this instanceof DataArtifact && ((DataArtifact) this).isProperty();
    }

    /**
     * @return true if this artifact is an Expression; otherwise false
     */
    @Override
    public boolean isExpression() {
        return this instanceof Expression;
    }

    protected boolean verifyExpression(Expression expression) {
        return verifyExpression(expression, null);
    }

    protected boolean verifyExpression(Expression expression, Artifact artifact) {
        return verifyExpression(expression, artifact, true);
    }

    protected boolean verifyExpression(Expression expression, Artifact artifact, boolean calculableless) {
        boolean ok = true;
        String locator = artifact == null ? "at " + getFullName() : "for " + artifact.getFullName();
        if (expression == null) {
            String message = "null expression defined " + locator;
            logger.error(message);
            TLC.getProject().getParser().increaseErrorCount();
            ok = false;
        } else {
            String expname = StringUtils.isBlank(expression.getName()) ? "" : expression.getName() + " ";
            String message = "invalid expression " + expname + "defined " + locator;
            if (expression instanceof VariantX) {
                boolean b1 = this instanceof Entity && depth() == 0;
                boolean b2 = this instanceof Operation; // Operation --> depth() == 0
                if (b1 || b2) {
                    message += "; expression is not properly defined";
                    logger.error(message);
                    TLC.getProject().getParser().increaseErrorCount();
                    ok = false;
                }
            } else {
                ok &= verifyExpression(this, expression, message, calculableless);
            }
        }
        return ok;
    }

    private boolean verifyExpression(Artifact container, Expression expression, String message, boolean calculableless) {
        boolean ok = true;
        if (expression != null && container != null) {
            boolean b1, b2, calculable;
            Artifact artifact;
            List<Artifact> pathList;
            Expression operandExpression;
            String fullName;
            Object[] operands = expression.getOperands();
            if (operands != null && operands.length > 0) {
                for (Object operand : operands) {
                    if (operand instanceof DataArtifact) {
                        artifact = (Artifact) operand;
                        pathList = artifact.getPathList();
                        if (pathList.contains(container)) {
                            b1 = container instanceof Entity && container.depth() == 0;
                            b2 = container instanceof Operation;
                            if (b1 || b2) {
                                calculable = ((DataArtifact) artifact).isProperty() && ((Property) artifact).isCalculable();
                                if (calculable) {
                                    if (calculableless || b2) {
                                        fullName = artifact.getFullName();
                                        logger.error(message + "; operand " + fullName + " is a calculable property");
                                        TLC.getProject().getParser().increaseErrorCount();
                                        ok = false;
                                    } else if (b1 && !artifact.getDeclaringEntity().equals(container)) {
                                        fullName = artifact.getFullName();
                                        logger.error(message + "; operand " + fullName + " is a foreign calculable property");
                                        TLC.getProject().getParser().increaseErrorCount();
                                        ok = false;
                                    }
                                }
                            }
                        } else {
                            fullName = artifact.getFullName();
                            logger.error(message + "; operand " + fullName + " is out of scope");
                            TLC.getProject().getParser().increaseErrorCount();
                            ok = false;
                        }
                    } else if (operand instanceof VariantX) {
                        artifact = (Artifact) operand;
                        fullName = artifact.getFullName();
                        logger.error(message + "; operand " + fullName + " is out of scope");
                        TLC.getProject().getParser().increaseErrorCount();
                        ok = false;
                    } else if (operand instanceof Expression) {
                        operandExpression = (Expression) operand;
                        ok &= verifyExpression(container, operandExpression, message, calculableless);
                    }
                }
            }
        }
        return ok;
    }

//  protected void verifyNames(Class<?> top) {
//      verifyNames(top, Artifact.class);
//  }
//
    protected void verifyNames(Class<?> top, Class<?> clazz) {
        String message;
        String fieldName, artifactName;
        Object object;
        AbstractArtifact artifact;
        Class<?> fieldType;
        Class<?> dac = getClass();
        for (Field field : XS1.getFields(dac, top)) {
            field.setAccessible(true);
            fieldName = field.getName();
            fieldType = field.getType();
            if (isAssignableFrom(clazz, fieldType) && isNotRestricted(field)) {
                try {
                    object = field.get(this);
                    if (object instanceof AbstractArtifact) {
                        artifact = (AbstractArtifact) object;
                        artifactName = artifact.getName();
                        if (artifactName == null) {
                            artifact.setName(fieldName);
                        }
                    }
                } catch (IllegalArgumentException | IllegalAccessException ex) {
                    message = "failed to get field \"" + field + "\" at " + this;
                    logger.error(message, ThrowableUtils.getCause(ex));
                    TLC.getProject().getParser().increaseErrorCount();
                }
            }
        }
    }

    private boolean isAssignableFrom(Class<?> clazz, Class<?> type) {
        if (Expression.class.isAssignableFrom(type)) {
            if (Parameter.class.isAssignableFrom(type) || Property.class.isAssignableFrom(type)) {
                return false;
            }
        }
        return clazz.isAssignableFrom(type);
    }

    private boolean isNotRestricted(Field field) {
        int modifiers = field.getModifiers();
        return !(Modifier.isPrivate(modifiers) || Modifier.isStatic(modifiers) || Modifier.isFinal(modifiers));
    }

    /**
     * the wrapper
     */
    private Wrapper _wrapper;

    /**
     * @return the wrapper
     */
    @Override
    public Wrapper getWrapper() {
        if (_wrapper == null) {
            Class<?> thisClass = getClass();
            Class<? extends Wrapper> wrapperClass = getWrapperClass(thisClass);
            if (wrapperClass != null) {
                Class<?> parameterType = XS1.getConstructorParameterType(wrapperClass, thisClass);
                if (parameterType != null) {
                    String pattern = "failed to wrap {0} using {1}({2})";
                    String message = MessageFormat.format(pattern, thisClass, wrapperClass, parameterType);
                    try {
                        _wrapper = wrapperClass.getConstructor(parameterType).newInstance(this);
                    } catch (NoSuchMethodException | SecurityException | InstantiationException | IllegalAccessException | IllegalArgumentException | InvocationTargetException ex) {
                        throw new RuntimeException(message, ex);
                    }
                } else {
                    String pattern = "{1} is not a valid wrapper for {0}";
                    String message = MessageFormat.format(pattern, thisClass, wrapperClass);
                    throw new RuntimeException(message);
                }
            }
        }
        return _wrapper;
    }

    @SuppressWarnings("unchecked") // unchecked cast
    private Class<? extends Wrapper> getWrapperClass(Class<?> clazz) {
        if (clazz != null && Wrappable.class.isAssignableFrom(clazz)) {
            Class<? extends Wrappable> wrappableClass = (Class<? extends Wrappable>) clazz; // unchecked cast
            Class<? extends Wrapper> wrapperClass = TLB.getWrapperClass(wrappableClass);
            return wrapperClass != null ? wrapperClass : getWrapperClass(clazz.getSuperclass());
        }
        return getDefaultWrapperClass();
    }

    /**
     * @return the default wrapper class
     */
    @Override
    public Class<? extends ArtifactWrapper> getDefaultWrapperClass() {
        return ArtifactWrapper.class;
    }

    // <editor-fold defaultstate="collapsed" desc="toString">
    @Override
    public String toString() {
        String str1 = _name;
        String str2 = getNamedClass().getSimpleName();
        String str3 = str1 == null || str1.equals(str2) ? "" : "[" + str1 + "]";
//      return str2 + str3 + "@" + Integer.toHexString(hashCode());
        return str2 + str3;
    }

    @Override
    public String toString(int n) {
        return toString(n, null);
    }

    @Override
    public String toString(int n, String key) {
        return toString(n, key, Project.isVerbose());
    }

    @Override
    public String toString(int n, String key, boolean verbose) {
        return toString(n, key, verbose, verbose, verbose);
    }

    @Override
    public String toString(int n, String key, boolean verbose, boolean fields, boolean maps) {
        String r4n = StringUtils.repeat(" ", 4 * n);
        String tab = verbose ? StringUtils.repeat(" ", 4) : "";
        String fee = verbose ? StringUtils.repeat(tab, n) : "";
//      String faa = " = ";
        String foo = verbose ? EOL : "";
        String eol = verbose ? EOL : ", ";
        String string = EOL;
        String c = classToString(n, key, verbose);
        String f = fieldsToString(n, key, verbose, fields, maps);
        String m = mapsToString(n, key, verbose, fields, maps);
        string += c;
        string += " " + "{" + " " + this + eol;
        string += f;
        string += m;
        if (!verbose) {
            string = StringUtils.removeEnd(string, eol);
            if (c.contains(EOL) || f.contains(EOL) || m.contains(EOL)) {
                fee = r4n;
            }
        }
        string += fee + "}" + EOL;
        return string.replaceAll(EOL + EOL, EOL);
    }

    protected String classToString(int n, String key, boolean verbose) {
        String tab = StringUtils.repeat(" ", 4);
        String fee = StringUtils.repeat(tab, n);
//      String faa = " = ";
//      String foo = EOL;
        String string = "";
        String s1 = getDeclaringField() == null ? "" : getDeclaringField().getType().getSimpleName();
        String s2 = getNamedClass().getSimpleName();
        String s3 = StringUtils.isBlank(s1) || s1.equals(s2) ? s2 : StringUtils.removeEnd(s1, "[]") + " " + "(" + s2 + ")";
        String s4 = StringUtils.isBlank(key) ? "" : " " + key;
//      String s5 = "@" + Integer.toHexString(hashCode());
//      string += fee + s3 + s4 + s5;
        string += fee + s3 + s4;
        return string;
    }

    protected String fieldsToString(int n, String key, boolean verbose, boolean fields, boolean maps) {
        String tab = verbose ? StringUtils.repeat(" ", 4) : "";
        String fee = verbose ? StringUtils.repeat(tab, n) : "";
        String faa = " = ";
        String foo = verbose ? EOL : ", ";
        String string = "";
        if (fields || verbose) {
            if (verbose) {
                if (_declaringArtifact != null) {
                    string += fee + tab + "declaringArtifact" + faa + _declaringArtifact + foo;
                    if (_declaringField != null) {
                        string += fee + tab + "declaringField" + faa + _declaringField + foo;
                        if (_declaringField.getType().isArray()) {
                            string += fee + tab + "declaringFieldIndex" + faa + _declaringFieldIndex + foo;
                        }
                    }
                    string += fee + tab + "path" + faa + getPathString() + foo;
                }
                if (this instanceof Entity) {
                    string += fee + tab + "depth" + faa + _depth + foo;
                    string += fee + tab + "round" + faa + _round + foo;
                }
                if (_name != null) {
                    string += fee + tab + "name" + faa + _name + foo;
                }
                if (_alias != null) {
                    string += fee + tab + "alias" + faa + _alias + foo;
                }
            } else {
                string += fee + tab + "path" + faa + getPathString() + foo;
            }

        }
        return string;
    }

    protected String mapsToString(int n, String key, boolean verbose, boolean fields, boolean maps) {
        return "";
    }
    // </editor-fold>

}
