/*
 * Copyright 2017 Jorge Campins y David Uzcategui
 *
 * Este archivo forma parte de Adalid.
 *
 * Adalid es software libre; usted puede redistribuirlo y/o modificarlo bajo los terminos de la
 * licencia "GNU General Public License" publicada por la Fundacion "Free Software Foundation".
 * Adalid se distribuye con la esperanza de que pueda ser util, pero SIN NINGUNA GARANTIA; sin
 * siquiera las garantias implicitas de COMERCIALIZACION e IDONEIDAD PARA UN PROPOSITO PARTICULAR.
 *
 * Para mas detalles vea la licencia "GNU General Public License" en http://www.gnu.org/licenses
 */
package adalid.core;

import adalid.commons.util.BitUtils;
import adalid.commons.util.StrUtils;
import adalid.commons.util.TimeUtils;
import adalid.core.annotations.BaseField;
import adalid.core.annotations.BigDecimalField;
import adalid.core.annotations.BooleanDataGen;
import adalid.core.annotations.BooleanField;
import adalid.core.annotations.BusinessKey;
import adalid.core.annotations.CharacterDataGen;
import adalid.core.annotations.CharacterKey;
import adalid.core.annotations.ColumnField;
import adalid.core.annotations.DateField;
import adalid.core.annotations.DescriptionProperty;
import adalid.core.annotations.DiscriminatorColumn;
import adalid.core.annotations.FileReference;
import adalid.core.annotations.GraphicImage;
import adalid.core.annotations.ImageProperty;
import adalid.core.annotations.InactiveIndicator;
import adalid.core.annotations.InstanceReference;
import adalid.core.annotations.NameProperty;
import adalid.core.annotations.NumericDataGen;
import adalid.core.annotations.NumericField;
import adalid.core.annotations.NumericKey;
import adalid.core.annotations.OwnerProperty;
import adalid.core.annotations.ParameterField;
import adalid.core.annotations.ParentProperty;
import adalid.core.annotations.PrimaryKey;
import adalid.core.annotations.PropertyAggregation;
import adalid.core.annotations.PropertyField;
import adalid.core.annotations.SegmentProperty;
import adalid.core.annotations.SequenceProperty;
import adalid.core.annotations.StateProperty;
import adalid.core.annotations.StringField;
import adalid.core.annotations.TemporalDataGen;
import adalid.core.annotations.TimeField;
import adalid.core.annotations.TimestampField;
import adalid.core.annotations.UniformResourceLocator;
import adalid.core.annotations.UniqueKey;
import adalid.core.annotations.UrlProperty;
import adalid.core.annotations.VariantString;
import adalid.core.annotations.VersionProperty;
import adalid.core.data.types.BigDecimalData;
import adalid.core.data.types.BigIntegerData;
import adalid.core.data.types.BinaryData;
import adalid.core.data.types.BooleanData;
import adalid.core.data.types.ByteData;
import adalid.core.data.types.CharacterData;
import adalid.core.data.types.DateData;
import adalid.core.data.types.DoubleData;
import adalid.core.data.types.FloatData;
import adalid.core.data.types.IntegerData;
import adalid.core.data.types.LongData;
import adalid.core.data.types.ShortData;
import adalid.core.data.types.StringData;
import adalid.core.data.types.TimeData;
import adalid.core.data.types.TimestampData;
import adalid.core.enums.AggregateFunction;
import adalid.core.enums.BooleanDisplayType;
import adalid.core.enums.Checkpoint;
import adalid.core.enums.DataGenNumericAction;
import adalid.core.enums.DataGenTemporalInterval;
import adalid.core.enums.DataGenType;
import adalid.core.enums.DefaultCondition;
import adalid.core.enums.DivisorRule;
import adalid.core.enums.KeyProperty;
import adalid.core.enums.MimeType;
import adalid.core.enums.OperationKind;
import adalid.core.enums.PropertyAccess;
import adalid.core.enums.SpecialEntityValue;
import adalid.core.enums.SpecialTemporalValue;
import adalid.core.enums.StandardRelationalOp;
import adalid.core.enums.UploadStorageOption;
import adalid.core.enums.UrlDisplayType;
import adalid.core.enums.UrlType;
import adalid.core.interfaces.Artifact;
import adalid.core.interfaces.BooleanExpression;
import adalid.core.interfaces.CalculableProperty;
import adalid.core.interfaces.ContextualEntity;
import adalid.core.interfaces.DataArtifact;
import adalid.core.interfaces.DatabaseEntity;
import adalid.core.interfaces.Entity;
import adalid.core.interfaces.EnumerationEntity;
import adalid.core.interfaces.Expression;
import adalid.core.interfaces.IntervalizedArtifact;
import adalid.core.interfaces.Parameter;
import adalid.core.interfaces.PersistentEntity;
import adalid.core.interfaces.PersistentEnumerationEntity;
import adalid.core.interfaces.Property;
import adalid.core.parameters.BigDecimalParameter;
import adalid.core.parameters.BooleanParameter;
import adalid.core.parameters.DateParameter;
import adalid.core.parameters.StringParameter;
import adalid.core.parameters.TimeParameter;
import adalid.core.parameters.TimestampParameter;
import adalid.core.primitives.BinaryPrimitive;
import adalid.core.primitives.BooleanPrimitive;
import adalid.core.primitives.CharacterPrimitive;
import adalid.core.primitives.NumericPrimitive;
import adalid.core.primitives.TemporalPrimitive;
import adalid.core.properties.BigDecimalProperty;
import adalid.core.properties.BigIntegerProperty;
import adalid.core.properties.BinaryProperty;
import adalid.core.properties.BooleanProperty;
import adalid.core.properties.ByteProperty;
import adalid.core.properties.CharacterProperty;
import adalid.core.properties.DateProperty;
import adalid.core.properties.DoubleProperty;
import adalid.core.properties.FloatProperty;
import adalid.core.properties.IntegerProperty;
import adalid.core.properties.LongProperty;
import adalid.core.properties.ShortProperty;
import adalid.core.properties.StringProperty;
import adalid.core.properties.TimeProperty;
import adalid.core.properties.TimestampProperty;
import adalid.core.wrappers.DataArtifactWrapper;
import java.lang.annotation.Annotation;
import java.lang.reflect.Field;
import java.math.BigDecimal;
import java.sql.Date;
import java.sql.Time;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;
import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;

/**
 * @author Jorge Campins
 */
public abstract class AbstractDataArtifact extends AbstractArtifact implements DataArtifact, Parameter, Property, CalculableProperty {

    // <editor-fold defaultstate="collapsed" desc="field declarations">
    /**
     *
     */
    private static final Logger logger = Logger.getLogger(DataArtifact.class);

    private static final String EOL = "\n";

    public static final String CONVERTER_REGEX = "^[a-zA-Z]\\w*$";

    public static final String VALIDATOR_REGEX = "^[a-zA-Z]\\w*$";

    /**
     * dataClass
     */
    private Class<?> _dataClass;

    /**
     * dataType
     */
    private Class<?> _dataType;

    /**
     *
     */
    private PropertyAccess _propertyAccess;

    /**
     *
     */
    private boolean _auditable;

    /**
     *
     */
    private boolean _password;

    /**
     *
     */
    private Boolean _required;

    /**
     *
     */
    private boolean _hiddenField;

    /**
     *
     */
    private Boolean _createField;

    /**
     *
     */
    private Boolean _updateField;

    /**
     *
     */
    private Boolean _searchField;

    /**
     *
     */
    private Boolean _filterField;

    /**
     *
     */
    private Boolean _sortField;

    /**
     *
     */
    private Boolean _tableField;

    /**
     *
     */
    private boolean _detailField;

    /**
     *
     */
    private Boolean _columnField;

    /**
     *
     */
    private Boolean _reportField;

    /**
     *
     */
    private Boolean _exportField;

    /**
     *
     */
    private boolean _submitField;

    /**
     *
     */
    private boolean _headertextlessField;

    /**
     *
     */
    private Boolean _headingField;

    /**
     *
     */
    private Boolean _overlayField;

    /**
     *
     */
    private Boolean _immutableField;

    /**
     *
     */
    private DefaultCondition _defaultCondition;

    /**
     *
     */
    private Checkpoint _defaultCheckpoint;

    /**
     *
     */
    private int _sequenceNumber;

    /**
     *
     */
    private AggregateFunction _aggregateFunction;

    /**
     *
     */
    private String _aggregateTitle;

    /**
     *
     */
    private boolean _calculable;

    /**
     *
     */
    private boolean _nullable;

    /**
     *
     */
    private boolean _insertable;

    /**
     *
     */
    private boolean _updateable;

    /**
     *
     */
    private boolean _unique;

    /**
     *
     */
    private String _linkedFieldName;

    /**
     *
     */
    private Field _linkedField;

    /**
     *
     */
    private Property _linkedProperty;

    /**
     *
     */
    private String _linkedColumnName;

    /**
     *
     */
    private StandardRelationalOp _linkedColumnOperator;

    /**
     *
     */
    DataGenType _dataGenType;

    /**
     *
     */
    int _dataGenSeriesStart;

    /**
     *
     */
    int _dataGenSeriesStop;

    /**
     *
     */
    int _dataGenSeriesStep;

    /**
     *
     */
    String _dataGenFunction;

    /**
     *
     */
    int _dataGenNullable;

    /**
     *
     */
    private int _dataGenTrueable;

    /**
     *
     */
    private String _dataGenPattern;

    /**
     *
     */
    private String _dataGenPrefix;

    /**
     *
     */
    private String _dataGenSuffix;

    /**
     *
     */
    private Object _dataGenMin;

    /**
     *
     */
    private Object _dataGenMax;

    /**
     *
     */
    private DataGenTemporalInterval _dataGenTemporalInterval;

    /**
     *
     */
    private DataGenNumericAction _dataGenNumericAction;

    /**
     *
     */
    private Object _dataGenFactor;

    /**
     *
     */
    private BooleanExpression _renderingFilter;

    /**
     *
     */
    private BooleanExpression _requiringFilter;

    /**
     *
     */
    private BooleanExpression _modifyingFilter;

    /**
     *
     */
    private BooleanExpression _nullifyingFilter;

    /**
     * annotated with BaseField
     */
    private boolean _annotatedWithBaseField;

    /**
     * annotated with PrimaryKey
     */
    private boolean _annotatedWithPrimaryKey;

    /**
     * annotated with SequenceProperty
     */
    private boolean _annotatedWithSequenceProperty;

    /**
     * annotated with VersionProperty
     */
    private boolean _annotatedWithVersionProperty;

    /**
     * annotated with NumericKey
     */
    private boolean _annotatedWithNumericKey;

    /**
     * annotated with CharacterKey
     */
    private boolean _annotatedWithCharacterKey;

    /**
     * annotated with NameProperty
     */
    private boolean _annotatedWithNameProperty;

    /**
     * annotated with DescriptionProperty
     */
    private boolean _annotatedWithDescriptionProperty;

    /**
     * annotated with ImageProperty
     */
    private boolean _annotatedWithImageProperty;

    /**
     * annotated with InactiveIndicator
     */
    private boolean _annotatedWithInactiveIndicator;

    /**
     * annotated with UrlProperty
     */
    private boolean _annotatedWithUrlProperty;

    /**
     * annotated with ParentProperty
     */
    private boolean _annotatedWithParentProperty;

    /**
     * annotated with OwnerProperty
     */
    private boolean _annotatedWithOwnerProperty;

    /**
     * annotated with SegmentProperty
     */
    private boolean _annotatedWithSegmentProperty;

    /**
     * annotated with UniqueKey
     */
    private boolean _annotatedWithUniqueKey;

    /**
     * annotated with BusinessKey
     */
    private boolean _annotatedWithBusinessKey;

    /**
     * annotated with DiscriminatorColumn
     */
    private boolean _annotatedWithDiscriminatorColumn;

    /**
     * annotated with StateProperty
     */
    private boolean _annotatedWithStateProperty;

    /**
     * annotated with ColumnField
     */
    private boolean _annotatedWithColumnField;

    /**
     * annotated with BigDecimalField
     */
    private boolean _annotatedWithBigDecimalField;

    /**
     * annotated with BooleanField
     */
    private boolean _annotatedWithBooleanField;

    /**
     * annotated with BooleanField
     */
    private boolean _annotatedWithNumericField;

    /**
     * annotated with StringField
     */
    private boolean _annotatedWithStringField;

    /**
     * annotated with DateField
     */
    private boolean _annotatedWithDateField;

    /**
     * annotated with TimeField
     */
    private boolean _annotatedWithTimeField;

    /**
     * annotated with TimestampField
     */
    private boolean _annotatedWithTimestampField;

    /**
     * annotated with ParameterField
     */
    private boolean _annotatedWithParameterField;

    /**
     * annotated with PropertyField
     */
    private boolean _annotatedWithPropertyField;

    /**
     * annotated with PropertyAggregation
     */
    private boolean _annotatedWithPropertyAggregation;

    /**
     * annotated with InstanceReference
     */
    private boolean _annotatedWithInstanceReference;

    /**
     * annotated with FileReference
     */
    private boolean _annotatedWithFileReference;

    /**
     * annotated with GraphicImage
     */
    private boolean _annotatedWithGraphicImage;

    /**
     * annotated with UniformResourceLocator
     */
    private boolean _annotatedWithUniformResourceLocator;

    /**
     * annotated with VariantString
     */
    private boolean _annotatedWithVariantString;

    /**
     * annotated with DataGen
     */
    private boolean _annotatedWithDataGen;
    // </editor-fold>

    // <editor-fold defaultstate="collapsed" desc="field getters and setters">
    /**
     * @return the SQL name
     */
    @Override
    public String getSqlName() {
        if (isProperty()) {
            Entity declaringEntity = getDeclaringEntity();
            if (declaringEntity.isRootInstance()) {
                return super.getSqlName();
            }
            String name = getName();
            if (name != null) {
                Entity root = declaringEntity.getRoot();
                for (Property rootProperty : root.getPropertiesList()) {
                    if (name.equals(rootProperty.getName())) {
                        return rootProperty.getSqlName();
                    }
                }
            }
        }
        return super.getSqlName();
    }

    /**
     * @return the corresponding property at the declaring entity's root instance
     */
    @Override
    public Property getPropertyAtRoot() {
        if (isProperty()) {
            Entity declaringEntity = getDeclaringEntity();
            if (declaringEntity.isRootInstance()) {
                return this;
            }
            String name = getName();
            if (name != null) {
                Entity root = declaringEntity.getRoot();
                for (Property property : root.getPropertiesList()) {
                    if (name.equals(property.getName())) {
                        return property;
                    }
                }
            }
        }
        return null;
    }

    /**
     * @return the data class
     */
    @Override
    public Class<?> getDataClass() {
        return _dataClass;
    }

    /**
     * @param clazz the data class to set
     */
    void setDataClass(Class<?> clazz) {
        _dataClass = clazz;
    }

    /**
     * @return the data type
     */
    @Override
    public Class<?> getDataType() {
        return _dataType;
    }

    /**
     * @param type the data type to set
     */
    void setDataType(Class<?> type) {
        _dataType = type;
    }

    /**
     * @return true if the field defines a base field
     */
    @Override
    public boolean isBaseField() {
        return _annotatedWithBaseField || isPrimaryKeyProperty() || isSequenceProperty() || isVersionProperty();
    }

    /**
     * @return true if the field defines a key field
     */
    @Override
    public boolean isKeyField() {
        return isPrimaryKeyProperty()
            || isSequenceProperty()
            || isUniqueKeyProperty()
            || isBusinessKeyProperty()
            || isCharacterKeyProperty()
            || isNumericKeyProperty();
    }

    /**
     * @return the property access
     */
    @Override
    public PropertyAccess getPropertyAccess() {
        /*
        return PropertyAccess.UNSPECIFIED.equals(_propertyAccess)
            ? _password ? PropertyAccess.RESTRICTED_READING : PropertyAccess.UNRESTRICTED
            : _propertyAccess;
        **/
        return _password ? PropertyAccess.RESTRICTED_READING
            : (_propertyAccess == null || _propertyAccess.equals(PropertyAccess.UNSPECIFIED)) ? PropertyAccess.UNRESTRICTED : _propertyAccess;
    }

    private boolean restrictedReadingAccess() {
        return PropertyAccess.RESTRICTED_READING.equals(getPropertyAccess());
    }

    /**
     * @return the auditable indicator
     */
    @Override
    public boolean isAuditable() {
        return _auditable && !_password && !isBinaryData();
    }

    /**
     * @return the password indicator
     */
    @Override
    public boolean isPassword() {
        return _password;
    }

    /**
     * @return the read only indicator
     */
    @Override
    public boolean isReadOnly() {
        return !isCreateField() && !isUpdateField();
    }

    /**
     * @return the required parameter indicator
     */
    @Override
    public boolean isRequiredParameter() {
        return !isHiddenField() && !isBinaryData() && BitUtils.valueOf(_required);
    }

    /**
     * @return the required property indicator
     */
    @Override
    public boolean isRequiredProperty() {
        return !isHiddenField() && !isBinaryData() && BitUtils.valueOf(_required, implicitRequiredProperty());
    }

    private boolean implicitRequiredProperty() {
        return !_calculable
            && !_nullable
            && !isDiscriminatorProperty()
            && getDefaultValue() == null;
    }

    /**
     * @return the hidden field indicator
     */
    @Override
    public boolean isHiddenField() {
        return _hiddenField;
    }

    /**
     * @return the create field indicator
     */
    @Override
    public boolean isCreateField() {
        return !isHiddenField() && !isPrimaryKeyProperty() && !isSequenceProperty() && !isVersionProperty() && !isBinaryData()
            && (_calculable || _insertable) && BitUtils.valueOf(_createField, implicitCreateField());
    }

    private boolean implicitCreateField() {
        return isInsertable() && !defaultUnconditionallyOnInsert() && isUnlinkedProperty() && isRequiredProperty();
    }

    private boolean defaultUnconditionallyOnInsert() {
        return getDefaultValue() != null && (DefaultCondition.UNCONDITIONALLY.equals(_defaultCondition)
            || DefaultCondition.UNCONDITIONALLY_ON_INSERT.equals(_defaultCondition));
    }

    /**
     * @return the update field indicator
     */
    @Override
    public boolean isUpdateField() {
        return !isHiddenField() && !isPrimaryKeyProperty() && !isSequenceProperty() && !isVersionProperty() && !isBinaryData()
            && (_calculable || _updateable) && BitUtils.valueOf(_updateField, implicitUpdateField());
    }

    private boolean implicitUpdateField() {
        return isUpdateable() && !defaultUnconditionallyOnUpdate() && isUnlinkedProperty();
    }

    private boolean defaultUnconditionallyOnUpdate() {
        return getDefaultValue() != null && (DefaultCondition.UNCONDITIONALLY.equals(_defaultCondition)
            || DefaultCondition.UNCONDITIONALLY_ON_UPDATE.equals(_defaultCondition));
    }

    private boolean isUnlinkedProperty() {
        if (isProperty()) {
            List<Parameter> parametersList;
            Entity declaringEntity = getDeclaringEntity();
            List<Operation> businessOperationsList = declaringEntity.getBusinessOperationsList();
            for (Operation operation : businessOperationsList) {
                if (operation instanceof ProcessOperation && OperationKind.INSTANCE.equals(operation.getOperationKind())) {
                    parametersList = operation.getParametersList();
                    for (Parameter parameter : parametersList) {
                        if (this.equals(parameter.getLinkedProperty())) {
                            return false;
                        }
                    }
                }
            }
            return true;
        }
        return false;
    }

    /**
     * @return the search field indicator
     */
    @Override
    public boolean isSearchField() {
        return !isHiddenField() && !isBinaryData() && !_password && !_calculable && BitUtils.valueOf(_searchField, implicitSearchField());
    }

    private boolean implicitSearchField() {
        return implicitSearchFieldDisjunction() && unrestrictedReadingField();
    }

    private boolean implicitSearchFieldDisjunction() {
        return isTableField()
            || isUniqueKeyProperty()
            || isBusinessKeyProperty()
            || isCharacterKeyProperty()
            || isNumericKeyProperty()
            || isDiscriminatorProperty()
            || isNameProperty()
            || isInactiveIndicatorProperty();
    }

    /**
     * @return the filter field indicator
     */
    @Override
    public boolean isFilterField() {
        return !isHiddenField() && !isBinaryData() && !_password && !_calculable && BitUtils.valueOf(_filterField, implicitFilterField());
    }

    private boolean implicitFilterField() {
//      if (isSearchField()) {
//          return true;
//      }
//      if (isStringData()) {
//          StringData stringData = (StringData) this;
//          Integer maxLength = stringData.getMaxLength();
//          return maxLength != null && maxLength <= 200;
//      }
        return isSearchField() || unrestrictedReadingField();
    }

    /**
     * @return the sort field indicator
     */
    @Override
    public boolean isSortField() {
        return !isHiddenField() && !isBinaryData() && !_password && BitUtils.valueOf(_sortField, unrestrictedReadingField());
    }

    /**
     * @return the table field indicator
     */
    @Override
    public boolean isTableField() {
        return !isHiddenField() && !isUndisplayableBinaryData() && !_password && BitUtils.valueOf(_tableField, implicitTableField());
    }

    private boolean implicitTableField() {
        return isRequiredProperty()
            || isUniqueKeyProperty()
            || isBusinessKeyProperty()
            || isCharacterKeyProperty()
            || isNumericKeyProperty()
            || isDiscriminatorProperty()
            || isNameProperty()
            || isInactiveIndicatorProperty();
    }

    private boolean requiredTableField() {
        return isTableField() && isRequiredProperty();
    }

    /**
     * @return the detail field indicator
     */
    @Override
    public boolean isDetailField() {
        return !isHiddenField() && !isUndisplayableBinaryData() && _detailField;
    }

    /**
     * @return the column field indicator
     */
    @Override
    public boolean isColumnField() {
        return !isHiddenField() && !isBinaryData() && !_password && BitUtils.valueOf(_columnField, !isFileReferenceField());
    }

    /**
     * @return the report field indicator
     */
    @Override
    public boolean isReportField() {
        return !isHiddenField() && !isBinaryData() && !_password && BitUtils.valueOf(_reportField, implicitReportField());
    }

    private boolean implicitReportField() {
        return implicitReportFieldDisjunction() && unrestrictedReadingField();
    }

    private boolean implicitReportFieldDisjunction() {
        return isRequiredProperty()
            || isUniqueKeyProperty()
            || isBusinessKeyProperty()
            || isCharacterKeyProperty()
            || isNumericKeyProperty()
            || isDiscriminatorProperty()
            || isNameProperty()
            || isInactiveIndicatorProperty();
    }

    /**
     * @return the export field indicator
     */
    @Override
    public boolean isExportField() {
        return !isHiddenField() && !isBinaryData() && !_password && BitUtils.valueOf(_exportField, unrestrictedReadingField());
    }

    /**
     * @return the submit field indicator
     */
    @Override
    public boolean isSubmitField() {
        return !isHiddenField() && !isBinaryData() && !_password && _submitField;
    }

    /**
     * @return the headertextless field indicator
     */
    @Override
    public boolean isHeadertextlessField() {
        return !isHiddenField() && !isBinaryData() && _headertextlessField;
    }

    /**
     * @return the heading field indicator
     */
    @Override
    public boolean isHeadingField() {
        return !isHiddenField() && !isUndisplayableBinaryData() && !restrictedReadingAccess() && BitUtils.valueOf(_headingField, implicitHeadingField());
    }

    private boolean implicitHeadingField() {
        return isBusinessKeyProperty()
            || isNameProperty()
            || isDiscriminatorProperty()
            || isInactiveIndicatorProperty()
            || isUniqueKeyProperty()
            || isCharacterKeyProperty()
            || isNumericKeyProperty()
            || isStateProperty()
            || isCodelessNamelessEntityUniqueKeyProperty();
    }

    /**
     * @return the overlay field indicator
     */
    @Override
    public boolean isOverlayField() {
        return !isHiddenField() && !isUndisplayableBinaryData() && unrestrictedReadingField() && BitUtils.valueOf(_overlayField, implicitOverlayField());
    }

    private boolean implicitOverlayField() {
        return implicitOverlayFieldDisjunction(); // && unrestrictedReadingField();
    }

    private boolean implicitOverlayFieldDisjunction() {
        if (isBusinessKeyProperty() || isNameProperty()) {
            return false;
        }
        return isImageProperty()
            || isDiscriminatorProperty()
            || isStateProperty()
            || isInactiveIndicatorProperty()
            || isUniqueKeyProperty()
            || isCharacterKeyProperty()
            || isNumericKeyProperty()
            || isCodelessNamelessEntityUniqueKeyProperty();
    }

    private boolean isCodelessNamelessEntityUniqueKeyProperty() {
        PersistentEntity pent = getDeclaringPersistentEntityRoot();
        return pent != null
            && pent.getBusinessKeyProperty() == null
            && pent.getNameProperty() == null
            && pent.getUniqueKeyPropertiesList().contains(this);
    }

    /**
     * A prominent field should be visible at all times.
     *
     * @return the prominent field indicator
     */
    public boolean isProminentField() {
        return requiredTableField() // isRequiredProperty or requiredTableField ?
            || isBusinessKeyProperty()
            || isCharacterKeyProperty()
            || isNumericKeyProperty()
            || isUniqueKeyProperty()
            || isDiscriminatorProperty()
            || isInactiveIndicatorProperty()
            || isNameProperty()
            || isOwnerProperty()
            || isParentProperty()
            || isSegmentProperty()
            || isStateProperty()
            || isHeadingField();
    }

    private boolean unrestrictedReadingField() {
        return !restrictedReadingField();
    }

    private boolean restrictedReadingField() {
        return restrictedReadingAccess() || isFileReferenceField();
    }

    /**
     * @return the immutable field indicator
     */
    @Override
    public boolean isImmutableField() {
        return !isUpdateField() && BitUtils.valueOf(_immutableField);
    }

    /**
     * @return the default condition
     */
    @Override
    public DefaultCondition getDefaultCondition() {
        return _defaultCondition;
    }

    /**
     * @return the default checkpoint
     */
    @Override
    public Checkpoint getDefaultCheckpoint() {
        return _defaultCheckpoint;
    }

    /**
     * @return the sequence number
     */
    @Override
    public int getSequenceNumber() {
        if (_sequenceNumber == 0) {
            if (isParameter()) {
                Operation declaringOperation = getDeclaringOperation();
                List<Parameter> parameters = declaringOperation == null ? null : declaringOperation.getParametersList();
                return parameters == null || parameters.isEmpty() ? 0 : parameters.indexOf(this) + 1;
            }
            if (isProperty()) {
                Entity declaringEntity = getDeclaringEntity();
                List<Property> properties = declaringEntity == null ? null : declaringEntity.getPropertiesList();
                return properties == null || properties.isEmpty() ? 0 : properties.indexOf(this) + 1;
            }
        }
        return _sequenceNumber;
    }

    /**
     * @return the aggregate function
     */
    @Override
    public AggregateFunction getAggregateFunction() {
        return _aggregateFunction;
    }

    /**
     * @return the aggregate title
     */
    @Override
    public String getAggregateTitle() {
        return _aggregateTitle;
    }

    /**
     * @return the calculable indicator
     */
    @Override
    public boolean isCalculable() {
        return _calculable;
    }

    /**
     * @return the nullable indicator
     */
    @Override
    public boolean isNullable() {
        return _nullable;
    }

    /**
     * @return the insertable indicator
     */
    @Override
    public boolean isInsertable() {
        return _insertable;
    }

    /**
     * @return the updateable indicator
     */
    @Override
    public boolean isUpdateable() {
        return _updateable;
    }

    /**
     * @return the unique indicator
     */
    @Override
    public boolean isUnique() {
        return _annotatedWithUniqueKey || _unique
            || isPrimaryKeyProperty()
            || isSequenceProperty()
            || isBusinessKeyProperty()
            || isCharacterKeyProperty()
            || isNumericKeyProperty();
    }

    /**
     * @return the instance reference indicator
     */
    @Override
    public boolean isInstanceReferenceField() {
        return _annotatedWithInstanceReference;
    }

    /**
     * @return the file reference indicator
     */
    @Override
    public boolean isFileReferenceField() {
        return _annotatedWithFileReference;
    }

    /**
     * @return the graphic image indicator
     */
    @Override
    public boolean isGraphicImageField() {
        return _annotatedWithGraphicImage || _annotatedWithImageProperty;
    }

    /**
     * @return the URL indicator
     */
    @Override
    public boolean isUniformResourceLocatorField() {
        return _annotatedWithUniformResourceLocator;
    }

    /**
     * @return the variant string indicator
     */
    @Override
    public boolean isVariantStringField() {
        return _annotatedWithVariantString;
    }

    /**
     * @return the linked field name
     */
    @Override
    public String getLinkedFieldName() {
        return _linkedFieldName;
    }

    /**
     * @return the linked field
     */
    @Override
    public Field getLinkedField() {
        return _linkedField;
    }

    /**
     * @return the linked property
     */
    @Override
    public Property getLinkedProperty() {
        return _linkedProperty;
    }

    /**
     * @return the linked column name
     */
    @Override
    public String getLinkedColumnName() {
        return _linkedColumnName;
    }

    /**
     * @return the linked column operator
     */
    @Override
    public StandardRelationalOp getLinkedColumnOperator() {
        return _linkedColumnOperator;
    }

    /**
     * @return the data generation type
     */
//  @Override
    public DataGenType getDataGenType() {
        return _dataGenType;
    }

    /**
     * @return the series start
     */
//  @Override
    public int getDataGenSeriesStart() {
        return _dataGenSeriesStart;
    }

    /**
     * @return the series stop
     */
//  @Override
    public int getDataGenSeriesStop() {
        return _dataGenSeriesStop;
    }

    /**
     * @return the series step
     */
//  @Override
    public int getDataGenSeriesStep() {
        return _dataGenSeriesStep;
    }

    /**
     * @return the entity data generation enabled indicator
     */
//  @Override
    public boolean isDataGenSeriesEnabled() {
        return _annotatedWithDataGen && _dataGenSeriesStart <= _dataGenSeriesStop && _dataGenSeriesStep > 0;
    }

    /**
     * @return the data generation user-defined function name
     */
//  @Override
    public String getDataGenFunction() {
        return _dataGenFunction;
    }

    /**
     * @return the data nullable
     */
//  @Override
    public int getDataGenNullable() {
        return _dataGenNullable;
    }

    /**
     * @return the data trueable percentage
     */
//  @Override
    public int getDataGenTrueable() {
        return _dataGenTrueable;
    }

    /**
     * @return the data generation pattern
     */
//  @Override
    public String getDataGenPattern() {
        return _dataGenPattern;
    }

    /**
     * @return the data generation prefix
     */
//  @Override
    public String getDataGenPrefix() {
        return _dataGenPrefix;
    }

    /**
     * @return the data generation suffix
     */
//  @Override
    public String getDataGenSuffix() {
        return _dataGenSuffix;
    }

    /**
     * @return the data min
     */
//  @Override
    public Object getDataGenMin() {
        return _dataGenMin;
    }

    /**
     * @return the data max
     */
//  @Override
    public Object getDataGenMax() {
        return _dataGenMax;
    }

    /**
     * @return the data gen temporal interval
     */
//  @Override
    public DataGenTemporalInterval getDataGenTemporalInterval() {
        return _dataGenTemporalInterval;
    }

    /**
     * @return the data gen numeric action
     */
//  @Override
    public DataGenNumericAction getDataGenNumericAction() {
        return _dataGenNumericAction;
    }

    /**
     * @return the data factor
     */
//  @Override
    public Object getDataGenFactor() {
        return _dataGenFactor == null && isNumericPrimitive() ? 1 : _dataGenFactor;
    }

    /**
     * @return the rendering filter
     */
    @Override
    public BooleanExpression getRenderingFilter() {
        return _renderingFilter;
    }

    /**
     * El método setRenderingFilter se utiliza para establecer el filtro de presentación de propiedades en vistas (páginas) de registro, y de
     * parámetros en vistas (páginas) de ejecución de operaciones de negocio. Solo si se cumplen los criterios del filtro, el valor de la propiedad o
     * el parámetro será visible en las vistas (páginas) de registro o ejecución de operaciones de negocio.
     *
     * @param renderingFilter expresión booleana que se utiliza como filtro
     */
    @Override
    public void setRenderingFilter(BooleanExpression renderingFilter) {
        _renderingFilter = renderingFilter instanceof BooleanPrimitive ? renderingFilter.isTrue() : renderingFilter;
    }

    /**
     * @return the requiring filter
     */
    @Override
    public BooleanExpression getRequiringFilter() {
        return _requiringFilter;
    }

    /**
     * El método setRequiringFilter se utiliza para establecer el filtro de obligatoriedad de propiedades en vistas (páginas) de registro, y de
     * parámetros en vistas (páginas) de ejecución de operaciones de negocio. Solo si se cumplen los criterios del filtro, el valor de la propiedad o
     * el parámetro será obligatoriamente requerido en las vistas (páginas) de registro o ejecución de operaciones de negocio.
     *
     * @param requiringFilter expresión booleana que se utiliza como filtro
     */
    @Override
    public void setRequiringFilter(BooleanExpression requiringFilter) {
        _requiringFilter = requiringFilter instanceof BooleanPrimitive ? requiringFilter.isTrue() : requiringFilter;
    }

    /**
     * @return the modifying filter
     */
    @Override
    public BooleanExpression getModifyingFilter() {
        return _modifyingFilter;
    }

    /**
     * El método setModifyingFilter se utiliza para establecer el filtro de modificación de propiedades en vistas (páginas) de registro, y de
     * parámetros en vistas (páginas) de ejecución de operaciones de negocio. Solo si se cumplen los criterios del filtro, el valor de la propiedad o
     * el parámetro podrá ser modificado en las vistas (páginas) de registro o ejecución de operaciones de negocio.
     *
     * @param modifyingFilter expresión booleana que se utiliza como filtro
     */
    @Override
    public void setModifyingFilter(BooleanExpression modifyingFilter) {
        _modifyingFilter = modifyingFilter instanceof BooleanPrimitive ? modifyingFilter.isTrue() : modifyingFilter;
    }

    /**
     * @return the nullifying filter
     */
    @Override
    public BooleanExpression getNullifyingFilter() {
        return _nullifyingFilter;
    }

    /**
     * El método setNullifyingFilter se utiliza para establecer el filtro de anulación de propiedades en vistas (páginas) de registro, y de parámetros
     * en vistas (páginas) de ejecución de operaciones de negocio. Solo si se cumplen los criterios del filtro, el valor de la propiedad o el
     * parámetro será anulado en las vistas (páginas) de registro o ejecución de operaciones de negocio.
     *
     * @param nullifyingFilter expresión booleana que se utiliza como filtro
     */
    @Override
    public void setNullifyingFilter(BooleanExpression nullifyingFilter) {
        _nullifyingFilter = nullifyingFilter instanceof BooleanPrimitive ? nullifyingFilter.isTrue() : nullifyingFilter;
    }

    /**
     * @return the enclosing tabs list
     */
    @Override
    public List<Tab> getEnclosingTabs() {
        List<Tab> list = new ArrayList<>();
        if (isProperty()) {
            Entity declaringEntity = getDeclaringEntity();
            List<Tab> tabsList = declaringEntity.getTabsList();
            List<TabField> tabFieldsList;
            for (Tab tab : tabsList) {
                tabFieldsList = tab.getTabFieldsList();
                for (TabField tabField : tabFieldsList) {
                    if (this.equals(tabField.getProperty())) {
                        list.add(tab);
                        break;
                    }
                }
            }
        }
        return list;
    }

    public boolean isEnclosedInAtLeastOneTab() {
        if (isProperty()) {
            Entity declaringEntity = getDeclaringEntity();
            List<Tab> tabsList = declaringEntity.getTabsList();
            List<TabField> tabFieldsList;
            for (Tab tab : tabsList) {
                tabFieldsList = tab.getTabFieldsList();
                for (TabField tabField : tabFieldsList) {
                    if (this.equals(tabField.getProperty())) {
                        return true;
                    }
                }
            }
        }
        return false;
    }

    /**
     * @return the BaseField annotation indicator
     */
//  @Override
    public boolean isAnnotatedWithBaseField() {
        return _annotatedWithBaseField;
    }

    /**
     * @return PrimaryKey annotation indicator
     */
//  @Override
    public boolean isAnnotatedWithPrimaryKey() {
        return _annotatedWithPrimaryKey;
    }

    /**
     * @return SequenceProperty annotation indicator
     */
//  @Override
    public boolean isAnnotatedWithSequenceProperty() {
        return _annotatedWithSequenceProperty;
    }

    /**
     * @return VersionProperty annotation indicator
     */
//  @Override
    public boolean isAnnotatedWithVersionProperty() {
        return _annotatedWithVersionProperty;
    }

    /**
     * @return NumericKey annotation indicator
     */
//  @Override
    public boolean isAnnotatedWithNumericKey() {
        return _annotatedWithNumericKey;
    }

    /**
     * @return CharacterKey annotation indicator
     */
//  @Override
    public boolean isAnnotatedWithCharacterKey() {
        return _annotatedWithCharacterKey;
    }

    /**
     * @return NameProperty annotation indicator
     */
//  @Override
    public boolean isAnnotatedWithNameProperty() {
        return _annotatedWithNameProperty;
    }

    /**
     * @return DescriptionProperty annotation indicator
     */
//  @Override
    public boolean isAnnotatedWithDescriptionProperty() {
        return _annotatedWithDescriptionProperty;
    }

    /**
     * @return ImageProperty annotation indicator
     */
//  @Override
    public boolean isAnnotatedWithImageProperty() {
        return _annotatedWithImageProperty;
    }

    /**
     * @return InactiveIndicator annotation indicator
     */
//  @Override
    public boolean isAnnotatedWithInactiveIndicator() {
        return _annotatedWithInactiveIndicator;
    }

    /**
     * @return UrlProperty annotation indicator
     */
//  @Override
    public boolean isAnnotatedWithUrlProperty() {
        return _annotatedWithUrlProperty;
    }

    /**
     * @return ParentProperty annotation indicator
     */
//  @Override
    public boolean isAnnotatedWithParentProperty() {
        return _annotatedWithParentProperty;
    }

    /**
     * @return OwnerProperty annotation indicator
     */
//  @Override
    public boolean isAnnotatedWithOwnerProperty() {
        return _annotatedWithOwnerProperty;
    }

    /**
     * @return SegmentProperty annotation indicator
     */
//  @Override
    public boolean isAnnotatedWithSegmentProperty() {
        return _annotatedWithSegmentProperty;
    }

    /**
     * @return UniqueKey annotation indicator
     */
//  @Override
    public boolean isAnnotatedWithUniqueKey() {
        return _annotatedWithUniqueKey;
    }

    /**
     * @return BusinessKey annotation indicator
     */
//  @Override
    public boolean isAnnotatedWithBusinessKey() {
        return _annotatedWithBusinessKey;
    }

    /**
     * @return DiscriminatorColumn annotation indicator
     */
//  @Override
    public boolean isAnnotatedWithDiscriminatorColumn() {
        return _annotatedWithDiscriminatorColumn;
    }

    /**
     * @return StateProperty annotation indicator
     */
//  @Override
    public boolean isAnnotatedWithStateProperty() {
        return _annotatedWithStateProperty;
    }

    /**
     * @return the ColumnField annotation indicator
     */
//  @Override
    public boolean isAnnotatedWithColumnField() {
        return _annotatedWithColumnField;
    }

    /**
     * @return the BigDecimalField annotation indicator
     */
//  @Override
    public boolean isAnnotatedWithBigDecimalField() {
        return _annotatedWithBigDecimalField;
    }

    /**
     * @return the BooleanField annotation indicator
     */
//  @Override
    public boolean isAnnotatedWithBooleanField() {
        return _annotatedWithBooleanField;
    }

    /**
     * @return the NumericField annotation indicator
     */
//  @Override
    public boolean isAnnotatedWithNumericField() {
        return _annotatedWithNumericField;
    }

    /**
     * @return the StringField annotation indicator
     */
//  @Override
    public boolean isAnnotatedWithStringField() {
        return _annotatedWithStringField;
    }

    /**
     * @return the TimeField annotation indicator
     */
//  @Override
    public boolean isAnnotatedWithTimeField() {
        return _annotatedWithTimeField;
    }

    /**
     * @return the TimestampField annotation indicator
     */
//  @Override
    public boolean isAnnotatedWithTimestampField() {
        return _annotatedWithTimestampField;
    }

    /**
     * @return the ParameterField annotation indicator
     */
//  @Override
    public boolean isAnnotatedWithParameterField() {
        return _annotatedWithParameterField;
    }

    /**
     * @return the PropertyField annotation indicator
     */
//  @Override
    public boolean isAnnotatedWithPropertyField() {
        return _annotatedWithPropertyField;
    }

    /**
     * @return the PropertyAggregation annotation indicator
     */
//  @Override
    public boolean isAnnotatedWithPropertyAggregation() {
        return _annotatedWithPropertyAggregation;
    }

    /**
     * @return the InstanceReference annotation indicator
     */
//  @Override
    public boolean isAnnotatedWithInstanceReference() {
        return _annotatedWithInstanceReference;
    }

    /**
     * @return the FileReference annotation indicator
     */
//  @Override
    public boolean isAnnotatedWithFileReference() {
        return _annotatedWithFileReference;
    }

    /**
     * @return the UniformResourceLocator annotation indicator
     */
//  @Override
    public boolean isAnnotatedWithUniformResourceLocator() {
        return _annotatedWithUniformResourceLocator;
    }

    /**
     * @return the VariantString annotation indicator
     */
//  @Override
    public boolean isAnnotatedWithVariantString() {
        return _annotatedWithVariantString;
    }

    /**
     * @return the DataGen annotation indicator
     */
//  @Override
    public boolean isAnnotatedWithDataGen() {
        return _annotatedWithDataGen;
    }
    // </editor-fold>

    // <editor-fold defaultstate="collapsed" desc="annotate">
    @Override
    void initializeAnnotations() {
        super.initializeAnnotations();
        _annotatedWithPrimaryKey = false;
        _annotatedWithSequenceProperty = false;
        _annotatedWithVersionProperty = false;
        _annotatedWithNumericKey = false;
        _annotatedWithCharacterKey = false;
        _annotatedWithNameProperty = false;
        _annotatedWithDescriptionProperty = false;
        _annotatedWithImageProperty = false;
        _annotatedWithInactiveIndicator = false;
        _annotatedWithUrlProperty = false;
        _annotatedWithParentProperty = false;
        _annotatedWithOwnerProperty = false;
        _annotatedWithSegmentProperty = false;
        _annotatedWithUniqueKey = false;
        _annotatedWithBusinessKey = false;
        _annotatedWithDiscriminatorColumn = false;
        _annotatedWithStateProperty = false;
        _annotatedWithBaseField = false;
        _annotatedWithColumnField = false;
        _annotatedWithBigDecimalField = false;
        _annotatedWithBooleanField = false;
        _annotatedWithNumericField = false;
        _annotatedWithStringField = false;
        _annotatedWithTimeField = false;
        _annotatedWithTimestampField = false;
        _annotatedWithFileReference = false;
        _annotatedWithGraphicImage = false;
        _annotatedWithUniformResourceLocator = false;
        _annotatedWithVariantString = false;
        _annotatedWithDataGen = false;
        _annotatedWithInstanceReference = false;
        _annotatedWithParameterField = false;
        _annotatedWithPropertyField = false;
        _annotatedWithPropertyAggregation = false;
        _propertyAccess = PropertyAccess.UNSPECIFIED;
        _auditable = true;
        _password = false;
        _required = null;
        _hiddenField = false;
        Entity declaringEntity = getDeclaringEntity();
        if (declaringEntity instanceof EnumerationEntity) {
            _createField = false;
            _updateField = false;
        } else {
            _createField = null;
            _updateField = null;
        }
        _searchField = null;
        _filterField = null;
        _sortField = null;
        _tableField = null;
        _detailField = true;
        _columnField = null;
        _reportField = null;
        _exportField = null;
        _submitField = false;
        _headertextlessField = false;
        _headingField = null;
        _overlayField = null;
        _immutableField = null;
        _defaultCondition = DefaultCondition.IF_NULL;
        _defaultCheckpoint = Checkpoint.WHEREVER_POSSIBLE;
        _sequenceNumber = 0;
        _aggregateFunction = null;
        _aggregateTitle = null;
        _calculable = false;
        _nullable = true;
        _insertable = true;
        _updateable = true;
        _unique = false;
        _linkedFieldName = null;
        _linkedField = null;
        _linkedProperty = null;
        _linkedColumnName = null;
        _linkedColumnOperator = StandardRelationalOp.EQ;
        _dataGenType = DataGenType.UNSPECIFIED;
        _dataGenSeriesStart = 1;
        _dataGenSeriesStop = 10000;
        _dataGenSeriesStep = 1;
        _dataGenFunction = null;
        _dataGenNullable = 10;
        _dataGenTrueable = 50;
        _dataGenPattern = null;
        _dataGenPrefix = null;
        _dataGenSuffix = null;
        _dataGenMin = null;
        _dataGenMax = null;
        _dataGenTemporalInterval = DataGenTemporalInterval.UNSPECIFIED;
        _dataGenNumericAction = DataGenNumericAction.UNSPECIFIED;
        _dataGenFactor = null;
    }

    @Override
    void annotate(Class<?> type) {
        super.annotate(type);
    }

    @Override
    void annotate(Field field) {
        super.annotate(field);
        if (field != null) {
            if (isParameter()) {
                annotateBigDecimalField(field);
                annotateBooleanField(field);
                annotateNumericField(field);
                annotateStringField(field);
                annotateDateField(field);
                annotateTimeField(field);
                annotateTimestampField(field);
                annotateParameterField(field); // must be executed before annotateFileReference
                annotateFileReference(field);
                annotateInstanceReference(field);
            }
            if (isProperty()) {
                annotateKeyProperties(field);
                annotateBaseField(field);
                annotateColumnField(field);
                annotateBigDecimalField(field);
                annotateBooleanField(field);
                annotateNumericField(field);
                annotateStringField(field);
                annotateDateField(field);
                annotateTimeField(field);
                annotateTimestampField(field);
                annotateFileReference(field);
                annotateGraphicImage(field);
                annotateUniformResourceLocator(field);
                annotateVariantString(field);
                annotatePropertyField(field);
                annotatePropertyAggregation(field);
                annotateDataGen(field);
            }
        }
    }

    private void annotateKeyProperties(Field field) {
        annotatePrimaryKey(field);
        annotateSequenceProperty(field);
        annotateVersionProperty(field);
        annotateNumericKey(field);
        annotateCharacterKey(field);
        annotateNameProperty(field);
        annotateDescriptionProperty(field);
        annotateImageProperty(field);
        annotateInactiveIndicator(field);
        annotateUrlProperty(field);
        annotateParentProperty(field);
        annotateOwnerProperty(field);
        annotateSegmentProperty(field);
        annotateUniqueKey(field);
        annotateBusinessKey(field);
        annotateDiscriminatorColumn(field);
        annotateStateProperty(field);
        initializeKeyPropertyAnnotations();
    }

    private void annotatePrimaryKey(Field field) {
        PrimaryKey annotation = field.getAnnotation(PrimaryKey.class);
        if (annotation != null && annotation.value()) {
            Class<? extends Annotation> annotationClass = PrimaryKey.class;
            boolean log = depth() == 0;
            boolean aye = XS1.checkKeyPropertyFieldAnnotation(log, field, KeyProperty.PRIMARY_KEY);
            if (aye) {
                Field previous = getDeclaringArtifact().put(annotationClass, field);
                if (previous == null) {
                    _annotatedWithPrimaryKey = true;
                } else if (log) {
                    XS1.logDuplicateAnnotation(field, annotationClass, previous);
                }
            }
        }
    }

    private void annotateSequenceProperty(Field field) {
        SequenceProperty annotation = field.getAnnotation(SequenceProperty.class);
        if (annotation != null && annotation.value()) {
            Class<? extends Annotation> annotationClass = SequenceProperty.class;
            boolean log = depth() == 0;
            boolean aye = XS1.checkKeyPropertyFieldAnnotation(log, field, KeyProperty.SEQUENCE);
            if (aye) {
                Field previous = getDeclaringArtifact().put(annotationClass, field);
                if (previous == null) {
                    _annotatedWithSequenceProperty = true;
                } else if (log) {
                    XS1.logDuplicateAnnotation(field, annotationClass, previous);
                }
            }
        }
    }

    private void annotateVersionProperty(Field field) {
        VersionProperty annotation = field.getAnnotation(VersionProperty.class);
        if (annotation != null && annotation.value()) {
            Class<? extends Annotation> annotationClass = VersionProperty.class;
            boolean log = depth() == 0;
            boolean aye = XS1.checkKeyPropertyFieldAnnotation(log, field, KeyProperty.VERSION);
            if (aye) {
                Field previous = getDeclaringArtifact().put(annotationClass, field);
                if (previous == null) {
                    _annotatedWithVersionProperty = true;
                } else if (log) {
                    XS1.logDuplicateAnnotation(field, annotationClass, previous);
                }
            }
        }
    }

    private void annotateNumericKey(Field field) {
        NumericKey annotation = field.getAnnotation(NumericKey.class);
        if (annotation != null && annotation.value()) {
            Class<? extends Annotation> annotationClass = NumericKey.class;
            boolean log = depth() == 0;
            boolean aye = XS1.checkKeyPropertyFieldAnnotation(log, field, KeyProperty.NUMERIC_KEY);
            if (aye) {
                Field previous = getDeclaringArtifact().put(annotationClass, field);
                if (previous == null) {
                    _annotatedWithNumericKey = true;
                } else if (log) {
                    XS1.logDuplicateAnnotation(field, annotationClass, previous);
                }
            }
        }
    }

    private void annotateCharacterKey(Field field) {
        CharacterKey annotation = field.getAnnotation(CharacterKey.class);
        if (annotation != null && annotation.value()) {
            Class<? extends Annotation> annotationClass = CharacterKey.class;
            boolean log = depth() == 0;
            boolean aye = XS1.checkKeyPropertyFieldAnnotation(log, field, KeyProperty.CHARACTER_KEY);
            if (aye) {
                Field previous = getDeclaringArtifact().put(annotationClass, field);
                if (previous == null) {
                    _annotatedWithCharacterKey = true;
                } else if (log) {
                    XS1.logDuplicateAnnotation(field, annotationClass, previous);
                }
            }
        }
    }

    private void annotateNameProperty(Field field) {
        NameProperty annotation = field.getAnnotation(NameProperty.class);
        if (annotation != null && annotation.value()) {
            Class<? extends Annotation> annotationClass = NameProperty.class;
            boolean log = depth() == 0;
            boolean aye = XS1.checkKeyPropertyFieldAnnotation(log, field, KeyProperty.NAME);
            if (aye) {
                Field previous = getDeclaringArtifact().put(annotationClass, field);
                if (previous == null) {
                    _annotatedWithNameProperty = true;
                } else if (log) {
                    XS1.logDuplicateAnnotation(field, annotationClass, previous);
                }
            }
        }
    }

    private void annotateDescriptionProperty(Field field) {
        DescriptionProperty annotation = field.getAnnotation(DescriptionProperty.class);
        if (annotation != null && annotation.value()) {
            Class<? extends Annotation> annotationClass = DescriptionProperty.class;
            boolean log = depth() == 0;
            boolean aye = XS1.checkKeyPropertyFieldAnnotation(log, field, KeyProperty.DESCRIPTION);
            if (aye) {
                Field previous = getDeclaringArtifact().put(annotationClass, field);
                if (previous == null) {
                    _annotatedWithDescriptionProperty = true;
                } else if (log) {
                    XS1.logDuplicateAnnotation(field, annotationClass, previous);
                }
            }
        }
    }

    private void annotateImageProperty(Field field) {
        Class<? extends Annotation> annotationClass = ImageProperty.class;
        boolean log = depth() == 0;
        boolean aye = field.isAnnotationPresent(annotationClass)
            && XS1.checkKeyPropertyFieldAnnotation(log, field, KeyProperty.IMAGE);
        /**/
        if (aye) {
            Field previous = getDeclaringArtifact().put(annotationClass, field);
            if (previous == null) {
                _annotatedWithImageProperty = true;
                ImageProperty annotation = field.getAnnotation(ImageProperty.class);
                int displayWidth = annotation.displayWidth();
                int displayHeight = annotation.displayHeight();
                boolean resizable = annotation.resizable().toBoolean(true);
                if (displayWidth <= 0 && displayHeight <= 0) {
                } else if (displayWidth <= 0) {
                    displayWidth = displayHeight;
                } else if (displayHeight <= 0) {
                    displayHeight = displayWidth;
                }
                BinaryData data = (BinaryData) this;
                data.setDisplayWidth(displayWidth);
                data.setDisplayHeight(displayHeight);
                data.setResizable(resizable);
            } else if (log) {
                XS1.logDuplicateAnnotation(field, annotationClass, previous);
            }
        }
    }

    private void annotateInactiveIndicator(Field field) {
        InactiveIndicator annotation = field.getAnnotation(InactiveIndicator.class);
        if (annotation != null && annotation.value()) {
            Class<? extends Annotation> annotationClass = InactiveIndicator.class;
            boolean log = depth() == 0;
            boolean aye = XS1.checkKeyPropertyFieldAnnotation(log, field, KeyProperty.INACTIVE_INDICATOR);
            if (aye) {
                Field previous = getDeclaringArtifact().put(annotationClass, field);
                if (previous == null) {
                    _annotatedWithInactiveIndicator = true;
                } else if (log) {
                    XS1.logDuplicateAnnotation(field, annotationClass, previous);
                }
            }
        }
    }

    private void annotateUrlProperty(Field field) {
        Class<? extends Annotation> annotationClass = UrlProperty.class;
        boolean log = depth() == 0;
        boolean aye = field.isAnnotationPresent(annotationClass)
            && XS1.checkKeyPropertyFieldAnnotation(log, field, KeyProperty.URL);
        /**/
        if (aye) {
            Field previous = getDeclaringArtifact().put(annotationClass, field);
            if (previous == null) {
                _annotatedWithUrlProperty = true;
                UrlProperty annotation = field.getAnnotation(UrlProperty.class);
                UrlType urlType = annotation.urlType();
                UrlDisplayType urlDisplayType = annotation.urlDisplayType();
                StringData data = (StringData) this;
                data.setUrlType(urlType);
                data.setUrlDisplayType(urlDisplayType);
            } else if (log) {
                XS1.logDuplicateAnnotation(field, annotationClass, previous);
            }
        }
    }

    private void annotateParentProperty(Field field) {
        ParentProperty annotation = field.getAnnotation(ParentProperty.class);
        if (annotation != null && annotation.value()) {
            Class<? extends Annotation> annotationClass = ParentProperty.class;
            Class<?> declaringClass = field.getDeclaringClass();
            Class<?>[] validTypes = new Class<?>[]{declaringClass};
            boolean log = depth() == 1;
            boolean aye = XS1.checkKeyPropertyFieldAnnotation(log, field, KeyProperty.PARENT, validTypes);
            if (aye) {
                Field previous = getDeclaringArtifact().put(annotationClass, field);
                if (previous == null) {
                    _annotatedWithParentProperty = true;
                } else if (log) {
                    XS1.logDuplicateAnnotation(field, annotationClass, previous);
                }
            }
        }
    }

    private void annotateOwnerProperty(Field field) {
        OwnerProperty annotation = field.getAnnotation(OwnerProperty.class);
        if (annotation != null && annotation.value()) {
            Class<? extends Annotation> annotationClass = OwnerProperty.class;
            boolean log = depth() == 1;
            boolean aye = XS1.checkKeyPropertyFieldAnnotation(log, field, KeyProperty.OWNER);
            if (aye) {
                Class<?> fieldType = field.getType();
                Class<? extends Entity> userEntityClass = TLC.getProject().getUserEntityClass();
                if (userEntityClass != null && userEntityClass.isAssignableFrom(fieldType)) {
                    Field previous = getDeclaringArtifact().put(annotationClass, field);
                    if (previous == null) {
                        _annotatedWithOwnerProperty = true;
                    } else if (log) {
                        XS1.logDuplicateAnnotation(field, annotationClass, previous);
                    }
                } else if (log) {
                    String message = userEntityClass + " is not assignable from " + fieldType;
                    XS1.logFieldAnnotationErrorMessage(field, annotationClass, message);
                }
            }
        }
    }

    private void annotateSegmentProperty(Field field) {
        SegmentProperty annotation = field.getAnnotation(SegmentProperty.class);
        if (annotation != null && annotation.value()) {
            Class<? extends Annotation> annotationClass = SegmentProperty.class;
            boolean log = depth() == 1;
            boolean aye = XS1.checkKeyPropertyFieldAnnotation(log, field, KeyProperty.SEGMENT);
            if (aye) {
                Field previous = getDeclaringArtifact().put(annotationClass, field);
                if (previous == null) {
                    _annotatedWithSegmentProperty = true;
                } else if (log) {
                    XS1.logDuplicateAnnotation(field, annotationClass, previous);
                }
            }
        }
    }

    private void annotateUniqueKey(Field field) {
        UniqueKey annotation = field.getAnnotation(UniqueKey.class);
        if (annotation != null && annotation.value()) {
            boolean log = this instanceof Entity ? depth() == 1 : depth() == 0;
            _annotatedWithUniqueKey = XS1.checkKeyPropertyFieldAnnotation(log, field, KeyProperty.UNIQUE_KEY);
        }
    }

    private void annotateBusinessKey(Field field) {
        BusinessKey annotation = field.getAnnotation(BusinessKey.class);
        if (annotation != null && annotation.value()) {
            Class<? extends Annotation> annotationClass = BusinessKey.class;
            boolean log = depth() == 0;
            boolean aye = XS1.checkKeyPropertyFieldAnnotation(log, field, KeyProperty.BUSINESS_KEY);
            if (aye) {
                Field previous = getDeclaringArtifact().put(annotationClass, field);
                if (previous == null) {
                    _annotatedWithBusinessKey = true;
                } else if (log) {
                    XS1.logDuplicateAnnotation(field, annotationClass, previous);
                }
            }
        }
    }

    private void annotateDiscriminatorColumn(Field field) {
        DiscriminatorColumn annotation = field.getAnnotation(DiscriminatorColumn.class);
        if (annotation != null && annotation.value()) {
            Class<? extends Annotation> annotationClass = DiscriminatorColumn.class;
            boolean log = this instanceof Entity ? depth() == 1 : depth() == 0;
            boolean aye = XS1.checkKeyPropertyFieldAnnotation(log, field, KeyProperty.DISCRIMINATOR);
            if (aye) {
                Field previous = getDeclaringArtifact().put(annotationClass, field);
                if (previous == null) {
                    _annotatedWithDiscriminatorColumn = true;
                } else if (log) {
                    XS1.logDuplicateAnnotation(field, annotationClass, previous);
                }
            }
        }
    }

    private void annotateStateProperty(Field field) {
        StateProperty annotation = field.getAnnotation(StateProperty.class);
        if (annotation != null && annotation.value()) {
            Class<? extends Annotation> annotationClass = StateProperty.class;
            boolean log = depth() == 1;
            boolean aye = XS1.checkKeyPropertyFieldAnnotation(log, field, KeyProperty.STATE);
            if (aye) {
                Field previous = getDeclaringArtifact().put(annotationClass, field);
                if (previous == null) {
                    _annotatedWithStateProperty = true;
                } else if (log) {
                    XS1.logDuplicateAnnotation(field, annotationClass, previous);
                }
            }
        }
    }

    private void initializeKeyPropertyAnnotations() {
        if (isKeyField()) {
            _nullable = false;
            _unique = true;
        }
        if (isPrimaryKeyProperty()) {
            _updateable = false;
            hideKeyProperty();
        }
        if (isSequenceProperty()) {
            _updateable = false;
            hideKeyProperty();
        }
        if (isVersionProperty()) {
            _nullable = false;
            hideKeyProperty();
            NumericPrimitive primitive = (NumericPrimitive) this;
            primitive.setInitialValue(0);
            primitive.setDefaultValue(0);
            primitive.setMinValue(0);
        }
        if (isNumericKeyProperty()) {
            NumericPrimitive primitive = (NumericPrimitive) this;
            primitive.setMinValue(0);
        }
        if (isCharacterKeyProperty()) {
            StringData data = (StringData) this;
            data.setMaxLength(Constants.DEFAULT_CHARACTER_KEY_MAX_LENGTH);
        }
        if (isBusinessKeyProperty()) {
            if (isNumericPrimitive()) {
                NumericPrimitive primitive = (NumericPrimitive) this;
                primitive.setMinValue(0);
            }
            if (isStringData()) {
                StringData data = (StringData) this;
                data.setMaxLength(Constants.DEFAULT_CHARACTER_KEY_MAX_LENGTH);
            }
        }
        if (isNameProperty()) {
            _nullable = false;
            StringData data = (StringData) this;
            data.setMaxLength(Constants.DEFAULT_NAME_PROPERTY_MAX_LENGTH);
        }
        if (isInactiveIndicatorProperty()) {
            _nullable = false;
            BooleanPrimitive primitive = (BooleanPrimitive) this;
            if (primitive.getInitialValue() == null) {
                primitive.setInitialValue(false);
            }
            if (primitive.getDefaultValue() == null) {
                primitive.setDefaultValue(false);
            }
        }
        if (isOwnerProperty()) {
            _nullable = false;
            _createField = false;
            _updateField = false;
            Entity owner = (Entity) this;
            owner.setInitialValue(SpecialEntityValue.CURRENT_USER);
            owner.setDefaultValue(SpecialEntityValue.CURRENT_USER);
        }
        if (isDiscriminatorProperty()) {
            _nullable = false;
            _updateable = false;
            _hiddenField = getDeclaringEntityRoot().getSubclassesMap().isEmpty();
            _createField = false;
            _updateField = false;
        }
    }

    private void hideKeyProperty() {
        _hiddenField = true;
        _createField = false;
        _updateField = false;
        _searchField = false;
        _filterField = false;
//      _sortField = false;
        _tableField = false;
//      _detailField = false;
        _columnField = false;
        _reportField = false;
        _exportField = false;
        _submitField = false;
        _headertextlessField = false;
        _headingField = false;
        _overlayField = false;
        _immutableField = false;
    }

    private void annotateBaseField(Field field) {
        BaseField annotation = field.getAnnotation(BaseField.class);
        if (annotation != null && annotation.value()) {
            Class<? extends Annotation> annotationClass = BaseField.class;
            Class<?>[] validTypes = new Class<?>[]{Property.class};
            boolean log = this instanceof Entity ? depth() == 1 : depth() == 0;
            _annotatedWithBaseField = XS1.checkFieldAnnotation(log, field, annotationClass, validTypes);
        }
    }

    private void annotateColumnField(Field field) {
        String fieldName = fullFieldName(field);
        Class<? extends Annotation> annotationClass = ColumnField.class;
        Class<?>[] validTypes = new Class<?>[]{Property.class};
        boolean log = this instanceof Entity ? depth() == 1 : depth() == 0;
        _annotatedWithColumnField = field.isAnnotationPresent(annotationClass)
            && XS1.checkFieldAnnotation(log, field, annotationClass, validTypes);
        /**/
        if (_annotatedWithColumnField) {
            ColumnField annotation = field.getAnnotation(ColumnField.class);
            _calculable = annotation.calculable().toBoolean(_calculable);
            _nullable = annotation.nullable().toBoolean(_calculable || _nullable);
            _insertable = annotation.insertable().toBoolean(!_calculable && _insertable);
            _updateable = annotation.updateable().toBoolean(!_calculable && _updateable);
            _unique = annotation.unique().toBoolean(!_calculable && _unique);
            if (log) {
                if (_calculable) {
                    if (!_nullable) {
                        logger.error(fieldName + " is calculable and therefore must be nullable");
                        TLC.getProject().getParser().increaseErrorCount();
                    }
                    if (_insertable) {
                        logger.error(fieldName + " is calculable and therefore cannot be insertable");
                        TLC.getProject().getParser().increaseErrorCount();
                    }
                    if (_updateable) {
                        logger.error(fieldName + " is calculable and therefore cannot be updateable");
                        TLC.getProject().getParser().increaseErrorCount();
                    }
                    if (isUnique()) {
                        logger.error(fieldName + " is calculable and therefore cannot be unique");
                        TLC.getProject().getParser().increaseErrorCount();
                    }
                } else {
                    if (_nullable && isUnique()) {
                        logger.error(fieldName + " is nullable and therefore cannot be unique");
                        TLC.getProject().getParser().increaseErrorCount();
                    }
                }
            }
        }
    }

    private void annotateBigDecimalField(Field field) {
        String fieldName = fullFieldName(field);
        Class<? extends Annotation> annotationClass = BigDecimalField.class;
        Class<?>[] validTypes = new Class<?>[]{BigDecimalParameter.class, BigDecimalProperty.class};
        boolean log = isParameter() || depth() == 0;
        _annotatedWithBigDecimalField = field.isAnnotationPresent(annotationClass)
            && XS1.checkFieldAnnotation(log, field, annotationClass, validTypes);
        /**/
        if (_annotatedWithBigDecimalField) {
            BigDecimalField annotation = field.getAnnotation(BigDecimalField.class);
            int precision = annotation.precision();
            int scale = annotation.scale();
            BigDecimalData data = (BigDecimalData) this;
            if (precision >= 1 && precision <= Constants.MAX_DECIMAL_PRECISION) {
                data.setPrecision(precision);
                if (scale <= precision) {
                    data.setScale(scale > 0 ? scale : 0);
                } else if (log) {
                    logger.error(fieldName + " has an invalid decimal scale");
                    TLC.getProject().getParser().increaseErrorCount();
                }
            } else if (log) {
                logger.error(fieldName + " has an invalid decimal precision");
                TLC.getProject().getParser().increaseErrorCount();
            }
        }
    }

    private void annotateBooleanField(Field field) {
        Class<? extends Annotation> annotationClass = BooleanField.class;
        Class<?>[] validTypes = new Class<?>[]{BooleanParameter.class, BooleanProperty.class};
        boolean log = isParameter() || depth() == 0;
        _annotatedWithBooleanField = field.isAnnotationPresent(annotationClass)
            && XS1.checkFieldAnnotation(log, field, annotationClass, validTypes);
        /**/
        if (_annotatedWithBooleanField) {
            BooleanField annotation = field.getAnnotation(BooleanField.class);
            BooleanDisplayType displayType = annotation.displayType();
            BooleanData data = (BooleanData) this;
            data.setBooleanDisplayType(displayType);
        }
    }

    private void annotateNumericField(Field field) {
        String fieldName = fullFieldName(field);
        Class<? extends Annotation> annotationClass = NumericField.class;
        Class<?>[] validTypes = new Class<?>[]{NumericPrimitive.class};
        boolean log = isParameter() || depth() == 0;
        _annotatedWithNumericField = field.isAnnotationPresent(annotationClass)
            && XS1.checkFieldAnnotation(log, field, annotationClass, validTypes);
        /**/
        if (_annotatedWithNumericField) {
            NumericField annotation = field.getAnnotation(NumericField.class);
            int divisor = Math.min(maxDivisor(), Math.max(1, annotation.divisor()));
            DivisorRule divisorRule = annotation.divisorRule();
            String converter = annotation.converter();
            String validator = annotation.validator();
            NumericPrimitive data = (NumericPrimitive) this;
            data.setDivisor(divisor);
            data.setDivisorRule(divisorRule);
            if (StringUtils.isNotBlank(converter)) {
                if (converter.matches(CONVERTER_REGEX)) {
                    data.setSpecialConverterName(converter);
                } else if (log) {
                    logger.error(fieldName + " has an invalid converter name");
                    TLC.getProject().getParser().increaseErrorCount();
                }
            }
            if (StringUtils.isNotBlank(validator)) {
                if (validator.matches(VALIDATOR_REGEX)) {
                    data.setSpecialValidatorName(validator);
                } else if (log) {
                    logger.error(fieldName + " has an invalid validator name");
                    TLC.getProject().getParser().increaseErrorCount();
                }
            }
        }
    }

    private int maxDivisor() {
        if (this instanceof ByteData) {
            return Constants.MAX_BYTE_DIVISOR;
        }
        if (this instanceof ShortData) {
            return Constants.MAX_SHORT_DIVISOR;
        }
        if (this instanceof NumericPrimitive) {
            return Constants.MAX_INTEGER_DIVISOR;
        }
        return 0;
    }

    private void annotateStringField(Field field) {
        String fieldName = fullFieldName(field);
        Class<? extends Annotation> annotationClass = StringField.class;
        Class<?>[] validTypes = new Class<?>[]{StringParameter.class, StringProperty.class};
        boolean log = isParameter() || depth() == 0;
        _annotatedWithStringField = field.isAnnotationPresent(annotationClass)
            && XS1.checkFieldAnnotation(log, field, annotationClass, validTypes);
        /**/
        if (_annotatedWithStringField) {
            StringField annotation = field.getAnnotation(StringField.class);
            int maxLength = annotation.maxLength();
            int minLength = annotation.minLength();
            String regex = annotation.regex();
            String converter = annotation.converter();
            String validator = annotation.validator();
            StringData data = (StringData) this;
            if (maxLength >= 1 && maxLength <= Constants.MAX_STRING_LENGTH) {
                data.setMaxLength(maxLength);
                data.setMinLength(minLength >= 0 ? minLength <= maxLength ? minLength : maxLength : 0);
            } else if (minLength >= 0 && minLength <= Constants.MAX_STRING_LENGTH) {
                data.setMinLength(minLength);
            }
            data.setLetterCase(annotation.letterCase());
            data.setAllowDiacritics(annotation.allowDiacritics().toBoolean(true));
            if (StringUtils.isNotBlank(regex)) {
                try {
                    data.setPattern(Pattern.compile(regex));
                } catch (PatternSyntaxException ex) {
                    if (log) {
                        logger.error(fieldName + " has an invalid regular expression");
                        TLC.getProject().getParser().increaseErrorCount();
                    }
                }
            }
            if (StringUtils.isNotBlank(converter)) {
                if (converter.matches(CONVERTER_REGEX)) {
                    data.setSpecialConverterName(converter);
                } else if (log) {
                    logger.error(fieldName + " has an invalid converter name");
                    TLC.getProject().getParser().increaseErrorCount();
                }
            }
            if (StringUtils.isNotBlank(validator)) {
                if (validator.matches(VALIDATOR_REGEX)) {
                    data.setSpecialValidatorName(validator);
                } else if (log) {
                    logger.error(fieldName + " has an invalid validator name");
                    TLC.getProject().getParser().increaseErrorCount();
                }
            }
        }
    }

    private void annotateDateField(Field field) {
        String fieldName = fullFieldName(field);
        Class<? extends Annotation> annotationClass = DateField.class;
        Class<?>[] validTypes = new Class<?>[]{DateParameter.class, DateProperty.class};
        boolean log = isParameter() || depth() == 0;
        _annotatedWithDateField = field.isAnnotationPresent(annotationClass)
            && XS1.checkFieldAnnotation(log, field, annotationClass, validTypes);
        /**/
        if (_annotatedWithDateField) {
            DateField annotation = field.getAnnotation(DateField.class);
            boolean disabledWeekends = annotation.disabledWeekends().toBoolean(false);
            boolean disabledWeekdays = annotation.disabledWeekdays().toBoolean(false);
            boolean disabledHolidays = annotation.disabledHolidays().toBoolean(false);
            int yearRange = annotation.yearRange();
            String converter = annotation.converter();
            String validator = annotation.validator();
            DateData data = (DateData) this;
            data.setDisabledWeekends(disabledWeekends);
            data.setDisabledWeekdays(disabledWeekdays);
            data.setDisabledHolidays(disabledHolidays);
            data.setYearRange(yearRange);
            if (StringUtils.isNotBlank(converter)) {
                if (converter.matches(CONVERTER_REGEX)) {
                    data.setSpecialConverterName(converter);
                } else if (log) {
                    logger.error(fieldName + " has an invalid converter name");
                    TLC.getProject().getParser().increaseErrorCount();
                }
            }
            if (StringUtils.isNotBlank(validator)) {
                if (validator.matches(VALIDATOR_REGEX)) {
                    data.setSpecialValidatorName(validator);
                } else if (log) {
                    logger.error(fieldName + " has an invalid validator name");
                    TLC.getProject().getParser().increaseErrorCount();
                }
            }
        }
    }

    private void annotateTimeField(Field field) {
        String fieldName = fullFieldName(field);
        Class<? extends Annotation> annotationClass = TimeField.class;
        Class<?>[] validTypes = new Class<?>[]{TimeParameter.class, TimeProperty.class};
        boolean log = isParameter() || depth() == 0;
        _annotatedWithTimeField = field.isAnnotationPresent(annotationClass)
            && XS1.checkFieldAnnotation(log, field, annotationClass, validTypes);
        /**/
        if (_annotatedWithTimeField) {
            TimeField annotation = field.getAnnotation(TimeField.class);
            int precision = annotation.precision();
            int minHour = annotation.minHour();
            int maxHour = annotation.maxHour();
            int stepHour = annotation.stepHour();
            int minMinute = annotation.minMinute();
            int maxMinute = annotation.maxMinute();
            int stepMinute = annotation.stepMinute();
            int minSecond = annotation.minSecond();
            int maxSecond = annotation.maxSecond();
            int stepSecond = annotation.stepSecond();
            String converter = annotation.converter();
            String validator = annotation.validator();
            TimeData data = (TimeData) this;
            if (precision >= 0 && precision <= Constants.MAX_TIME_PRECISION) {
                data.setPrecision(precision);
            }
            data.setMinHour(minHour);
            data.setMaxHour(maxHour);
            data.setStepHour(stepHour);
            data.setMinMinute(minMinute);
            data.setMaxMinute(maxMinute);
            data.setStepMinute(stepMinute);
            data.setMinSecond(minSecond);
            data.setMaxSecond(maxSecond);
            data.setStepSecond(stepSecond);
            if (StringUtils.isNotBlank(converter)) {
                if (converter.matches(CONVERTER_REGEX)) {
                    data.setSpecialConverterName(converter);
                } else if (log) {
                    logger.error(fieldName + " has an invalid converter name");
                    TLC.getProject().getParser().increaseErrorCount();
                }
            }
            if (StringUtils.isNotBlank(validator)) {
                if (validator.matches(VALIDATOR_REGEX)) {
                    data.setSpecialValidatorName(validator);
                } else if (log) {
                    logger.error(fieldName + " has an invalid validator name");
                    TLC.getProject().getParser().increaseErrorCount();
                }
            }
        }
    }

    private void annotateTimestampField(Field field) {
        String fieldName = fullFieldName(field);
        Class<? extends Annotation> annotationClass = TimestampField.class;
        Class<?>[] validTypes = new Class<?>[]{TimestampParameter.class, TimestampProperty.class};
        boolean log = isParameter() || depth() == 0;
        _annotatedWithTimestampField = field.isAnnotationPresent(annotationClass)
            && XS1.checkFieldAnnotation(log, field, annotationClass, validTypes);
        /**/
        if (_annotatedWithTimestampField) {
            TimestampField annotation = field.getAnnotation(TimestampField.class);
            int precision = annotation.precision();
            boolean disabledWeekends = annotation.disabledWeekends().toBoolean(false);
            boolean disabledWeekdays = annotation.disabledWeekdays().toBoolean(false);
            boolean disabledHolidays = annotation.disabledHolidays().toBoolean(false);
            int yearRange = annotation.yearRange();
            int minHour = annotation.minHour();
            int maxHour = annotation.maxHour();
            int stepHour = annotation.stepHour();
            int minMinute = annotation.minMinute();
            int maxMinute = annotation.maxMinute();
            int stepMinute = annotation.stepMinute();
            int minSecond = annotation.minSecond();
            int maxSecond = annotation.maxSecond();
            int stepSecond = annotation.stepSecond();
            String converter = annotation.converter();
            String validator = annotation.validator();
            TimestampData data = (TimestampData) this;
            if (precision >= 0 && precision <= Constants.MAX_TIME_PRECISION) {
                data.setPrecision(precision);
            }
            data.setDisabledWeekends(disabledWeekends);
            data.setDisabledWeekdays(disabledWeekdays);
            data.setDisabledHolidays(disabledHolidays);
            data.setYearRange(yearRange);
            data.setMinHour(minHour);
            data.setMaxHour(maxHour);
            data.setStepHour(stepHour);
            data.setMinMinute(minMinute);
            data.setMaxMinute(maxMinute);
            data.setStepMinute(stepMinute);
            data.setMinSecond(minSecond);
            data.setMaxSecond(maxSecond);
            data.setStepSecond(stepSecond);
            if (StringUtils.isNotBlank(converter)) {
                if (converter.matches(CONVERTER_REGEX)) {
                    data.setSpecialConverterName(converter);
                } else if (log) {
                    logger.error(fieldName + " has an invalid converter name");
                    TLC.getProject().getParser().increaseErrorCount();
                }
            }
            if (StringUtils.isNotBlank(validator)) {
                if (validator.matches(VALIDATOR_REGEX)) {
                    data.setSpecialValidatorName(validator);
                } else if (log) {
                    logger.error(fieldName + " has an invalid validator name");
                    TLC.getProject().getParser().increaseErrorCount();
                }
            }
        }
    }

    private void annotateFileReference(Field field) {
        String fieldName = fullFieldName(field);
        Class<? extends Annotation> annotationClass = FileReference.class;
        Class<?>[] validTypes = new Class<?>[]{StringParameter.class, StringProperty.class};
        boolean log = isParameter() || depth() == 0;
        _annotatedWithFileReference = field.isAnnotationPresent(annotationClass)
            && XS1.checkFieldAnnotation(log, field, annotationClass, validTypes);
        /**/
        if (_annotatedWithFileReference) {
            FileReference annotation = field.getAnnotation(FileReference.class);
            StringData data = (StringData) this;
            int max = annotation.max();
            MimeType[] types = annotation.types();
            String regex = annotation.regex();
            UploadStorageOption uploadStorageOption = annotation.storage();
            String blobFieldName = annotation.blobField();
            String joinFieldName = annotation.joinField();
            String loadFieldName = annotation.loadField();
//          Kleenean updateable = annotation.updateable();
            if (max >= 0) {
                data.setMaxInputFileSize(max);
            }
            if (types != null && types.length > 0) {
                data.setValidInputFileTypes(types);
            }
            if (StringUtils.isNotBlank(regex)) {
                try {
                    data.setValidInputFilePattern(Pattern.compile(regex));
                } catch (PatternSyntaxException ex) {
                    if (log) {
                        logger.error(fieldName + " has an invalid regular expression");
                        TLC.getProject().getParser().increaseErrorCount();
                    }
                }
            }
            data.setUploadStorageOption(uploadStorageOption);
            if (UploadStorageOption.FILE.equals(uploadStorageOption) && StringUtils.isNotBlank(blobFieldName)) {
                logger.error(fieldName + " has an invalid file reference: blob field cannot be specified if storage is FILE");
                TLC.getProject().getParser().increaseErrorCount();
            }
//          data.setUpdateableFileReference(updateable);
            setFieldReferenceBlobProperty(blobFieldName);
            setFieldReferenceJoinProperty(joinFieldName);
            setFieldReferenceLoadProperty(loadFieldName);
            setFieldReferenceLoadPropertyCurrentValue();
        }
    }

    private void setFieldReferenceBlobProperty(String blobFieldName) {
        StringData data = (StringData) this;
        if (StringUtils.isNotBlank(blobFieldName)) {
            data.setBlobFieldName(blobFieldName);
            Operation declaringOperation = getDeclaringOperation();
            Entity declaringEntity = declaringOperation == null ? getDeclaringEntity() : declaringOperation.getDeclaringEntity();
            Artifact declaringArtifact = declaringOperation == null ? declaringEntity : declaringOperation;
            String[] strings = new String[]{declaringArtifact.getName(), getName(), "blobField"};
            String role = StringUtils.join(strings, ".");
            Class<?>[] validTypes = new Class<?>[]{BinaryProperty.class};
            Field blobField = XS1.getField(true, role, blobFieldName, declaringEntity.getClass(), Entity.class, validTypes);
            if (blobField != null) {
                data.setBlobField(blobField);
                Property blobProperty = XS1.getProperty(blobField, declaringEntity);
                if (blobProperty != null) {
                    data.setBlobProperty(blobProperty);
                }
            }
        }
    }

    private void setFieldReferenceJoinProperty(String joinFieldName) {
        StringData data = (StringData) this;
        if (StringUtils.isNotBlank(joinFieldName)) {
            data.setJoinFieldName(joinFieldName);
            Operation declaringOperation = getDeclaringOperation();
            Entity declaringEntity = declaringOperation == null ? getDeclaringEntity() : declaringOperation.getDeclaringEntity();
            Artifact declaringArtifact = declaringOperation == null ? declaringEntity : declaringOperation;
            String[] strings = new String[]{declaringArtifact.getName(), getName(), "joinField"};
            String role = StringUtils.join(strings, ".");
            Class<?>[] validTypes = joinFieldValidTypes();
            Field joinField = XS1.getField(true, role, joinFieldName, declaringEntity.getClass(), Entity.class, validTypes);
            if (joinField != null) {
                data.setJoinField(joinField);
                Property joinProperty = XS1.getProperty(joinField, declaringEntity);
                if (joinProperty != null) {
                    data.setJoinProperty(joinProperty);
                }
            }
        }
    }

    private void setFieldReferenceLoadProperty(String loadFieldName) {
        StringData data = (StringData) this;
        if (StringUtils.isNotBlank(loadFieldName)) {
            data.setLoadFieldName(loadFieldName);
            Operation declaringOperation = getDeclaringOperation();
            Entity declaringEntity = declaringOperation == null ? getDeclaringEntity() : declaringOperation.getDeclaringEntity();
            Artifact declaringArtifact = declaringOperation == null ? declaringEntity : declaringOperation;
            String[] strings = new String[]{declaringArtifact.getName(), getName(), "loadField"};
            String role = StringUtils.join(strings, ".");
            Class<?>[] validTypes = new Class<?>[]{DateProperty.class, TimestampProperty.class};
            Field loadField = XS1.getField(true, role, loadFieldName, declaringEntity.getClass(), Entity.class, validTypes);
            if (loadField != null) {
                data.setLoadField(loadField);
                Property loadProperty = XS1.getProperty(loadField, declaringEntity);
                if (loadProperty != null) {
                    data.setLoadProperty(loadProperty);
                }
            }
        }
    }

    private void setFieldReferenceLoadPropertyCurrentValue() {
        StringData data = (StringData) this;
        Operation declaringOperation = getDeclaringOperation();
        if (declaringOperation != null) {
            Parameter instanceParameter = declaringOperation.getInstanceParameter();
            if (instanceParameter instanceof Entity) {
                Entity instanceEntity = (Entity) instanceParameter;
                Property loadProperty = data.getLoadProperty();
                if (loadProperty != null) {
                    String loadFieldName = loadProperty.getName();
                    if (StringUtils.isNotBlank(loadFieldName)) {
                        for (Property instanceProperty : instanceEntity.getPropertiesList()) {
                            if (loadFieldName.equals(instanceProperty.getName())) {
                                if (instanceProperty instanceof DateProperty) {
                                    ((DateProperty) instanceProperty).setCurrentValue(SpecialTemporalValue.CURRENT_DATE);
                                } else if (instanceProperty instanceof TimestampProperty) {
                                    ((TimestampProperty) instanceProperty).setCurrentValue(SpecialTemporalValue.CURRENT_TIMESTAMP);
                                }
                                break;
                            }
                        }
                    }
                }
            }
        }
    }

    private Class<?>[] joinFieldValidTypes() {
        Class<? extends Entity> clazz = TLC.getProject().getUploadedFileEntityClass();
        Class<? extends Entity> valid = clazz == null ? Entity.class : clazz;
        return new Class<?>[]{valid};
    }

    private void annotateGraphicImage(Field field) {
        Class<? extends Annotation> annotationClass = GraphicImage.class;
        Class<?>[] validTypes = new Class<?>[]{BinaryProperty.class};
        boolean log = depth() == 0;
        _annotatedWithGraphicImage = field.isAnnotationPresent(annotationClass)
            && XS1.checkFieldAnnotation(log, field, annotationClass, validTypes);
        /**/
        if (_annotatedWithGraphicImage) {
            GraphicImage annotation = field.getAnnotation(GraphicImage.class);
            int displayWidth = annotation.displayWidth();
            int displayHeight = annotation.displayHeight();
            boolean resizable = annotation.resizable().toBoolean(true);
            if (displayWidth <= 0 && displayHeight <= 0) {
            } else if (displayWidth <= 0) {
                displayWidth = displayHeight;
            } else if (displayHeight <= 0) {
                displayHeight = displayWidth;
            }
            BinaryData data = (BinaryData) this;
            data.setDisplayWidth(displayWidth);
            data.setDisplayHeight(displayHeight);
            data.setResizable(resizable);
        }
    }

    private void annotateUniformResourceLocator(Field field) {
        Class<? extends Annotation> annotationClass = UniformResourceLocator.class;
        Class<?>[] validTypes = new Class<?>[]{StringProperty.class};
        boolean log = depth() == 0;
        _annotatedWithUniformResourceLocator = field.isAnnotationPresent(annotationClass)
            && XS1.checkFieldAnnotation(log, field, annotationClass, validTypes);
        /**/
        if (_annotatedWithUniformResourceLocator) {
            UniformResourceLocator annotation = field.getAnnotation(UniformResourceLocator.class);
            UrlType urlType = annotation.urlType();
            UrlDisplayType urlDisplayType = annotation.urlDisplayType();
            StringData data = (StringData) this;
            data.setUrlType(urlType);
            data.setUrlDisplayType(urlDisplayType);
        }
    }

    private void annotateVariantString(Field field) {
        VariantString annotation = field.getAnnotation(VariantString.class);
        if (annotation != null && annotation.value()) {
            Class<? extends Annotation> annotationClass = VariantString.class;
            Class<?>[] validTypes = new Class<?>[]{StringProperty.class};
            boolean log = depth() == 0;
            _annotatedWithVariantString = XS1.checkFieldAnnotation(log, field, annotationClass, validTypes);
        }
    }

    private void annotateInstanceReference(Field field) {
        InstanceReference annotation = field.getAnnotation(InstanceReference.class);
        if (annotation != null && annotation.value()) {
            Class<? extends Annotation> annotationClass = InstanceReference.class;
            Operation declaringOperation = getDeclaringOperation();
            Class<?> declaringClass = declaringOperation.getDeclaringField().getDeclaringClass();
            Class<?>[] validTypes = new Class<?>[]{declaringClass};
            boolean log = true;
            boolean aye = XS1.checkFieldAnnotation(log, field, annotationClass, validTypes);
            if (aye) {
                Field previous = getDeclaringArtifact().put(annotationClass, field);
                if (previous == null) {
                    _annotatedWithInstanceReference = true;
                    declaringOperation.setOperationKind(OperationKind.INSTANCE);
                } else if (log) {
                    XS1.logDuplicateAnnotation(field, annotationClass, previous);
                }
            }
        }
        /**/
        if (_annotatedWithInstanceReference) {
            _required = true;
        }
    }

    private void annotateParameterField(Field field) {
        _annotatedWithParameterField = field.isAnnotationPresent(ParameterField.class);
        if (_annotatedWithParameterField) {
            ParameterField annotation = field.getAnnotation(ParameterField.class);
            _auditable = annotation.auditable().toBoolean(_auditable);
            _password = annotation.password().toBoolean(_password);
            _required = _annotatedWithInstanceReference ? Boolean.TRUE : annotation.required().toBoolean(_required);
            _hiddenField = !_annotatedWithInstanceReference && annotation.hidden().toBoolean(_hiddenField);
//          _submitField = annotation.submit().toBoolean(_submitField);
            _linkedFieldName = annotation.linkedField();
            _linkedColumnName = annotation.linkedColumn();
            _linkedColumnOperator = annotation.operator();
//          _sequenceNumber = Math.max(0, annotation.sequence());
            _sequenceNumber = annotation.sequence();
            if (StringUtils.isNotBlank(_linkedFieldName)) {
                Operation declaringOperation = getDeclaringOperation();
                Entity declaringEntity = declaringOperation.getDeclaringEntity();
                Class<?>[] validTypes = _dataClass == null ? new Class<?>[]{} : new Class<?>[]{_dataClass};
                String[] strings = new String[]{declaringOperation.getName(), getName(), "linkedField"};
                String role = StringUtils.join(strings, ".");
                _linkedField = XS1.getField(true, role, _linkedFieldName, declaringEntity.getClass(), Entity.class, validTypes);
                if (_linkedField != null) {
                    _linkedProperty = XS1.getProperty(_linkedField, declaringEntity);
                }
            }
        }
    }

    private void annotatePropertyField(Field field) {
        _annotatedWithPropertyField = field.isAnnotationPresent(PropertyField.class);
        if (_annotatedWithPropertyField) {
            PropertyField annotation = field.getAnnotation(PropertyField.class);
            _propertyAccess = annotation.access();
            _auditable = annotation.auditable().toBoolean(_auditable);
            _password = annotation.password().toBoolean(_password);
            _required = annotation.required().toBoolean(_required);
            _hiddenField = annotation.hidden().toBoolean(_hiddenField);
            _createField = annotation.create().toBoolean(_createField);
            _updateField = annotation.update().toBoolean(_updateField);
            _searchField = annotation.search().toBoolean(_searchField);
            _filterField = annotation.filter().toBoolean(_filterField);
            _sortField = annotation.sort().toBoolean(_sortField);
            _tableField = annotation.table().toBoolean(_tableField);
            _detailField = annotation.detail().toBoolean(_detailField);
            _columnField = annotation.column().toBoolean(_columnField);
            _reportField = annotation.report().toBoolean(_reportField);
            _exportField = annotation.export().toBoolean(_exportField);
//          _submitField = annotation.submit().toBoolean(_submitField);
            _headertextlessField = annotation.headertextless().toBoolean(_headertextlessField);
            _headingField = annotation.heading().toBoolean(_headingField);
            _overlayField = annotation.overlay().toBoolean(_overlayField);
            _immutableField = annotation.immutable().toBoolean(_immutableField);
            _defaultCondition = annotation.defaultCondition();
            _defaultCheckpoint = annotation.defaultCheckpoint();
//          _sequenceNumber = Math.max(0, annotation.sequence());
            _sequenceNumber = annotation.sequence();
            boolean log = depth() == 0;
            if (log && restrictedReadingAccess()) {
                if (BitUtils.valueOf(_headingField)) {
                    String message = getFullName() + " cannot be a heading property due to its restricted reading access";
                    logger.warn(message);
                    TLC.getProject().getParser().increaseWarningCount();
                }
                if (BitUtils.valueOf(_overlayField)) {
                    String message = getFullName() + " cannot be an overlay property due to its restricted reading access";
                    logger.warn(message);
                    TLC.getProject().getParser().increaseWarningCount();
                }
            }
        }
    }

    private void annotatePropertyAggregation(Field field) {
        _annotatedWithPropertyAggregation = field.isAnnotationPresent(PropertyAggregation.class);
        if (_annotatedWithPropertyAggregation) {
            PropertyAggregation annotation = field.getAnnotation(PropertyAggregation.class);
            _aggregateFunction = validAggregateFunction(annotation.function());
            _aggregateTitle = StrUtils.coalesce(annotation.title(), _aggregateFunction.getNameTag());
        }
    }

    private AggregateFunction validAggregateFunction(AggregateFunction aggregation) {
        if (isEntity() || isBooleanPrimitive() || isCharacterPrimitive()) {
            logNotCountAggregation(aggregation);
            return AggregateFunction.COUNT;
        } else if (isNumericPrimitive()) {
            return aggregation;
        } else if (isTemporalPrimitive()) {
            switch (aggregation) {
                case COUNT:
                case MINIMUM:
                case MAXIMUM:
                    return aggregation;
                default:
                    logNotCountAggregation(aggregation);
                    return AggregateFunction.COUNT;
            }
        } else {
            logNotValidAggregation();
            return aggregation;
        }
    }

    private void logNotCountAggregation(AggregateFunction aggregation) {
        if (!AggregateFunction.COUNT.equals(aggregation)) {
            String message = aggregation + " cannot be applied to " + getFullName() + "; using COUNT instead";
            logger.warn(message);
            TLC.getProject().getParser().increaseWarningCount();
        }
    }

    private void logNotValidAggregation() {
        String message = "no aggregation cannot be applied to " + getFullName();
        logger.error(message);
        TLC.getProject().getParser().increaseErrorCount();
    }

    private void annotateDataGen(Field field) {
        annotateBooleanDataGen(field);
        annotateCharacterDataGen(field);
        annotateNumericDataGen(field);
        annotateTemporalDataGen(field);
    }

    private void annotateBooleanDataGen(Field field) {
        Class<? extends Annotation> annotationClass = BooleanDataGen.class;
        Class<?>[] validTypes = new Class<?>[]{BooleanProperty.class};
        boolean log = depth() == 0;
        boolean annotated = field.isAnnotationPresent(annotationClass)
            && XS1.checkFieldAnnotation(log, field, annotationClass, validTypes);
        /**/
        if (annotated) {
            BooleanDataGen annotation = field.getAnnotation(BooleanDataGen.class);
            _dataGenType = annotation.type();
            _dataGenFunction = StringUtils.trimToNull(annotation.function());
            _dataGenNullable = Math.min(100, Math.max(0, annotation.nullable()));
            _dataGenTrueable = Math.min(100, Math.max(0, annotation.trueable()));
            if (_dataGenNullable + _dataGenTrueable > 100) {
                _dataGenNullable = 100 - _dataGenTrueable;
            }
            _annotatedWithDataGen = true;
        }
    }

    private void annotateCharacterDataGen(Field field) {
        Class<? extends Annotation> annotationClass = CharacterDataGen.class;
        Class<?>[] validTypes = new Class<?>[]{CharacterProperty.class, StringProperty.class};
        boolean log = depth() == 0;
        boolean annotated = field.isAnnotationPresent(annotationClass)
            && XS1.checkFieldAnnotation(log, field, annotationClass, validTypes);
        /**/
        if (annotated) {
            CharacterDataGen annotation = field.getAnnotation(CharacterDataGen.class);
            _dataGenType = annotation.type();
//          _dataGenSeriesStart = annotation.start();
//          _dataGenSeriesStop = annotation.stop();
//          _dataGenSeriesStep = annotation.step();
            _dataGenSeriesStart = Math.min(Constants.MAX_SERIES_START, Math.max(1, annotation.start()));
            _dataGenSeriesStop = Math.min(Constants.MAX_SERIES_STOP, Math.max(1, annotation.stop()));
            _dataGenSeriesStep = Math.min(Constants.MAX_SERIES_STEP, Math.max(1, annotation.step()));
            if (_dataGenSeriesStop < _dataGenSeriesStart + _dataGenSeriesStep) {
                _dataGenSeriesStop = Math.min(Constants.MAX_SERIES_STOP, _dataGenSeriesStart + Math.max(10000, _dataGenSeriesStep));
            }
            _dataGenFunction = StringUtils.trimToNull(annotation.function());
            _dataGenNullable = Math.min(100, Math.max(0, annotation.nullable()));
            _dataGenPattern = StringUtils.trimToNull(annotation.pattern());
            _dataGenPrefix = StrUtils.ltrimToNull(annotation.prefix());
            _dataGenSuffix = StrUtils.rtrimToNull(annotation.suffix());
            _annotatedWithDataGen = true;
        }
    }

    private void annotateNumericDataGen(Field field) {
        Class<? extends Annotation> annotationClass = NumericDataGen.class;
        Class<?>[] validTypes = new Class<?>[]{BigDecimalProperty.class, BigIntegerProperty.class,
            ByteProperty.class, ShortProperty.class, IntegerProperty.class, LongProperty.class,
            FloatProperty.class, DoubleProperty.class};
        boolean log = depth() == 0;
        boolean annotated = field.isAnnotationPresent(annotationClass)
            && XS1.checkFieldAnnotation(log, field, annotationClass, validTypes);
        /**/
        if (annotated) {
            NumericDataGen annotation = field.getAnnotation(NumericDataGen.class);
            _dataGenType = annotation.type();
//          _dataGenSeriesStart = annotation.start();
//          _dataGenSeriesStop = annotation.stop();
//          _dataGenSeriesStep = annotation.step();
            _dataGenSeriesStart = Math.min(Constants.MAX_SERIES_START, Math.max(1, annotation.start()));
            _dataGenSeriesStop = Math.min(Constants.MAX_SERIES_STOP, Math.max(1, annotation.stop()));
            _dataGenSeriesStep = Math.min(Constants.MAX_SERIES_STEP, Math.max(1, annotation.step()));
            if (_dataGenSeriesStop < _dataGenSeriesStart + _dataGenSeriesStep) {
                _dataGenSeriesStop = Math.min(Constants.MAX_SERIES_STOP, _dataGenSeriesStart + Math.max(10000, _dataGenSeriesStep));
            }
            _dataGenFunction = StringUtils.trimToNull(annotation.function());
            _dataGenNullable = Math.min(100, Math.max(0, annotation.nullable()));
            _dataGenMin = someIntegerValue(field, annotation.min());
            _dataGenMax = someIntegerValue(field, annotation.max());
            _dataGenNumericAction = annotation.action();
            _dataGenFactor = someBigDecimalValue(field, annotation.factor());
            _annotatedWithDataGen = true;
        }
    }

    private void annotateTemporalDataGen(Field field) {
        Class<? extends Annotation> annotationClass = TemporalDataGen.class;
        Class<?>[] validTypes = new Class<?>[]{DateProperty.class, TimeProperty.class, TimestampProperty.class};
        boolean log = depth() == 0;
        boolean annotated = field.isAnnotationPresent(annotationClass)
            && XS1.checkFieldAnnotation(log, field, annotationClass, validTypes);
        /**/
        if (annotated) {
            TemporalDataGen annotation = field.getAnnotation(TemporalDataGen.class);
            _dataGenType = annotation.type();
//          _dataGenSeriesStart = annotation.start();
//          _dataGenSeriesStop = annotation.stop();
//          _dataGenSeriesStep = annotation.step();
            _dataGenSeriesStart = Math.min(Constants.MAX_SERIES_START, Math.max(1, annotation.start()));
            _dataGenSeriesStop = Math.min(Constants.MAX_SERIES_STOP, Math.max(1, annotation.stop()));
            _dataGenSeriesStep = Math.min(Constants.MAX_SERIES_STEP, Math.max(1, annotation.step()));
            if (_dataGenSeriesStop < _dataGenSeriesStart + _dataGenSeriesStep) {
                _dataGenSeriesStop = Math.min(Constants.MAX_SERIES_STOP, _dataGenSeriesStart + Math.max(10000, _dataGenSeriesStep));
            }
            _dataGenFunction = StringUtils.trimToNull(annotation.function());
            _dataGenNullable = Math.min(100, Math.max(0, annotation.nullable()));
            _dataGenMin = someTemporalValue(field, annotation.min());
            _dataGenMax = someTemporalValue(field, annotation.max());
            _dataGenTemporalInterval = annotation.interval();
            _annotatedWithDataGen = true;
        }
    }
    // </editor-fold>

    // <editor-fold defaultstate="collapsed" desc="some methods">
    private Integer someIntegerValue(Field field, String string) {
        if (StringUtils.isBlank(string)) {
            return null;
        }
        try {
            return Integer.valueOf(string);
        } catch (NumberFormatException e) {
            logInvalidDataExpression(field, "integer", e);
        }
        return null;
    }

    private BigDecimal someBigDecimalValue(Field field, String string) {
        if (StringUtils.isBlank(string)) {
            return null;
        }
        try {
            return new BigDecimal(string);
        } catch (Exception e) {
            logInvalidDataExpression(field, "decimal", e);
        }
        return null;
    }

    private java.util.Date someTemporalValue(Field field, String string) {
        if (StringUtils.isBlank(string)) {
            return null;
        }
        if (this instanceof DateData) {
            return someDateValue(field, string);
        }
        if (this instanceof TimeData) {
            return someTimeValue(field, string);
        }
        if (this instanceof TimestampData) {
            return someTimestampValue(field, string);
        }
        return null;
    }

    private final Date currentDate = TimeUtils.currentDate();

    private Date someDateValue(Field field, String string) {
        if (StringUtils.isBlank(string)) {
            return null;
        }
        Date someRelativeDate = someRelativeDateValue(string);
        if (someRelativeDate != null) {
            return someRelativeDate;
        }
        try {
            return Date.valueOf(string);
        } catch (Exception e) {
            logInvalidDataExpression(field, "date", e);
        }
        return null;
    }

    private Date someRelativeDateValue(String string) {
        char[] validUnits = {'Y', 'M', 'D'};
        TemporalAddend addend = temporalAddendValueOf(string, validUnits, 'D');
        if (addend == null) {
            return null;
        }
        return TimeUtils.addDate(currentDate, addend.number, addend.unit);
    }

    private final Time currentTime = TimeUtils.currentTime();

    private Time someTimeValue(Field field, String string) {
        if (StringUtils.isBlank(string)) {
            return null;
        }
        Time someRelativeTime = someRelativeTimeValue(string);
        if (someRelativeTime != null) {
            return someRelativeTime;
        }
        try {
            return Time.valueOf(string);
        } catch (Exception e) {
            logInvalidDataExpression(field, "time", e);
        }
        return null;
    }

    private Time someRelativeTimeValue(String string) {
        char[] validUnits = {'h', 'm', 's'};
        TemporalAddend addend = temporalAddendValueOf(string, validUnits, 'h');
        if (addend == null) {
            return null;
        }
        return TimeUtils.addTime(currentTime, addend.number, addend.unit);
    }

    private final Timestamp currentTimestamp = TimeUtils.currentTimestamp();

    private Timestamp someTimestampValue(Field field, String string) {
        if (StringUtils.isBlank(string)) {
            return null;
        }
        Timestamp someRelativeTimestamp = someRelativeTimestampValue(string);
        if (someRelativeTimestamp != null) {
            return someRelativeTimestamp;
        }
        try {
            return Timestamp.valueOf(string);
        } catch (Exception e) {
            logInvalidDataExpression(field, "timestamp", e);
        }
        return null;
    }

    private Timestamp someRelativeTimestampValue(String string) {
        char[] validUnits = {'Y', 'M', 'D', 'h', 'm', 's'};
        TemporalAddend addend = temporalAddendValueOf(string, validUnits, 'D');
        if (addend == null) {
            return null;
        }
        return TimeUtils.addTimestamp(currentTimestamp, addend.number, addend.unit);
    }

    private void logInvalidDataExpression(Field field, String string, Exception e) {
        if (depth() == 0) {
            String fieldName = fullFieldName(field);
            logger.error(fieldName + " has an invalid " + string + " data expression");
            TLC.getProject().getParser().increaseErrorCount();
        }
    }

    private String fullFieldName(Field field) {
        return getDeclaringArtifactClassSimpleName() + "." + field.getName();
    }

    private TemporalAddend temporalAddendValueOf(String string, char[] validUnits, char defaultUnit) {
        String chopped;
        String trimmed = StringUtils.trimToNull(string);
        if (trimmed == null) {
            return null;
        }
        int end = trimmed.length() - 1;
        char unit = trimmed.charAt(end);
        if (Character.isDigit(unit)) {
            chopped = trimmed;
            unit = defaultUnit;
        } else if (end > 0 && ArrayUtils.contains(validUnits, unit)) {
            chopped = trimmed.substring(0, end);
        } else {
            return null;
        }
        try {
            Integer number = Integer.valueOf(chopped);
            return new TemporalAddend(number, unit);
        } catch (NumberFormatException e) {
            return null;
        }
    }

    private class TemporalAddend {

        int number;

        char unit;

        TemporalAddend(int i, char c) {
            number = i;
            unit = c;
        }

    }
    // </editor-fold>

    // <editor-fold defaultstate="collapsed" desc="instanceof">
    /**
     * @return true if is a Primitive; otherwise false
     */
    @Override
    public boolean isPrimitive() {
        return this instanceof Primitive;
    }

    /**
     * @return true if is a BinaryPrimitive; otherwise false
     */
    @Override
    public boolean isBinaryPrimitive() {
        return this instanceof BinaryPrimitive;
    }

    /**
     * @return true if is a BooleanPrimitive; otherwise false
     */
    @Override
    public boolean isBooleanPrimitive() {
        return this instanceof BooleanPrimitive;
    }

    /**
     * @return true if is a CharacterPrimitive; otherwise false
     */
    @Override
    public boolean isCharacterPrimitive() {
        return this instanceof CharacterPrimitive;
    }

    /**
     * @return true if is a NumericPrimitive; otherwise false
     */
    @Override
    public boolean isNumericPrimitive() {
        return this instanceof NumericPrimitive;
    }

    /**
     * @return true if is a TemporalPrimitive; otherwise false
     */
    @Override
    public boolean isTemporalPrimitive() {
        return this instanceof TemporalPrimitive;
    }

    /**
     * @return true if is a BigDecimalData; otherwise false
     */
    @Override
    public boolean isBigDecimalData() {
        return this instanceof BigDecimalData;
    }

    /**
     * @return true if is a BigIntegerData; otherwise false
     */
    @Override
    public boolean isBigIntegerData() {
        return this instanceof BigIntegerData;
    }

    /**
     * @return true if is a BinaryData; otherwise false
     */
    @Override
    public boolean isBinaryData() {
        return this instanceof BinaryData;
    }

    private boolean isUndisplayableBinaryData() {
        return isBinaryData() && !isGraphicImageField();
    }

    /**
     * @return true if is a BooleanData; otherwise false
     */
    @Override
    public boolean isBooleanData() {
        return this instanceof BooleanData;
    }

    /**
     * @return true if is a ByteData; otherwise false
     */
    @Override
    public boolean isByteData() {
        return this instanceof ByteData;
    }

    /**
     * @return true if is a CharacterData; otherwise false
     */
    @Override
    public boolean isCharacterData() {
        return this instanceof CharacterData;
    }

    /**
     * @return true if is a DateData; otherwise false
     */
    @Override
    public boolean isDateData() {
        return this instanceof DateData;
    }

    /**
     * @return true if is a DoubleData; otherwise false
     */
    @Override
    public boolean isDoubleData() {
        return this instanceof DoubleData;
    }

    /**
     * @return true if is a FloatData; otherwise false
     */
    @Override
    public boolean isFloatData() {
        return this instanceof FloatData;
    }

    /**
     * @return true if is a IntegerData; otherwise false
     */
    @Override
    public boolean isIntegerData() {
        return this instanceof IntegerData;
    }

    /**
     * @return true if is a LongData; otherwise false
     */
    @Override
    public boolean isLongData() {
        return this instanceof LongData;
    }

    /**
     * @return true if is a ShortData; otherwise false
     */
    @Override
    public boolean isShortData() {
        return this instanceof ShortData;
    }

    /**
     * @return true if is a StringData; otherwise false
     */
    @Override
    public boolean isStringData() {
        return this instanceof StringData;
    }

    /**
     * @return true if is a TimeData; otherwise false
     */
    @Override
    public boolean isTimeData() {
        return this instanceof TimeData;
    }

    /**
     * @return true if is a TimestampData; otherwise false
     */
    @Override
    public boolean isTimestampData() {
        return this instanceof TimestampData;
    }

    /**
     * @return true if is an entity; otherwise false
     */
    @Override
    public boolean isEntity() {
        return this instanceof Entity;
    }

    /**
     * @return true if is a contextual entity; otherwise false
     */
    @Override
    public boolean isContextualEntity() {
        return this instanceof ContextualEntity;
    }

    /**
     * @return true if is a enumeration entity; otherwise false
     */
    @Override
    public boolean isEnumerationEntity() {
        return this instanceof EnumerationEntity;
    }

    /**
     * @return true if is a database entity; otherwise false
     */
    @Override
    public boolean isDatabaseEntity() {
        return this instanceof DatabaseEntity;
    }

    /**
     * @return true if is a persistent entity; otherwise false
     */
    @Override
    public boolean isPersistentEntity() {
        return this instanceof PersistentEntity;
    }

    /**
     * @return true if is a persistent enumeration entity; otherwise false
     */
    @Override
    public boolean isPersistentEnumerationEntity() {
        return this instanceof PersistentEnumerationEntity;
    }

    /**
     * @return true if is a overlayable entity reference; otherwise false
     */
    @Override
    public boolean isOverlayableEntityReference() {
        return false;
    }

    /**
     * @return true if is a Parameter; otherwise false
     */
    @Override
    public boolean isParameter() {
        return this instanceof Parameter
            && getDeclaringField() != null
            && getDeclaringArtifact() instanceof Operation;
    }

    public boolean isInstanceParameter() {
        if (isParameter()) {
            Operation operation = (Operation) getDeclaringArtifact();
            Parameter parameter = (Parameter) this;
            return OperationKind.INSTANCE.equals(operation.getOperationKind()) && parameter.isInstanceReferenceField();
        }
        return false;
    }

    /**
     * @return true if is a Property; otherwise false
     */
    @Override
    public boolean isProperty() {
        return this instanceof Property
            && getDeclaringField() != null
            && getDeclaringArtifact() instanceof Entity;
    }
//
//  public boolean isParameterProperty() {
//      return isProperty() && isClassInPath(Operation.class);
//  }

    @Override
    public boolean isPrimaryKeyProperty() {
        return isProperty()
            && (_annotatedWithPrimaryKey
            || getDeclaringField().equals(getDeclaringEntity().getPrimaryKeyField()));
    }

    @Override
    public boolean isSequenceProperty() {
        return isProperty()
            && (_annotatedWithSequenceProperty
            || getDeclaringField().equals(getDeclaringEntity().getSequenceField()));
    }

    @Override
    public boolean isVersionProperty() {
        return isProperty()
            && (_annotatedWithVersionProperty
            || getDeclaringField().equals(getDeclaringEntity().getVersionField()));
    }

    @Override
    public boolean isNumericKeyProperty() {
        return isProperty()
            && (_annotatedWithNumericKey
            || getDeclaringField().equals(getDeclaringEntity().getNumericKeyField()));
    }

    @Override
    public boolean isCharacterKeyProperty() {
        return isProperty()
            && (_annotatedWithCharacterKey
            || getDeclaringField().equals(getDeclaringEntity().getCharacterKeyField()));
    }

    @Override
    public boolean isNameProperty() {
        return isProperty()
            && (_annotatedWithNameProperty
            || getDeclaringField().equals(getDeclaringEntity().getNameField()));
    }

    @Override
    public boolean isDescriptionProperty() {
        return isProperty()
            && (_annotatedWithDescriptionProperty
            || getDeclaringField().equals(getDeclaringEntity().getDescriptionField()));
    }

    @Override
    public boolean isImageProperty() {
        return isProperty()
            && (_annotatedWithImageProperty
            || getDeclaringField().equals(getDeclaringEntity().getImageField()));
    }

    @Override
    public boolean isInactiveIndicatorProperty() {
        return isProperty()
            && (_annotatedWithInactiveIndicator
            || getDeclaringField().equals(getDeclaringEntity().getInactiveIndicatorField()));
    }

    @Override
    public boolean isUrlProperty() {
        return isProperty()
            && (_annotatedWithUrlProperty
            || getDeclaringField().equals(getDeclaringEntity().getUrlField()));
    }

    @Override
    public boolean isParentProperty() {
        return isProperty()
            && (_annotatedWithParentProperty
            || getDeclaringField().equals(getDeclaringEntity().getParentField()));
    }

    @Override
    public boolean isOwnerProperty() {
        return isProperty()
            && (_annotatedWithOwnerProperty
            || getDeclaringField().equals(getDeclaringEntity().getOwnerField()));
    }

    @Override
    public boolean isSegmentProperty() {
        return isProperty()
            && (_annotatedWithSegmentProperty
            || getDeclaringField().equals(getDeclaringEntity().getSegmentField()));
    }

    @Override
    public boolean isUniqueKeyProperty() {
        return isProperty() && isUnique();
    }

    @Override
    public boolean isBusinessKeyProperty() {
        return isProperty()
            && (_annotatedWithBusinessKey
            || getDeclaringField().equals(getDeclaringEntity().getBusinessKeyField()));
    }

    @Override
    public boolean isDiscriminatorProperty() {
        return _annotatedWithDiscriminatorColumn;
    }

    @Override
    public boolean isStateProperty() {
        return _annotatedWithStateProperty;
    }
    // </editor-fold>

    /**
     * @return the parameter path list
     */
    @Override
    @SuppressWarnings("unchecked") // unchecked conversion
    public List<Artifact> getParameterPathList() {
        List list = new ArrayList<>();
        addToParameterPathList(list, this);
        return list;
    }

    private void addToParameterPathList(List<Artifact> list, Artifact artifact) {
        Artifact declaringArtifact = artifact.getDeclaringArtifact();
        if (declaringArtifact == null || declaringArtifact instanceof Operation) {
        } else {
            addToParameterPathList(list, declaringArtifact);
        }
        list.add(artifact);
    }

    /**
     * @return the property path list
     */
    @Override
    @SuppressWarnings("unchecked") // unchecked conversion
    public List<Artifact> getPropertyPathList() {
        List list = new ArrayList<>();
        addToPropertyPathList(list, this);
        return list;
    }

    private void addToPropertyPathList(List<Artifact> list, Artifact artifact) {
        Artifact declaringArtifact = artifact.getDeclaringArtifact();
        if (declaringArtifact == null) {
            if (artifact instanceof Entity) {
            } else {
                list.add(artifact);
            }
        } else {
            addToPropertyPathList(list, declaringArtifact);
            list.add(artifact);
        }
    }

    // <editor-fold defaultstate="collapsed" desc="initial value referencing siblings">
    /**
     * @return the list of artifacts referencing this artifact (if it is an instance parameter) in their initial value
     */
    public Map<String, ? extends DataArtifact> getInstanceParameterInitialValueReferencingSiblings() {
        return getInstanceParameterInitialValueReferencingSiblings(false);
    }

    /**
     * @param recursively
     * @return the list of artifacts referencing this artifact (if it is an instance parameter) in their initial value
     */
    public Map<String, ? extends DataArtifact> getInstanceParameterInitialValueReferencingSiblings(boolean recursively) {
        return isInstanceParameter() ? getInitialValueReferencingParameters(recursively) : new LinkedHashMap<>();
    }

    /**
     * @return the list of parameters referencing this parameter in their initial value
     */
    @Override
    public Map<String, Parameter> getInitialValueReferencingParameters() {
        return getInitialValueReferencingParameters(false);
    }

    /**
     * @param recursively
     * @return the list of parameters referencing this parameter in their initial value
     */
    @Override
    public Map<String, Parameter> getInitialValueReferencingParameters(boolean recursively) {
        Map<String, Parameter> initialValueReferencingParameters = new LinkedHashMap<>();
        String fullName = getFullName();
        Operation declaringOperation = getDeclaringOperation();
        List<Parameter> parametersList = declaringOperation.getParametersList();
        Expression initialValue;
        for (Parameter parameter : parametersList) {
            if (!parameter.getFullName().equals(fullName)) {
                initialValue = initialValueExpression(parameter.getInitialValue());
                if (referencedByExpression(initialValue)) {
                    logger.trace("parameter " + fullName + " is referenced by " + parameter.getName());
                    initialValueReferencingParameters.put(parameter.getPathString(), parameter);
                    if (recursively) {
                        initialValueReferencingParameters.putAll(parameter.getInitialValueReferencingParameters(recursively));
                    }
                }
            }
        }
        return initialValueReferencingParameters;
    }

    private Expression initialValueExpression(Object initialValueObject) {
        return initialValueObject instanceof Expression ? (Expression) initialValueObject
            : (initialValueObject instanceof Entity) ? ((Entity) initialValueObject).isNull() : null;
    }
    // </editor-fold>

    // <editor-fold defaultstate="collapsed" desc="max value referencing siblings">
    /**
     * @return the list of artifacts referencing this artifact in their max value
     */
    public Map<String, ? extends DataArtifact> getMaxValueReferencingSiblings() {
        return getMaxValueReferencingSiblings(false);
    }

    /**
     * @param recursively
     * @return the list of artifacts referencing this artifact in their max value
     */
    public Map<String, ? extends DataArtifact> getMaxValueReferencingSiblings(boolean recursively) {
        return isProperty() ? getMaxValueReferencingProperties(recursively)
            : isParameter() ? getMaxValueReferencingParameters(recursively) : new LinkedHashMap<>();
    }

    /**
     * @return the list of parameters referencing this parameter in their max value
     */
    @Override
    public Map<String, Parameter> getMaxValueReferencingParameters() {
        return getMaxValueReferencingParameters(false);
    }

    /**
     * @param recursively
     * @return the list of parameters referencing this parameter in their max value
     */
    @Override
    public Map<String, Parameter> getMaxValueReferencingParameters(boolean recursively) {
        Map<String, Parameter> maxValueReferencingParameters = new LinkedHashMap<>();
        String fullName = getFullName();
        Operation declaringOperation = getDeclaringOperation();
        List<Parameter> parametersList = declaringOperation.getParametersList();
        Expression maxValue;
        for (Parameter parameter : parametersList) {
            if (!parameter.getFullName().equals(fullName)) {
                maxValue = maxValueExpression(parameter);
                if (referencedByExpression(maxValue)) {
                    logger.trace("parameter " + fullName + " is referenced by " + parameter.getName());
                    maxValueReferencingParameters.put(parameter.getPathString(), parameter);
                    if (recursively) {
                        maxValueReferencingParameters.putAll(parameter.getMaxValueReferencingParameters(recursively));
                    }
                }
            }
        }
        return maxValueReferencingParameters;
    }

    /**
     * @return the list of properties referencing this property in their max value
     */
    @Override
    public Map<String, Property> getMaxValueReferencingProperties() {
        return getMaxValueReferencingProperties(false);
    }

    /**
     * @param recursively
     * @return the list of properties referencing this property in their max value
     */
    @Override
    public Map<String, Property> getMaxValueReferencingProperties(boolean recursively) {
        Map<String, Property> maxValueReferencingProperties = new LinkedHashMap<>();
        String fullName = getFullName();
        Entity declaringEntity = getDeclaringEntity();
        List<Property> propertiesList = declaringEntity.getPropertiesList();
        Expression maxValue;
        for (Property property : propertiesList) {
            if (!property.getFullName().equals(fullName)) {
                maxValue = maxValueExpression(property);
                if (referencedByExpression(maxValue)) {
                    logger.trace("property " + fullName + " is referenced by " + property.getName());
                    maxValueReferencingProperties.put(property.getPathString(), property);
                    if (recursively) {
                        maxValueReferencingProperties.putAll(property.getMaxValueReferencingProperties(recursively));
                    }
                }
            }
        }
        return maxValueReferencingProperties;
    }

    private Expression maxValueExpression(DataArtifact da) {
        if (da instanceof IntervalizedArtifact) {
            IntervalizedArtifact ia = (IntervalizedArtifact) da;
            Object maxValueObject = ia.getMaxValue();
            return maxValueObject instanceof Expression ? (Expression) maxValueObject
                : (maxValueObject instanceof Entity) ? ((Entity) maxValueObject).isNull() : null;
        }
        return null;
    }
    // </editor-fold>

    // <editor-fold defaultstate="collapsed" desc="min value referencing siblings">
    /**
     * @return the list of artifacts referencing this artifact in their min value
     */
    public Map<String, ? extends DataArtifact> getMinValueReferencingSiblings() {
        return getMinValueReferencingSiblings(false);
    }

    /**
     * @param recursively
     * @return the list of artifacts referencing this artifact in their min value
     */
    public Map<String, ? extends DataArtifact> getMinValueReferencingSiblings(boolean recursively) {
        return isProperty() ? getMinValueReferencingProperties(recursively)
            : isParameter() ? getMinValueReferencingParameters(recursively) : new LinkedHashMap<>();
    }

    /**
     * @return the list of parameters referencing this parameter in their min value
     */
    @Override
    public Map<String, Parameter> getMinValueReferencingParameters() {
        return getMinValueReferencingParameters(false);
    }

    /**
     * @param recursively
     * @return the list of parameters referencing this parameter in their min value
     */
    @Override
    public Map<String, Parameter> getMinValueReferencingParameters(boolean recursively) {
        Map<String, Parameter> minValueReferencingParameters = new LinkedHashMap<>();
        String fullName = getFullName();
        Operation declaringOperation = getDeclaringOperation();
        List<Parameter> parametersList = declaringOperation.getParametersList();
        Expression minValue;
        for (Parameter parameter : parametersList) {
            if (!parameter.getFullName().equals(fullName)) {
                minValue = minValueExpression(parameter);
                if (referencedByExpression(minValue)) {
                    logger.trace("parameter " + fullName + " is referenced by " + parameter.getName());
                    minValueReferencingParameters.put(parameter.getPathString(), parameter);
                    if (recursively) {
                        minValueReferencingParameters.putAll(parameter.getMinValueReferencingParameters(recursively));
                    }
                }
            }
        }
        return minValueReferencingParameters;
    }

    /**
     * @return the list of properties referencing this property in their min value
     */
    @Override
    public Map<String, Property> getMinValueReferencingProperties() {
        return getMinValueReferencingProperties(false);
    }

    /**
     * @param recursively
     * @return the list of properties referencing this property in their min value
     */
    @Override
    public Map<String, Property> getMinValueReferencingProperties(boolean recursively) {
        Map<String, Property> minValueReferencingProperties = new LinkedHashMap<>();
        String fullName = getFullName();
        Entity declaringEntity = getDeclaringEntity();
        List<Property> propertiesList = declaringEntity.getPropertiesList();
        Expression minValue;
        for (Property property : propertiesList) {
            if (!property.getFullName().equals(fullName)) {
                minValue = minValueExpression(property);
                if (referencedByExpression(minValue)) {
                    logger.trace("property " + fullName + " is referenced by " + property.getName());
                    minValueReferencingProperties.put(property.getPathString(), property);
                    if (recursively) {
                        minValueReferencingProperties.putAll(property.getMinValueReferencingProperties(recursively));
                    }
                }
            }
        }
        return minValueReferencingProperties;
    }

    private Expression minValueExpression(DataArtifact da) {
        if (da instanceof IntervalizedArtifact) {
            IntervalizedArtifact ia = (IntervalizedArtifact) da;
            Object minValueObject = ia.getMinValue();
            return minValueObject instanceof Expression ? (Expression) minValueObject
                : (minValueObject instanceof Entity) ? ((Entity) minValueObject).isNull() : null;
        }
        return null;
    }
    // </editor-fold>

    // <editor-fold defaultstate="collapsed" desc="modifying filter referencing siblings">
    /**
     * @return the list of artifacts referencing this artifact in their modifying filter
     */
    public Map<String, ? extends DataArtifact> getModifyingFilterReferencingSiblings() {
        return getModifyingFilterReferencingSiblings(false);
    }

    /**
     * @param recursively
     * @return the list of artifacts referencing this artifact in their modifying filter
     */
    public Map<String, ? extends DataArtifact> getModifyingFilterReferencingSiblings(boolean recursively) {
        return isProperty() ? getModifyingFilterReferencingProperties(recursively) : new LinkedHashMap<>();
    }

    /**
     * @return the list of properties referencing this property in their modifying filter
     */
    @Override
    public Map<String, Property> getModifyingFilterReferencingProperties() {
        return getModifyingFilterReferencingProperties(false);
    }

    /**
     * @param recursively
     * @return the list of properties referencing this property in their modifying filter
     */
    @Override
    public Map<String, Property> getModifyingFilterReferencingProperties(boolean recursively) {
        Map<String, Property> modifyingFilterReferencingProperties = new LinkedHashMap<>();
        String fullName = getFullName();
        Entity declaringEntity = getDeclaringEntity();
        List<Property> propertiesList = declaringEntity.getPropertiesList();
        BooleanExpression modifyingFilter;
        for (Property property : propertiesList) {
            if (!property.getFullName().equals(fullName)) {
                modifyingFilter = property.getModifyingFilter();
                if (referencedByExpression(modifyingFilter)) {
                    logger.trace("property " + fullName + " is referenced by " + property.getName());
                    modifyingFilterReferencingProperties.put(property.getPathString(), property);
                    if (recursively) {
                        modifyingFilterReferencingProperties.putAll(property.getModifyingFilterReferencingProperties(recursively));
                    }
                }
            }
        }
        return modifyingFilterReferencingProperties;
    }
    // </editor-fold>

    // <editor-fold defaultstate="collapsed" desc="rendering filter referencing siblings">
    /**
     * @return the list of artifacts referencing this artifact in their rendering filter
     */
    public Map<String, ? extends DataArtifact> getRenderingFilterReferencingSiblings() {
        return getRenderingFilterReferencingSiblings(false);
    }

    /**
     * @param recursively
     * @return the list of artifacts referencing this artifact in their rendering filter
     */
    public Map<String, ? extends DataArtifact> getRenderingFilterReferencingSiblings(boolean recursively) {
        return isProperty() ? getRenderingFilterReferencingProperties(recursively)
            : isParameter() ? getRenderingFilterReferencingParameters(recursively) : new LinkedHashMap<>();
    }

    /**
     * @return the list of parameters referencing this parameter in their rendering filter
     */
    @Override
    public Map<String, Parameter> getRenderingFilterReferencingParameters() {
        return getRenderingFilterReferencingParameters(false);
    }

    /**
     * @param recursively
     * @return the list of parameters referencing this parameter in their rendering filter
     */
    @Override
    public Map<String, Parameter> getRenderingFilterReferencingParameters(boolean recursively) {
        Map<String, Parameter> renderingFilterReferencingParameters = new LinkedHashMap<>();
        String fullName = getFullName();
        Operation declaringOperation = getDeclaringOperation();
        List<Parameter> parametersList = declaringOperation.getParametersList();
        BooleanExpression renderingFilter;
        for (Parameter parameter : parametersList) {
            if (!parameter.getFullName().equals(fullName)) {
                renderingFilter = parameter.getRenderingFilter();
                if (referencedByExpression(renderingFilter)) {
                    logger.trace("parameter " + fullName + " is referenced by " + parameter.getName());
                    renderingFilterReferencingParameters.put(parameter.getPathString(), parameter);
                    if (recursively) {
                        renderingFilterReferencingParameters.putAll(parameter.getRenderingFilterReferencingParameters(recursively));
                    }
                }
            }
        }
        return renderingFilterReferencingParameters;
    }

    /**
     * @return the list of properties referencing this property in their rendering filter
     */
    @Override
    public Map<String, Property> getRenderingFilterReferencingProperties() {
        return getRenderingFilterReferencingProperties(false);
    }

    /**
     * @param recursively
     * @return the list of properties referencing this property in their rendering filter
     */
    @Override
    public Map<String, Property> getRenderingFilterReferencingProperties(boolean recursively) {
        Map<String, Property> renderingFilterReferencingProperties = new LinkedHashMap<>();
        String fullName = getFullName();
        Entity declaringEntity = getDeclaringEntity();
        List<Property> propertiesList = declaringEntity.getPropertiesList();
        BooleanExpression renderingFilter;
        for (Property property : propertiesList) {
            if (!property.getFullName().equals(fullName)) {
                renderingFilter = property.getRenderingFilter();
                if (referencedByExpression(renderingFilter)) {
                    logger.trace("property " + fullName + " is referenced by " + property.getName());
                    renderingFilterReferencingProperties.put(property.getPathString(), property);
                    if (recursively) {
                        renderingFilterReferencingProperties.putAll(property.getRenderingFilterReferencingProperties(recursively));
                    }
                }
            }
        }
        return renderingFilterReferencingProperties;
    }
    // </editor-fold>

    // <editor-fold defaultstate="collapsed" desc="requiring filter referencing siblings">
    /**
     * @return the list of artifacts referencing this artifact in their requiring filter
     */
    public Map<String, ? extends DataArtifact> getRequiringFilterReferencingSiblings() {
        return getRequiringFilterReferencingSiblings(false);
    }

    /**
     * @param recursively
     * @return the list of artifacts referencing this artifact in their requiring filter
     */
    public Map<String, ? extends DataArtifact> getRequiringFilterReferencingSiblings(boolean recursively) {
        return isProperty() ? getRequiringFilterReferencingProperties(recursively)
            : isParameter() ? getRequiringFilterReferencingParameters(recursively) : new LinkedHashMap<>();
    }

    /**
     * @return the list of parameters referencing this parameter in their requiring filter
     */
    @Override
    public Map<String, Parameter> getRequiringFilterReferencingParameters() {
        return getRequiringFilterReferencingParameters(false);
    }

    /**
     * @param recursively
     * @return the list of parameters referencing this parameter in their requiring filter
     */
    @Override
    public Map<String, Parameter> getRequiringFilterReferencingParameters(boolean recursively) {
        Map<String, Parameter> requiringFilterReferencingParameters = new LinkedHashMap<>();
        String fullName = getFullName();
        Operation declaringOperation = getDeclaringOperation();
        List<Parameter> parametersList = declaringOperation.getParametersList();
        BooleanExpression requiringFilter;
        for (Parameter parameter : parametersList) {
            if (!parameter.getFullName().equals(fullName)) {
                requiringFilter = parameter.getRequiringFilter();
                if (referencedByExpression(requiringFilter)) {
                    logger.trace("parameter " + fullName + " is referenced by " + parameter.getName());
                    requiringFilterReferencingParameters.put(parameter.getPathString(), parameter);
                    if (recursively) {
                        requiringFilterReferencingParameters.putAll(parameter.getRequiringFilterReferencingParameters(recursively));
                    }
                }
            }
        }
        return requiringFilterReferencingParameters;
    }

    /**
     * @return the list of properties referencing this property in their requiring filter
     */
    @Override
    public Map<String, Property> getRequiringFilterReferencingProperties() {
        return getRequiringFilterReferencingProperties(false);
    }

    /**
     * @param recursively
     * @return the list of properties referencing this property in their requiring filter
     */
    @Override
    public Map<String, Property> getRequiringFilterReferencingProperties(boolean recursively) {
        Map<String, Property> requiringFilterReferencingProperties = new LinkedHashMap<>();
        String fullName = getFullName();
        Entity declaringEntity = getDeclaringEntity();
        List<Property> propertiesList = declaringEntity.getPropertiesList();
        BooleanExpression requiringFilter;
        for (Property property : propertiesList) {
            if (!property.getFullName().equals(fullName)) {
                requiringFilter = property.getRequiringFilter();
                if (referencedByExpression(requiringFilter)) {
                    logger.trace("property " + fullName + " is referenced by " + property.getName());
                    requiringFilterReferencingProperties.put(property.getPathString(), property);
                    if (recursively) {
                        requiringFilterReferencingProperties.putAll(property.getRequiringFilterReferencingProperties(recursively));
                    }
                }
            }
        }
        return requiringFilterReferencingProperties;
    }
    // </editor-fold>

    // <editor-fold defaultstate="collapsed" desc="search query filter referencing siblings">
    /**
     * @return the list of artifacts referencing this artifact in their search query filter
     */
    public Map<String, ? extends DataArtifact> getSearchQueryFilterReferencingSiblings() {
        return getSearchQueryFilterReferencingSiblings(false);
    }

    /**
     * @param recursively
     * @return the list of artifacts referencing this artifact in their search query filter
     */
    public Map<String, ? extends DataArtifact> getSearchQueryFilterReferencingSiblings(boolean recursively) {
        return isProperty() ? getSearchQueryFilterReferencingProperties(recursively)
            : isParameter() ? getSearchQueryFilterReferencingParameters(recursively) : new LinkedHashMap<>();
    }

    /**
     * @return the list of parameters referencing this parameter in their search query filter
     */
    @Override
    public Map<String, Parameter> getSearchQueryFilterReferencingParameters() {
        return getSearchQueryFilterReferencingParameters(false);
    }

    /**
     * @param recursively
     * @return the list of parameters referencing this parameter in their search query filter
     */
    @Override
    public Map<String, Parameter> getSearchQueryFilterReferencingParameters(boolean recursively) {
        Map<String, Parameter> searchQueryFilterReferencingParameters = new LinkedHashMap<>();
        String fullName = getFullName();
        Operation declaringOperation = getDeclaringOperation();
        List<Parameter> parametersList = declaringOperation.getParametersList();
        Entity entity;
        BooleanExpression searchQueryFilter;
        for (Parameter parameter : parametersList) {
            if (parameter instanceof Entity && !parameter.getFullName().equals(fullName)) {
                entity = (Entity) parameter;
                searchQueryFilter = entity.getSearchQueryFilter();
                if (referencedByExpression(searchQueryFilter)) {
                    logger.trace("parameter " + fullName + " is referenced by " + parameter.getName());
                    searchQueryFilterReferencingParameters.put(parameter.getPathString(), parameter);
                    if (recursively) {
                        searchQueryFilterReferencingParameters.putAll(parameter.getSearchQueryFilterReferencingParameters(recursively));
                    }
                }
            }
        }
        return searchQueryFilterReferencingParameters;
    }

    /**
     * @return the list of properties referencing this property in their search query filter
     */
    @Override
    public Map<String, Property> getSearchQueryFilterReferencingProperties() {
        return getSearchQueryFilterReferencingProperties(false);
    }

    /**
     * @param recursively
     * @return the list of properties referencing this property in their search query filter
     */
    @Override
    public Map<String, Property> getSearchQueryFilterReferencingProperties(boolean recursively) {
        Map<String, Property> searchQueryFilterReferencingProperties = new LinkedHashMap<>();
        String fullName = getFullName();
        Entity declaringEntity = getDeclaringEntity();
        List<Property> propertiesList = declaringEntity.getPropertiesList();
        Entity entity;
        BooleanExpression searchQueryFilter;
        for (Property property : propertiesList) {
            if (property instanceof Entity && !property.getFullName().equals(fullName)) {
                entity = (Entity) property;
                searchQueryFilter = entity.getSearchQueryFilter();
                if (referencedByExpression(searchQueryFilter)) {
                    logger.trace("property " + fullName + " is referenced by " + property.getName());
                    searchQueryFilterReferencingProperties.put(property.getPathString(), property);
                    if (recursively) {
                        searchQueryFilterReferencingProperties.putAll(property.getSearchQueryFilterReferencingProperties(recursively));
                    }
                }
            }
        }
        return searchQueryFilterReferencingProperties;
    }
    // </editor-fold>

    private boolean referencedByExpression(Expression expression) {
        if (expression != null) {
            String pathString = getPathString();
//          return expression.getReferencedColumnsMap().containsKey(pathString);
            Set<String> keys = expression.getReferencedColumnsMap().keySet();
            for (String key : keys) {
                if (key.startsWith(pathString)) {
                    return true;
                }
            }
        }
        return false;
    }

    /**
     * @return the default wrapper class
     */
    @Override
    public Class<? extends DataArtifactWrapper> getDefaultWrapperClass() {
        return DataArtifactWrapper.class;
    }

    /**
     * @return the calculable value
     */
    @Override
    public Object getCalculableValue() {
        return null;
    }

    // <editor-fold defaultstate="collapsed" desc="tag fields getters and setters">
    /**
     * @return the calculable value tag
     */
    @Override
    public String getCalculableValueTag() {
        return getLocalizedCalculableValueTag(null);
    }

    /**
     * El método setCalculableValueTag se utiliza para establecer la descripción del valor calculable de la propiedad que se almacena en el archivo de
     * recursos por defecto. En caso de que el archivo de recursos para el idioma seleccionado por el usuario no esté disponible, la interfaz de la
     * aplicación utiliza el archivo de recursos por defecto para obtener el valor de la descripción.
     *
     * @param tag una o más oraciones que describen muy brevemente el valor calculable de la propiedad
     */
    @Override
    public void setCalculableValueTag(String tag) {
        setLocalizedCalculableValueTag(null, tag);
    }

    /**
     * @return the initial value tag
     */
    @Override
    public String getInitialValueTag() {
        return getLocalizedInitialValueTag(null);
    }

    /**
     * El método setInitialValueTag se utiliza para establecer la descripción del valor inicial de propiedades y parámetros que se almacena en el
     * archivo de recursos por defecto. En caso de que el archivo de recursos para el idioma seleccionado por el usuario no esté disponible, la
     * interfaz de la aplicación utiliza el archivo de recursos por defecto para obtener el valor de la descripción.
     *
     * @param tag una o más oraciones que describen muy brevemente el valor inicial de la propiedad o el parámetro
     */
    @Override
    public void setInitialValueTag(String tag) {
        setLocalizedInitialValueTag(null, tag);
    }

    /**
     * @return the default value tag
     */
    @Override
    public String getDefaultValueTag() {
        return getLocalizedDefaultValueTag(null);
    }

    /**
     * El método setDefaultValueTag se utiliza para establecer la descripción del valor por omisión de propiedades y parámetros que se almacena en el
     * archivo de recursos por defecto. En caso de que el archivo de recursos para el idioma seleccionado por el usuario no esté disponible, la
     * interfaz de la aplicación utiliza el archivo de recursos por defecto para obtener el valor de la descripción.
     *
     * @param tag una o más oraciones que describen muy brevemente el valor por omisión de la propiedad o el parámetro
     */
    @Override
    public void setDefaultValueTag(String tag) {
        setLocalizedDefaultValueTag(null, tag);
    }

    /**
     * @return the current value tag
     */
    @Override
    public String getCurrentValueTag() {
        return getLocalizedCurrentValueTag(null);
    }

    /**
     * El método setCurrentValueTag se utiliza para establecer la descripción del valor actual de propiedades y parámetros que se almacena en el
     * archivo de recursos por defecto. En caso de que el archivo de recursos para el idioma seleccionado por el usuario no esté disponible, la
     * interfaz de la aplicación utiliza el archivo de recursos por defecto para obtener el valor de la descripción.
     *
     * @param tag una o más oraciones que describen muy brevemente el valor actual de la propiedad o el parámetro
     */
    @Override
    public void setCurrentValueTag(String tag) {
        setLocalizedCurrentValueTag(null, tag);
    }

    /**
     * A descriptive word or phrase applied to the value expression as a label or identifier.
     */
    private final Map<Locale, String> _localizedCalculableValueTag = new LinkedHashMap<>();

    /**
     * @param locale the locale for the tag
     * @return the calculable value tag
     */
//  @Override
    public String getLocalizedCalculableValueTag(Locale locale) {
        Locale l = localeReadingKey(locale);
        return _localizedCalculableValueTag.get(l);
    }

    /**
     * El método setLocalizedCalculableValueTag se utiliza para establecer la descripción del valor calculable de la propiedad que se almacena en el
     * archivo de recursos de configuración regional. En caso de que el archivo de recursos para el idioma seleccionado por el usuario no esté
     * disponible, la interfaz de la aplicación utiliza el archivo de recursos por defecto para obtener el valor de la descripción.
     *
     * @param locale configuración regional
     * @param tag una o más oraciones que describen muy brevemente el valor calculable de la propiedad
     */
//  @Override
    public void setLocalizedCalculableValueTag(Locale locale, String tag) {
        Locale l = localeWritingKey(locale);
        if (tag == null) {
            _localizedCalculableValueTag.remove(l);
        } else {
            _localizedCalculableValueTag.put(l, tag);
        }
    }

    /**
     * A descriptive word or phrase applied to the initial value as a label or identifier.
     */
    private final Map<Locale, String> _localizedInitialValueTag = new LinkedHashMap<>();

    /**
     * @param locale the locale for the tag
     * @return the initial value tag
     */
//  @Override
    public String getLocalizedInitialValueTag(Locale locale) {
        Locale l = localeReadingKey(locale);
        return _localizedInitialValueTag.get(l);
    }

    /**
     * El método setLocalizedInitialValueTag se utiliza para establecer la descripción del valor inicial de propiedades y parámetros que se almacena
     * en el archivo de recursos de configuración regional. En caso de que el archivo de recursos para el idioma seleccionado por el usuario no esté
     * disponible, la interfaz de la aplicación utiliza el archivo de recursos por defecto para obtener el valor de la descripción.
     *
     * @param locale configuración regional
     * @param tag una o más oraciones que describen muy brevemente el valor inicial de la propiedad o el parámetro
     */
//  @Override
    public void setLocalizedInitialValueTag(Locale locale, String tag) {
        Locale l = localeWritingKey(locale);
        if (tag == null) {
            _localizedInitialValueTag.remove(l);
        } else {
            _localizedInitialValueTag.put(l, tag);
        }
    }

    /**
     * A descriptive word or phrase applied to the default value as a label or identifier.
     */
    private final Map<Locale, String> _localizedDefaultValueTag = new LinkedHashMap<>();

    /**
     * @param locale the locale for the tag
     * @return the default value tag
     */
//  @Override
    public String getLocalizedDefaultValueTag(Locale locale) {
        Locale l = localeReadingKey(locale);
        return _localizedDefaultValueTag.get(l);
    }

    /**
     * El método setLocalizedDefaultValueTag se utiliza para establecer la descripción del valor por omisión de propiedades y parámetros que se
     * almacena en el archivo de recursos de configuración regional. En caso de que el archivo de recursos para el idioma seleccionado por el usuario
     * no esté disponible, la interfaz de la aplicación utiliza el archivo de recursos por defecto para obtener el valor de la descripción.
     *
     * @param locale configuración regional
     * @param tag una o más oraciones que describen muy brevemente el valor por omisión de la propiedad o el parámetro
     */
//  @Override
    public void setLocalizedDefaultValueTag(Locale locale, String tag) {
        Locale l = localeWritingKey(locale);
        if (tag == null) {
            _localizedDefaultValueTag.remove(l);
        } else {
            _localizedDefaultValueTag.put(l, tag);
        }
    }

    /**
     * A descriptive word or phrase applied to the current value as a label or identifier.
     */
    private final Map<Locale, String> _localizedCurrentValueTag = new LinkedHashMap<>();

    /**
     * @param locale the locale for the tag
     * @return the current value tag
     */
//  @Override
    public String getLocalizedCurrentValueTag(Locale locale) {
        Locale l = localeReadingKey(locale);
        return _localizedCurrentValueTag.get(l);
    }

    /**
     * El método setLocalizedCurrentValueTag se utiliza para establecer la descripción del valor actual de propiedades y parámetros que se almacena en
     * el archivo de recursos de configuración regional. En caso de que el archivo de recursos para el idioma seleccionado por el usuario no esté
     * disponible, la interfaz de la aplicación utiliza el archivo de recursos por defecto para obtener el valor de la descripción.
     *
     * @param locale configuración regional
     * @param tag una o más oraciones que describen muy brevemente el valor actual de la propiedad o el parámetro
     */
//  @Override
    public void setLocalizedCurrentValueTag(Locale locale, String tag) {
        Locale l = localeWritingKey(locale);
        if (tag == null) {
            _localizedCurrentValueTag.remove(l);
        } else {
            _localizedCurrentValueTag.put(l, tag);
        }
    }
    // </editor-fold>

    protected boolean validInitialValue(Object object) {
//      if (isParameterProperty()) {
//          return true;
//      }
        if (isCalculable()) {
            logger.error(getFullName() + " is calculable and therefore does not support initial value");
            TLC.getProject().getParser().increaseErrorCount();
            return false;
        }
        if (object == null) {
            logger.error("null is not a valid initial value for " + getFullName());
            TLC.getProject().getParser().increaseErrorCount();
            return false;
        }
        if (object instanceof SpecialTemporalValue && !validSpecialTemporalValue((SpecialTemporalValue) object)) {
            logger.error(object + " is not a valid initial value for " + getFullName());
            TLC.getProject().getParser().increaseErrorCount();
            return false;
        }
        /*
        if (isParameter()) {
            if (object instanceof Instance) {
            } else if (object instanceof Artifact) {
                Artifact artifact = (Artifact) object;
                logger.error(artifact.getFullName() + " is not a valid initial value for parameter " + getFullName());
                TLC.getProject().getParser().increaseErrorCount();
                return false;
            } else if (object instanceof NamedValue && this instanceof Entity) {
                NamedValue namedValue = (NamedValue) object;
                logger.error(namedValue.name() + " is not a valid initial value for parameter " + getFullName());
                TLC.getProject().getParser().increaseErrorCount();
                return false;
            }
        }
        **/
        return true;
    }

    protected boolean validDefaultValue(Object object) {
//      if (isParameterProperty()) {
//          return true;
//      }
        if (isCalculable()) {
            logger.error(getFullName() + " is calculable and therefore does not support default value");
            TLC.getProject().getParser().increaseErrorCount();
            return false;
        }
        if (object == null) {
            logger.error("null is not a valid default value for " + getFullName());
            TLC.getProject().getParser().increaseErrorCount();
            return false;
        }
        if (object instanceof SpecialTemporalValue && !validSpecialTemporalValue((SpecialTemporalValue) object)) {
            logger.error(object + " is not a valid default value for " + getFullName());
            TLC.getProject().getParser().increaseErrorCount();
            return false;
        }
        return true;
    }

    protected boolean validCurrentValue(Object object) {
        if (object == null) {
            logger.error("null is not a valid current value for " + getFullName());
            TLC.getProject().getParser().increaseErrorCount();
            return false;
        }
        if (object instanceof SpecialTemporalValue && !validSpecialTemporalValue((SpecialTemporalValue) object)) {
            logger.error(object + " is not a valid current value for " + getFullName());
            TLC.getProject().getParser().increaseErrorCount();
            return false;
        }
        return true;
    }

    protected boolean validSpecialTemporalValue(SpecialTemporalValue value) {
        switch (value) {
            case CURRENT_DATE:
                return isDateData() || isTimestampData();
            case CURRENT_TIME:
                return isTimeData() || isTimestampData();
            case CURRENT_TIMESTAMP:
                return isTimestampData();
            default:
                return isTemporalPrimitive();
        }
    }

    /*
    private static final int COLUMN_WIDTH = 960;
    **/
    /**
     * @return the property size in pixels
     */
    @Override
    public int getPixels() {
        if (isEnumerationEntityPrimaryKey()) {
            return 48;
        } else if (isEntity() || isHiddenField() || isBinaryData() || isPassword()) {
            return 0;
        } else if (this instanceof BooleanPrimitive) {
            return 48; // 50/1000
        } else if (this instanceof CharacterData) {
            return 48; // 50/1000
        } else if (this instanceof StringData) {
            if (isVariantStringField()) {
                return 120; // 125/1000
            }
            StringData string = (StringData) this;
            Integer maxLength = string.getMaxLength();
            if (maxLength == null) {
                return 384; // 400/1000
            } else if (maxLength > 100) {
                return 240; // 250/1000
            } else if (maxLength > 50) {
                return 192; // 200/1000
            } else if (maxLength > 20) {
                return 144; // 150/1000
            } else if (maxLength > 10) {
                return 120; // 125/1000
            } else {
                return 72; // 75/1000
            }
        } else if (this instanceof ByteData || this instanceof ShortData || this instanceof IntegerData) {
            return 72; // 75/1000
        } else if (this instanceof LongData) {
            return 128;
        } else if (this instanceof NumericPrimitive) {
            return 120; // 125/1000
        } else if (this instanceof TimestampData) {
            return 120; // 125/1000
        } else if (this instanceof TemporalPrimitive) {
            return 72; // 75/1000
        } else {
            return 0;
        }
    }

    @Override
    public int getColumnPixels() {
        return isEnumerationEntityPrimaryKey() || isColumnField() ? getPixels() : 0;
    }

    private boolean isEnumerationEntityPrimaryKey() {
        return isPrimaryKeyProperty() && getDeclaringEntity() instanceof EnumerationEntity;
    }

    public boolean isSinglePropertyOfUniqueKey() {
        Entity root = getDeclaringEntityRoot();
        if (root != null) {
            List<Key> keys = root.getKeysList();
            for (Key key : keys) {
                if (key.isUnique() && key.isSingleProperty() && this.equals(key.getTheProperty())) {
                    return true;
                }
            }
        }
        return false;
    }

    // <editor-fold defaultstate="collapsed" desc="toString">
    @Override
    protected String fieldsToString(int n, String key, boolean verbose, boolean fields, boolean maps) {
        String tab = verbose ? StringUtils.repeat(" ", 4) : "";
        String fee = verbose ? StringUtils.repeat(tab, n) : "";
        String faa = " = ";
        String foo = verbose ? EOL : ", ";
        String string = super.fieldsToString(n, key, verbose, fields, maps);
        if (fields || verbose) {
            if (verbose) {
                string += fee + tab + "dataType" + faa + _dataType + foo;
                if (getDeclaringField() != null && getDeclaringArtifact() != null) {
//                  string += fee + tab + "baseField" + faa + _annotatedWithBaseField + foo;
//                  string += fee + tab + "argumentField" + faa + _annotatedWithArgumentField + foo;
//                  string += fee + tab + "columnField" + faa + _annotatedWithColumnField + foo;
//                  string += fee + tab + "PropertyField" + faa + _annotatedWithPropertyField + foo;
//                  string += fee + tab + "PrimaryKey" + faa + _annotatedWithPrimaryKey + foo;
//                  string += fee + tab + "SequenceProperty" + faa + _annotatedWithSequenceProperty + foo;
//                  string += fee + tab + "VersionProperty" + faa + _annotatedWithVersionProperty + foo;
//                  string += fee + tab + "NumericKey" + faa + _annotatedWithNumericKey + foo;
//                  string += fee + tab + "CharacterKey" + faa + _annotatedWithCharacterKey + foo;
//                  string += fee + tab + "NameProperty" + faa + _annotatedWithNameProperty + foo;
//                  string += fee + tab + "DescriptionProperty" + faa + _annotatedWithDescriptionProperty + foo;
//                  string += fee + tab + "ImageProperty" + faa + _annotatedWithImageProperty + foo;
//                  string += fee + tab + "InactiveIndicator" + faa + _annotatedWithInactiveIndicator + foo;
//                  string += fee + tab + "UrlProperty" + faa + _annotatedWithUrlProperty + foo;
//                  string += fee + tab + "ParentProperty" + faa + _annotatedWithParentProperty + foo;
//                  string += fee + tab + "OwnerProperty" + faa + _annotatedWithOwnerProperty + foo;
//                  string += fee + tab + "SegmentProperty" + faa + _annotatedWithSegmentProperty + foo;
//                  string += fee + tab + "UniqueKey" + faa + _annotatedWithUniqueKey + foo;
//                  string += fee + tab + "BusinessKey" + faa + _annotatedWithBusinessKey + foo;
//                  string += fee + tab + "DiscriminatorColumn" + faa + _annotatedWithDiscriminatorColumn + foo;
//                  string += fee + tab + "StateProperty" + faa + _annotatedWithStateProperty + foo;
                    if (isParameter()) {
                        string += fee + tab + "auditable" + faa + _auditable + foo;
                        string += fee + tab + "password" + faa + _password + foo;
                        string += fee + tab + "required" + faa + _required + foo;
                        string += fee + tab + "linkedFieldName" + faa + _linkedFieldName + foo;
                        string += fee + tab + "linkedField" + faa + _linkedField + foo;
                        string += fee + tab + "linkedProperty" + faa + _linkedProperty + foo;
                        string += fee + tab + "linkedColumnName" + faa + _linkedColumnName + foo;
                        string += fee + tab + "linkedColumnOperator" + faa + _linkedColumnOperator + foo;
                    }
                    if (isProperty()) {
                        string += fee + tab + "baseField" + faa + isBaseField() + foo;
                        string += fee + tab + "keyField" + faa + isKeyField() + foo;
                        string += fee + tab + "auditable" + faa + _auditable + foo;
                        string += fee + tab + "password" + faa + _password + foo;
                        string += fee + tab + "required" + faa + _required + foo;
                        string += fee + tab + "hiddenField" + faa + _hiddenField + foo;
                        string += fee + tab + "createField" + faa + _createField + foo;
                        string += fee + tab + "updateField" + faa + _updateField + foo;
                        string += fee + tab + "searchField" + faa + _searchField + foo;
                        string += fee + tab + "filterField" + faa + _filterField + foo;
                        string += fee + tab + "tableField" + faa + _tableField + foo;
                        string += fee + tab + "detailField" + faa + _detailField + foo;
                        string += fee + tab + "columnField" + faa + _columnField + foo;
                        string += fee + tab + "reportField" + faa + _reportField + foo;
                        string += fee + tab + "exportField" + faa + _exportField + foo;
                        string += fee + tab + "submitField" + faa + _submitField + foo;
                        string += fee + tab + "calculable" + faa + _calculable + foo;
                        string += fee + tab + "nullable" + faa + _nullable + foo;
                        string += fee + tab + "insertable" + faa + _insertable + foo;
                        string += fee + tab + "updateable" + faa + _updateable + foo;
                        string += fee + tab + "unique" + faa + _unique + foo;
                    }
                }
            }
        }
        return string;
    }
    // </editor-fold>

}
