/*
 * Copyright 2017 Jorge Campins y David Uzcategui
 *
 * Este archivo forma parte de Adalid.
 *
 * Adalid es software libre; usted puede redistribuirlo y/o modificarlo bajo los terminos de la
 * licencia "GNU General Public License" publicada por la Fundacion "Free Software Foundation".
 * Adalid se distribuye con la esperanza de que pueda ser util, pero SIN NINGUNA GARANTIA; sin
 * siquiera las garantias implicitas de COMERCIALIZACION e IDONEIDAD PARA UN PROPOSITO PARTICULAR.
 *
 * Para mas detalles vea la licencia "GNU General Public License" en http://www.gnu.org/licenses
 */
package adalid.core;

import adalid.commons.bundles.Bundle;
import adalid.commons.properties.PropertiesHandler;
import adalid.commons.util.BitUtils;
import adalid.commons.util.ColUtils;
import adalid.commons.util.IntUtils;
import adalid.commons.util.StrUtils;
import adalid.commons.util.ThrowableUtils;
import adalid.core.annotations.AbstractClass;
import adalid.core.annotations.Allocation;
import adalid.core.annotations.AllocationOverride;
import adalid.core.annotations.AllocationOverrides;
import adalid.core.annotations.BaseField;
import adalid.core.annotations.BusinessKey;
import adalid.core.annotations.CastingField;
import adalid.core.annotations.CharacterKey;
import adalid.core.annotations.ColumnField;
import adalid.core.annotations.DescriptionProperty;
import adalid.core.annotations.EntityClass;
import adalid.core.annotations.EntityCodeGen;
import adalid.core.annotations.EntityConsoleView;
import adalid.core.annotations.EntityDataGen;
import adalid.core.annotations.EntityDeleteOperation;
import adalid.core.annotations.EntityDetailView;
import adalid.core.annotations.EntityDocGen;
import adalid.core.annotations.EntityExportOperation;
import adalid.core.annotations.EntityInsertOperation;
import adalid.core.annotations.EntityReferenceDataGen;
import adalid.core.annotations.EntityReferenceSearch;
import adalid.core.annotations.EntityReportOperation;
import adalid.core.annotations.EntitySelectOperation;
import adalid.core.annotations.EntityTableView;
import adalid.core.annotations.EntityTreeView;
import adalid.core.annotations.EntityUpdateOperation;
import adalid.core.annotations.EntityWarnings;
import adalid.core.annotations.Filter;
import adalid.core.annotations.GraphicImage;
import adalid.core.annotations.ImageProperty;
import adalid.core.annotations.InactiveIndicator;
import adalid.core.annotations.InstanceReference;
import adalid.core.annotations.ManyToOne;
import adalid.core.annotations.NameProperty;
import adalid.core.annotations.NumericKey;
import adalid.core.annotations.OneToOne;
import adalid.core.annotations.OwnerProperty;
import adalid.core.annotations.ParameterField;
import adalid.core.annotations.ParentProperty;
import adalid.core.annotations.PrimaryKey;
import adalid.core.annotations.PropertyAggregation;
import adalid.core.annotations.PropertyField;
import adalid.core.annotations.QueryMapping;
import adalid.core.annotations.SegmentProperty;
import adalid.core.annotations.SequenceProperty;
import adalid.core.annotations.StateProperty;
import adalid.core.annotations.UniqueKey;
import adalid.core.annotations.UrlProperty;
import adalid.core.annotations.VersionProperty;
import adalid.core.data.types.IntegerData;
import adalid.core.data.types.StringData;
import adalid.core.enums.BusinessKeyType;
import adalid.core.enums.CascadeType;
import adalid.core.enums.DisplayMode;
import adalid.core.enums.FetchType;
import adalid.core.enums.HierarchyNodeType;
import adalid.core.enums.KeyProperty;
import adalid.core.enums.ListStyle;
import adalid.core.enums.MasterDetailView;
import adalid.core.enums.Navigability;
import adalid.core.enums.OperationAccess;
import adalid.core.enums.OperationLogging;
import adalid.core.enums.OperationType;
import adalid.core.enums.QuickAddingFilter;
import adalid.core.enums.ResourceGender;
import adalid.core.enums.ResourceType;
import adalid.core.enums.SearchType;
import adalid.core.enums.SelectOnloadOption;
import adalid.core.enums.SortOption;
import adalid.core.enums.SpecialEntityValue;
import adalid.core.exceptions.UnexpectedRuntimeException;
import adalid.core.expressions.BooleanComparisonX;
import adalid.core.expressions.BooleanDataAggregateX;
import adalid.core.expressions.BooleanOrderedPairX;
import adalid.core.expressions.BooleanScalarX;
import adalid.core.expressions.CharacterOrderedPairX;
import adalid.core.expressions.CharacterScalarX;
import adalid.core.expressions.EntityOrderedPairX;
import adalid.core.expressions.NumericScalarX;
import adalid.core.expressions.TemporalScalarX;
import adalid.core.expressions.XB;
import adalid.core.interfaces.Artifact;
import adalid.core.interfaces.BooleanExpression;
import adalid.core.interfaces.CalculableProperty;
import adalid.core.interfaces.CharacterExpression;
import adalid.core.interfaces.DataArtifact;
import adalid.core.interfaces.Entity;
import adalid.core.interfaces.EntityExpression;
import adalid.core.interfaces.EntityReference;
import adalid.core.interfaces.Expression;
import adalid.core.interfaces.NamedValue;
import adalid.core.interfaces.Parameter;
import adalid.core.interfaces.PersistentEntityReference;
import adalid.core.interfaces.Property;
import adalid.core.interfaces.State;
import adalid.core.operations.DeleteOperation;
import adalid.core.operations.InsertOperation;
import adalid.core.operations.SelectOperation;
import adalid.core.operations.UpdateOperation;
import adalid.core.predicates.IsAccesibleOperation;
import adalid.core.primitives.BooleanPrimitive;
import adalid.core.primitives.CharacterPrimitive;
import adalid.core.primitives.NumericPrimitive;
import adalid.core.primitives.TemporalPrimitive;
import adalid.core.properties.BinaryProperty;
import adalid.core.properties.BooleanProperty;
import adalid.core.properties.IntegerProperty;
import adalid.core.properties.LongProperty;
import adalid.core.properties.StringProperty;
import adalid.core.wrappers.EntityWrapper;
import java.lang.annotation.Annotation;
import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.EnumSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.TreeSet;
import org.apache.commons.collections.ExtendedProperties;
import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;

/**
 * @author Jorge Campins
 */
public abstract class AbstractEntity extends AbstractDataArtifact implements EntityReference {

    // <editor-fold defaultstate="collapsed" desc="expression building protected static methods">
    /**
     * La conjunción (AND) es un operador lógico que resulta en verdadero si ambos operandos son verdadero.
     *
     * @param operand1
     * @param operand2
     * @return verdadero si ambos operandos son verdadero; de lo contrario, falso.
     */
    protected static BooleanOrderedPairX and(BooleanExpression operand1, BooleanExpression operand2) {
        return XB.Boolean.OrderedPair.and(operand1, operand2);
    }

    /**
     * La negación alternativa (NAND) es un operador lógico que resulta en verdadero si uno de los operandos es falso. Es equivalente a la negación
     * (NOT) de la conjunción (AND).
     *
     * @param operand1
     * @param operand2
     * @return verdadero si uno de los operandos es falso; de lo contrario, falso.
     */
    protected static BooleanOrderedPairX nand(BooleanExpression operand1, BooleanExpression operand2) {
        return XB.Boolean.OrderedPair.nand(operand1, operand2);
    }

    /**
     * La disyunción (OR) es un operador lógico que resulta en verdadero si uno de los operandos es verdadero.
     *
     * @param operand1
     * @param operand2
     * @return verdadero si uno de los operandos es verdadero; de lo contrario, falso.
     */
    protected static BooleanOrderedPairX or(BooleanExpression operand1, BooleanExpression operand2) {
        return XB.Boolean.OrderedPair.or(operand1, operand2);
    }

    /**
     * La negación conjunta (NOR) es un operador lógico que resulta en verdadero si ambos operandos son falso. Es equivalente a la negación (NOT) de
     * la disyunción (OR).
     *
     * @param operand1
     * @param operand2
     * @return verdadero si ambos operandos son falso; de lo contrario, falso.
     */
    protected static BooleanOrderedPairX nor(BooleanExpression operand1, BooleanExpression operand2) {
        return XB.Boolean.OrderedPair.nor(operand1, operand2);
    }

    /**
     * La disyunción exclusiva (XOR) es un operador lógico que resulta en verdadero si uno y solo uno de los operandos es verdadero.
     *
     * @param operand1
     * @param operand2
     * @return verdadero si uno y solo uno de los operandos es verdadero; de lo contrario, falso.
     */
    protected static BooleanOrderedPairX xor(BooleanExpression operand1, BooleanExpression operand2) {
        return XB.Boolean.OrderedPair.xor(operand1, operand2);
    }

    /**
     * La equivalencia (XNOR) es un operador lógico que resulta en verdadero si ambos operandos son verdadero ó si ambos son falso. Es equivalente a
     * la negación (NOT) de la disyunción exclusiva (XOR).
     *
     * @param operand1
     * @param operand2
     * @return verdadero si ambos operandos son verdadero ó si ambos son falso; de lo contrario, falso.
     */
    protected static BooleanOrderedPairX xnor(BooleanExpression operand1, BooleanExpression operand2) {
        return XB.Boolean.OrderedPair.xnor(operand1, operand2);
    }

    /**
     * La condicional material (X_IMPLIES_Y) es un operador lógico que resulta en falso solo si X es verdadero y Y es falso. Es equivalente a la
     * disyunción (OR) de la negación (NOT) de X con Y (NOT X OR Y).
     *
     * @param operandX
     * @param operandY
     * @return falso solo si X es verdadero y Y es falso; de lo contrario, verdadero.
     */
    protected static BooleanOrderedPairX xImpliesY(BooleanExpression operandX, BooleanExpression operandY) {
        return XB.Boolean.OrderedPair.xImpliesY(operandX, operandY);
    }

    /**
     * La conjunción (AND) es un operador lógico que resulta en verdadero si todos los operandos son verdadero.
     *
     * @param operand1
     * @param operand2
     * @param extraOperands
     * @return verdadero si todos los operandos son verdadero; de lo contrario, falso.
     */
    protected static BooleanDataAggregateX and(BooleanExpression operand1, BooleanExpression operand2, BooleanExpression... extraOperands) {
        return XB.Boolean.DataAggregate.and(operand1, operand2, extraOperands);
    }

    /**
     * La negación alternativa (NAND) es un operador lógico que resulta en verdadero si al menos uno de los operandos es falso. Es equivalente a la
     * negación (NOT) de la conjunción (AND).
     *
     * @param operand1
     * @param operand2
     * @param extraOperands
     * @return verdadero si al menos uno de los operandos es falso; de lo contrario, falso.
     */
    protected static BooleanDataAggregateX nand(BooleanExpression operand1, BooleanExpression operand2, BooleanExpression... extraOperands) {
        return XB.Boolean.DataAggregate.nand(operand1, operand2, extraOperands);
    }

    /**
     * La disyunción (OR) es un operador lógico que resulta en verdadero si al menos uno de los operandos es verdadero.
     *
     * @param operand1
     * @param operand2
     * @param extraOperands
     * @return verdadero si al menos uno de los operandos es verdadero; de lo contrario, falso.
     */
    protected static BooleanDataAggregateX or(BooleanExpression operand1, BooleanExpression operand2, BooleanExpression... extraOperands) {
        return XB.Boolean.DataAggregate.or(operand1, operand2, extraOperands);
    }

    /**
     * La negación conjunta (NOR) es un operador lógico que resulta en verdadero si todos los operandos son falso. Es equivalente a la negación (NOT)
     * de la disyunción (OR).
     *
     * @param operand1
     * @param operand2
     * @param extraOperands
     * @return verdadero si todos los operandos son falso; de lo contrario, falso.
     */
    protected static BooleanDataAggregateX nor(BooleanExpression operand1, BooleanExpression operand2, BooleanExpression... extraOperands) {
        return XB.Boolean.DataAggregate.nor(operand1, operand2, extraOperands);
    }

    /**
     * La disyunción exclusiva no-asociativa (NAXOR) es un operador lógico que resulta en verdadero si uno y solo uno de los operandos es verdadero.
     *
     * @param operand1
     * @param operand2
     * @param extraOperands
     * @return verdadero si uno y solo uno de los operandos es verdadero; de lo contrario, falso.
     */
    protected static BooleanDataAggregateX naxor(BooleanExpression operand1, BooleanExpression operand2, BooleanExpression... extraOperands) {
        return XB.Boolean.DataAggregate.naxor(operand1, operand2, extraOperands);
    }

    /**
     * La equivalencia no-asociativa (NAXNOR) es un operador lógico que resulta en verdadero si todos los operandos son verdadero ó si todos son
     * falso. Es equivalente a la negación (NOT) de la disyunción exclusiva no-asociativa (NAXOR).
     *
     * @param operand1
     * @param operand2
     * @param extraOperands
     * @return verdadero si todos los operandos son verdadero ó si todos son falso; de lo contrario, falso.
     */
    protected static BooleanDataAggregateX naxnor(BooleanExpression operand1, BooleanExpression operand2, BooleanExpression... extraOperands) {
        return XB.Boolean.DataAggregate.naxnor(operand1, operand2, extraOperands);
    }

    /**
     * NOR_OR_NAXOR es un operador lógico compuesto que resulta en verdadero si uno y solo uno de los operandos es verdadero ó si todos son falso. Es
     * equivalente a la disyunción (OR) de la negación conjunta (NOR) y la disyunción exclusiva no-asociativa (NAXOR). Con solo dos operandos también
     * es equivalente a la negación alternativa (NAND).
     *
     * @param operand1
     * @param operand2
     * @param extraOperands
     * @return verdadero si uno y solo uno de los operandos es verdadero ó si todos son falso; de lo contrario, falso.
     */
    protected static BooleanDataAggregateX norOrNaxor(BooleanExpression operand1, BooleanExpression operand2, BooleanExpression... extraOperands) {
        return XB.Boolean.DataAggregate.norOrNaxor(operand1, operand2, extraOperands);
    }

    protected static BooleanScalarX not(BooleanExpression x) {
        return x.not();
    }
    // </editor-fold>

    // <editor-fold defaultstate="collapsed" desc="field declarations">
    /**
     *
     */
    private static final Logger logger = Logger.getLogger(Entity.class);

    private static final String EOL = "\n";

    /**
     *
     */
    private boolean _initialised;

    /**
     *
     */
    private boolean _settled;

    /**
     *
     */
    private boolean _finalised;

    /**
     *
     */
    private boolean _finished;

    /**
     *
     */
    private char _settler = '?';

    /**
     *
     */
    private Project _project;

    /**
     *
     */
    private boolean _rootInstance;

    /**
     *
     */
    private boolean _explicitlyDeclared;

    /**
     *
     */
    private boolean _implicitlyDeclared;

    /**
     *
     */
    private int _referenceIndex;

    /**
     *
     */
//  private final Map<EntityReference, String> _defaultLabelByReferenceMap = new LinkedHashMap<>();
    private final Map<LocaleEntityReference, String> _localizedLabelByReferenceMap = new LinkedHashMap<>();

    /**
     *
     */
//  private final Map<EntityReference, String> _defaultShortLabelByReferenceMap = new LinkedHashMap<>();
    private final Map<LocaleEntityReference, String> _localizedShortLabelByReferenceMap = new LinkedHashMap<>();

    /**
     *
     */
//  private final Map<EntityReference, String> _defaultCollectionLabelByReferenceMap = new LinkedHashMap<>();
    private final Map<LocaleEntityReference, String> _localizedCollectionLabelByReferenceMap = new LinkedHashMap<>();

    /**
     *
     */
//  private final Map<EntityReference, String> _defaultCollectionShortLabelByReferenceMap = new LinkedHashMap<>();
    private final Map<LocaleEntityReference, String> _localizedCollectionShortLabelByReferenceMap = new LinkedHashMap<>();

    /**
     *
     */
    private final EntityAtlas _atlas = new EntityAtlas(this);

    /**
     * baseClass: concrete entity superclass, if any (null otherwise)
     */
    private Class<?> _baseClass;

    /**
     * subclasses: direct known subclasses
     */
    private final Map<String, Class<?>> _subclasses = new LinkedHashMap<>();

    /**
     *
     */
    private final Map<String, AllocationOverride> _allocationOverrides = new LinkedHashMap<>();

    /**
     *
     */
    private String _primaryKeyFieldName;

    /**
     *
     */
    private Field _primaryKeyField;

    /**
     * primaryKeyProperty: reference to the entity primary key property, if any (null otherwise)
     */
    private Property _primaryKeyProperty;

    /**
     *
     */
    private String _sequenceFieldName;

    /**
     *
     */
    private Field _sequenceField;

    /**
     * sequenceProperty: reference to the entity sequence property, if any (null otherwise)
     */
    private LongProperty _sequenceProperty;

    /**
     *
     */
    private String _versionFieldName;

    /**
     *
     */
    private Field _versionField;

    /**
     * versionProperty: reference to the entity version property, if any (null otherwise)
     */
    private LongProperty _versionProperty;

    /**
     *
     */
    private String _numericKeyFieldName;

    /**
     *
     */
    private Field _numericKeyField;

    /**
     * numericKeyProperty: reference to the entity numeric key property, if any (null otherwise)
     */
    private IntegerProperty _numericKeyProperty;

    /**
     *
     */
    private String _characterKeyFieldName;

    /**
     *
     */
    private Field _characterKeyField;

    /**
     * characterKeyProperty: reference to the entity character key property, if any (null otherwise)
     */
    private StringProperty _characterKeyProperty;

    /**
     *
     */
    private String _nameFieldName;

    /**
     *
     */
    private Field _nameField;

    /**
     * nameProperty: reference to the entity name (short description) property, if any (null otherwise)
     */
    private StringProperty _nameProperty;

    /**
     *
     */
    private String _descriptionFieldName;

    /**
     *
     */
    private Field _descriptionField;

    /**
     * descriptionProperty: reference to the entity description property, if any (null otherwise)
     */
    private StringProperty _descriptionProperty;

    /**
     *
     */
    private String _imageFieldName;

    /**
     *
     */
    private Field _imageField;

    /**
     * imageProperty: reference to the entity image property, if any (null otherwise)
     */
    private BinaryProperty _imageProperty;

    /**
     *
     */
    private String _inactiveIndicatorFieldName;

    /**
     *
     */
    private Field _inactiveIndicatorField;

    /**
     * inactiveIndicatorProperty: reference to the entity inactive indicator property, if any (null otherwise)
     */
    private BooleanProperty _inactiveIndicatorProperty;

    /**
     *
     */
    private boolean _foreignInactiveIndicatorProperty;

    /**
     *
     */
    private String _urlFieldName;

    /**
     *
     */
    private Field _urlField;

    /**
     * urlProperty: reference to the entity URL property, if any (null otherwise)
     */
    private StringProperty _urlProperty;

    /**
     *
     */
    private String _parentFieldName;

    /**
     *
     */
    private Field _parentField;

    /**
     * parentProperty: parent entity reference, if any (null otherwise)
     */
    private EntityReference _parentProperty;

    /**
     *
     */
    private String _ownerFieldName;

    /**
     *
     */
    private Field _ownerField;

    /**
     * ownerProperty: owner (user) entity reference, if any (null otherwise)
     */
    private EntityReference _ownerProperty;

    /**
     *
     */
    private boolean _foreignOwnerProperty;

    /**
     *
     */
    private String _segmentFieldName;

    /**
     *
     */
    private Field _segmentField;

    /**
     * segmentProperty: reference to the entity segment property, if any (null otherwise)
     */
    private EntityReference _segmentProperty;

    /**
     *
     */
    private boolean _foreignSegmentProperty;

    /**
     *
     */
    private String _businessKeyFieldName;

    /**
     *
     */
    private Field _businessKeyField;

    /**
     * businessKeyProperty: reference to the entity business key property, if any (null otherwise)
     */
    private Property _businessKeyProperty;

    /**
     * businessKeyType: character key, numeric key or unspecified
     */
    private BusinessKeyType _businessKeyType;

    /**
     *
     */
    private String _stateFieldName;

    /**
     *
     */
    private Field _stateField;

    /**
     *
     */
    private EntityReference _stateProperty;

    /**
     * basicOperationEntity: basic operation entity indicator
     */
    private boolean _basicOperationEntity;

    /**
     * catalogEntity: catalog entity indicator
     */
    private boolean _catalogEntity;

    /**
     * existentiallyIndependent: existentially independent entity indicator
     */
    private boolean _existentiallyIndependent;

    /**
     * variantEntity: variant entity indicator
     */
    private boolean _variantEntity;

    /**
     * resourceType: configuration, operation or unspecified
     */
    private ResourceType _resourceType;

    /**
     * resourceGender: masculine, feminine, common, neuter or unspecified
     */
    private ResourceGender _resourceGender;

    /**
     *
     */
    private String _propertiesPrefix;

    /**
     *
     */
    private String _propertiesSuffix;

    /**
     *
     */
    private String _helpFileName;

    /**
     *
     */
    private int _startWith;

    /**
     *
     */
    private int _seriesStart;

    /**
     *
     */
    private int _seriesStop;

    /**
     *
     */
    private int _seriesStep;

    /**
     *
     */
    private boolean _selectEnabled;

//  /**
//   *
//   */
//  private boolean _selectRestricted;
//
    /**
     *
     */
    private OperationAccess _selectOperationAccess;

    /**
     *
     */
    private SelectOnloadOption _selectOnloadOption;

    /**
     *
     */
    private int _selectRowsLimit;

    /**
     *
     */
    SortOption _selectSortOption;

    /**
     *
     */
    private boolean _insertEnabled;

//  /**
//   *
//   */
//  private boolean _insertRestricted;
//
    /**
     *
     */
    private OperationAccess _insertOperationAccess;

    /**
     *
     */
    private OperationLogging _insertLogging;

    /**
     *
     */
    private boolean _updateEnabled;

//  /**
//   *
//   */
//  private boolean _updateRestricted;
//
    /**
     *
     */
    private OperationAccess _updateOperationAccess;

    /**
     *
     */
    private OperationLogging _updateLogging;

    /**
     *
     */
    private boolean _deleteEnabled;

//  /**
//   *
//   */
//  private boolean _deleteRestricted;
//
    /**
     *
     */
    private OperationAccess _deleteOperationAccess;

    /**
     *
     */
    private OperationLogging _deleteLogging;

    /**
     *
     */
    private boolean _reportEnabled;

    /**
     *
     */
    private int _reportRowsLimit;

    /**
     *
     */
    SortOption _reportSortOption;

    /**
     *
     */
    private boolean _exportEnabled;

    /**
     *
     */
    private int _exportRowsLimit;

    /**
     *
     */
    SortOption _exportSortOption;

    /**
     *
     */
    private Boolean _tableViewEnabled;

    /**
     *
     */
    private boolean _tableViewWithInsertEnabled;

    /**
     *
     */
    private boolean _tableViewWithUpdateEnabled;

    /**
     *
     */
    private boolean _tableViewWithDeleteEnabled;

    /**
     *
     */
    private boolean _tableViewWithMasterHeading;

    /**
     *
     */
    private int _tableViewRowsLimit;

    /**
     *
     */
    private int _tableViewRows;
//
//  /**
//   *
//   */
//  private int _tableViewWidth;

    /**
     *
     */
    private boolean _detailViewEnabled;

    /**
     *
     */
    private boolean _detailViewWithMasterHeading;
//
//  /**
//   *
//   */
//  private int _detailViewWidth;

    /**
     *
     */
    private boolean _treeViewEnabled;

    /**
     *
     */
    private boolean _consoleViewEnabled;
//
//  /**
//   *
//   */
//  private int _consoleViewWidth;

    /**
     *
     */
    private boolean _warningsEnabled;

    /**
     *
     */
    private boolean _bplCodeGenEnabled;

    /**
     *
     */
    private boolean _bwsCodeGenEnabled;

    /**
     *
     */
    private boolean _fwsCodeGenEnabled;

    /**
     *
     */
    private boolean _sqlCodeGenEnabled;

    /**
     *
     */
    private boolean _entityStateCodeGenEnabled;

    /**
     *
     */
    private boolean _entityClassDiagramGenEnabled;

    /**
     *
     */
    private boolean _entityStateDiagramGenEnabled;

    /**
     *
     */
    private boolean _entityInsertActivityDiagramGenEnabled;

    /**
     *
     */
    private boolean _entityUpdateActivityDiagramGenEnabled;

    /**
     * annotated with EntityReferenceSearch
     */
    private SearchType _searchType;

    /**
     * annotated with EntityReferenceSearch
     */
    private ListStyle _listStyle;

    /**
     * annotated with EntityReferenceSearch
     */
    private DisplayMode _searchDisplayMode;

    /**
     * search query filter
     */
    private BooleanExpression _searchQueryFilter;

    /**
     * select filter
     */
    private BooleanExpression _selectFilter;

    /**
     * insert filter
     */
    private final Map<EntityReference, BooleanExpression> _insertFilterByReferenceMap = new LinkedHashMap<>();

    /**
     * update filter
     */
    private BooleanExpression _updateFilter;

    /**
     * delete filter
     */
    private BooleanExpression _deleteFilter;

    /**
     * master/detail filter
     */
    private final Map<EntityReference, BooleanExpression> _masterDetailFilterByReferenceMap = new LinkedHashMap<>();

    /**
     *
     */
    private boolean _filterInactiveIndicatorProperty;

    /**
     *
     */
    private boolean _filterOwnerProperty;

    /**
     *
     */
    private boolean _filterSegmentProperty;

    /**
     *
     */
    private FetchType _fetchType;

    /**
     *
     */
    private CascadeType[] _cascadeType;

    /**
     *
     */
    private Navigability _navigability;

    /**
     *
     */
    private MasterDetailView _masterDetailView;

    /**
     *
     */
    private QuickAddingFilter _quickAddingFilter;

    /**
     *
     */
    private boolean _keyPropertiesQueryMappingEnabled;

    /**
     *
     */
    private Object _initialValue;

    /**
     *
     */
    private Object _defaultValue;

    /**
     *
     */
    private Object _currentValue;

    /**
     *
     */
    private Object _orderBy;

    /**
     *
     */
    private Tab _defaultTab;

    /**
     * annotated with AbstractClass
     */
    private boolean _annotatedWithAbstractClass;

    /**
     * annotated with AllocationOverride
     */
    private boolean _annotatedWithAllocationOverride;

    /**
     * annotated with AllocationOverrides
     */
    private boolean _annotatedWithAllocationOverrides;

    /**
     * annotated with EntityPrimaryKey
     */
    private boolean _annotatedWithEntityPrimaryKey;

    /**
     * annotated with EntitySequenceProperty
     */
    private boolean _annotatedWithEntitySequenceProperty;

    /**
     * annotated with EntityVersionProperty
     */
    private boolean _annotatedWithEntityVersionProperty;

    /**
     * annotated with EntityNumericKey
     */
    private boolean _annotatedWithEntityNumericKey;

    /**
     * annotated with EntityCharacterKey
     */
    private boolean _annotatedWithEntityCharacterKey;

    /**
     * annotated with EntityNameProperty
     */
    private boolean _annotatedWithEntityNameProperty;

    /**
     * annotated with EntityDescriptionProperty
     */
    private boolean _annotatedWithEntityDescriptionProperty;

    /**
     * annotated with EntityInactiveIndicator
     */
    private boolean _annotatedWithEntityInactiveIndicator;

    /**
     * annotated with EntityUrlProperty
     */
    private boolean _annotatedWithEntityUrlProperty;

    /**
     * annotated with EntityParentProperty
     */
    private boolean _annotatedWithEntityParentProperty;

    /**
     * annotated with EntityOwnerProperty
     */
    private boolean _annotatedWithEntityOwnerProperty;

    /**
     * annotated with EntitySegmentProperty
     */
    private boolean _annotatedWithEntitySegmentProperty;

    /**
     * annotated with EntityBusinessKey
     */
    private boolean _annotatedWithEntityBusinessKey;

    /**
     * annotated with EntityClass
     */
    private boolean _annotatedWithEntityClass;

    /**
     * annotated with EntityDataGen
     */
    private boolean _annotatedWithEntityDataGen;

    /**
     * annotated with EntitySelectOperation
     */
    private boolean _annotatedWithEntitySelectOperation;

    /**
     * annotated with EntityInsertOperation
     */
    private boolean _annotatedWithEntityInsertOperation;

    /**
     * annotated with EntityUpdateOperation
     */
    private boolean _annotatedWithEntityUpdateOperation;

    /**
     * annotated with EntityDeleteOperation
     */
    private boolean _annotatedWithEntityDeleteOperation;

    /**
     * annotated with EntityReportOperation
     */
    private boolean _annotatedWithEntityReportOperation;

    /**
     * annotated with EntityExportOperation
     */
    private boolean _annotatedWithEntityExportOperation;

    /**
     * annotated with EntityTableView
     */
    private boolean _annotatedWithEntityTableView;

    /**
     * annotated with EntityDetailView
     */
    private boolean _annotatedWithEntityDetailView;

    /**
     * annotated with EntityTreeView
     */
    private boolean _annotatedWithEntityTreeView;

    /**
     * annotated with EntityConsoleView
     */
    private boolean _annotatedWithEntityConsoleView;

    /**
     * annotated with EntityWarnings
     */
    private boolean _annotatedWithEntityWarnings;

    /**
     * annotated with EntityCodeGen
     */
    private boolean _annotatedWithEntityCodeGen;

    /**
     * annotated with EntityDocGen
     */
    private boolean _annotatedWithEntityDocGen;

    /**
     * annotated with EntityReferenceSearch
     */
    private boolean _annotatedWithEntityReferenceSearch;

    /**
     * annotated with Filter
     */
    private boolean _annotatedWithFilter;

    /**
     * annotated with Extension
     */
    private boolean _annotatedWithExtension;

    /**
     * annotated with OneToOne
     */
    private boolean _annotatedWithOneToOne;

    /**
     * annotated with ManyToOne
     */
    private boolean _annotatedWithManyToOne;

    /**
     * annotated with QueryMapping
     */
    private boolean _annotatedWithQueryMapping;

    /**
     * annotated with EntityReferenceDataGen
     */
    private boolean _annotatedWithEntityReferenceDataGen;
    // </editor-fold>

    // <editor-fold defaultstate="collapsed" desc="field getters and setters">
    /**
     * @return the initialised indicator
     */
    @Override
    public boolean isInitialised() {
        return _initialised;
    }

    /**
     * @return the settled indicator
     */
    @Override
    public boolean isSettled() {
        return _settled;
    }

    /**
     * @return the finalised indicator
     */
    @Override
    public boolean isFinalised() {
        return _finalised;
    }

    /**
     * @return the finished indicator
     */
    @Override
    public boolean isFinished() {
        return _finished;
    }

    /**
     * @return the settler
     */
    @Override
    protected char settler() {
        return _settler;
    }

    /**
     * @return the root-instance indicator
     */
    @Override
    public boolean isRootInstance() {
        return _rootInstance;
    }

    /**
     * @return the explicitly declared indicator
     */
    @Override
    public boolean isExplicitlyDeclared() {
        return _explicitlyDeclared;
    }

    /**
     * @return the implicitly declared indicator
     */
    @Override
    public boolean isImplicitlyDeclared() {
        return _implicitlyDeclared;
    }

    /**
     * @return the reference index
     */
    @Override
    public int getReferenceIndex() {
        return _referenceIndex;
    }

    /**
     * @param referenceIndex the reference index to set
     */
    void setReferenceIndex(int referenceIndex) {
        _referenceIndex = referenceIndex;
    }

    /**
     * @param reference the referenced entity
     * @return the default label
     */
    @Override
    public String getDefaultLabel(EntityReference reference) {
        return getLocalizedLabel(null, reference);
    }

    /**
     * El método setDefaultLabel se utiliza para establecer la etiqueta de la entidad en las vistas (páginas) Maestro/Detalle, con el maestro
     * identificado por el parámetro reference, que se almacena en el archivo de recursos por defecto. En caso de que el archivo de recursos para el
     * idioma seleccionado por el usuario no esté disponible, la interfaz de la aplicación utiliza el archivo de recursos por defecto para obtener el
     * valor de la etiqueta.
     *
     * @param reference entidad que hace la función de maestro en la vista Maestro/Detalle
     * @param defaultLabel sustantivo singular que se usa como etiqueta de la entidad
     */
//  @Override
    public void setDefaultLabel(EntityReference reference, String defaultLabel) {
        setLocalizedLabel(null, reference, defaultLabel);
    }

    /**
     * @param reference the referenced entity
     * @return the default short label
     */
    @Override
    public String getDefaultShortLabel(EntityReference reference) {
        return getLocalizedShortLabel(null, reference);
    }

    /**
     * El método setDefaultShortLabel se utiliza para establecer la etiqueta corta de la entidad en las vistas (páginas) Maestro/Detalle, con el
     * maestro identificado por el parámetro reference, que se almacena en el archivo de recursos por defecto. En caso de que el archivo de recursos
     * para el idioma seleccionado por el usuario no esté disponible, la interfaz de la aplicación utiliza el archivo de recursos por defecto para
     * obtener el valor de la etiqueta.
     *
     * @param reference entidad que hace la función de maestro en la vista Maestro/Detalle
     * @param defaultShortLabel sustantivo singular, preferiblemente sin complementos, que se usa como etiqueta corta de la entidad
     */
//  @Override
    public void setDefaultShortLabel(EntityReference reference, String defaultShortLabel) {
        setLocalizedShortLabel(null, reference, defaultShortLabel);
    }

    /**
     * @param reference the referenced entity
     * @return the default collection label
     */
    @Override
    public String getDefaultCollectionLabel(EntityReference reference) {
        return getLocalizedCollectionLabel(null, reference);
    }

    /**
     * El método setDefaultCollectionLabel se utiliza para establecer la etiqueta de colección de la entidad en las vistas (páginas) Maestro/Detalle,
     * con el maestro identificado por el parámetro reference, que se almacena en el archivo de recursos por defecto. En caso de que el archivo de
     * recursos para el idioma seleccionado por el usuario no esté disponible, la interfaz de la aplicación utiliza el archivo de recursos por defecto
     * para obtener el valor de la etiqueta.
     *
     * @param reference entidad que hace la función de maestro en la vista Maestro/Detalle
     * @param defaultCollectionLabel sustantivo plural que se usa como etiqueta de colección de la entidad
     */
//  @Override
    public void setDefaultCollectionLabel(EntityReference reference, String defaultCollectionLabel) {
        setLocalizedCollectionLabel(null, reference, defaultCollectionLabel);
    }

    /**
     * @param reference the referenced entity
     * @return the default collection short label
     */
    @Override
    public String getDefaultCollectionShortLabel(EntityReference reference) {
        return getLocalizedCollectionShortLabel(null, reference);
    }

    /**
     * El método setDefaultCollectionShortLabel se utiliza para establecer la etiqueta corta de colección de la entidad en las vistas (páginas)
     * Maestro/Detalle, con el maestro identificado por el parámetro reference, que se almacena en el archivo de recursos por defecto. En caso de que
     * el archivo de recursos para el idioma seleccionado por el usuario no esté disponible, la interfaz de la aplicación utiliza el archivo de
     * recursos por defecto para obtener el valor de la etiqueta.
     *
     * @param reference entidad que hace la función de maestro en la vista Maestro/Detalle
     * @param defaultCollectionShortLabel sustantivo plural, preferiblemente sin complementos, que se usa como etiqueta corta de colección de la
     * entidad
     */
//  @Override
    public void setDefaultCollectionShortLabel(EntityReference reference, String defaultCollectionShortLabel) {
        setLocalizedCollectionShortLabel(null, reference, defaultCollectionShortLabel);
    }

    /**
     * @param locale the locale for the label
     * @param reference the referenced entity
     * @return the localized label
     */
    @Override
    public String getLocalizedLabel(Locale locale, EntityReference reference) {
        LocaleEntityReference ler = localeEntityReferenceReadingKey(locale, reference);
        return _localizedLabelByReferenceMap.get(ler);
    }

    /**
     * El método setLocalizedLabel se utiliza para establecer la etiqueta de la entidad en las vistas (páginas) Maestro/Detalle, con el maestro
     * identificado por el parámetro reference, que se almacena en el archivo de recursos de configuración regional. En caso de que el archivo de
     * recursos para el idioma seleccionado por el usuario no esté disponible, la interfaz de la aplicación utiliza el archivo de recursos por defecto
     * para obtener el valor de la etiqueta.
     *
     * @param locale configuración regional
     * @param reference entidad que hace la función de maestro en la vista Maestro/Detalle
     * @param localizedLabel sustantivo singular que se usa como etiqueta de la entidad
     */
//  @Override
    public void setLocalizedLabel(Locale locale, EntityReference reference, String localizedLabel) {
        LocaleEntityReference ler = localeEntityReferenceWritingKey(locale, reference);
        if (localizedLabel == null) {
            _localizedLabelByReferenceMap.remove(ler);
        } else {
            _localizedLabelByReferenceMap.put(ler, localizedLabel);
        }
    }

    /**
     * @param locale the locale for the label
     * @param reference the referenced entity
     * @return the localized short label
     */
    @Override
    public String getLocalizedShortLabel(Locale locale, EntityReference reference) {
        LocaleEntityReference ler = localeEntityReferenceReadingKey(locale, reference);
        return _localizedShortLabelByReferenceMap.get(ler);
    }

    /**
     * El método setLocalizedShortLabel se utiliza para establecer la etiqueta corta de la entidad en las vistas (páginas) Maestro/Detalle, con el
     * maestro identificado por el parámetro reference, que se almacena en el archivo de recursos de configuración regional. En caso de que el archivo
     * de recursos para el idioma seleccionado por el usuario no esté disponible, la interfaz de la aplicación utiliza el archivo de recursos por
     * defecto para obtener el valor de la etiqueta.
     *
     * @param locale configuración regional
     * @param reference entidad que hace la función de maestro en la vista Maestro/Detalle
     * @param localizedShortLabel sustantivo singular, preferiblemente sin complementos, que se usa como etiqueta corta de la entidad
     */
//  @Override
    public void setLocalizedShortLabel(Locale locale, EntityReference reference, String localizedShortLabel) {
        LocaleEntityReference ler = localeEntityReferenceWritingKey(locale, reference);
        if (localizedShortLabel == null) {
            _localizedShortLabelByReferenceMap.remove(ler);
        } else {
            _localizedShortLabelByReferenceMap.put(ler, localizedShortLabel);
        }
    }

    /**
     * @param locale the locale for the label
     * @param reference the referenced entity
     * @return the localized collection label
     */
    @Override
    public String getLocalizedCollectionLabel(Locale locale, EntityReference reference) {
        LocaleEntityReference ler = localeEntityReferenceReadingKey(locale, reference);
        return _localizedCollectionLabelByReferenceMap.get(ler);
    }

    /**
     * El método setLocalizedCollectionLabel se utiliza para establecer la etiqueta de colección de la entidad en las vistas (páginas)
     * Maestro/Detalle, con el maestro identificado por el parámetro reference, que se almacena en el archivo de recursos de configuración regional.
     * En caso de que el archivo de recursos para el idioma seleccionado por el usuario no esté disponible, la interfaz de la aplicación utiliza el
     * archivo de recursos por defecto para obtener el valor de la etiqueta.
     *
     * @param locale configuración regional
     * @param reference entidad que hace la función de maestro en la vista Maestro/Detalle
     * @param localizedCollectionLabel sustantivo plural que se usa como etiqueta de colección de la entidad
     */
//  @Override
    public void setLocalizedCollectionLabel(Locale locale, EntityReference reference, String localizedCollectionLabel) {
        LocaleEntityReference ler = localeEntityReferenceWritingKey(locale, reference);
        if (localizedCollectionLabel == null) {
            _localizedCollectionLabelByReferenceMap.remove(ler);
        } else {
            _localizedCollectionLabelByReferenceMap.put(ler, localizedCollectionLabel);
        }
    }

    /**
     * @param locale the locale for the label
     * @param reference the referenced entity
     * @return the localized collection short label
     */
    @Override
    public String getLocalizedCollectionShortLabel(Locale locale, EntityReference reference) {
        LocaleEntityReference ler = localeEntityReferenceReadingKey(locale, reference);
        return _localizedCollectionShortLabelByReferenceMap.get(ler);
    }

    /**
     * El método setLocalizedCollectionShortLabel se utiliza para establecer la etiqueta corta de colección de la entidad en las vistas (páginas)
     * Maestro/Detalle, con el maestro identificado por el parámetro reference, que se almacena en el archivo de recursos de configuración regional.
     * En caso de que el archivo de recursos para el idioma seleccionado por el usuario no esté disponible, la interfaz de la aplicación utiliza el
     * archivo de recursos por defecto para obtener el valor de la etiqueta.
     *
     * @param locale configuración regional
     * @param reference entidad que hace la función de maestro en la vista Maestro/Detalle
     * @param localizedCollectionShortLabel sustantivo plural, preferiblemente sin complementos, que se usa como etiqueta corta de colección de la
     * entidad
     */
//  @Override
    public void setLocalizedCollectionShortLabel(Locale locale, EntityReference reference, String localizedCollectionShortLabel) {
        LocaleEntityReference ler = localeEntityReferenceWritingKey(locale, reference);
        if (localizedCollectionShortLabel == null) {
            _localizedCollectionShortLabelByReferenceMap.remove(ler);
        } else {
            _localizedCollectionShortLabelByReferenceMap.put(ler, localizedCollectionShortLabel);
        }
    }

    @Override
    public List<Property> getPropertiesList() {
        return _atlas.getPropertiesList();
    }

    @Override
    public List<Property> getReferencesList() {
        return _atlas.getReferencesList();
    }

    @Override
    public List<Parameter> getParameterReferencesList() {
        return _atlas.getParameterReferencesList();
    }

    @Override
    public List<Key> getKeysList() {
        return _atlas.getKeysList();
    }

    @Override
    public List<Tab> getTabsList() {
        return _atlas.getTabsList();
    }

    @Override
    public List<View> getViewsList() {
        return _atlas.getViewsList();
    }

    @Override
    public List<Instance> getInstancesList() {
        return _atlas.getInstancesList();
    }

    @Override
    public List<NamedValue> getNamedValuesList() {
        return _atlas.getNamedValuesList();
    }

    @Override
    public List<Expression> getExpressionsList() {
        return _atlas.getExpressionsList();
    }

    @Override
    public List<Transition> getTransitionsList() {
        return _atlas.getTransitionsList();
    }

    @Override
    public List<Operation> getOperationsList() {
        return _atlas.getOperationsList();
    }

    @Override
    public List<Class<?>> getOperationClassesList() {
        return new ArrayList<>(getOperationClassesMap().values());
    }

    @Override
    public List<Trigger> getTriggersList() {
        return _atlas.getTriggersList();
    }

    @Override
    public List<Property> getOverlayPropertiesList() {
        return new ArrayList<>(getOverlayPropertiesMap().values());
    }

    @Override
    public List<Property> getQueryPropertiesList() {
        return new ArrayList<>(getQueryPropertiesMap().values());
    }

    @Override
    public Map<String, Property> getPropertiesMap() {
        return _atlas.getPropertiesMap();
    }

    @Override
    public Map<String, Property> getReferencesMap() {
        return _atlas.getReferencesMap();
    }

    @Override
    public Map<String, Parameter> getParameterReferencesMap() {
        return _atlas.getParameterReferencesMap();
    }

    @Override
    public Map<String, Key> getKeysMap() {
        return _atlas.getKeysMap();
    }

    @Override
    public Map<String, Tab> getTabsMap() {
        return _atlas.getTabsMap();
    }

    @Override
    public Map<String, View> getViewsMap() {
        return _atlas.getViewsMap();
    }

    @Override
    public Map<String, Instance> getInstancesMap() {
        return _atlas.getInstancesMap();
    }

    @Override
    public Map<String, NamedValue> getNamedValuesMap() {
        return _atlas.getNamedValuesMap();
    }

    @Override
    public Map<String, Expression> getExpressionsMap() {
        return _atlas.getExpressionsMap();
    }

    @Override
    public Map<String, Transition> getTransitionsMap() {
        return _atlas.getTransitionsMap();
    }

    @Override
    public Map<String, Operation> getOperationsMap() {
        return _atlas.getOperationsMap();
    }

    @Override
    public Map<String, Class<?>> getOperationClassesMap() {
        return _atlas.getOperationClassesMap();
    }

    @Override
    public Map<String, Trigger> getTriggersMap() {
        return _atlas.getTriggersMap();
    }

    @Override
    public Map<String, Property> getOverlayPropertiesMap() {
        Map<String, Property> map = new LinkedHashMap<>();
        List<Property> list = getPropertiesList();
        if (list.isEmpty()) {
            return map;
        }
        for (Property property : list) {
            if (property.isOverlayField()) {
                map.put(property.getPathString(), property);
            }
        }
        return map;
    }

    @Override
    public Map<String, Property> getQueryPropertiesMap() {
        Map<String, Property> map = new LinkedHashMap<>();
        List<Property> list = getPropertiesList();
        if (list.isEmpty()) {
            return map;
        }
        for (Property property : list) {
            mapQueryProperty(map, property);
        }
        for (Property property : list) {
            if (property.isCalculable()) {
                continue;
            }
            if (property instanceof EntityReference) {
                mapEntityReferenceKeyProperties(map, (EntityReference) property);
            }
        }
        if (_inactiveIndicatorProperty != null) {
            mapQueryProperty(map, _inactiveIndicatorProperty);
        }
        if (_ownerProperty != null) {
            mapQueryProperty(map, _ownerProperty);
            mapEntityReferenceKeyProperties(map, _ownerProperty);
        }
        if (_segmentProperty != null) {
            mapQueryProperty(map, _segmentProperty);
            mapEntityReferenceKeyProperties(map, _segmentProperty);
        }
        if (_orderBy != null) {
            Property orderByProperty = getOrderByProperty();
            if (orderByProperty != null) {
                mapQueryProperty(map, orderByProperty);
            } else {
                Property[] orderByProperties = getOrderByProperties();
                if (orderByProperties != null) {
                    for (Property property : orderByProperties) {
                        if (property != null) {
                            mapQueryProperty(map, property);
                        }
                    }
                }
            }
        }
        List<View> views = getViewsList();
        if (views != null && !views.isEmpty()) {
            Property column;
            List<ViewField> fields;
            for (View view : views) {
                fields = view.getViewFields();
                if (fields != null && !fields.isEmpty()) {
                    for (ViewField field : fields) {
                        column = field.getColumn();
                        if (column != null) {
                            mapQueryProperty(map, column);
                        }
                    }
                }
            }
        }
        Property foreignQueryProperty;
        for (String key : _foreignQueryProperties.keySet()) {
            foreignQueryProperty = _foreignQueryProperties.get(key);
            mapQueryProperty(map, foreignQueryProperty);
        }
        return map;
    }

    private void mapQueryProperty(Map<String, Property> map, Property property) {
        String key = property.getPathString();
        if (map.containsKey(key)) {
        } else {
            map.put(key, property);
            if (property.isCalculable()) {
                if (property instanceof CalculableProperty) {
                    mapCalculableQueryProperty(map, (CalculableProperty) property);
                }
            }
        }
    }

    private void mapCalculableQueryProperty(Map<String, Property> map, CalculableProperty calculableProperty) {
        Object calculableValue = calculableProperty.getCalculableValue();
        if (calculableValue == null) {
        } else if (calculableValue instanceof Property) {
            boolean calculable = false;
            Property property = (Property) calculableValue;
            List<Artifact> pathList = property.getPathList();
            for (Artifact artifact : pathList) {
                if (artifact instanceof Property) {
                    calculable = ((Property) artifact).isCalculable();
                    if (calculable) {
                        break;
                    }
                }
            }
            if (calculable) {
            } else {
                mapQueryProperty(map, property);
            }
        } else if (calculableValue instanceof Expression) {
            Expression calculableValueExpression = (Expression) calculableValue;
            boolean ok = verifyExpression(calculableValueExpression, calculableProperty);
            if (ok) {
                List<Property> referencedColumnsList = calculableValueExpression.getReferencedColumnsList();
                for (Property property : referencedColumnsList) {
                    mapQueryProperty(map, property);
                }
            }
        }
    }

    private void mapEntityReferenceKeyProperties(Map<String, Property> map, EntityReference reference) {
        if (reference.isVariantEntity()) {
        } else if (reference.isKeyPropertiesQueryMappingEnabled()) {
            boolean enumeration = reference.isEnumerationEntity();
            List<Property> list = reference.getPropertiesList();
            for (Property property : list) {
                if ((property.isPrimaryKeyProperty() && enumeration) || property.isBusinessKeyProperty() || property.isNameProperty()) {
                    mapQueryProperty(map, property);
                }
            }
        }
    }

    /**
     * @return true if the entity is an abstract entity class
     */
    @Override
    public boolean isAbstractClass() {
        return _annotatedWithAbstractClass;
    }

    /**
     * @return the baseClass
     */
    @Override
    public Class<?> getBaseClass() {
        return _baseClass;
    }

    /**
     * @return the direct known subclasses list
     */
    @Override
    public List<Class<?>> getSubclassesList() {
        return new ArrayList<>(getSubclassesMap().values());
    }

    /**
     * @return the direct known subclasses map
     */
    @Override
    public Map<String, Class<?>> getSubclassesMap() {
        return _subclasses;
    }

    /**
     * @return the allocation overrides list
     */
    @Override
    public List<AllocationOverride> getAllocationOverridesList() {
        return new ArrayList<>(getAllocationOverridesMap().values());
    }

    /**
     * @return the allocation overrides map
     */
    @Override
    public Map<String, AllocationOverride> getAllocationOverridesMap() {
        return _allocationOverrides;
    }

    /**
     * @return the primaryKeyFieldName
     */
    @Override
    public String getPrimaryKeyFieldName() {
        return _primaryKeyFieldName;
    }

    /**
     * @return the primaryKeyField
     */
    @Override
    public Field getPrimaryKeyField() {
        return _primaryKeyField;
    }

    /**
     * @return the primaryKeyProperty
     */
    @Override
    public Property getPrimaryKeyProperty() {
        return _primaryKeyProperty;
    }

    /**
     * @return the sequenceFieldName
     */
    @Override
    public String getSequenceFieldName() {
        return _sequenceFieldName;
    }

    /**
     * @return the sequenceField
     */
    @Override
    public Field getSequenceField() {
        return _sequenceField;
    }

    /**
     * @return the sequenceProperty
     */
    @Override
    public LongProperty getSequenceProperty() {
        return _sequenceProperty;
    }

    /**
     * @return the versionFieldName
     */
    @Override
    public String getVersionFieldName() {
        return _versionFieldName;
    }

    /**
     * @return the versionField
     */
    @Override
    public Field getVersionField() {
        return _versionField;
    }

    /**
     * @return the versionProperty
     */
    @Override
    public LongProperty getVersionProperty() {
        return _versionProperty;
    }

    /**
     * @return the numericKeyFieldName
     */
    @Override
    public String getNumericKeyFieldName() {
        return _numericKeyFieldName;
    }

    /**
     * @return the numericKeyField
     */
    @Override
    public Field getNumericKeyField() {
        return _numericKeyField;
    }

    /**
     * @return the numericKeyProperty
     */
    @Override
    public IntegerProperty getNumericKeyProperty() {
        return _numericKeyProperty;
    }

    /**
     * @return the characterKeyFieldName
     */
    @Override
    public String getCharacterKeyFieldName() {
        return _characterKeyFieldName;
    }

    /**
     * @return the characterKeyField
     */
    @Override
    public Field getCharacterKeyField() {
        return _characterKeyField;
    }

    /**
     * @return the characterKeyProperty
     */
    @Override
    public StringProperty getCharacterKeyProperty() {
        return _characterKeyProperty;
    }

    /**
     * @return the nameFieldName
     */
    @Override
    public String getNameFieldName() {
        return _nameFieldName;
    }

    /**
     * @return the nameField
     */
    @Override
    public Field getNameField() {
        return _nameField;
    }

    /**
     * @return the nameProperty
     */
    @Override
    public StringProperty getNameProperty() {
        return _nameProperty;
    }

    /**
     * @return the descriptionFieldName
     */
    @Override
    public String getDescriptionFieldName() {
        return _descriptionFieldName;
    }

    /**
     * @return the descriptionField
     */
    @Override
    public Field getDescriptionField() {
        return _descriptionField;
    }

    /**
     * @return the descriptionProperty
     */
    @Override
    public StringProperty getDescriptionProperty() {
        return _descriptionProperty;
    }

    /**
     * @return the imageFieldName
     */
    @Override
    public String getImageFieldName() {
        return _imageFieldName;
    }

    /**
     * @return the imageField
     */
    @Override
    public Field getImageField() {
        return _imageField;
    }

    /**
     * @return the imageProperty
     */
    @Override
    public BinaryProperty getImageProperty() {
        return _imageProperty;
    }

    /**
     * @return the inactiveIndicatorFieldName
     */
    @Override
    public String getInactiveIndicatorFieldName() {
        return _inactiveIndicatorFieldName;
    }

    /**
     * @return the inactiveIndicatorField
     */
    @Override
    public Field getInactiveIndicatorField() {
        return _inactiveIndicatorField;
    }

    /**
     * @return the inactive indicator property
     */
    @Override
    public BooleanProperty getInactiveIndicatorProperty() {
        return _inactiveIndicatorProperty;
    }

    /**
     * @return the urlFieldName
     */
    @Override
    public String getUrlFieldName() {
        return _urlFieldName;
    }

    /**
     * @return the urlField
     */
    @Override
    public Field getUrlField() {
        return _urlField;
    }

    /**
     * @return the url property
     */
    @Override
    public StringProperty getUrlProperty() {
        return _urlProperty;
    }

    /**
     * @return the parentFieldName
     */
    @Override
    public String getParentFieldName() {
        return _parentFieldName;
    }

    /**
     * @return the parentField
     */
    @Override
    public Field getParentField() {
        return _parentField;
    }

    /**
     * @return the parentProperty
     */
    @Override
    public Entity getParentProperty() {
        return _parentProperty;
    }

    /**
     * @return the ownerFieldName
     */
    @Override
    public String getOwnerFieldName() {
        return _ownerFieldName;
    }

    /**
     * @return the ownerField
     */
    @Override
    public Field getOwnerField() {
        return _ownerField;
    }

    /**
     * @return the ownerProperty
     */
    @Override
    public Entity getOwnerProperty() {
        return _ownerProperty;
    }

    /**
     * @return the segmentFieldName
     */
    @Override
    public String getSegmentFieldName() {
        return _segmentFieldName;
    }

    /**
     * @return the segmentField
     */
    @Override
    public Field getSegmentField() {
        return _segmentField;
    }

    /**
     * @return the segmentProperty
     */
    @Override
    public Entity getSegmentProperty() {
        return _segmentProperty;
    }

    /**
     * @return the businessKeyFieldName
     */
    @Override
    public String getBusinessKeyFieldName() {
        return _businessKeyFieldName;
    }

    /**
     * @return the businessKeyField
     */
    @Override
    public Field getBusinessKeyField() {
        return _businessKeyField;
    }

    /**
     * @return the businessKeyProperty
     */
    @Override
    public Property getBusinessKeyProperty() {
        return _businessKeyProperty;
    }

    /**
     * @return the stateFieldName
     */
    @Override
    public String getStateFieldName() {
        return _stateFieldName;
    }

    /**
     * @return the stateField
     */
    @Override
    public Field getStateField() {
        return _stateField;
    }

    /**
     * @return the stateProperty
     */
    @Override
    public Entity getStateProperty() {
        return _stateProperty;
    }

    /**
     * @return true if the inactive indicator is a foreign property
     */
    public boolean isForeignInactiveIndicatorProperty() {
        return _foreignInactiveIndicatorProperty;
    }

    /**
     * @return true if the owner is a foreign property
     */
    public boolean isForeignOwnerProperty() {
        return _foreignOwnerProperty;
    }

    /**
     * @return true if the segment is a foreign property
     */
    public boolean isForeignSegmentProperty() {
        return _foreignSegmentProperty;
    }

    /**
     * @return the businessKeyType
     */
    @Override
    public BusinessKeyType getBusinessKeyType() {
        return _businessKeyType;
    }

    /**
     * @return the basic operation entity indicator
     */
    @Override
    public boolean isBasicOperationEntity() {
        return _basicOperationEntity;
    }

    /**
     * @return the catalog entity indicator
     */
    @Override
    public boolean isCatalogEntity() {
        return _catalogEntity;
    }

    /**
     * @return the existentially independent indicator
     */
    @Override
    public boolean isExistentiallyIndependent() {
        return _existentiallyIndependent;
    }

    /**
     * @return the variant entity indicator
     */
    @Override
    public boolean isVariantEntity() {
        return _variantEntity;
    }

    public boolean isInvariantEntity() {
        return !isVariantEntity();
    }

    /**
     * @return the resource type
     */
    @Override
    public ResourceType getResourceType() {
        return _resourceType != null && !_resourceType.equals(ResourceType.UNSPECIFIED) ? _resourceType
            : isEnumerationEntity() ? ResourceType.CONFIGURATION : ResourceType.OPERATION;
    }

    /**
     * @return the resource gender
     */
    @Override
    public ResourceGender getResourceGender() {
        return _resourceGender;
    }

    /**
     * @return the properties prefix
     */
    @Override
    public String getPropertiesPrefix() {
        return _propertiesPrefix;
    }

    /**
     * @return the properties suffix
     */
    @Override
    public String getPropertiesSuffix() {
        return _propertiesSuffix;
    }

    /**
     * @return the help file name
     */
    @Override
    public String getHelpFileName() {
        return _helpFileName;
    }

    /**
     * sets the help file name.
     *
     * @param helpFileName
     */
    @Override
    public void setHelpFileName(String helpFileName) {
        _helpFileName = helpFileName;
    }

    /**
     * @return the start-with
     */
    @Override
    public int getStartWith() {
        return _startWith;
    }

    /**
     * @return the series start
     */
//  @Override
    public int getSeriesStart() {
        return _seriesStart;
    }

    /**
     * @return the series stop
     */
//  @Override
    public int getSeriesStop() {
        return _seriesStop;
    }

    /**
     * @return the series step
     */
//  @Override
    public int getSeriesStep() {
        return _seriesStep;
    }

    /**
     * @return the entity data generation enabled indicator
     */
    public boolean isDataGenEnabled() {
        return _annotatedWithEntityDataGen && _seriesStart <= _seriesStop && _seriesStep > 0;
    }

    /**
     * @return the select enabled indicator
     */
    @Override
    public boolean isSelectEnabled() {
        return _selectEnabled;
    }

//  /**
//   * @return the select restricted indicator
//   */
//  @Override
//  public boolean isSelectRestricted() {
//      return _selectRestricted;
//  }
//
    /**
     * @return the select operation access mode
     */
    @Override
    public OperationAccess isSelectOperationAccess() {
        return _selectOperationAccess;
    }

    /**
     * @return the select onload option
     */
    @Override
    public SelectOnloadOption getSelectOnloadOption() {
        return _selectOnloadOption != null && !_selectOnloadOption.equals(SelectOnloadOption.DEFAULT) ? _selectOnloadOption
            : isEnumerationEntity() ? SelectOnloadOption.EXECUTE : SelectOnloadOption.PROMPT;
    }

    /**
     * @return the select rows limit
     */
    @Override
    public int getSelectRowsLimit() {
        return _selectRowsLimit;
    }

    /**
     * @return the select sort option
     */
    @Override
    public SortOption getSelectSortOption() {
        return _selectSortOption;
    }

    /**
     * @return the insert enabled indicator
     */
    @Override
    public boolean isInsertEnabled() {
        return _insertEnabled && !_annotatedWithAbstractClass;
    }

//  /**
//   * @return the insert restricted indicator
//   */
//  @Override
//  public boolean isInsertRestricted() {
//      return _insertRestricted;
//  }
//
    /**
     * @return the insert operation access mode
     */
    @Override
    public OperationAccess isInsertOperationAccess() {
        return _insertOperationAccess;
    }

    /**
     * @return the insert logging mode
     */
    @Override
    public OperationLogging getInsertLogging() {
        return _insertLogging;
    }

    /**
     * @return the update enabled indicator
     */
    @Override
    public boolean isUpdateEnabled() {
        return _updateEnabled;
    }

//  /**
//   * @return the update restricted indicator
//   */
//  @Override
//  public boolean isUpdateRestricted() {
//      return _updateRestricted;
//  }
//
    /**
     * @return the update operation access mode
     */
    @Override
    public OperationAccess isUpdateOperationAccess() {
        return _updateOperationAccess;
    }

    /**
     * @return the update logging mode
     */
    @Override
    public OperationLogging getUpdateLogging() {
        return _updateLogging;
    }

    /**
     * @return the delete enabled indicator
     */
    @Override
    public boolean isDeleteEnabled() {
        return _deleteEnabled;
    }

//  /**
//   * @return the delete restricted indicator
//   */
//  @Override
//  public boolean isDeleteRestricted() {
//      return _deleteRestricted;
//  }
//
    /**
     * @return the delete operation access mode
     */
    @Override
    public OperationAccess isDeleteOperationAccess() {
        return _deleteOperationAccess;
    }

    /**
     * @return the delete logging mode
     */
    @Override
    public OperationLogging getDeleteLogging() {
        return _deleteLogging;
    }

    /**
     * @return the report enabled indicator
     */
    @Override
    public boolean isReportEnabled() {
        return _reportEnabled;
    }

    /**
     * @return the report rows limit
     */
    @Override
    public int getReportRowsLimit() {
        return _reportRowsLimit;
    }

    /**
     * @return the report sort option
     */
    @Override
    public SortOption getReportSortOption() {
        return _reportSortOption;
    }

    /**
     * @return the export enabled indicator
     */
    @Override
    public boolean isExportEnabled() {
        return _exportEnabled;
    }

    /**
     * @return the export rows limit
     */
    @Override
    public int getExportRowsLimit() {
        return _exportRowsLimit;
    }

    /**
     * @return the export sort option
     */
    @Override
    public SortOption getExportSortOption() {
        return _exportSortOption;
    }

    /**
     * @return the foreign entity class indicator
     */
    @Override
    public boolean isForeignEntityClass() {
        Set<Class<?>> foreignEntityClasses = TLC.getProject().getForeignEntityClasses();
        return foreignEntityClasses.contains(getNamedClass());
    }

    /**
     * @return the private entity class indicator
     */
    @Override
    public boolean isPrivateEntityClass() {
        Set<Class<?>> privateEntityClasses = TLC.getProject().getPrivateEntityClasses();
        return privateEntityClasses.contains(getNamedClass());
    }

    /**
     * @return the table-view enabled indicator
     */
    @Override
    public boolean isTableViewEnabled() {
        return BitUtils.valueOf(_tableViewEnabled) && !isPrivateEntityClass();
    }

    /**
     * @return the table-view-with-insert-enabled indicator
     */
//  @Override
    public boolean isTableViewWithInsertEnabled() {
        return _tableViewWithInsertEnabled && isInsertEnabled() && tableViewHasAtLeastOneCreateField();
    }

    private boolean tableViewHasAtLeastOneCreateField() {
        List<Property> list = getPropertiesList();
        for (Property property : list) {
            if (property.isTableField() && property.isCreateField()) {
                return true;
            }
        }
        return false;
    }

    /**
     * @return the table-view-with-update-enabled indicator
     */
//  @Override
    public boolean isTableViewWithUpdateEnabled() {
        return _tableViewWithUpdateEnabled && isUpdateEnabled() && tableViewHasAtLeastOneUpdateField();
    }

    private boolean tableViewHasAtLeastOneUpdateField() {
        List<Property> list = getPropertiesList();
        for (Property property : list) {
            if (property.isTableField() && property.isUpdateField()) {
                return true;
            }
        }
        return false;
    }

    /**
     * @return the table-view-with-delete-enabled indicator
     */
//  @Override
    public boolean isTableViewWithDeleteEnabled() {
        return _tableViewWithDeleteEnabled && isDeleteEnabled();
    }

    /**
     * @return the table-view-with-master-heading indicator
     */
//  @Override
    public boolean isTableViewWithMasterHeading() {
        return _tableViewWithMasterHeading;
    }

    /**
     * @return the table view rows limit
     */
//  @Override
    public int getTableViewRowsLimit() {
        return _tableViewRowsLimit;
    }

    /**
     * @return the table view rows
     */
//  @Override
    public int getTableViewRows() {
        return _tableViewRows;
    }
//
//  /**
//   * @return the table view width
//   */
////@Override
//  public int getTableViewWidth() {
//      return _tableViewWidth;
//  }

    /**
     * @return the detail-view enabled indicator
     */
    @Override
    public boolean isDetailViewEnabled() {
        return _detailViewEnabled && !isPrivateEntityClass();
    }

    /**
     * @return the detail-view-with-master-heading indicator
     */
//  @Override
    public boolean isDetailViewWithMasterHeading() {
        return _detailViewWithMasterHeading;
    }
//
//  /**
//   * @return the detail view width
//   */
////@Override
//  public int getDetailViewWidth() {
//      return _detailViewWidth;
//  }

    /**
     * @return the tree-view enabled indicator
     */
    @Override
    public boolean isTreeViewEnabled() {
        return _treeViewEnabled && _parentProperty != null && _ownerProperty == null && _segmentProperty == null && !isPrivateEntityClass();
    }

    /**
     * @return the console-view enabled indicator
     */
    @Override
    public boolean isConsoleViewEnabled() {
        return _consoleViewEnabled && !isPrivateEntityClass();
    }
//
//  /**
//   * @return the console view width
//   */
////@Override
//  public int getConsoleViewWidth() {
//      return _consoleViewWidth;
//  }

    /**
     * @return the warnings enabled indicator
     */
    @Override
    public boolean isWarningsEnabled() {
        return _warningsEnabled;
    }

    /**
     * @return the business process logic code generation enabled indicator
     */
    @Override
    public boolean isBplCodeGenEnabled() {
        return _bplCodeGenEnabled;
    }

    /**
     * @return the business web service code generation enabled indicator
     */
    @Override
    public boolean isBwsCodeGenEnabled() {
        return _bwsCodeGenEnabled;
    }

    /**
     * @return the facade web service code generation enabled indicator
     */
    @Override
    public boolean isFwsCodeGenEnabled() {
        return _fwsCodeGenEnabled;
    }

    /**
     * @return the SQL code generation enabled indicator
     */
    @Override
    public boolean isSqlCodeGenEnabled() {
        return _sqlCodeGenEnabled && !isForeignEntityClass();
    }

    /**
     * @return the entity state machine code generation enabled indicator
     */
    @Override
    public boolean isEntityStateCodeGenEnabled() {
        return _entityStateCodeGenEnabled && _stateProperty != null && !_stateProperty.isInherited();
    }

    /**
     * @return the entity class diagram generation indicator
     */
    @Override
    public boolean isEntityClassDiagramGenEnabled() {
        return _entityClassDiagramGenEnabled;
    }

    /**
     * @return the entity state diagram generation indicator
     */
    @Override
    public boolean isEntityStateDiagramGenEnabled() {
        return _entityStateDiagramGenEnabled && !getStatesMap().isEmpty();
    }

    /**
     * @return the entity insert activity diagram generation indicator
     */
    @Override
    public boolean isEntityInsertActivityDiagramGenEnabled() {
        return _entityInsertActivityDiagramGenEnabled;
    }

    /**
     * @return the entity update activity diagram generation indicator
     */
    @Override
    public boolean isEntityUpdateActivityDiagramGenEnabled() {
        return _entityUpdateActivityDiagramGenEnabled;
    }

    /**
     * @return the search type
     */
    @Override
    public SearchType getSearchType() {
        return _searchType != null && !_searchType.equals(SearchType.UNSPECIFIED) ? _searchType
            : isEnumerationEntity() ? SearchType.LIST : SearchType.DISPLAY;
    }

    /**
     * @return the list style
     */
    @Override
    public ListStyle getListStyle() {
        return _listStyle;
    }

    /**
     * @return the search display mode
     */
    @Override
    public DisplayMode getSearchDisplayMode() {
        return _searchDisplayMode;
    }

    /**
     * @return the search query filter
     */
    @Override
    public BooleanExpression getSearchQueryFilter() {
        BooleanExpression exp = isInstanceReferenceField() ? defaultInstanceParameterSearchQueryFilter() : null;
        if (_searchQueryFilter == null) {
            if (exp != null) {
                return exp;
            }
            if (_filterInactiveIndicatorProperty || _filterOwnerProperty || _filterSegmentProperty) {
                return isNotNull();
            }
            if (QuickAddingFilter.MISSING.equals(_quickAddingFilter)) {
                return isNotNull();
            }
        }
        if (exp != null) {
            return exp.and(_searchQueryFilter);
        }
        return _searchQueryFilter;
    }

    private BooleanExpression defaultInstanceParameterSearchQueryFilter() {
        Operation operation = getDeclaringOperation();
        List<State> list = operation.getInitialStatesList();
        int size = list.size();
        if (size == 0) {
            return null;
        }
        Expression exp;
        Map<String, Expression> map = getExpressionsMap();
        State[] array = new State[size];
        int i = 0;
        for (State state : list) {
            exp = map.get(state.getName());
            array[i++] = (State) exp;
        }
        switch (array.length) {
            case 1:
                return array[0];
            case 2:
                return or(array[0], array[1]);
            default:
                return or(array[0], array[1], (State[]) ArrayUtils.subarray(array, 2, array.length));
        }
    }

    /**
     * El método setSearchQueryFilter se utiliza para establecer el filtro de búsqueda del valor de la referencia (propiedad o parámetro que hace
     * referencia a otra entidad) en vistas (páginas) de registro o de ejecución de operaciones de negocio de la entidad.
     *
     * @param filter expresión booleana que se utiliza como filtro; solo las instancias que satisfacen los criterios del filtro se pueden utilizar
     * como valor de la propiedad o parámetro
     */
    public void setSearchQueryFilter(BooleanExpression filter) {
        String message = "failed to set search query filter of " + getFullName();
        if (filter == null) {
            message += "; supplied expression is null";
            logger.error(message);
            TLC.getProject().getParser().increaseErrorCount();
        } else if (filter instanceof BooleanPrimitive) {
            _searchQueryFilter = filter.isTrue();
        } else {
            _searchQueryFilter = filter;
        }
    }

    /**
     * @return the select filter
     */
    @Override
    public BooleanExpression getSelectFilter() {
        return _selectFilter;
    }

    /**
     * El método setSelectFilter se utiliza para establecer el filtro de selección de las operaciones select, export y report de las vistas (páginas)
     * de consulta y/o registro de la entidad. Solo las instancias de la entidad que cumplen con los criterios del filtro son incluidas en el
     * resultado de estas operaciones.
     *
     * @param filter expresión booleana que se utiliza como filtro
     */
    public void setSelectFilter(BooleanExpression filter) {
        String message = "failed to set select filter of " + getFullName();
        if (filter == null) {
            message += "; supplied expression is null";
            logger.error(message);
            TLC.getProject().getParser().increaseErrorCount();
        } else if (filter instanceof BooleanPrimitive) {
            _selectFilter = filter.isTrue();
        } else {
            _selectFilter = filter;
        }
    }

    /**
     * @param reference
     * @return the insert filter
     */
    @Override
    public BooleanExpression getInsertFilter(EntityReference reference) {
        return _insertFilterByReferenceMap.get(reference);
    }

    /**
     * El método setInsertFilter se utiliza para establecer el filtro de selección de la operación insert de las vistas (páginas) de registro,
     * Maestro/Detalle, de la entidad. Solo las instancias de la entidad referenciada que cumplen con los criterios del filtro podrán ser utilizadas
     * como maestro por la operación insert.
     *
     * @param filter expresión booleana, definida en una entidad referenciada (por una relación con cardinalidad varios-a-uno), que se utiliza como
     * filtro
     */
    public void setInsertFilter(BooleanExpression filter) {
        String message = "failed to set insert filter of " + getFullName();
        if (filter == null) {
            message += "; supplied expression is null";
            logger.error(message);
            TLC.getProject().getParser().increaseErrorCount();
            return;
        }
        Entity master = filter.getDeclaringEntity();
        if (master == null) {
            message += "; supplied expression declaring entity is null";
            logger.error(message);
            TLC.getProject().getParser().increaseErrorCount();
            return;
        }
        EntityReference reference;
        if (master instanceof EntityReference) {
            reference = (EntityReference) master;
        } else {
            message += "; supplied expression declaring entity is not an entity reference";
            logger.error(message);
            TLC.getProject().getParser().increaseErrorCount();
            return;
        }
        Entity detail = master.getDeclaringEntity();
        if (detail == null || !detail.equals(this)) {
            message += "; supplied expression declaring entity is not a valid entity reference";
            logger.error(message);
            TLC.getProject().getParser().increaseErrorCount();
            return;
        }
        if (filter instanceof BooleanPrimitive) {
            _insertFilterByReferenceMap.put(reference, filter.isTrue());
        } else {
            _insertFilterByReferenceMap.put(reference, filter);
        }
    }

    /**
     * @return the insert filter by reference map
     */
    public Map<EntityReference, BooleanExpression> getInsertFilterByReferenceMap() {
        return _insertFilterByReferenceMap;
    }

    /**
     * @return the update filter
     */
    @Override
    public BooleanExpression getUpdateFilter() {
        return _updateFilter;
    }

    /**
     * El método setUpdateFilter se utiliza para establecer el filtro de selección de la operación update de las vistas (páginas) de registro de la
     * entidad. Solo las instancias de la entidad que cumplen con los criterios del filtro podrán ser modificadas con la operación update.
     *
     * @param filter expresión booleana que se utiliza como filtro
     */
    public void setUpdateFilter(BooleanExpression filter) {
        String message = "failed to set update filter of " + getFullName();
        if (filter == null) {
            message += "; supplied expression is null";
            logger.error(message);
            TLC.getProject().getParser().increaseErrorCount();
        } else if (filter instanceof BooleanPrimitive) {
            _updateFilter = filter.isTrue();
        } else {
            _updateFilter = filter;
        }
    }

    /**
     * @return the delete filter
     */
    @Override
    public BooleanExpression getDeleteFilter() {
        return _deleteFilter;
    }

    /**
     * El método setDeleteFilter se utiliza para establecer el filtro de selección de la operación delete de las vistas (páginas) de registro de la
     * entidad. Solo las instancias de la entidad que cumplen con los criterios del filtro podrán ser eliminadas con la operación delete.
     *
     * @param filter expresión booleana que se utiliza como filtro
     */
    public void setDeleteFilter(BooleanExpression filter) {
        String message = "failed to set delete filter of " + getFullName();
        if (filter == null) {
            message += "; supplied expression is null";
            logger.error(message);
            TLC.getProject().getParser().increaseErrorCount();
        } else if (filter instanceof BooleanPrimitive) {
            _deleteFilter = filter.isTrue();
        } else {
            _deleteFilter = filter;
        }
    }

    /**
     * @param reference
     * @return the master/detail filter
     */
    @Override
    public BooleanExpression getMasterDetailFilter(EntityReference reference) {
        return _masterDetailFilterByReferenceMap.get(reference);
    }

    /**
     * El método setMasterDetailFilter se utiliza para establecer el filtro de selección de la operación select de las vistas (páginas) de consulta y
     * registro, Maestro/Detalle, de la entidad. Solo las instancias de la entidad referenciada que cumplen con los criterios del filtro podrán ser
     * utilizadas como maestro por la operación select.
     *
     * @param filter expresión booleana, definida en una entidad referenciada (por una relación con cardinalidad varios-a-uno), que se utiliza como
     * filtro
     */
    public void setMasterDetailFilter(BooleanExpression filter) {
        String message = "failed to set master/detail filter of " + getFullName();
        if (filter == null) {
            message += "; supplied expression is null";
            logger.error(message);
            TLC.getProject().getParser().increaseErrorCount();
            return;
        }
        Entity master = filter.getDeclaringEntity();
        if (master == null) {
            message += "; supplied expression declaring entity is null";
            logger.error(message);
            TLC.getProject().getParser().increaseErrorCount();
            return;
        }
        EntityReference reference;
        if (master instanceof EntityReference) {
            reference = (EntityReference) master;
        } else {
            message += "; supplied expression declaring entity is not an entity reference";
            logger.error(message);
            TLC.getProject().getParser().increaseErrorCount();
            return;
        }
        Entity detail = master.getDeclaringEntity();
        if (detail == null || !detail.equals(this)) {
            message += "; supplied expression declaring entity is not a valid entity reference";
            logger.error(message);
            TLC.getProject().getParser().increaseErrorCount();
            return;
        }
        if (filter instanceof BooleanPrimitive) {
            _masterDetailFilterByReferenceMap.put(reference, filter.isTrue());
        } else {
            _masterDetailFilterByReferenceMap.put(reference, filter);
        }
    }

    /**
     * @return the master/detail filter by reference map
     */
    public Map<EntityReference, BooleanExpression> getMasterDetailFilterByReferenceMap() {
        return _masterDetailFilterByReferenceMap;
    }

    /**
     * @return the filter inactive indicator property
     */
    public boolean isFilterInactiveIndicatorProperty() {
        return _filterInactiveIndicatorProperty;
    }

    /**
     * @return the filter owner property
     */
    public boolean isFilterOwnerProperty() {
        return _filterOwnerProperty;
    }

    /**
     * @return the filter segment property
     */
    public boolean isFilterSegmentProperty() {
        return _filterSegmentProperty;
    }

    /**
     * @return true if the entity is an extension of the declaring entity
     */
//  @Override
    public boolean isExtension() {
        return _annotatedWithExtension;
    }

    /**
     * @return true if the entity defines a one-to-one relationship
     */
    @Override
    public boolean isOneToOne() {
        return _annotatedWithOneToOne;
    }

    /**
     * @return true if the entity defines a manty-to-one relationship
     */
    @Override
    public boolean isManyToOne() {
        return _annotatedWithManyToOne;
    }

    /**
     * @return the fetch type
     */
    @Override
    public FetchType getFetchType() {
        return _fetchType;
    }

    /**
     * @return the cascade type
     */
    @Override
    public CascadeType[] getCascadeType() {
        return _cascadeType;
    }

    /**
     * @return the cascade type
     */
    @Override
    public String getCascadeTypeString() {
        if (_cascadeType == null || _cascadeType.length == 0) {
            return null;
        }
        Set<CascadeType> set = EnumSet.noneOf(CascadeType.class);
        set.addAll(Arrays.asList(_cascadeType)); // duplicates are ignored
        List<String> list = new ArrayList<>();
        int n = 0;
        for (CascadeType ct : set) {
            if (CascadeType.UNSPECIFIED.equals(ct)) {
                continue;
            }
            if (CascadeType.ALL.equals(ct)) {
                return "CascadeType.ALL";
            }
            list.add("CascadeType." + ct.name());
            n++;
        }
        return n == 0 ? null : n == 1 ? list.get(0) : "{" + StringUtils.join(list, ", ") + "}";
    }

    /**
     * @return the navigability
     */
    @Override
    public Navigability getNavigability() {
        return _navigability;
    }

    /**
     * @return the master/detail view
     */
    @Override
    public MasterDetailView getMasterDetailView() {
        MasterDetailView masterDetailView = _masterDetailView;
        if (MasterDetailView.UNSPECIFIED.equals(masterDetailView)) {
            masterDetailView = MasterDetailView.NONE;
            if (isManyToOne()) {
                ResourceType rt1 = getResourceType();
                if (rt1 != null && !rt1.equals(ResourceType.UNSPECIFIED)) {
                    Entity declaringEntity = getDeclaringEntity();
                    ResourceType rt2 = declaringEntity == null ? null : declaringEntity.getResourceType();
                    if (declaringEntity != null && rt1.equals(rt2)) {
                        if (declaringEntity.isExistentiallyIndependent()) {
                            masterDetailView = MasterDetailView.TABLE;
                        } else {
                            masterDetailView = MasterDetailView.TABLE_AND_DETAIL;
                        }
                    }
                }
            }
        }
        return masterDetailView;
    }

    /**
     * @return the quick-adding filter
     */
    @Override
    public QuickAddingFilter getQuickAddingFilter() {
        return _quickAddingFilter;
    }

    /**
     * @return the key-properties-query-mapping indicator
     */
    @Override
    public boolean isKeyPropertiesQueryMappingEnabled() {
        return _keyPropertiesQueryMappingEnabled;
    }

    /**
     * the calculable value.
     */
    private Object _calculableValue;

    /**
     * @return the calculable value
     */
    @Override
    public Object getCalculableValue() {
        return _calculableValue;
    }

    /**
     * El método linkCalculableValueProperty enlaza el valor la propiedad a la entidad referenciada. El valor de una referencia (propiedad que hace
     * referencia a una entidad) definida como columna calculable (mediante el elemento calculable de la anotación ColumnField) se puede enlazar a
     * otra referencia (no calculable) mediante el método linkCalculableValueProperty de la propiedad.
     *
     * @param entity referencia (no calculable) a una entidad
     */
    public void linkCalculableValueProperty(Entity entity) {
        _calculableValue = entity;
    }

    /**
     * El método setCalculableValueExpression se utiliza para establecer la expresión para el cálculo del valor de propiedades definidas como columnas
     * calculables (mediante el elemento calculable de la anotación ColumnField).
     *
     * @param expression expresión para el cálculo del valor
     */
    public void setCalculableValueExpression(EntityExpression expression) {
        _calculableValue = expression;
    }

    /**
     * @return the initial value
     */
    @Override
    public Object getInitialValue() {
        return _initialValue;
    }

    /**
     * El método setInitialValue se utiliza para establecer el valor inicial de propiedades y parámetros. El valor inicial de las propiedades se
     * utiliza para inicializar el valor de la propiedad en la operación insert de las vistas (páginas) de registro. El valor inicial de los
     * parámetros se utiliza para inicializar el valor del parámetro al ejecutar la operación.
     *
     * @param initialValue valor inicial de la propiedad o el parámetro
     */
    @Override
    public void setInitialValue(Entity initialValue) {
//      _initialValue = validInitialValue(initialValue) ? XB.Entity.Scalar.self(initialValue) : null;
        _initialValue = validInitialValue(initialValue) ? initialValue : null;
    }

    /**
     * El método setInitialValue se utiliza para establecer el valor inicial de propiedades y parámetros. El valor inicial de las propiedades se
     * utiliza para inicializar el valor de la propiedad en la operación insert de las vistas (páginas) de registro. El valor inicial de los
     * parámetros se utiliza para inicializar el valor del parámetro al ejecutar la operación.
     *
     * @param initialValue valor inicial de la propiedad o el parámetro
     */
    @Override
    public void setInitialValue(Instance initialValue) {
        _initialValue = validInitialValue(initialValue) ? initialValue : null;
    }

    /**
     * El método setInitialValue se utiliza para establecer el valor inicial de propiedades y parámetros. El valor inicial de las propiedades se
     * utiliza para inicializar el valor de la propiedad en la operación insert de las vistas (páginas) de registro. El valor inicial de los
     * parámetros se utiliza para inicializar el valor del parámetro al ejecutar la operación.
     *
     * @param initialValue valor inicial de la propiedad o el parámetro
     */
    @Override
    public void setInitialValue(EntityExpression initialValue) {
        _initialValue = validInitialValue(initialValue) ? initialValue : null;
    }

    /**
     * El método setInitialValue se utiliza para establecer el valor inicial de propiedades y parámetros. El valor inicial de las propiedades se
     * utiliza para inicializar el valor de la propiedad en la operación insert de las vistas (páginas) de registro. El valor inicial de los
     * parámetros se utiliza para inicializar el valor del parámetro al ejecutar la operación.
     *
     * @param initialValue valor inicial de la propiedad o el parámetro
     */
    @Override
    public void setInitialValue(SpecialEntityValue initialValue) {
        _initialValue = validInitialValue(initialValue) && validSpecialEntityValue(initialValue) ? initialValue : null;
    }

    /**
     * @return the default value
     */
    @Override
    public Object getDefaultValue() {
        return _defaultValue;
    }

    /**
     * El método setDefaultValue se utiliza para establecer el valor por omisión de propiedades y parámetros. El valor por omisión de las propiedades
     * se utiliza al almacenar el valor de la propiedad en la base de datos, dependiendo de la opción seleccionada para el elemento defaultCondition
     * de la anotación PropertyField. El valor por omisión de los parámetros se utiliza al ejecutar la operación.
     *
     * @param defaultValue valor por omisión de la propiedad o el parámetro
     */
    @Override
    public void setDefaultValue(Entity defaultValue) {
//      _defaultValue = validInitialValue(defaultValue) ? XB.Entity.Scalar.self(defaultValue) : null;
        _defaultValue = validDefaultValue(defaultValue) ? defaultValue : null;
    }

    /**
     * El método setDefaultValue se utiliza para establecer el valor por omisión de propiedades y parámetros. El valor por omisión de las propiedades
     * se utiliza al almacenar el valor de la propiedad en la base de datos, dependiendo de la opción seleccionada para el elemento defaultCondition
     * de la anotación PropertyField. El valor por omisión de los parámetros se utiliza al ejecutar la operación.
     *
     * @param defaultValue valor por omisión de la propiedad o el parámetro
     */
    @Override
    public void setDefaultValue(Instance defaultValue) {
        _defaultValue = validDefaultValue(defaultValue) ? defaultValue : null;
    }

    /**
     * El método setDefaultValue se utiliza para establecer el valor por omisión de propiedades y parámetros. El valor por omisión de las propiedades
     * se utiliza al almacenar el valor de la propiedad en la base de datos, dependiendo de la opción seleccionada para el elemento defaultCondition
     * de la anotación PropertyField. El valor por omisión de los parámetros se utiliza al ejecutar la operación.
     *
     * @param defaultValue valor por omisión de la propiedad o el parámetro
     */
    @Override
    public void setDefaultValue(EntityExpression defaultValue) {
        _defaultValue = validDefaultValue(defaultValue) ? defaultValue : null;
    }

    /**
     * El método setDefaultValue se utiliza para establecer el valor por omisión de propiedades y parámetros. El valor por omisión de las propiedades
     * se utiliza al almacenar el valor de la propiedad en la base de datos, dependiendo de la opción seleccionada para el elemento defaultCondition
     * de la anotación PropertyField. El valor por omisión de los parámetros se utiliza al ejecutar la operación.
     *
     * @param defaultValue valor por omisión de la propiedad o el parámetro
     */
    @Override
    public void setDefaultValue(SpecialEntityValue defaultValue) {
        _defaultValue = validDefaultValue(defaultValue) && validSpecialEntityValue(defaultValue) ? defaultValue : null;
    }

    /**
     * @return the current value
     */
    @Override
    public Object getCurrentValue() {
        return _currentValue;
    }

    /**
     * El método setCurrentValue se utiliza para establecer el valor actual del parámetro.
     *
     * @param currentValue valor actual
     */
    public void setCurrentValue(Entity currentValue) {
//      _currentValue = validInitialValue(currentValue) ? XB.Entity.Scalar.self(currentValue) : null;
        _currentValue = validCurrentValue(currentValue) ? currentValue : null;
    }

    /**
     * El método setCurrentValue se utiliza para establecer el valor actual del parámetro.
     *
     * @param currentValue valor actual
     */
    public void setCurrentValue(Instance currentValue) {
        _currentValue = validCurrentValue(currentValue) ? currentValue : null;
    }

    /**
     * El método setCurrentValue se utiliza para establecer el valor actual del parámetro.
     *
     * @param currentValue valor actual
     */
    public void setCurrentValue(EntityExpression currentValue) {
        _currentValue = validCurrentValue(currentValue) ? currentValue : null;
    }

    /**
     * El método setCurrentValue se utiliza para establecer el valor actual del parámetro.
     *
     * @param currentValue valor actual
     */
    public void setCurrentValue(SpecialEntityValue currentValue) {
        _currentValue = validCurrentValue(currentValue) && validSpecialEntityValue(currentValue) ? currentValue : null;
    }

    private boolean validSpecialEntityValue(SpecialEntityValue value) {
        Class<? extends Entity> userEntityClass;
        if (value != null) {
            switch (value) {
                case CURRENT_USER:
                    userEntityClass = TLC.getProject().getUserEntityClass();
                    if (userEntityClass != null && userEntityClass.isAssignableFrom(getClass())) {
                        return true;
                    }
                    break;
                default:
                    return true;
            }
        }
        String message = value + " is not a valid value for " + getFullName();
        logger.error(message);
        TLC.getProject().getParser().increaseErrorCount();
        return false;
    }

    /**
     * @return the order by object
     */
    public Object getOrderBy() {
        return _orderBy;
    }

    /**
     * @return the order by property
     */
    public Property getOrderByProperty() {
        return _orderBy instanceof Property ? (Property) _orderBy : null;
    }

    /**
     * @return the order by properties
     */
    public Property[] getOrderByProperties() {
        return _orderBy instanceof Property[] ? (Property[]) _orderBy : null;
    }

    /**
     * @return the order by key
     */
    public Key getOrderByKey() {
        return _orderBy instanceof Key ? (Key) _orderBy : null;
    }

    /**
     * El método setOrderBy se utiliza para establecer el criterio de ordenamiento de las operaciones select, export y report de las vistas (páginas)
     * de consulta y/o registro de la entidad.
     *
     * @param orderBy propiedad que se utiliza como criterio de ordenamiento
     */
    public void setOrderBy(Property orderBy) {
        _orderBy = orderBy;
    }

    /**
     * El método setOrderBy se utiliza para establecer el criterio de ordenamiento de las operaciones select, export y report de las vistas (páginas)
     * de consulta y/o registro de la entidad.
     *
     * @param orderBy una o más propiedades que se utilizan como criterio de ordenamiento, en la misma secuencia en la que son escritas
     */
    public void setOrderBy(Property... orderBy) {
        _orderBy = orderBy;
    }

    /**
     * El método setOrderBy se utiliza para establecer el criterio de ordenamiento de las operaciones select, export y report de las vistas (páginas)
     * de consulta y/o registro de la entidad.
     *
     * @param orderBy clave de acceso cuyas propiedades se utilizan como criterio de ordenamiento, en la misma secuencia en la que aparecen en la
     * clave
     */
    public void setOrderBy(Key orderBy) {
        _orderBy = orderBy;
    }

    /**
     * @return the default tab
     */
    public Tab getDefaultTab() {
        if (_defaultTab == null) {
            List<Tab> tabs = getTabsList();
            if (tabs != null && !tabs.isEmpty()) {
                _defaultTab = tabs.get(0);
            }
        }
        return _defaultTab;
    }

    /**
     * El método setDefaultTab de la entidad se utiliza para establecer la pestaña (tab) por defecto, es decir, la pestaña que recibe el enfoque al
     * abrir una vista (página) con pestañas de la entidad.
     *
     * @param tab pestaña que recibe el enfoque al abrir una vista con pestañas de la entidad
     */
    public void setDefaultTab(Tab tab) {
        _defaultTab = tab;
    }

    /**
     * @return the default tab sequence number
     */
    public int getDefaultTabSequenceNumber() {
        Tab tab = getDefaultTab();
        return tab == null ? 0 : tab.getSequenceNumber();
    }

    /**
     * @return true if the entity is master of at least one detail
     */
    @Override
    public boolean isWritingPageMaster() {
        EntityReference reference;
        MasterDetailView masterDetailView;
        Class<? extends Entity> declaring;
        Entity detail;
        List<Property> properties = getReferencesList();
        for (Property property : properties) {
            reference = (EntityReference) property;
            if (reference.isManyToOne()) {
                masterDetailView = reference.getMasterDetailView();
                switch (masterDetailView) {
                    case TABLE:
//                  case DETAIL:
                    case TABLE_AND_DETAIL:
                        declaring = reference.getDeclaringEntity().getClass();
                        if (declaring != null) {
                            detail = _project.getEntity(declaring);
                            if (detail != null && !detail.isPrivateEntityClass()) { // && detail.isSelectEnabled()
                                if (detail.isInsertEnabled() || detail.isUpdateEnabled() || detail.isDeleteEnabled()) {
                                    return true;
                                }
                            }
                        }
                        break;
                }
            }
        }
        return false;
    }

    /**
     * @return the AbstractClass annotation indicator
     */
    public boolean isAnnotatedWithAbstractClass() {
        return _annotatedWithAbstractClass;
    }

    /**
     * @return the AllocationOverride annotation indicator
     */
    public boolean isAnnotatedWithAllocationOverride() {
        return _annotatedWithAllocationOverride;
    }

    /**
     * @return the AllocationOverrides annotation indicator
     */
    public boolean isAnnotatedWithAllocationOverrides() {
        return _annotatedWithAllocationOverrides;
    }

    /**
     * @return the EntityPrimaryKey annotation indicator
     */
    public boolean isAnnotatedWithEntityPrimaryKey() {
        return _annotatedWithEntityPrimaryKey;
    }

    /**
     * @return the EntitySequenceProperty annotation indicator
     */
    public boolean isAnnotatedWithEntitySequenceProperty() {
        return _annotatedWithEntitySequenceProperty;
    }

    /**
     * @return the EntityVersionProperty annotation indicator
     */
    public boolean isAnnotatedWithEntityVersionProperty() {
        return _annotatedWithEntityVersionProperty;
    }

    /**
     * @return the EntityNumericKey annotation indicator
     */
    public boolean isAnnotatedWithEntityNumericKey() {
        return _annotatedWithEntityNumericKey;
    }

    /**
     * @return the EntityCharacterKey annotation indicator
     */
    public boolean isAnnotatedWithEntityCharacterKey() {
        return _annotatedWithEntityCharacterKey;
    }

    /**
     * @return the EntityNameProperty annotation indicator
     */
    public boolean isAnnotatedWithEntityNameProperty() {
        return _annotatedWithEntityNameProperty;
    }

    /**
     * @return the EntityDescriptionProperty annotation indicator
     */
    public boolean isAnnotatedWithEntityDescriptionProperty() {
        return _annotatedWithEntityDescriptionProperty;
    }

    /**
     * @return the EntityInactiveIndicator annotation indicator
     */
    public boolean isAnnotatedWithEntityInactiveIndicator() {
        return _annotatedWithEntityInactiveIndicator;
    }

    /**
     * @return the EntityUrlProperty annotation indicator
     */
    public boolean isAnnotatedWithEntityUrlProperty() {
        return _annotatedWithEntityUrlProperty;
    }

    /**
     * @return the EntityParentProperty annotation indicator
     */
    public boolean isAnnotatedWithEntityParentProperty() {
        return _annotatedWithEntityParentProperty;
    }

    /**
     * @return the EntityOwnerProperty annotation indicator
     */
    public boolean isAnnotatedWithEntityOwnerProperty() {
        return _annotatedWithEntityOwnerProperty;
    }

    /**
     * @return the EntitySegmentProperty annotation indicator
     */
    public boolean isAnnotatedWithEntitySegmentProperty() {
        return _annotatedWithEntitySegmentProperty;
    }

    /**
     * @return the EntityBusinessKey annotation indicator
     */
    public boolean isAnnotatedWithEntityBusinessKey() {
        return _annotatedWithEntityBusinessKey;
    }

    /**
     * @return the EntityClass annotation indicator
     */
    public boolean isAnnotatedWithEntityClass() {
        return _annotatedWithEntityClass;
    }

    /**
     * @return the EntityDataGen annotation indicator
     */
    public boolean isAnnotatedWithEntityDataGen() {
        return _annotatedWithEntityDataGen;
    }

    /**
     * @return the EntitySelectOperation annotation indicator
     */
    public boolean isAnnotatedWithEntitySelectOperation() {
        return _annotatedWithEntitySelectOperation;
    }

    /**
     * @return the EntityInsertOperation annotation indicator
     */
    public boolean isAnnotatedWithEntityInsertOperation() {
        return _annotatedWithEntityInsertOperation;
    }

    /**
     * @return the EntityUpdateOperation annotation indicator
     */
    public boolean isAnnotatedWithEntityUpdateOperation() {
        return _annotatedWithEntityUpdateOperation;
    }

    /**
     * @return the EntityDeleteOperation annotation indicator
     */
    public boolean isAnnotatedWithEntityDeleteOperation() {
        return _annotatedWithEntityDeleteOperation;
    }

    /**
     * @return the EntityReportOperation annotation indicator
     */
    public boolean isAnnotatedWithEntityReportOperation() {
        return _annotatedWithEntityReportOperation;
    }

    /**
     * @return the EntityExportOperation annotation indicator
     */
    public boolean isAnnotatedWithEntityExportOperation() {
        return _annotatedWithEntityExportOperation;
    }

    /**
     * @return the EntityTableView annotation indicator
     */
    public boolean isAnnotatedWithEntityTableView() {
        return _annotatedWithEntityTableView;
    }

    /**
     * @return the EntityDetailView annotation indicator
     */
    public boolean isAnnotatedWithEntityDetailView() {
        return _annotatedWithEntityDetailView;
    }

    /**
     * @return the EntityTreeView annotation indicator
     */
    public boolean isAnnotatedWithEntityTreeView() {
        return _annotatedWithEntityTreeView;
    }

    /**
     * @return the EntityConsoleView annotation indicator
     */
    public boolean isAnnotatedWithEntityConsoleView() {
        return _annotatedWithEntityConsoleView;
    }

    /**
     * @return the EntityWarnings annotation indicator
     */
    public boolean isAnnotatedWithEntityWarnings() {
        return _annotatedWithEntityWarnings;
    }

    /**
     * @return the EntityCodeGen annotation indicator
     */
    public boolean isAnnotatedWithEntityCodeGen() {
        return _annotatedWithEntityCodeGen;
    }

    /**
     * @return the EntityDocGen annotation indicator
     */
    public boolean isAnnotatedWithEntityDocGen() {
        return _annotatedWithEntityDocGen;
    }

    /**
     * @return the EntityReferenceSearch annotation indicator
     */
    public boolean isAnnotatedWithEntityReferenceSearch() {
        return _annotatedWithEntityReferenceSearch;
    }

    /**
     * @return the Filter annotation indicator
     */
    public boolean isAnnotatedWithFilter() {
        return _annotatedWithFilter;
    }

    /**
     * @return the Extension annotation indicator
     */
    public boolean isAnnotatedWithExtension() {
        return _annotatedWithExtension;
    }

    /**
     * @return the OneToOne annotation indicator
     */
    public boolean isAnnotatedWithOneToOne() {
        return _annotatedWithOneToOne;
    }

    /**
     * @return the ManyToOne annotation indicator
     */
    public boolean isAnnotatedWithManyToOne() {
        return _annotatedWithManyToOne;
    }

    /**
     * @return the QueryMapping annotation indicator
     */
    public boolean isAnnotatedWithQueryMapping() {
        return _annotatedWithQueryMapping;
    }

    /**
     * @return the EntityReferenceDataGen annotation indicator
     */
    public boolean isAnnotatedWithEntityReferenceDataGen() {
        return _annotatedWithEntityReferenceDataGen;
    }
    // </editor-fold>

    // <editor-fold defaultstate="collapsed" desc="CRUD operations">
    protected SelectOperation select;

    protected InsertOperation insert;

    protected UpdateOperation update;

    protected DeleteOperation delete;

    /**
     * @return the select operation
     */
    public SelectOperation getSelectOperation() {
        return select;
    }

    /**
     * @return the insert operation
     */
    public InsertOperation getInsertOperation() {
        return insert;
    }

    /**
     * @return the update operation
     */
    public UpdateOperation getUpdateOperation() {
        return update;
    }

    /**
     * @return the delete operation
     */
    public DeleteOperation getDeleteOperation() {
        return delete;
    }
    // </editor-fold>

    // <editor-fold defaultstate="collapsed" desc="LocaleEntityReferenceKey">
    protected class LocaleEntityReference {

        Locale loc;

        EntityReference ref;

        private LocaleEntityReference(Locale locale, EntityReference reference) {
            loc = locale;
            ref = reference;
        }

        @Override
        public boolean equals(Object obj) {
            if (obj instanceof LocaleEntityReference) {
                LocaleEntityReference ler = (LocaleEntityReference) obj;
                return loc.equals(ler.loc) && ref.equals(ler.ref);
            }
            return false;
        }

        @Override
        public int hashCode() {
            return Objects.hash(loc, ref);
        }

    }

    protected LocaleEntityReference localeEntityReferenceReadingKey(Locale locale, EntityReference reference) {
        if (reference == null) {
            throw new IllegalArgumentException("null entity reference");
        }
        Locale l = locale == null ? Bundle.getLocale() : locale;
        return new LocaleEntityReference(l, reference);
    }

    protected LocaleEntityReference localeEntityReferenceWritingKey(Locale locale, EntityReference reference) {
        if (reference == null) {
            throw new IllegalArgumentException("null entity reference");
        }
        Locale l = locale == null ? Bundle.getLocale() : Bundle.isSupportedLocale(locale) ? locale : null;
        if (l == null) {
            throw new IllegalArgumentException("Locale " + locale + " not supported yet.");
        }
        return new LocaleEntityReference(l, reference);
    }
    // </editor-fold>

    public AbstractEntity(Artifact declaringArtifact, Field declaringField) {
        super();
        init(declaringArtifact, declaringField);
    }

    private void init(Artifact declaringArtifact, Field declaringField) {
        if (declaringArtifact == null) {
            throw new IllegalArgumentException("null declaring artifact");
        }
        copyFromBootstrappingProperties();
        Class<?> namedClass = getNamedClass();
        String className = namedClass.getSimpleName();
        String fieldName = declaringField == null ? className : declaringField.getName();
        setDataClass(namedClass);
        setDataType(namedClass);
        if (declaringArtifact instanceof Project) {
            _project = (Project) declaringArtifact;
            _rootInstance = true;
            setDeclaredIndicators();
            /*
             * data type, project and the root indicator must be set before calling setDeclared
             * they are all used at initializeInheritanceFields, which is called by setDeclared
             */
            setDeclared(className);
            track("allocate");
        } else {
            Project currentProject = TLC.getProject();
            if (currentProject == null) {
                throw new UnexpectedRuntimeException("null current project");
            }
            _project = currentProject;
            _rootInstance = false;
            setDeclaredIndicators();
            setName(fieldName);
            setDeclaringArtifact(declaringArtifact);
            setDeclaringField(declaringField);
            track("allocate");
            initialise();
            settle();
        }
    }

    private static final String REUSE_WHEN_CONTAINS = "reuse.default.entity.labels.when.class.name.contains.property.field.name";

    private boolean _reuseWhenContains = true;

    private void copyFromBootstrappingProperties() {
        ExtendedProperties bootstrapping = PropertiesHandler.getBootstrapping();
        if (bootstrapping != null && !bootstrapping.isEmpty()) {
            _reuseWhenContains = BitUtils.valueOf(bootstrapping.getString(REUSE_WHEN_CONTAINS, "true"));
        }
    }

    private void setDeclaredIndicators() {
        ProjectEntityReference reference = _project.getEntityReference(getDataType());
        _explicitlyDeclared = reference.isExplicit();
        _implicitlyDeclared = reference.isImplicit();
    }

    // <editor-fold defaultstate="collapsed" desc="check">
//  private void check(Artifact declaringArtifact) {
//      if (declaringArtifact == null) {
//          Class<?> namedClass = getNamedClass();
//          String methodName = namedClass.getName() + '.' + XS.GET_INSTANCE;
//          int lineNumber = XS.getLineNumber(methodName, true);
//          if (lineNumber == 0) {
//              methodName = namedClass.getName() + '.' + "<init>";
//              lineNumber = XS.getLineNumber(methodName, false);
//              String msg = "null declaring artifact";
//              if (lineNumber > 0) {
//                  msg += " at " + methodName + "(" + namedClass.getSimpleName() + ".java:" + lineNumber + ")";
//              }
//              throw new IllegalArgumentException(msg);
//          }
//      }
//      String message = "";
//      message += getAbstractSuperclass().getSimpleName() + " ";
//      message += getNamedClass().getSimpleName() + " " + hashCode();
//      if (declaringArtifact != null) {
//          message += " @ ";
//          message += XS.getNamedClass(declaringArtifact).getSimpleName() + " " + declaringArtifact.hashCode();
//      }
//      logger.trace(message);
//  }
    // </editor-fold>
//
    @Override
    public final void initialise() {
        XS1.checkAccess();
        if (_initialised) {
            return;
        }
        _initialised = true;
        Artifact declaringArtifact = getDeclaringArtifact();
        if (declaringArtifact == null) {
            TLC.removeAllocationSettings();
            _atlas.initialiseFields(Property.class);
            _atlas.initialiseFields(Key.class);
            _atlas.initialiseFields(Tab.class);
            _atlas.initialiseFields(View.class);
            _atlas.initialiseFields(Instance.class);
            _atlas.initialiseFields(NamedValue.class);
            _atlas.initialiseFields(Expression.class);
            _atlas.initialiseFields(Transition.class);
            _atlas.checkOperationClasses();
            _atlas.initialiseFields(Operation.class);
            _atlas.checkOperationFields();
            _atlas.initialiseFields(Trigger.class);
        } else {
            _atlas.initialiseFields(Property.class);
            _atlas.initialiseFields(Instance.class);
            _atlas.initialiseFields(NamedValue.class);
            _atlas.initialiseFields(Expression.class);
        }
    }

    @Override
    public final void settle() {
        XS1.checkAccess();
        if (_settled) {
            return;
        }
        _settled = true;
        Artifact declaringArtifact = getDeclaringArtifact();
        settleAttributes();
        if (declaringArtifact == null) {
            settleProperties();
            settleLinks();
            settleKeys();
            settleTabs();
            settleViews();
            settleInstances();
//          settleNamedValues();
            settleExpressions();
            verifyNames(Entity.class, Expression.class);
            settleFilters();
            settleTransitions();
            clinchOperations();
            settleOperations();
            settleTriggers();
//          verifyNames(Entity.class);
        } else {
            if (declaringArtifact instanceof Operation) {
                settleExpressions();
            }
            verifyLabels();
            verifyDescriptions();
        }
        _settler = '?';
    }

    private void verifyLabels() {
        String className = StrUtils.getHumplessCase(getDataClass().getSimpleName());
        String fieldName = StrUtils.getHumplessCase(getName());
        String shortName = StrUtils.removeWords(fieldName, EntityReference.class, "_");
        boolean preserve = className.equals(shortName) || (_reuseWhenContains && className.contains(shortName));
        if (!preserve) {
            for (Locale locale : Bundle.getSupportedLocales()) {
                setLocalizedLabel(locale, null);
                setLocalizedShortLabel(locale, null);
            }
        }
    }

    private void verifyDescriptions() {
        String className = StrUtils.getHumplessCase(getDataClass().getSimpleName());
        String fieldName = StrUtils.getHumplessCase(getName());
        String shortName = StrUtils.removeWords(fieldName, EntityReference.class, "_");
        boolean preserve = className.equals(shortName);
        if (!preserve) {
            for (Locale locale : Bundle.getSupportedLocales()) {
                setLocalizedDescription(locale, null);
                setLocalizedShortDescription(locale, null);
            }
        }
    }

    // <editor-fold defaultstate="collapsed" desc="settle">
    protected void settleAttributes() {
        track("settleAttributes");
        _settler = 'A';
    }

    protected void settleProperties() {
        track("settleProperties");
        _settler = 'P';
    }

    protected void settleLinks() {
        track("settleLinks");
        _settler = 'L';
    }

    protected void settleKeys() {
        track("settleKeys");
        _settler = 'K';
    }

    protected void settleTabs() {
        track("settleTabs");
        _settler = 'T';
    }

    protected void settleViews() {
        track("settleViews");
        _settler = 'V';
    }

    protected void settleInstances() {
        track("settleInstances");
        _settler = 'I';
    }

    protected void settleExpressions() {
        track("settleExpressions");
        _settler = 'X';
    }

    protected void settleFilters() {
        track("settleFilters");
        _settler = 'F';
    }

    protected void settleTransitions() {
        track("settleTransitions");
        _settler = 'R';
    }

    protected void settleOperations() {
        track("settleOperations");
        _settler = 'O';
    }

    protected void clinchOperations() {
        track("clinchOperations");
        Class<?> type;
        Object o;
        Operation operation;
        Class<?> clazz = getClass();
        Class<?> top = Entity.class;
        for (Field field : XS1.getFields(clazz, top)) {
            field.setAccessible(true);
            type = field.getType();
            if (Operation.class.isAssignableFrom(type) && isNotRestricted(field)) {
                String errmsg = "failed to get field \"" + field + "\" at " + this;
                try {
                    o = field.get(this);
                    if (o instanceof Operation) {
                        operation = (Operation) o;
                        operation.settle();
                    }
                } catch (IllegalArgumentException | IllegalAccessException ex) {
                    logger.error(errmsg, ThrowableUtils.getCause(ex));
                    TLC.getProject().getParser().increaseErrorCount();
                }
            }
        }
    }

    protected void settleTriggers() {
        track("settleTriggers");
        _settler = 'G';
    }
    // </editor-fold>

    private final Map<String, Property> _foreignQueryProperties = new LinkedHashMap<>();

    public List<Property> getForeignQueryPropertiesList() {
        return new ArrayList<>(_foreignQueryProperties.values());
    }

    public Map<String, Property> getForeignQueryPropertiesMap() {
        return _foreignQueryProperties;
    }

    /**
     * El método linkForeignQueryProperty agrega una propiedad no calculable a la vista de la entidad. La vista predeterminada de una entidad incluye,
     * además de todas sus propiedades, propiedades de entidades referenciadas que sean necesarias para la interfaz de usuario, el control de acceso y
     * otras funciones. Es posible agregar propiedades a la vista predeterminada de manera explícita, mediante los métodos linkForeignQueryProperty y
     * linkForeignQueryEntityReference.
     *
     * @param property propiedad no calculable de alguna entidad referenciada (directa o indirectamente)
     */
    public void linkForeignQueryProperty(Property property) {
        String message = "failed to link foreign query property to " + getFullName();
        if (property == null) {
            message += "; supplied foreign property is null";
            logger.error(message);
            TLC.getProject().getParser().increaseErrorCount();
        } else {
            String key = property.getPathString();
            if (_foreignQueryProperties.containsKey(key)) {
            } else {
                _foreignQueryProperties.put(key, property);
            }
        }
    }

    /**
     * El método linkForeignQueryProperty agrega una propiedad no calculable a la vista de la entidad. La vista predeterminada de una entidad incluye,
     * además de todas sus propiedades, propiedades de entidades referenciadas que sean necesarias para la interfaz de usuario, el control de acceso y
     * otras funciones. Es posible agregar propiedades a la vista predeterminada de manera explícita, mediante los métodos linkForeignQueryProperty y
     * linkForeignQueryEntityReference.
     *
     * @param properties una o más propiedades no calculables de alguna entidad referenciada (directa o indirectamente)
     */
    public void linkForeignQueryProperty(Property... properties) {
        for (Property property : properties) {
            linkForeignQueryProperty(property);
        }
    }

    /**
     * El método linkForeignQueryProperty agrega una propiedad no calculable a la vista de la entidad. La vista predeterminada de una entidad incluye,
     * además de todas sus propiedades, propiedades de entidades referenciadas que sean necesarias para la interfaz de usuario, el control de acceso y
     * otras funciones. Es posible agregar propiedades a la vista predeterminada de manera explícita, mediante los métodos linkForeignQueryProperty y
     * linkForeignQueryEntityReference.
     *
     * @param properties lista de propiedades no calculables de alguna entidad referenciada (directa o indirectamente)
     */
    public void linkForeignQueryProperty(List<Property> properties) {
        for (Property property : properties) {
            linkForeignQueryProperty(property);
        }
    }

    /**
     * El método linkForeignQueryEntityReference agrega las propiedades no calculables de una o más referencias (no calculables) a la vista de la
     * entidad. La vista predeterminada de una entidad incluye, además de todas sus propiedades, propiedades de entidades referenciadas que sean
     * necesarias para la interfaz de usuario, el control de acceso y otras funciones. Es posible agregar propiedades a la vista predeterminada de
     * manera explícita, mediante los métodos linkForeignQueryProperty y linkForeignQueryEntityReference.
     *
     * @param references una o más referencias (no calculables) a entidades referenciadas directa o indirectamente
     */
    public void linkForeignQueryEntityReference(EntityReference... references) {
        List<Property> propertiesList;
        for (EntityReference reference : references) {
            linkForeignQueryProperty(reference);
            propertiesList = reference.getPropertiesList();
            linkForeignQueryProperty(propertiesList);
        }
    }

    /**
     * El método linkForeignInactiveIndicatorProperty enlaza el indicador de inactividad al indicador de inactividad de una entidad referenciada. El
     * indicador de inactividad (eliminación lógica) de la entidad puede ser el indicador de inactividad de una de sus entidades referenciadas. Cuando
     * éste sea el caso, en lugar de utilizar la anotación InactiveIndicator se utiliza el método linkForeignInactiveIndicatorProperty.
     *
     * @param foreignInactiveIndicatorProperty indicador de inactividad de una entidad referenciada (vea Anotación InactiveIndicator)
     */
    public void linkForeignInactiveIndicatorProperty(BooleanProperty foreignInactiveIndicatorProperty) {
//      if (isParameter() || isParameterProperty()) {
//          return;
//      }
        String message = "failed to link foreign inactive indicator property of " + getFullName();
        if (foreignInactiveIndicatorProperty == null) {
            message += "; supplied foreign property is null";
            logger.error(message);
            TLC.getProject().getParser().increaseErrorCount();
        } else {
            _inactiveIndicatorProperty = foreignInactiveIndicatorProperty;
        }
    }

    /**
     * El método linkForeignOwnerProperty enlaza la propiedad propietario a la propiedad propietario de una entidad referenciada. La propiedad
     * propietario de la entidad puede ser la propiedad propietario de una de sus entidades referenciadas. Cuando éste sea el caso, en lugar de
     * utilizar la anotación OwnerProperty se utiliza el método linkForeignOwnerProperty.
     *
     * @param foreignOwnerProperty propiedad propietario de una entidad referenciada (vea Anotación OwnerProperty)
     */
    public void linkForeignOwnerProperty(EntityReference foreignOwnerProperty) {
//      if (isParameter() || isParameterProperty()) {
//          return;
//      }
        String message = "failed to link foreign owner property of " + getFullName();
        if (foreignOwnerProperty == null) {
            message += "; supplied foreign property is null";
            logger.error(message);
            TLC.getProject().getParser().increaseErrorCount();
        } else {
            Class<?> foreignOwnerPropertyClass = foreignOwnerProperty.getClass();
            Class<? extends Entity> userEntityClass = TLC.getProject().getUserEntityClass();
            if (userEntityClass != null && userEntityClass.isAssignableFrom(foreignOwnerPropertyClass)) {
                Field field = foreignOwnerProperty.getDeclaringField();
                boolean aye = field.isAnnotationPresent(OwnerProperty.class);
                if (aye) {
                    _ownerProperty = foreignOwnerProperty;
                } else {
                    message += "; " + field.getDeclaringClass().getSimpleName() + "." + field.getName() + " is not an owner property";
                    logger.error(message);
                    TLC.getProject().getParser().increaseErrorCount();
                }
            } else {
                message += "; " + userEntityClass + " is not assignable from " + foreignOwnerPropertyClass;
                logger.error(message);
                TLC.getProject().getParser().increaseErrorCount();
            }
        }
    }

    /**
     * El método linkForeignSegmentProperty enlaza la propiedad segmento a la propiedad segmento de una entidad referenciada. La propiedad segmento de
     * la entidad puede ser la propiedad segmento de una de sus entidades referenciadas. Cuando éste sea el caso, en lugar de utilizar la anotación
     * SegmentProperty se utiliza el método linkForeignSegmentProperty.
     *
     * @param foreignSegmentProperty propiedad segmento de una entidad referenciada (vea Anotación SegmentProperty)
     */
    public void linkForeignSegmentProperty(EntityReference foreignSegmentProperty) {
//      if (isParameter() || isParameterProperty()) {
//          return;
//      }
        String message = "failed to link foreign segment property of " + getFullName();
        if (foreignSegmentProperty == null) {
            message += "; supplied foreign property is null";
            logger.error(message);
            TLC.getProject().getParser().increaseErrorCount();
        } else {
            Field field = foreignSegmentProperty.getDeclaringField();
            boolean aye = field.isAnnotationPresent(SegmentProperty.class);
            if (aye) {
                _segmentProperty = foreignSegmentProperty;
            } else {
                message += "; " + field.getDeclaringClass().getSimpleName() + "." + field.getName() + " is not a segment property";
                logger.error(message);
                TLC.getProject().getParser().increaseErrorCount();
            }
        }
    }

    private boolean isNotRestricted(Field field) {
        int modifiers = field.getModifiers();
        return !(Modifier.isPrivate(modifiers) || Modifier.isStatic(modifiers) || Modifier.isFinal(modifiers));
    }

    @Override
    public void finalise() {
        if (_finalised) {
            logger.warn(getFullName() + " already finalised! ");
            return;
        }
        XS1.checkAccess();
        track("finalise");
        _finalised = true;
        _atlas.finalise();
        setKeyFields();
        setBusinessKey();
        setKeyProperties();
        setForeignKeyFields();
        setTableViewEnabledIndicator();
        setBasicDatabaseOperationsAttributes();
//      check(); --> moved to finish() on 10/17/2017
    }

    @Override
    public void finish() {
        if (_finished) {
            logger.warn(getFullName() + " already finished! ");
            return;
        }
        XS1.checkAccess();
        track("finish");
        _finished = true;
        check();
    }

    /**
     * El método addAttribute permite agregar un atributo a la lista de atributos extraordinarios de la entidad. Los atributos extraordinarios son
     * parejas clave/valor, de modo que si se agregan varios atributos con la misma clave a una entidad, el valor de tal atributo será el último valor
     * agregado.
     *
     * @param property meta-propiedad (de la entidad) a la que corresponde el atributo.
     * @param name clave del atributo
     * @param value valor del atributo
     * @return el valor anterior asociado con la clave, o nulo si no había una asignación para la clave, o si la implementación admite valores nulos.
     */
    @Override
    public Object addAttribute(Property property, String name, Object value) {
        return property == null ? null : property.addAttribute(name, value);
    }

    void setKeyFields() {
        setPrimaryKeyField();
        setSequenceField();
        setVersionField();
        setNumericKeyField();
        setCharacterKeyField();
        setNameField();
        setDescriptionField();
        setImageField();
        setInactiveIndicatorField();
        setUrlField();
        setParentField();
        setOwnerField();
        setSegmentField();
        setBusinessKeyField();
        setUniqueKeyFields();
        setStateField();
    }

    void setPrimaryKeyField() {
        Field field = getAnnotations().get(PrimaryKey.class);
        if (field != null) {
            Class<?> type = getDataType();
            Class<?> fieldType = field.getType();
            String fieldName = field.getName();
            if (field.equals(getPrimaryKeyField(fieldName, type))) {
                _primaryKeyFieldName = fieldName;
                _primaryKeyField = field;
                if (IntegerProperty.class.isAssignableFrom(fieldType) && _numericKeyField == null) {
                    _numericKeyFieldName = fieldName;
                    _numericKeyField = field;
                }
            }
        }
    }

    void setSequenceField() {
        Field field = getAnnotations().get(SequenceProperty.class);
        if (field != null) {
            Class<?> type = getDataType();
            Class<?> fieldType = field.getType();
            String fieldName = field.getName();
            if (field.equals(getSequenceField(fieldName, type))) {
                _sequenceFieldName = fieldName;
                _sequenceField = field;
            }
        }
    }

    void setVersionField() {
        Field field = getAnnotations().get(VersionProperty.class);
        if (field != null) {
            Class<?> type = getDataType();
            String fieldName = field.getName();
            if (field.equals(getVersionField(fieldName, type))) {
                _versionFieldName = fieldName;
                _versionField = field;
            }
        }
    }

    void setNumericKeyField() {
        Field field = getAnnotations().get(NumericKey.class);
        if (field != null) {
            Class<?> type = getDataType();
            String fieldName = field.getName();
            if (field.equals(getNumericKeyField(fieldName, type))) {
                _numericKeyFieldName = fieldName;
                _numericKeyField = field;
            }
        }
    }

    void setCharacterKeyField() {
        Field field = getAnnotations().get(CharacterKey.class);
        if (field != null) {
            Class<?> type = getDataType();
            String fieldName = field.getName();
            if (field.equals(getCharacterKeyField(fieldName, type))) {
                _characterKeyFieldName = fieldName;
                _characterKeyField = field;
            }
        }
    }

    void setNameField() {
        Field field = getAnnotations().get(NameProperty.class);
        if (field != null) {
            Class<?> type = getDataType();
            String fieldName = field.getName();
            if (field.equals(getNameField(fieldName, type))) {
                _nameFieldName = fieldName;
                _nameField = field;
            }
        }
    }

    void setDescriptionField() {
        Field field = getAnnotations().get(DescriptionProperty.class);
        if (field != null) {
            Class<?> type = getDataType();
            String fieldName = field.getName();
            if (field.equals(getDescriptionField(fieldName, type))) {
                _descriptionFieldName = fieldName;
                _descriptionField = field;
            }
        }
    }

    void setImageField() {
        Field field = getAnnotations().get(ImageProperty.class);
        if (field != null) {
            Class<?> type = getDataType();
            String fieldName = field.getName();
            if (field.equals(getImageField(fieldName, type))) {
                _imageFieldName = fieldName;
                _imageField = field;
            }
        }
    }

    void setInactiveIndicatorField() {
        Field field = getAnnotations().get(InactiveIndicator.class);
        if (field != null) {
            Class<?> type = getDataType();
            String fieldName = field.getName();
            if (field.equals(getInactiveIndicatorField(fieldName, type))) {
                _inactiveIndicatorFieldName = fieldName;
                _inactiveIndicatorField = field;
            }
        }
    }

    void setUrlField() {
        Field field = getAnnotations().get(UrlProperty.class);
        if (field != null) {
            Class<?> type = getDataType();
            String fieldName = field.getName();
            if (field.equals(getUrlField(fieldName, type))) {
                _urlFieldName = fieldName;
                _urlField = field;
            }
        }
    }

    void setParentField() {
        Field field = getAnnotations().get(ParentProperty.class);
        if (field != null) {
            String fieldName = field.getName();
            if (field.equals(getParentField(fieldName, field.getDeclaringClass()))) {
                _parentFieldName = fieldName;
                _parentField = field;
            }
        }
    }

    void setOwnerField() {
        Field field = getAnnotations().get(OwnerProperty.class);
        if (field != null) {
            Class<?> type = getDataType();
            String fieldName = field.getName();
            if (field.equals(getOwnerField(fieldName, type))) {
                _ownerFieldName = fieldName;
                _ownerField = field;
            }
        }
    }

    void setSegmentField() {
        Field field = getAnnotations().get(SegmentProperty.class);
        if (field != null) {
            Class<?> type = getDataType();
            String fieldName = field.getName();
            if (field.equals(getSegmentField(fieldName, type))) {
                _segmentFieldName = fieldName;
                _segmentField = field;
            }
        }
    }

    void setBusinessKeyField() {
        Field field = getAnnotations().get(BusinessKey.class);
        if (field != null) {
            Class<?> type = getDataType();
            Class<?> fieldType = field.getType();
            String fieldName = field.getName();
            if (field.equals(getBusinessKeyField(fieldName, type))) {
                _businessKeyFieldName = fieldName;
                _businessKeyField = field;
                if (IntegerProperty.class.isAssignableFrom(fieldType)) {
                    _numericKeyFieldName = fieldName;
                    _numericKeyField = field;
                }
                if (StringProperty.class.isAssignableFrom(fieldType)) {
                    _characterKeyFieldName = fieldName;
                    _characterKeyField = field;
                }
            }
        }
    }

    void setUniqueKeyFields() {
        Class<?> type;
        Object o;
        Class<?> clazz = getClass();
        Class<?> top = Entity.class;
        for (Field field : XS1.getFields(clazz, top)) {
            field.setAccessible(true);
            type = field.getType();
            if (Property.class.isAssignableFrom(type) && isNotRestricted(field)) {
                String errmsg = "failed to get field \"" + field + "\" at " + this;
                try {
                    o = field.get(this);
                    if (o instanceof Property) {
                        setKeyField(field);
                    }
                } catch (IllegalArgumentException | IllegalAccessException ex) {
                    logger.error(errmsg, ThrowableUtils.getCause(ex));
                    TLC.getProject().getParser().increaseErrorCount();
                }
            }
        }
    }

    void setStateField() {
        Field field = getAnnotations().get(StateProperty.class);
        if (field != null) {
            Class<?> type = getDataType();
            String fieldName = field.getName();
            if (field.equals(getStateField(fieldName, type))) {
                StateProperty annotation = field.getAnnotation(StateProperty.class);
                if (annotation != null) {
                    _stateFieldName = fieldName;
                    _stateField = field;
                }
            }
        }
    }

    void setKeyField(Field field) {
        Class<?> type = getDataType();
        Class<?> fieldType = field.getType();
        String fieldName = field.getName();
        if (field.isAnnotationPresent(UniqueKey.class)) {
            if (field.equals(getUniqueKeyField(fieldName, type))) {
                if (IntegerProperty.class.isAssignableFrom(fieldType) && _numericKeyField == null) {
                    _numericKeyFieldName = fieldName;
                    _numericKeyField = field;
                }
                if (StringProperty.class.isAssignableFrom(fieldType) && _characterKeyField == null) {
                    _characterKeyFieldName = fieldName;
                    _characterKeyField = field;
                }
            }
        }
    }

    void setKeyProperties() {
        Object keyProperty;
        keyProperty = getKeyProperty(_primaryKeyField);
        if (keyProperty instanceof Property) {
            _primaryKeyProperty = (Property) keyProperty;
        }
        keyProperty = getKeyProperty(_sequenceField);
        if (keyProperty instanceof LongProperty) {
            _sequenceProperty = (LongProperty) keyProperty;
        }
        keyProperty = getKeyProperty(_versionField);
        if (keyProperty instanceof LongProperty) {
            _versionProperty = (LongProperty) keyProperty;
        }
        keyProperty = getKeyProperty(_numericKeyField);
        if (keyProperty instanceof IntegerProperty) {
            _numericKeyProperty = (IntegerProperty) keyProperty;
        }
        keyProperty = getKeyProperty(_characterKeyField);
        if (keyProperty instanceof StringProperty) {
            _characterKeyProperty = (StringProperty) keyProperty;
        }
        keyProperty = getKeyProperty(_nameField);
        if (keyProperty instanceof StringProperty) {
            _nameProperty = (StringProperty) keyProperty;
        }
        keyProperty = getKeyProperty(_descriptionField);
        if (keyProperty instanceof StringProperty) {
            _descriptionProperty = (StringProperty) keyProperty;
        }
        keyProperty = getKeyProperty(_imageField);
        if (keyProperty instanceof BinaryProperty) {
            _imageProperty = (BinaryProperty) keyProperty;
        }
        keyProperty = getKeyProperty(_inactiveIndicatorField);
        if (keyProperty instanceof BooleanProperty) {
            _inactiveIndicatorProperty = (BooleanProperty) keyProperty;
        }
        keyProperty = getKeyProperty(_urlField);
        if (keyProperty instanceof StringProperty) {
            _urlProperty = (StringProperty) keyProperty;
        }
        keyProperty = getKeyProperty(_parentField);
        if (keyProperty instanceof Entity) {
            _parentProperty = (EntityReference) keyProperty;
        }
        keyProperty = getKeyProperty(_ownerField);
        if (keyProperty instanceof Entity) {
            _ownerProperty = (EntityReference) keyProperty;
        }
        keyProperty = getKeyProperty(_segmentField);
        if (keyProperty instanceof Entity) {
            _segmentProperty = (EntityReference) keyProperty;
        }
        keyProperty = getKeyProperty(_businessKeyField);
        if (keyProperty instanceof Property) {
            _businessKeyProperty = (Property) keyProperty;
        }
        keyProperty = getKeyProperty(_stateField);
        if (keyProperty instanceof Entity) {
            _stateProperty = (EntityReference) keyProperty;
        }
    }

    Object getKeyProperty(Field field) {
        if (field != null) {
            try {
                return field.get(this);
            } catch (IllegalArgumentException | IllegalAccessException ex) {
                Throwable cause = ThrowableUtils.getCause(ex);
                String message = ex.equals(cause) ? null : ex.getMessage();
                logger.error(message, cause);
                TLC.getProject().getParser().increaseErrorCount();
            }
        }
        return null;
    }

    private void setForeignKeyFields() {
        setForeignInactiveIndicatorProperty();
        setForeignOwnerProperty();
        setForeignSegmentProperty();
    }

    private void setForeignInactiveIndicatorProperty() {
        if (_inactiveIndicatorField == null && _inactiveIndicatorProperty != null) {
            String name = _inactiveIndicatorProperty.getName();
            String role = "inactive indicator property";
            String rootName = getRoot().getName();
            String message = "failed to link foreign " + role + " at entity " + rootName;
            Entity declaringEntity = _inactiveIndicatorProperty.getDeclaringEntity();
            if (declaringEntity == null) {
                message += "; declaring entity of \"" + name + "\" is null";
                logger.error(message);
                TLC.getProject().getParser().increaseErrorCount();
            } else {
                String declaringEntityName = declaringEntity.getRoot().getName();
                Field declaringField = _inactiveIndicatorProperty.getDeclaringField();
                if (declaringField == null) {
                    message += "; declaring field of \"" + name + "\" is null";
                    logger.error(message);
                    TLC.getProject().getParser().increaseErrorCount();
                } else {
                    String declaringFieldName = declaringField.getName();
                    BooleanProperty inactiveIndicatorProperty = declaringEntity.getInactiveIndicatorProperty();
                    if (_inactiveIndicatorProperty.equals(inactiveIndicatorProperty)) {
                        _inactiveIndicatorField = declaringField;
                        _inactiveIndicatorFieldName = declaringFieldName;
                        _foreignInactiveIndicatorProperty = true;
                    } else {
                        message += "; " + declaringFieldName + " is not the " + role + " of " + declaringEntityName;
                        logger.error(message);
                        TLC.getProject().getParser().increaseErrorCount();
                    }
                }
            }
        }
    }

    private void setForeignOwnerProperty() {
        if (_ownerField == null && _ownerProperty != null) {
            String name = _ownerProperty.getName();
            String role = "owner property";
            String rootName = getRoot().getName();
            String message = "failed to link foreign " + role + " at entity " + rootName;
            Entity declaringEntity = _ownerProperty.getDeclaringEntity();
            if (declaringEntity == null) {
                message += "; declaring entity of \"" + name + "\" is null";
                logger.error(message);
                TLC.getProject().getParser().increaseErrorCount();
            } else {
                String declaringEntityName = declaringEntity.getRoot().getName();
                Field declaringField = _ownerProperty.getDeclaringField();
                if (declaringField == null) {
                    message += "; declaring field of \"" + name + "\" is null";
                    logger.error(message);
                    TLC.getProject().getParser().increaseErrorCount();
                } else {
                    String declaringFieldName = declaringField.getName();
                    Entity ownerProperty = declaringEntity.getOwnerProperty();
                    if (_ownerProperty.equals(ownerProperty)) {
                        _ownerField = declaringField;
                        _ownerFieldName = declaringFieldName;
                        _foreignOwnerProperty = true;
                    } else {
                        message += "; " + declaringFieldName + " is not the " + role + " of " + declaringEntityName;
                        logger.error(message);
                        TLC.getProject().getParser().increaseErrorCount();
                    }
                }
            }
        }
    }

    private void setForeignSegmentProperty() {
        if (_segmentField == null && _segmentProperty != null) {
            String name = _segmentProperty.getName();
            String role = "segment property";
            String rootName = getRoot().getName();
            String message = "failed to link foreign " + role + " at entity " + rootName;
            Entity declaringEntity = _segmentProperty.getDeclaringEntity();
            if (declaringEntity == null) {
                message += "; declaring entity of \"" + name + "\" is null";
                logger.error(message);
                TLC.getProject().getParser().increaseErrorCount();
            } else {
                String declaringEntityName = declaringEntity.getRoot().getName();
                Field declaringField = _segmentProperty.getDeclaringField();
                if (declaringField == null) {
                    message += "; declaring field of \"" + name + "\" is null";
                    logger.error(message);
                    TLC.getProject().getParser().increaseErrorCount();
                } else {
                    String declaringFieldName = declaringField.getName();
                    Entity segmentProperty = declaringEntity.getSegmentProperty();
                    if (_segmentProperty.equals(segmentProperty)) {
                        _segmentField = declaringField;
                        _segmentFieldName = declaringFieldName;
                        _foreignSegmentProperty = true;
                    } else {
                        message += "; " + declaringFieldName + " is not the " + role + " of " + declaringEntityName;
                        logger.error(message);
                        TLC.getProject().getParser().increaseErrorCount();
                    }
                }
            }
        }
    }

    private void setTableViewEnabledIndicator() {
        if (_tableViewEnabled == null) {
            _tableViewEnabled = !isEnumerationEntity() || isUpdateEnabled();
        }
    }

    private void setBasicDatabaseOperationsAttributes() {
        Operation operation;
        /**/
        operation = (Operation) select;
        if (select != null) {
            operation.setOperationAccess(_selectOperationAccess);
        }
        /**/
        operation = (Operation) insert;
        if (insert != null) {
            operation.setOperationAccess(_insertOperationAccess);
            operation.setOperationLogging(_insertLogging);
        }
        /**/
        operation = (Operation) update;
        if (update != null) {
            operation.setOperationAccess(_updateOperationAccess);
            operation.setOperationLogging(_updateLogging);
        }
        /**/
        operation = (Operation) delete;
        if (delete != null) {
            operation.setOperationAccess(_deleteOperationAccess);
            operation.setOperationLogging(_deleteLogging);
        }
    }

    private void check() {
        checkExpressions();
        if (_rootInstance) {
            checkBinaryFields();
            checkNumericFields();
            checkStringFields();
            checkCalculableFields();
            checkInstances();
            if (_warningsEnabled) {
                checkTriggers();
                checkTreeView();
                checkVisibleFields();
                if (_businessKeyProperty == null) {
                    checkBusinessKeyProperty();
                } else {
                    checkBusinessKeyPropertyDefaultValue();
                }
            }
        }
    }

    private void checkExpressions() {
        Object o;
        Expression e;
        e = getSelectFilter();
        if (e != null) {
            verifyExpression(e);
        }
        e = getUpdateFilter();
        if (e != null) {
            verifyExpression(e);
        }
        e = getDeleteFilter();
        if (e != null) {
            verifyExpression(e);
        }
        for (Property property : getPropertiesList()) {
            e = property.getRenderingFilter();
            if (e != null) {
                verifyExpression(e, property, false);
            }
            e = property.getRequiringFilter();
            if (e != null) {
                verifyExpression(e, property, false);
            }
            e = property.getModifyingFilter();
            if (e != null) {
                verifyExpression(e, property, false);
            }
            e = property.getNullifyingFilter();
            if (e != null) {
                verifyExpression(e, property, false);
            }
            if (property instanceof Entity) {
                Entity entity = (Entity) property;
                e = entity.getSearchQueryFilter();
                if (e != null) {
                    verifyExpression(e, property, false);
                }
            }
            o = property.getInitialValue();
            if (o instanceof Expression) {
                e = (Expression) o;
                verifyExpression(e, property);
            }
            o = property.getDefaultValue();
            if (o instanceof Expression) {
                e = (Expression) o;
                verifyExpression(e, property);
            }
            o = property.getCurrentValue();
            if (o instanceof Expression) {
                e = (Expression) o;
                verifyExpression(e, property);
            }
        }
        for (Tab tab : _atlas.getTabsList()) {
            e = tab.getRenderingFilter();
            if (e != null) {
                verifyExpression(e, tab, false);
            }
        }
        for (Expression expression : _atlas.getExpressionsList()) {
            if (expression != null) {
                verifyExpression(expression);
            }
        }
    }

    private void checkBinaryFields() {
        String name = getName();
        String msg1 = name + " has the following mutually exclusive annotations: ";
        Set<String> annotations = new TreeSet<>();
        Field field;
        List<Property> list = getPropertiesList();
        for (Property property : list) {
            if (property.isBinaryData()) {
                annotations.clear();
                field = property.getDeclaringField();
                if (field != null) {
                    if (field.isAnnotationPresent(GraphicImage.class)) {
                        annotations.add("GraphicImage");
                    }
                    if (field.isAnnotationPresent(ImageProperty.class)) {
                        annotations.add("ImageProperty");
                    }
                }
                if (annotations.size() > 1) {
                    logger.error(msg1 + StringUtils.join(annotations, ", "));
                    TLC.getProject().getParser().increaseErrorCount();
                }
            }
        }
    }

    private void checkNumericFields() {
        String name = getName();
        String msg1 = name + " has the following mutually exclusive annotations: ";
        Set<String> annotations = new TreeSet<>();
        List<Property> list = getPropertiesList();
        for (Property property : list) {
            if (property.isNumericPrimitive()) {
                annotations.clear();
                if (property.isPrimaryKeyProperty()) {
                    annotations.add("(PrimaryKey)");
                }
                if (property.isSequenceProperty()) {
                    annotations.add("SequenceProperty");
                }
                if (annotations.size() > 1) {
                    logger.error(msg1 + StringUtils.join(annotations, ", "));
                    TLC.getProject().getParser().increaseErrorCount();
                }
                annotations.clear();
                if (property.isPrimaryKeyProperty() || property.isNumericKeyProperty() || property.isSequenceProperty() || property.isUniqueKeyProperty()) {
                    annotations.add("(PrimaryKey|NumericKey|SequenceProperty|UniqueKey)");
                }
                if (property.isDiscriminatorProperty()) {
                    annotations.add("DiscriminatorColumn");
                }
                if (property.isVersionProperty()) {
                    annotations.add("VersionProperty");
                }
                if (annotations.size() > 1) {
                    logger.error(msg1 + StringUtils.join(annotations, ", "));
                    TLC.getProject().getParser().increaseErrorCount();
                }
            }
        }
    }

    private void checkStringFields() {
        String name = getName();
        String msg1 = name + " has the following mutually exclusive annotations: ";
        Set<String> annotations = new TreeSet<>();
        List<Property> list = getPropertiesList();
        for (Property property : list) {
            if (property.isStringData()) {
                annotations.clear();
                if (property.isDiscriminatorProperty()) {
                    annotations.add("DiscriminatorColumn");
                }
                if (property.isFileReferenceField()) {
                    annotations.add("FileReference");
                }
                if (property.isUniformResourceLocatorField()) {
                    annotations.add("UniformResourceLocator");
                }
                if (property.isUrlProperty()) {
                    annotations.add("UrlProperty");
                }
                if (property.isVariantStringField()) {
                    annotations.add("VariantString");
                }
                if (annotations.size() > 1) {
                    logger.error(msg1 + StringUtils.join(annotations, ", "));
                    TLC.getProject().getParser().increaseErrorCount();
                }
                annotations.clear();
                if (property.isBusinessKeyProperty() || property.isCharacterKeyProperty() || property.isUniqueKeyProperty()) {
                    annotations.add("(BusinessKey|CharacterKey|UniqueKey)");
                }
                if (property.isDescriptionProperty()) {
                    annotations.add("DescriptionProperty");
                }
                if (property.isDiscriminatorProperty()) {
                    annotations.add("DiscriminatorColumn");
                }
                if (property.isNameProperty()) {
                    annotations.add("NameProperty");
                }
                if (property.isVariantStringField()) {
                    annotations.add("VariantString");
                }
                if (annotations.size() > 1) {
                    logger.error(msg1 + StringUtils.join(annotations, ", "));
                    TLC.getProject().getParser().increaseErrorCount();
                }
            }
        }
    }

    private void checkInstances() {
        Property property;
        StringProperty stringProperty;
        Object object;
        String string;
        for (Instance instance : getInstancesList()) {
            for (InstanceField field : instance.getInstanceFieldsList()) {
                property = field.getProperty();
                if (property instanceof StringProperty) {
                    stringProperty = (StringProperty) property;
                    object = field.getValue();
                    string = object instanceof String ? (String) object : null;
                    if (string == null) {
                        logger.error("null string value for field " + instance.getFullName() + "." + property.getName());
                        TLC.getProject().getParser().increaseErrorCount();
                    } else if (string.length() < IntUtils.valueOf(stringProperty.getMinLength(), 0)) {
                        logger.error("string value too small for field " + instance.getFullName() + "." + property.getName());
                        TLC.getProject().getParser().increaseErrorCount();
                    } else if (string.length() > IntUtils.valueOf(stringProperty.getMaxLength(), Integer.MAX_VALUE)) {
                        logger.error("string value too large for field " + instance.getFullName() + "." + property.getName());
                        TLC.getProject().getParser().increaseErrorCount();
                    }
                }
            }
        }
    }

    private void checkCalculableFields() {
        final String msg1 = " is calculable and therefore does not support the following annotations: ";
        final String msg2 = " is calculable and has no calculable-value expression or reference ";
        final String msg3 = " is calculable and its calculable-value references calculable property ";
        final String msg4 = " is calculable but not assignable from ";
        Set<String> annotations = new TreeSet<>();
        EntityReference reference;
        PersistentEntityReference persistentEntityReference;
        Object calculableValue;
        DataArtifact dataArtifact;
        List<Artifact> pathList;
        boolean b1, b2;
        Class<?> class1, class2;
        String simple1, simple2;
        List<Property> properties = getPropertiesList();
        for (Property property : properties) {
            if (property.isCalculable()) {
                annotations.clear();
                if (property.isPrimaryKeyProperty()) {
                    annotations.add("PrimaryKey");
                }
                if (property.isSequenceProperty()) {
                    annotations.add("SequenceProperty");
                }
                if (property.isVersionProperty()) {
                    annotations.add("VersionProperty");
                }
                if (property.isNumericKeyProperty()) {
                    annotations.add("NumericKey");
                }
                if (property.isCharacterKeyProperty()) {
                    annotations.add("CharacterKey");
                }
                if (property.isNameProperty()) {
                    annotations.add("NameProperty");
                }
                if (property.isDescriptionProperty()) {
                    annotations.add("DescriptionProperty");
                }
                if (property.isImageProperty()) {
                    annotations.add("ImageProperty");
                }
                if (property.isInactiveIndicatorProperty()) {
                    annotations.add("InactiveIndicator");
                }
                if (property.isUrlProperty()) {
                    annotations.add("UrlProperty");
                }
                if (property.isParentProperty()) {
                    annotations.add("ParentProperty");
                }
                if (property.isOwnerProperty()) {
                    annotations.add("OwnerProperty");
                }
                if (property.isSegmentProperty()) {
                    annotations.add("SegmentProperty");
                }
//              if (property.isUniqueKeyProperty()) {
//                  annotations.add("UniqueKey");
//              }
                if (property.isBusinessKeyProperty()) {
                    annotations.add("BusinessKey");
                }
                if (property.isDiscriminatorProperty()) {
                    annotations.add("DiscriminatorColumn");
                }
                if (property.isStateProperty()) {
                    annotations.add("StateProperty");
                }
                if (property instanceof EntityReference) {
                    reference = (EntityReference) property;
                    if (reference.isManyToOne()) {
                        annotations.add("ManyToOne");
                    }
                    if (reference.isOneToOne()) {
                        annotations.add("OneToOne");
                    }
                }
                if (property instanceof PersistentEntityReference) {
                    persistentEntityReference = (PersistentEntityReference) property;
                    if (persistentEntityReference.isForeignKey()) {
                        annotations.add("ForeignKey");
                    }
                }
                if (annotations.size() > 1) {
                    logger.error(property.getFullName() + msg1 + StringUtils.join(annotations, ", "));
                    TLC.getProject().getParser().increaseErrorCount();
                }
                if (property instanceof CalculableProperty) {
                    calculableValue = ((CalculableProperty) property).getCalculableValue();
                    if (calculableValue == null) {
                        logger.error(property.getFullName() + msg2);
                        TLC.getProject().getParser().increaseErrorCount();
                    } else if (calculableValue instanceof DataArtifact) {
                        dataArtifact = (DataArtifact) calculableValue;
                        pathList = dataArtifact.getPathList();
                        for (Artifact artifact : pathList) {
                            if (artifact instanceof Property) {
                                if (((Property) artifact).isCalculable()) {
                                    logger.error(property.getFullName() + msg3 + artifact.getFullName());
                                    TLC.getProject().getParser().increaseErrorCount();
                                }
                            }
                        }
                        b1 = property instanceof Entity;
                        b2 = dataArtifact instanceof Entity;
                        class1 = property.getDataClass();
                        class2 = dataArtifact.getDataClass();
                        simple1 = property.getDataClass().getSimpleName();
                        simple2 = dataArtifact.getDataClass().getSimpleName();
                        if (class1.isAssignableFrom(class2) || (b1 && b2 && simple1.equals(simple2))) {
                        } else {
                            logger.error(property.getFullName() + msg4 + dataArtifact.getFullName());
                            TLC.getProject().getParser().increaseErrorCount();
                        }
                    } else if (calculableValue instanceof Expression) {
                        verifyExpression((Expression) calculableValue, property);
                    }
                }
            }
        }
        checkCalculableKeyFields();
        checkCalculableForeignQueryProperties();
    }

    private void checkCalculableKeyFields() {
        final String msg5 = " is calculable and therefore cannot be a field of key ";
        List<Key> keys = getKeysList();
        List<KeyField> fields;
        Property property;
        for (Key key : keys) {
            fields = key.getKeyFieldsList();
            for (KeyField field : fields) {
                property = field.getProperty();
                if (property.isCalculable()) {
                    logger.error(property.getFullName() + msg5 + key.getFullName());
                    TLC.getProject().getParser().increaseErrorCount();
                }
            }
        }
    }

    private void checkCalculableForeignQueryProperties() {
        String message = "failed to link foreign query property to " + getFullName();
        Property property;
        for (String key : _foreignQueryProperties.keySet()) {
            property = _foreignQueryProperties.get(key);
            boolean calculable = property.isCalculable();
            if (calculable) {
                logger.error(message + "; " + property.getName() + " in " + property.getFullName() + " is a calculable property");
                TLC.getProject().getParser().increaseErrorCount();
            } else {
                List<Artifact> pathList = property.getPathList();
                for (Artifact artifact : pathList) {
                    if (artifact instanceof Property) {
                        calculable = ((Property) artifact).isCalculable();
                        if (calculable) {
                            logger.error(message + "; " + artifact.getName() + " in " + property.getFullName() + " is a calculable property");
                            TLC.getProject().getParser().increaseErrorCount();
                            break;
                        }
                    }
                }
            }
        }
    }

    private void checkTriggers() {
        String tag;
        String message;
        State state;
        ProcessOperation operation;
        List<State> initialStatesList;
        List<Trigger> triggers = getTriggersList();
        for (Trigger trigger : triggers) {
            tag = " trigger " + trigger.getName() + " at entity " + getName();
            state = trigger.getState();
            operation = trigger.getOperation();
            if (state == null) {
                message = "invalid" + tag;
                message += "; trigger state is null";
                logger.warn(message);
                TLC.getProject().getParser().increaseWarningCount();
            } else if (operation == null) {
                message = "invalid" + tag;
                message += "; trigger operation is null";
                logger.warn(message);
                TLC.getProject().getParser().increaseWarningCount();
            } else {
                initialStatesList = operation.getInitialStatesList();
                if (initialStatesList == null || !initialStatesList.contains(state)) {
                    message = "suspicious" + tag;
                    message += "; " + state.getName() + " is not an initial state of " + operation.getName();
                    logger.warn(message);
                    TLC.getProject().getParser().increaseWarningCount();
                }
            }
        }
    }

    private void checkTreeView() {
        if (_treeViewEnabled) {
            if (_parentProperty == null) {
                String message = getName() + " tree view will not be enabled because this entity does not have a parent property";
                logger.warn(message);
                TLC.getProject().getParser().increaseWarningCount();
            }
            if (_ownerProperty != null) {
                String message = getName() + " tree view will not be enabled because this entity has an owner property";
                logger.warn(message);
                TLC.getProject().getParser().increaseWarningCount();
            }
            if (_segmentProperty != null) {
                String message = getName() + " tree view will not be enabled because this entity has a segment property";
                logger.warn(message);
                TLC.getProject().getParser().increaseWarningCount();
            }
        }
        if (isTreeViewEnabled()) {
            if (_selectRowsLimit != 0) {
                String message = getName() + " tree view will ignore the select rows limit set for this entity";
                logger.warn(message);
                TLC.getProject().getParser().increaseWarningCount();
            }
        }
    }

    private void checkVisibleFields() {
        String msg1;
        String name = getName();
        int visibleTableField = 0;
        int visibleReportField = 0;
        int visibleRequired = 0;
        List<Property> list = getPropertiesList();
        for (Property property : list) {
            if (property.isHiddenField()) {
                continue;
            }
            if (property.isTableField()) {
                visibleTableField++;
            }
            if (property.isReportField()) {
                visibleReportField++;
            }
            if (property.isRequiredProperty()) {
                visibleRequired++;
            }
        }
        if (visibleTableField == 0) {
            msg1 = name + " does not have any visible table fields";
            logger.warn(msg1);
            TLC.getProject().getParser().increaseWarningCount();
        }
        if (visibleReportField == 0) {
            msg1 = name + " does not have any visible report fields";
            logger.warn(msg1);
            TLC.getProject().getParser().increaseWarningCount();
        }
        if (visibleRequired == 0 && isInsertEnabled()) {
            msg1 = name + " does not have any visible required properties";
            logger.warn(msg1);
            TLC.getProject().getParser().increaseWarningCount();
        }
    }

    private void checkBusinessKeyProperty() {
        String warn = getName() + " does not have a business key property and is referenced by ";
        String referenceFullName = null;
        EntityReference entityReference;
        List<Property> references = _atlas.getReferencesList();
        int size = references.size();
        if (size > 0) {
            size = 0;
            Entity declaringEntity;
            List<Property> declaringEntityProperties;
            for (Property reference : references) {
                if (reference.isHiddenField() || reference.isInherited()) {
                    continue;
                }
                declaringEntity = reference.getDeclaringEntity();
                if (declaringEntity.isExistentiallyIndependent()) {
                    size++;
                    referenceFullName = reference.getFullName();
                    continue;
                } else {
                    entityReference = (EntityReference) reference;
                    if (MasterDetailView.NONE.equals(entityReference.getMasterDetailView())) {
                        size++;
                        referenceFullName = reference.getFullName();
                        continue;
                    }
                }
                declaringEntityProperties = declaringEntity.getPropertiesList();
                for (Property property : declaringEntityProperties) {
                    if (property.equals(reference)) {
                        continue;
                    }
                    if (property instanceof EntityReference) {
                        entityReference = (EntityReference) property;
                        if (MasterDetailView.NONE.equals(entityReference.getMasterDetailView())) {
                            continue;
                        }
                        size++;
                        referenceFullName = reference.getFullName();
                        break;
                    }
                }
            }
            if (size > 0) {
                String by = size == 1 ? referenceFullName : size + " non-exclusively-existentially-dependent properties";
                logger.warn(warn + by);
                TLC.getProject().getParser().increaseWarningCount();
            }
        }
        List<Parameter> parameterReferencesList = _atlas.getParameterReferencesList();
        size = parameterReferencesList.size();
        if (size > 0) {
            size = 0;
            for (Parameter reference : parameterReferencesList) {
                if (reference.isInstanceReferenceField()) {
                    entityReference = (EntityReference) reference;
                    if (!entityReference.isConsoleViewEnabled()) {
                        continue;
                    }
                }
                size++;
                referenceFullName = reference.getFullName();
            }
            if (size > 0) {
                String by = size == 1 ? referenceFullName : size + " parameters";
                logger.warn(warn + by);
                TLC.getProject().getParser().increaseWarningCount();
            }
        }
    }

    private void checkBusinessKeyPropertyDefaultValue() {
        if (_insertEnabled) {
            if (_businessKeyProperty != null && !_businessKeyProperty.isCreateField() && _businessKeyProperty.getDefaultValue() == null) {
                String bkfn = _businessKeyProperty.getFullName();
                String pkfn = _primaryKeyProperty == null ? null : _primaryKeyProperty.getFullName();
                String message;
                if (_primaryKeyProperty == null) {
                    message = "property " + bkfn + " seems to need a default value";
                } else if (_businessKeyProperty.isStringData()) {
                    StringData bksd = (StringData) _businessKeyProperty;
                    bksd.setDefaultValue(XB.toCharString(_primaryKeyProperty));
                    if (_businessKeyProperty.getDefaultValueTag() == null) {
                        _businessKeyProperty.setDefaultValueTag("ID");
                    }
                    message = "property " + bkfn + " seemed to need a default value so it was set to \"" + pkfn + ".toCharString()\"";
                } else if (_businessKeyProperty.isIntegerData() && _primaryKeyProperty.isIntegerData() && _businessKeyProperty != _primaryKeyProperty) {
                    IntegerData bkid = (IntegerData) _businessKeyProperty;
                    IntegerData pkid = (IntegerData) _primaryKeyProperty;
                    bkid.setDefaultValue(pkid);
                    if (_businessKeyProperty.getDefaultValueTag() == null) {
                        _businessKeyProperty.setDefaultValueTag("ID");
                    }
                    message = "property " + bkfn + " seemed to need a default value so it was set to \"" + pkfn + "\"";
                } else {
                    message = "property " + bkfn + " seems to need a default value";
                }
                logger.warn(message);
                TLC.getProject().getParser().increaseWarningCount();
            }
        }
    }

    @Override
    void setDeclared(String name, Artifact declaringArtifact, Field declaringField, int declaringFieldIndex) {
        super.setDeclared(name, declaringArtifact, declaringField, declaringFieldIndex);
        initializeInheritanceFields();
    }

    // <editor-fold defaultstate="collapsed" desc="annotate">
    @Override
    void initializeAnnotations() {
        super.initializeAnnotations();
        _annotatedWithAbstractClass = false;
        _annotatedWithAllocationOverride = false;
        _annotatedWithAllocationOverrides = false;
        _annotatedWithEntityPrimaryKey = false;
        _annotatedWithEntitySequenceProperty = false;
        _annotatedWithEntityVersionProperty = false;
        _annotatedWithEntityNumericKey = false;
        _annotatedWithEntityCharacterKey = false;
        _annotatedWithEntityNameProperty = false;
        _annotatedWithEntityDescriptionProperty = false;
        _annotatedWithEntityInactiveIndicator = false;
        _annotatedWithEntityUrlProperty = false;
        _annotatedWithEntityParentProperty = false;
        _annotatedWithEntityOwnerProperty = false;
        _annotatedWithEntitySegmentProperty = false;
        _annotatedWithEntityBusinessKey = false;
        _annotatedWithEntityClass = false;
        _annotatedWithEntityDataGen = false;
        _annotatedWithEntitySelectOperation = false;
        _annotatedWithEntityInsertOperation = false;
        _annotatedWithEntityUpdateOperation = false;
        _annotatedWithEntityDeleteOperation = false;
        _annotatedWithEntityReportOperation = false;
        _annotatedWithEntityExportOperation = false;
        _annotatedWithEntityTableView = false;
        _annotatedWithEntityDetailView = false;
        _annotatedWithEntityTreeView = false;
        _annotatedWithEntityConsoleView = false;
        _annotatedWithEntityWarnings = false;
        _annotatedWithEntityCodeGen = false;
        _annotatedWithEntityDocGen = false;
        _annotatedWithEntityReferenceSearch = false;
        _annotatedWithFilter = false;
        _annotatedWithExtension = false;
        _annotatedWithOneToOne = false;
        _annotatedWithManyToOne = false;
        _annotatedWithQueryMapping = false;
        _annotatedWithEntityReferenceDataGen = false;
        _businessKeyType = BusinessKeyType.UNSPECIFIED;
        _basicOperationEntity = false;
        _catalogEntity = false;
        _existentiallyIndependent = true;
        _variantEntity = false;
        _resourceType = ResourceType.UNSPECIFIED;
        _resourceGender = ResourceGender.UNSPECIFIED;
        _propertiesPrefix = "";
        _propertiesSuffix = "";
        _helpFileName = "";
        _startWith = 1;
        _seriesStart = 1;
        _seriesStop = 100;
        _seriesStep = 1;
        _selectEnabled = true;
//      _selectRestricted = true;
        _selectOperationAccess = OperationAccess.UNSPECIFIED;
        _selectOnloadOption = SelectOnloadOption.DEFAULT;
        _selectRowsLimit = Constants.DEFAULT_SELECT_ROWS_LIMIT;
        _selectSortOption = SortOption.ASC;
        _insertEnabled = !isEnumerationEntity();
//      _insertRestricted = true;
        _insertOperationAccess = OperationAccess.UNSPECIFIED;
        _insertLogging = OperationLogging.UNSPECIFIED;
        _updateEnabled = true;
//      _updateRestricted = true;
        _updateOperationAccess = OperationAccess.UNSPECIFIED;
        _updateLogging = OperationLogging.UNSPECIFIED;
        _deleteEnabled = !isEnumerationEntity();
//      _deleteRestricted = true;
        _deleteOperationAccess = OperationAccess.UNSPECIFIED;
        _deleteLogging = OperationLogging.UNSPECIFIED;
        _reportEnabled = !isEnumerationEntity();
        _reportRowsLimit = Constants.DEFAULT_REPORT_ROWS_LIMIT;
        _reportSortOption = SortOption.ASC;
        _exportEnabled = !isEnumerationEntity();
        _exportRowsLimit = Constants.DEFAULT_EXPORT_ROWS_LIMIT;
        _exportSortOption = SortOption.ASC;
        _tableViewEnabled = null;
        _tableViewWithInsertEnabled = true;
        _tableViewWithUpdateEnabled = true;
        _tableViewWithDeleteEnabled = true;
        _tableViewWithMasterHeading = true;
        _tableViewRowsLimit = Constants.DEFAULT_ROWS_PER_PAGE_LIMIT;
        _tableViewRows = Constants.DEFAULT_ROWS_PER_PAGE;
//      _tableViewWidth = 1200;
        _detailViewEnabled = !isEnumerationEntity();
        _detailViewWithMasterHeading = true;
//      _detailViewWidth = 1200;
        _treeViewEnabled = false;
        _consoleViewEnabled = !isEnumerationEntity();
//      _consoleViewWidth = 1200;
        _warningsEnabled = true;
        _bplCodeGenEnabled = true;
        _bwsCodeGenEnabled = !isEnumerationEntity();
        _fwsCodeGenEnabled = !isEnumerationEntity();
        _sqlCodeGenEnabled = true;
        _entityStateCodeGenEnabled = true;
        _entityClassDiagramGenEnabled = true;
        _entityStateDiagramGenEnabled = true;
        _entityInsertActivityDiagramGenEnabled = true;
        _entityUpdateActivityDiagramGenEnabled = true;
        _searchType = SearchType.UNSPECIFIED;
        _listStyle = ListStyle.UNSPECIFIED;
        _searchDisplayMode = DisplayMode.UNSPECIFIED;
        _filterInactiveIndicatorProperty = false;
        _filterOwnerProperty = false;
        _filterSegmentProperty = false;
        _fetchType = FetchType.UNSPECIFIED;
        _cascadeType = new CascadeType[]{CascadeType.UNSPECIFIED};
        _navigability = Navigability.UNSPECIFIED;
        _masterDetailView = MasterDetailView.UNSPECIFIED;
        _quickAddingFilter = QuickAddingFilter.NONE;
        _keyPropertiesQueryMappingEnabled = true;
        _primaryKeyFieldName = null;
        _primaryKeyField = null;
        _primaryKeyProperty = null;
        _sequenceFieldName = null;
        _sequenceField = null;
        _sequenceProperty = null;
        _versionFieldName = null;
        _versionField = null;
        _versionProperty = null;
        _numericKeyFieldName = null;
        _numericKeyField = null;
        _numericKeyProperty = null;
        _characterKeyFieldName = null;
        _characterKeyField = null;
        _characterKeyProperty = null;
        _nameFieldName = null;
        _nameField = null;
        _nameProperty = null;
        _descriptionFieldName = null;
        _descriptionField = null;
        _descriptionProperty = null;
        _imageFieldName = null;
        _imageField = null;
        _imageProperty = null;
        _inactiveIndicatorFieldName = null;
        _inactiveIndicatorField = null;
        _inactiveIndicatorProperty = null;
        _foreignInactiveIndicatorProperty = false;
        _urlFieldName = null;
        _urlField = null;
        _urlProperty = null;
        _parentFieldName = null;
        _parentField = null;
        _parentProperty = null;
        _ownerFieldName = null;
        _ownerField = null;
        _ownerProperty = null;
        _foreignOwnerProperty = false;
        _segmentFieldName = null;
        _segmentField = null;
        _segmentProperty = null;
        _foreignSegmentProperty = false;
        _businessKeyFieldName = null;
        _businessKeyField = null;
        _businessKeyProperty = null;
        _stateFieldName = null;
        _stateField = null;
        _stateProperty = null;
    }

    @Override
    void annotate(Class<?> type) {
        super.annotate(type);
        if (type != null) {
            annotateAbstractClass(type);
            annotateAllocationOverride(type);
            annotateAllocationOverrides(type);
//          annotateEntityPrimaryKey(type);
//          annotateEntitySequenceProperty(type);
//          annotateEntityVersionProperty(type);
//          annotateEntityNumericKey(type);
//          annotateEntityCharacterKey(type);
//          annotateEntityNameProperty(type);
//          annotateEntityDescriptionProperty(type);
//          annotateEntityInactiveIndicator(type);
//          annotateEntityUrlProperty(type);
//          annotateEntityParentProperty(type);
//          annotateEntityOwnerProperty(type);
//          annotateEntitySegmentProperty(type);
//          annotateEntityBusinessKey(type);
            annotateEntityClass(type);
            annotateEntityDataGen(type);
            annotateEntitySelectOperation(type);
            annotateEntityInsertOperation(type);
            annotateEntityUpdateOperation(type);
            annotateEntityDeleteOperation(type);
            annotateEntityReportOperation(type);
            annotateEntityExportOperation(type);
            annotateEntityTableView(type);
            annotateEntityDetailView(type);
            annotateEntityTreeView(type);
            annotateEntityConsoleView(type);
            annotateEntityWarnings(type);
            annotateEntityCodeGen(type);
            annotateEntityDocGen(type);
            annotateEntityReferenceSearch(type);
        }
    }

    @Override
    void annotate(Field field) {
        super.annotate(field);
        if (field != null) {
            if (isEntityReference()) {
                annotateEntityReferenceDataGen(field);
                annotateEntityReferenceSearch(field);
                annotateFilter(field);
//              annotateExtension(field);
                if (!isExtension()) {
                    annotateOneToOne(field);
                    if (!isOneToOne()) {
                        annotateManyToOne(field);
                    }
                }
                annotateQueryMapping(field);
            }
        }
    }

    @Override
//  @SuppressWarnings("deprecation")
    protected List<Class<? extends Annotation>> getValidTypeAnnotations() {
        List<Class<? extends Annotation>> valid = super.getValidTypeAnnotations();
        valid.add(AbstractClass.class);
        valid.add(AllocationOverride.class);
        valid.add(AllocationOverrides.class);
//      valid.add(EntityBusinessKey.class);
//      valid.add(EntityCharacterKey.class);
        valid.add(EntityClass.class);
        valid.add(EntityCodeGen.class);
        valid.add(EntityConsoleView.class);
        valid.add(EntityDeleteOperation.class);
//      valid.add(EntityDescriptionProperty.class);
        valid.add(EntityDetailView.class);
        valid.add(EntityDocGen.class);
        valid.add(EntityExportOperation.class);
//      valid.add(EntityInactiveIndicator.class);
        valid.add(EntityInsertOperation.class);
//      valid.add(EntityNameProperty.class);
//      valid.add(EntityNumericKey.class);
//      valid.add(EntityOwnerProperty.class);
//      valid.add(EntityParentProperty.class);
//      valid.add(EntityPrimaryKey.class);
        valid.add(EntityReferenceSearch.class);
        valid.add(EntityReportOperation.class);
//      valid.add(EntitySegmentProperty.class);
//      valid.add(EntitySequenceProperty.class);
        valid.add(EntitySelectOperation.class);
        valid.add(EntityTableView.class);
        valid.add(EntityTreeView.class);
        valid.add(EntityUpdateOperation.class);
//      valid.add(EntityUrlProperty.class);
//      valid.add(EntityVersionProperty.class);
        valid.add(EntityWarnings.class);
        if (isEnumerationEntity()) {
        } else {
            valid.add(EntityDataGen.class);
        }
        return valid;
    }

    @Override
//  @SuppressWarnings("deprecation")
    protected List<Class<? extends Annotation>> getValidFieldAnnotations() {
        List<Class<? extends Annotation>> valid = super.getValidFieldAnnotations();
        valid.add(Allocation.class);
        if (isParameter()) {
            valid.add(EntityReferenceSearch.class);
            valid.add(Filter.class);
            valid.add(InstanceReference.class);
            valid.add(ParameterField.class);
        }
        if (isProperty()) {
            valid.add(BaseField.class);
            valid.add(CastingField.class);
            valid.add(ColumnField.class);
            valid.add(EntityReferenceDataGen.class);
            valid.add(EntityReferenceSearch.class);
//          valid.add(Extension.class);
            valid.add(Filter.class);
//          valid.add(LastUserProperty.class);
            valid.add(ManyToOne.class);
            valid.add(OneToOne.class);
            valid.add(OwnerProperty.class);
            valid.add(ParentProperty.class);
            valid.add(PropertyField.class);
            valid.add(PropertyAggregation.class);
            valid.add(QueryMapping.class);
            valid.add(SegmentProperty.class);
            valid.add(UniqueKey.class);
        }
        return valid;
    }

    private void annotateAbstractClass(Class<?> type) {
        /*
         * AbstractClass annotation cannot be "inherited"
         */
        AbstractClass annotation = type.getAnnotation(AbstractClass.class);
        _annotatedWithAbstractClass = annotation != null && annotation.value();
    }

    private void annotateAllocationOverride(Class<?> type) {
        Class<?> annotatedClass = XS1.getAnnotatedClass(type, AllocationOverride.class);
        if (annotatedClass != null) {
            AllocationOverride annotation = annotatedClass.getAnnotation(AllocationOverride.class);
            if (annotation != null) {
                _annotatedWithAllocationOverride = putAllocationOverride(annotation, type);
            }
        }
    }

    private void annotateAllocationOverrides(Class<?> type) {
        Class<?> annotatedClass = XS1.getAnnotatedClass(type, AllocationOverrides.class);
        if (annotatedClass != null) {
            AllocationOverrides annotation = annotatedClass.getAnnotation(AllocationOverrides.class);
            if (annotation != null) {
                for (AllocationOverride value : annotation.value()) {
                    _annotatedWithAllocationOverrides |= putAllocationOverride(value, type);
                }
            }
        }
    }

    private boolean putAllocationOverride(AllocationOverride annotation, Class<?> type) {
        if (annotation != null) {
            String name = annotation.field();
            if (StringUtils.isNotBlank(name)) {
                Field field = XS1.getEntityField(name, type);
                if (field != null) {
                    _allocationOverrides.put(name, annotation);
                    return true;
                }
            }
        }
        return false;
    }

    // <editor-fold defaultstate="collapsed" desc="deprecated methods">
    /*
    @SuppressWarnings("deprecation")
    private void annotateEntityPrimaryKey(Class<?> type) {
        Class<?> annotatedClass = XS1.getAnnotatedClass(type, EntityPrimaryKey.class);
        if (annotatedClass != null) {
            EntityPrimaryKey annotation = annotatedClass.getAnnotation(EntityPrimaryKey.class);
            if (annotation != null) {
                _primaryKeyFieldName = annotation.value();
                if (StringUtils.isBlank(_primaryKeyFieldName)) {
                    _primaryKeyFieldName = null;
                } else {
                    _primaryKeyField = getPrimaryKeyField(_primaryKeyFieldName, type);
                    _annotatedWithEntityPrimaryKey = _primaryKeyField != null;
                }
            }
        }
    }

    @SuppressWarnings("deprecation")
    private void annotateEntitySequenceProperty(Class<?> type) {
        Class<?> annotatedClass = XS1.getAnnotatedClass(type, EntitySequenceProperty.class);
        if (annotatedClass != null) {
            EntitySequenceProperty annotation = annotatedClass.getAnnotation(EntitySequenceProperty.class);
            if (annotation != null) {
                _sequenceFieldName = annotation.value();
                if (StringUtils.isBlank(_sequenceFieldName)) {
                    _sequenceFieldName = null;
                } else {
                    _sequenceField = getSequenceField(_sequenceFieldName, type);
                    _annotatedWithEntitySequenceProperty = _sequenceField != null;
                }
            }
        }
    }

    @SuppressWarnings("deprecation")
    private void annotateEntityVersionProperty(Class<?> type) {
        Class<?> annotatedClass = XS1.getAnnotatedClass(type, EntityVersionProperty.class);
        if (annotatedClass != null) {
            EntityVersionProperty annotation = annotatedClass.getAnnotation(EntityVersionProperty.class);
            if (annotation != null) {
                _versionFieldName = annotation.value();
                if (StringUtils.isBlank(_versionFieldName)) {
                    _versionFieldName = null;
                } else {
                    _versionField = getVersionField(_versionFieldName, type);
                    _annotatedWithEntityVersionProperty = _versionField != null;
                }
            }
        }
    }

    @SuppressWarnings("deprecation")
    private void annotateEntityNumericKey(Class<?> type) {
        Class<?> annotatedClass = XS1.getAnnotatedClass(type, EntityNumericKey.class);
        if (annotatedClass != null) {
            EntityNumericKey annotation = annotatedClass.getAnnotation(EntityNumericKey.class);
            if (annotation != null) {
                _numericKeyFieldName = annotation.value();
                if (StringUtils.isBlank(_numericKeyFieldName)) {
                    _numericKeyFieldName = null;
                } else {
                    _numericKeyField = getNumericKeyField(_numericKeyFieldName, type);
                    _annotatedWithEntityNumericKey = _numericKeyField != null;
                }
            }
        }
    }

    @SuppressWarnings("deprecation")
    private void annotateEntityCharacterKey(Class<?> type) {
        Class<?> annotatedClass = XS1.getAnnotatedClass(type, EntityCharacterKey.class);
        if (annotatedClass != null) {
            EntityCharacterKey annotation = annotatedClass.getAnnotation(EntityCharacterKey.class);
            if (annotation != null) {
                _characterKeyFieldName = annotation.value();
                if (StringUtils.isBlank(_characterKeyFieldName)) {
                    _characterKeyFieldName = null;
                } else {
                    _characterKeyField = getCharacterKeyField(_characterKeyFieldName, type);
                    _annotatedWithEntityCharacterKey = _characterKeyField != null;
                }
            }
        }
    }

    @SuppressWarnings("deprecation")
    private void annotateEntityNameProperty(Class<?> type) {
        Class<?> annotatedClass = XS1.getAnnotatedClass(type, EntityNameProperty.class);
        if (annotatedClass != null) {
            EntityNameProperty annotation = annotatedClass.getAnnotation(EntityNameProperty.class);
            if (annotation != null) {
                _nameFieldName = annotation.value();
                if (StringUtils.isBlank(_nameFieldName)) {
                    _nameFieldName = null;
                } else {
                    _nameField = getNameField(_nameFieldName, type);
                    _annotatedWithEntityNameProperty = _nameField != null;
                }
            }
        }
    }

    @SuppressWarnings("deprecation")
    private void annotateEntityDescriptionProperty(Class<?> type) {
        Class<?> annotatedClass = XS1.getAnnotatedClass(type, EntityDescriptionProperty.class);
        if (annotatedClass != null) {
            EntityDescriptionProperty annotation = annotatedClass.getAnnotation(EntityDescriptionProperty.class);
            if (annotation != null) {
                _descriptionFieldName = annotation.value();
                if (StringUtils.isBlank(_descriptionFieldName)) {
                    _descriptionFieldName = null;
                } else {
                    _descriptionField = getDescriptionField(_descriptionFieldName, type);
                    _annotatedWithEntityDescriptionProperty = _descriptionField != null;
                }
            }
        }
    }

    @SuppressWarnings("deprecation")
    private void annotateEntityInactiveIndicator(Class<?> type) {
        Class<?> annotatedClass = XS1.getAnnotatedClass(type, EntityInactiveIndicator.class);
        if (annotatedClass != null) {
            EntityInactiveIndicator annotation = annotatedClass.getAnnotation(EntityInactiveIndicator.class);
            if (annotation != null) {
                _inactiveIndicatorFieldName = annotation.value();
                if (StringUtils.isBlank(_inactiveIndicatorFieldName)) {
                    _inactiveIndicatorFieldName = null;
                } else {
                    _inactiveIndicatorField = getInactiveIndicatorField(_inactiveIndicatorFieldName, type);
                    _annotatedWithEntityInactiveIndicator = _inactiveIndicatorField != null;
                }
            }
        }
    }

    @SuppressWarnings("deprecation")
    private void annotateEntityUrlProperty(Class<?> type) {
        Class<?> annotatedClass = XS1.getAnnotatedClass(type, EntityUrlProperty.class);
        if (annotatedClass != null) {
            EntityUrlProperty annotation = annotatedClass.getAnnotation(EntityUrlProperty.class);
            if (annotation != null) {
                _urlFieldName = annotation.value();
                if (StringUtils.isBlank(_urlFieldName)) {
                    _urlFieldName = null;
                } else {
                    _urlField = getUrlField(_urlFieldName, type);
                    _annotatedWithEntityUrlProperty = _urlField != null;
                }
            }
        }
    }

    @SuppressWarnings("deprecation")
    private void annotateEntityParentProperty(Class<?> type) {
        Class<?> annotatedClass = XS1.getAnnotatedClass(type, EntityParentProperty.class);
        if (annotatedClass != null) {
            EntityParentProperty annotation = annotatedClass.getAnnotation(EntityParentProperty.class);
            if (annotation != null) {
                _parentFieldName = annotation.value();
                if (StringUtils.isBlank(_parentFieldName)) {
                    _parentFieldName = null;
                } else {
                    Field field = getParentField(_parentFieldName, Entity.class);
                    _parentField = field == null ? null : getParentField(_parentFieldName, field.getDeclaringClass());
                    _annotatedWithEntityParentProperty = _parentField != null;
                }
            }
        }
    }

    @SuppressWarnings("deprecation")
    private void annotateEntityOwnerProperty(Class<?> type) {
        Class<?> annotatedClass = XS1.getAnnotatedClass(type, EntityOwnerProperty.class);
        if (annotatedClass != null) {
            EntityOwnerProperty annotation = annotatedClass.getAnnotation(EntityOwnerProperty.class);
            if (annotation != null) {
                _ownerFieldName = annotation.value();
                if (StringUtils.isBlank(_ownerFieldName)) {
                    _ownerFieldName = null;
                } else {
                    _ownerField = getOwnerField(_ownerFieldName, type);
                    _annotatedWithEntityOwnerProperty = _ownerField != null;
                }
            }
        }
    }

    @SuppressWarnings("deprecation")
    private void annotateEntitySegmentProperty(Class<?> type) {
        Class<?> annotatedClass = XS1.getAnnotatedClass(type, EntitySegmentProperty.class);
        if (annotatedClass != null) {
            EntitySegmentProperty annotation = annotatedClass.getAnnotation(EntitySegmentProperty.class);
            if (annotation != null) {
                _segmentFieldName = annotation.value();
                if (StringUtils.isBlank(_segmentFieldName)) {
                    _segmentFieldName = null;
                } else {
                    _segmentField = getSegmentField(_segmentFieldName, type);
                    _annotatedWithEntitySegmentProperty = _segmentField != null;
                }
            }
        }
    }

    @SuppressWarnings("deprecation")
    private void annotateEntityBusinessKey(Class<?> type) {
        Class<?> annotatedClass = XS1.getAnnotatedClass(type, EntityBusinessKey.class);
        if (annotatedClass != null) {
            EntityBusinessKey annotation = annotatedClass.getAnnotation(EntityBusinessKey.class);
            if (annotation != null) {
                _businessKeyType = annotation.value();
                _annotatedWithEntityBusinessKey = true;
            }
        }
        setBusinessKey();
    }
    **/
    // </editor-fold>
//
    private void setBusinessKey() {
        if (_businessKeyFieldName == null || _businessKeyField == null) {
            switch (_businessKeyType) {
                case CHARACTER_KEY_PROPERTY:
                    if (_characterKeyField != null) {
                        _businessKeyFieldName = _characterKeyFieldName;
                        _businessKeyField = _characterKeyField;
                    }
                    break;
                case NUMERIC_KEY_PROPERTY:
                    if (_numericKeyField != null) {
                        _businessKeyFieldName = _numericKeyFieldName;
                        _businessKeyField = _numericKeyField;
                    }
                    break;
                case UNSPECIFIED:
                    if (_characterKeyField != null) {
                        _businessKeyFieldName = _characterKeyFieldName;
                        _businessKeyField = _characterKeyField;
                    } else if (_numericKeyField != null) {
                        _businessKeyFieldName = _numericKeyFieldName;
                        _businessKeyField = _numericKeyField;
                    }
                    break;
            }
        }
    }

    private void annotateEntityClass(Class<?> type) {
        Class<?> annotatedClass = XS1.getAnnotatedClass(type, EntityClass.class);
        if (annotatedClass != null) {
            EntityClass annotation = annotatedClass.getAnnotation(EntityClass.class);
            if (annotation != null) {
                _basicOperationEntity = annotation.base().toBoolean(_basicOperationEntity);
                _catalogEntity = annotation.catalog().toBoolean(_catalogEntity);
                _existentiallyIndependent = annotation.independent().toBoolean(_existentiallyIndependent);
                _variantEntity = annotation.variant().toBoolean(_variantEntity);
                _resourceType = annotation.resourceType();
                _resourceGender = annotation.resourceGender();
//              _propertiesPrefix = annotation.propertiesPrefix();
//              _propertiesSuffix = annotation.propertiesSuffix();
                _helpFileName = annotation.helpFile();
                _startWith = Math.max(0, annotation.startWith());
                _annotatedWithEntityClass = true;
            }
        }
    }

    private void annotateEntityDataGen(Class<?> type) {
        Class<?> annotatedClass = XS1.getAnnotatedClass(type, EntityDataGen.class);
        if (annotatedClass != null) {
            EntityDataGen annotation = annotatedClass.getAnnotation(EntityDataGen.class);
            if (annotation != null) {
                _seriesStart = Math.min(Constants.MAX_ENTITY_SERIES_START, Math.max(0, annotation.start()));
                _seriesStop = Math.min(Constants.MAX_ENTITY_SERIES_STOP, Math.max(0, annotation.stop()));
                _seriesStep = Math.min(Constants.MAX_ENTITY_SERIES_STEP, Math.max(0, annotation.step()));
                _annotatedWithEntityDataGen = true;
            }
        }
    }

    private void annotateEntitySelectOperation(Class<?> type) {
        final int MAX = Constants.MAXIMUM_SELECT_ROWS_LIMIT;
        final int MIN = 0;
        Class<?> annotatedClass = XS1.getAnnotatedClass(type, EntitySelectOperation.class);
        if (annotatedClass != null) {
            EntitySelectOperation annotation = annotatedClass.getAnnotation(EntitySelectOperation.class);
            if (annotation != null) {
                _selectEnabled = annotation.enabled().toBoolean(_selectEnabled);
//              _selectRestricted = annotation.restricted().toBoolean(_selectRestricted);
                _selectOperationAccess = annotation.access();
                _selectOnloadOption = annotation.onload();
                _selectRowsLimit = Math.min(MAX, Math.max(MIN, annotation.rowsLimit()));
                _selectSortOption = annotation.sortOption();
                _annotatedWithEntitySelectOperation = true;
            }
        }
    }

    private void annotateEntityInsertOperation(Class<?> type) {
        Class<?> annotatedClass = XS1.getAnnotatedClass(type, EntityInsertOperation.class);
        if (annotatedClass != null) {
            EntityInsertOperation annotation = annotatedClass.getAnnotation(EntityInsertOperation.class);
            if (annotation != null) {
                _insertEnabled = annotation.enabled().toBoolean(_insertEnabled);
//              _insertRestricted = annotation.restricted().toBoolean(_insertRestricted);
                _insertOperationAccess = annotation.access();
                _insertLogging = annotation.logging();
                _annotatedWithEntityInsertOperation = true;
            }
        }
    }

    private void annotateEntityUpdateOperation(Class<?> type) {
        Class<?> annotatedClass = XS1.getAnnotatedClass(type, EntityUpdateOperation.class);
        if (annotatedClass != null) {
            EntityUpdateOperation annotation = annotatedClass.getAnnotation(EntityUpdateOperation.class);
            if (annotation != null) {
                _updateEnabled = annotation.enabled().toBoolean(_updateEnabled);
//              _updateRestricted = annotation.restricted().toBoolean(_updateRestricted);
                _updateOperationAccess = annotation.access();
                _updateLogging = annotation.logging();
                _annotatedWithEntityUpdateOperation = true;
            }
        }
    }

    private void annotateEntityDeleteOperation(Class<?> type) {
        Class<?> annotatedClass = XS1.getAnnotatedClass(type, EntityDeleteOperation.class);
        if (annotatedClass != null) {
            EntityDeleteOperation annotation = annotatedClass.getAnnotation(EntityDeleteOperation.class);
            if (annotation != null) {
                _deleteEnabled = annotation.enabled().toBoolean(_deleteEnabled);
//              _deleteRestricted = annotation.restricted().toBoolean(_deleteRestricted);
                _deleteOperationAccess = annotation.access();
                _deleteLogging = annotation.logging();
                _annotatedWithEntityDeleteOperation = true;
            }
        }
    }

    private void annotateEntityReportOperation(Class<?> type) {
        final int MAX = Constants.MAXIMUM_REPORT_ROWS_LIMIT;
        final int MIN = 0;
        Class<?> annotatedClass = XS1.getAnnotatedClass(type, EntityReportOperation.class);
        if (annotatedClass != null) {
            EntityReportOperation annotation = annotatedClass.getAnnotation(EntityReportOperation.class);
            if (annotation != null) {
                _reportEnabled = annotation.enabled().toBoolean(_reportEnabled);
                _reportRowsLimit = Math.min(MAX, Math.max(MIN, annotation.rowsLimit()));
                _reportSortOption = annotation.sortOption();
                _annotatedWithEntityReportOperation = true;
            }
        }
    }

    private void annotateEntityExportOperation(Class<?> type) {
        final int MAX = Constants.MAXIMUM_EXPORT_ROWS_LIMIT;
        final int MIN = 0;
        Class<?> annotatedClass = XS1.getAnnotatedClass(type, EntityExportOperation.class);
        if (annotatedClass != null) {
            EntityExportOperation annotation = annotatedClass.getAnnotation(EntityExportOperation.class);
            if (annotation != null) {
                _exportEnabled = annotation.enabled().toBoolean(_exportEnabled);
                _exportRowsLimit = Math.min(MAX, Math.max(MIN, annotation.rowsLimit()));
                _exportSortOption = annotation.sortOption();
                _annotatedWithEntityExportOperation = true;
            }
        }
    }

    private void annotateEntityTableView(Class<?> type) {
        final int MAX = Constants.MAXIMUM_ROWS_PER_PAGE_LIMIT;
        final int MIN = Constants.MINIMUM_ROWS_PER_PAGE_LIMIT;
        Class<?> annotatedClass = XS1.getAnnotatedClass(type, EntityTableView.class);
        if (annotatedClass != null) {
            EntityTableView annotation = annotatedClass.getAnnotation(EntityTableView.class);
            if (annotation != null) {
                _tableViewEnabled = annotation.enabled().toBoolean();
                _tableViewWithInsertEnabled = annotation.inserts().toBoolean(_tableViewWithInsertEnabled);
                _tableViewWithUpdateEnabled = annotation.updates().toBoolean(_tableViewWithUpdateEnabled);
                _tableViewWithDeleteEnabled = annotation.deletes().toBoolean(_tableViewWithDeleteEnabled);
                _tableViewWithMasterHeading = annotation.heading().toBoolean(_tableViewWithMasterHeading);
                _tableViewRowsLimit = Math.min(MAX, Math.max(MIN, annotation.rowsLimit()));
                _tableViewRowsLimit = (_tableViewRowsLimit + 9) / 10 * 10;
                _tableViewRows = Math.min(_tableViewRowsLimit, Math.max(1, annotation.rows()));
                _tableViewRows = (_tableViewRows + 4) / 5 * 5;
//              _tableViewWidth = Math.min(2400, Math.max(800, annotation.width()));
                _annotatedWithEntityTableView = true;
            }
        }
    }

    private void annotateEntityDetailView(Class<?> type) {
        Class<?> annotatedClass = XS1.getAnnotatedClass(type, EntityDetailView.class);
        if (annotatedClass != null) {
            EntityDetailView annotation = annotatedClass.getAnnotation(EntityDetailView.class);
            if (annotation != null) {
                _detailViewEnabled = annotation.enabled().toBoolean(_detailViewEnabled);
                _annotatedWithEntityDetailView = true;
                _detailViewWithMasterHeading = annotation.heading().toBoolean(_detailViewWithMasterHeading);
//              _detailViewWidth = Math.min(2400, Math.max(800, annotation.width()));
            }
        }
    }

    private void annotateEntityTreeView(Class<?> type) {
        Class<?> annotatedClass = XS1.getAnnotatedClass(type, EntityTreeView.class);
        if (annotatedClass != null) {
            EntityTreeView annotation = annotatedClass.getAnnotation(EntityTreeView.class);
            if (annotation != null) {
                _treeViewEnabled = annotation.enabled().toBoolean(_treeViewEnabled);
                _annotatedWithEntityTreeView = true;
            }
        }
    }

    private void annotateEntityConsoleView(Class<?> type) {
        Class<?> annotatedClass = XS1.getAnnotatedClass(type, EntityConsoleView.class);
        if (annotatedClass != null) {
            EntityConsoleView annotation = annotatedClass.getAnnotation(EntityConsoleView.class);
            if (annotation != null) {
                _consoleViewEnabled = annotation.enabled().toBoolean(_consoleViewEnabled);
                _annotatedWithEntityConsoleView = true;
//              _consoleViewWidth = Math.min(2400, Math.max(800, annotation.width()));
            }
        }
    }

    private void annotateEntityWarnings(Class<?> type) {
        Class<?> annotatedClass = XS1.getAnnotatedClass(type, EntityWarnings.class);
        if (annotatedClass != null) {
            EntityWarnings annotation = annotatedClass.getAnnotation(EntityWarnings.class);
            if (annotation != null) {
                _warningsEnabled = annotation.enabled().toBoolean(_warningsEnabled);
                _annotatedWithEntityWarnings = true;
            }
        }
    }

    private void annotateEntityCodeGen(Class<?> type) {
        Class<?> annotatedClass = XS1.getAnnotatedClass(type, EntityCodeGen.class);
        if (annotatedClass != null) {
            EntityCodeGen annotation = annotatedClass.getAnnotation(EntityCodeGen.class);
            if (annotation != null) {
                _bplCodeGenEnabled = annotation.bpl().toBoolean(_bplCodeGenEnabled);
                _bwsCodeGenEnabled = annotation.bws().toBoolean(_bwsCodeGenEnabled);
                _fwsCodeGenEnabled = annotation.fws().toBoolean(_fwsCodeGenEnabled);
                _sqlCodeGenEnabled = annotation.sql().toBoolean(_sqlCodeGenEnabled);
                _entityStateCodeGenEnabled = annotation.state().toBoolean(_entityStateCodeGenEnabled);
                _annotatedWithEntityCodeGen = true;
            }
        }
    }

    private void annotateEntityDocGen(Class<?> type) {
        Class<?> annotatedClass = XS1.getAnnotatedClass(type, EntityDocGen.class);
        if (annotatedClass != null) {
            EntityDocGen annotation = annotatedClass.getAnnotation(EntityDocGen.class);
            if (annotation != null) {
                _entityClassDiagramGenEnabled = annotation.classDiagram().toBoolean(_entityClassDiagramGenEnabled);
                _entityStateDiagramGenEnabled = annotation.stateDiagram().toBoolean(_entityStateDiagramGenEnabled);
                _entityInsertActivityDiagramGenEnabled = annotation.insertActivityDiagram().toBoolean(_entityInsertActivityDiagramGenEnabled);
                _entityUpdateActivityDiagramGenEnabled = annotation.updateActivityDiagram().toBoolean(_entityUpdateActivityDiagramGenEnabled);
                _annotatedWithEntityDocGen = true;
            }
        }
    }

    private void annotateEntityReferenceSearch(Class<?> type) {
        Class<?> annotatedClass = XS1.getAnnotatedClass(type, EntityReferenceSearch.class);
        if (annotatedClass != null) {
            EntityReferenceSearch annotation = annotatedClass.getAnnotation(EntityReferenceSearch.class);
            if (annotation != null) {
                _searchType = annotation.searchType();
                _listStyle = annotation.listStyle();
                _searchDisplayMode = annotation.displayMode();
                _annotatedWithEntityReferenceSearch = true;
            }
        }
    }

    private void annotateEntityReferenceSearch(Field field) {
        _annotatedWithEntityReferenceSearch = field.isAnnotationPresent(EntityReferenceSearch.class);
        if (_annotatedWithEntityReferenceSearch) {
            EntityReferenceSearch annotation = field.getAnnotation(EntityReferenceSearch.class);
            _searchType = annotation.searchType();
            _listStyle = annotation.listStyle();
            _searchDisplayMode = annotation.displayMode();
        }
    }

    private void annotateFilter(Field field) {
        _annotatedWithFilter = field.isAnnotationPresent(Filter.class);
        if (_annotatedWithFilter) {
            Filter annotation = field.getAnnotation(Filter.class);
            _filterInactiveIndicatorProperty = annotation.inactive().toBoolean(_filterInactiveIndicatorProperty);
            _filterOwnerProperty = annotation.owner().toBoolean(_filterOwnerProperty);
            _filterSegmentProperty = annotation.segment().toBoolean(_filterSegmentProperty);
        }
    }

    // <editor-fold defaultstate="collapsed" desc="deprecated methods">
    /*
    // </editor-fold>
    @SuppressWarnings("deprecation")
    private void annotateExtension(Field field) {
        _annotatedWithExtension = field.isAnnotationPresent(Extension.class);
//      if (_annotatedWithExtension) {
//          Extension annotation = field.getAnnotation(Extension.class);
//      }
    }
    **/
    // </editor-fold>
//
    private void annotateOneToOne(Field field) {
        _annotatedWithOneToOne = field.isAnnotationPresent(OneToOne.class);
        if (_annotatedWithOneToOne) {
            OneToOne annotation = field.getAnnotation(OneToOne.class);
            _fetchType = annotation.fetch();
            _cascadeType = annotation.cascade();
            _navigability = annotation.navigability();
        }
    }

    private void annotateManyToOne(Field field) {
        _annotatedWithManyToOne = field.isAnnotationPresent(ManyToOne.class);
        if (_annotatedWithManyToOne) {
            ManyToOne annotation = field.getAnnotation(ManyToOne.class);
            _fetchType = annotation.fetch();
            _cascadeType = annotation.cascade();
            _navigability = annotation.navigability();
            _masterDetailView = annotation.view();
            _quickAddingFilter = annotation.quickAdding();
        }
    }

    private void annotateQueryMapping(Field field) {
        _annotatedWithQueryMapping = field.isAnnotationPresent(QueryMapping.class);
        if (_annotatedWithQueryMapping) {
            QueryMapping annotation = field.getAnnotation(QueryMapping.class);
            _keyPropertiesQueryMappingEnabled = annotation.mapKeyProperties().toBoolean(_keyPropertiesQueryMappingEnabled);
        }
    }

    private void annotateEntityReferenceDataGen(Field field) {
        _annotatedWithEntityReferenceDataGen = field.isAnnotationPresent(EntityReferenceDataGen.class);
        if (_annotatedWithEntityReferenceDataGen) {
            EntityReferenceDataGen annotation = field.getAnnotation(EntityReferenceDataGen.class);
            _dataGenType = annotation.type();
            _dataGenNullable = Math.min(100, Math.max(0, annotation.nullable()));
        }
    }
    // </editor-fold>

    void initializeInheritanceFields() {
        Class<?> type = getDataType();
        if (_rootInstance) {
            _baseClass = XS1.getConcreteSuperclass(getClass());
            if (_baseClass != null) {
                Entity baseEntity = _project.getEntity(_baseClass);
                if (baseEntity != null) {
                    baseEntity.getSubclassesMap().put(type.getName(), type);
                }
            }
        } else {
            Entity rootEntity = _project.getEntity(type);
            _baseClass = rootEntity.getBaseClass();
        }
    }

    @Override
    public Project getDeclaringProject() {
        return _project;
    }

    /**
     * @return the root entity instance of the same class
     */
    @Override
    public Entity getRoot() {
        return getClassMainInstance();
    }

    /**
     * @return the root entity instance of the base class
     */
    @Override
    public Entity getBaseRoot() {
        return getBaseClassMainInstance();
    }

    /**
     * @return the root entity instance of the same class
     */
    @Override
    public Entity getHierarchyRoot() {
        return _baseClass == null ? getRoot() : getBaseClassMainInstance().getHierarchyRoot();
    }

    /**
     * @return the class hierarchy node type; null if the entity is not part of a hierarchy
     */
    @Override
    public HierarchyNodeType getHierarchyNodeType() {
        return _baseClass == null
            ? _subclasses.isEmpty() ? null : HierarchyNodeType.ROOT
            : _subclasses.isEmpty() ? HierarchyNodeType.LEAF : HierarchyNodeType.BRANCH;
    }

    /**
     * @return the direct known extensions list
     */
    @Override
    public List<Entity> getExtensionsList() {
        return new ArrayList<>(getExtensionsMap().values());
    }

    /**
     * @return the direct known extensions map
     */
    @Override
    public Map<String, Entity> getExtensionsMap() {
        Map<String, Entity> entities = new LinkedHashMap<>();
        Entity ext;
//      PersistentEntity pent;
//      InheritanceMappingStrategy ims;
        List<Class<?>> subclasses = getMainInstanceSubclassesList();
        if (subclasses != null) {
            for (Class<?> subclass : subclasses) {
                ext = _project.getEntity(subclass);
                entities.put(subclass.getName(), ext);
//              pent = ext instanceof PersistentEntity ? (PersistentEntity) ext : null;
//              ims = pent == null ? null : pent.getInheritanceMappingStrategy();
//              if (InheritanceMappingStrategy.SINGLE_TABLE.equals(ims)) {
                entities.putAll(ext.getExtensionsMap());
//              }
            }
        }
        return entities;
    }

    public List<State> getStatesList() {
        List<State> list = new ArrayList<>();
        list.addAll(getStatesMap().values());
        return list;
    }

    public List<State> getInitialStatesList() {
        List<State> list = new ArrayList<>();
        List<State> states = getStatesList();
        List<Operation> operations = getOperationsList();
        List<Transition> transitions;
        boolean add;
        for (State state : states) {
            add = true;
            operationsLoop:
            for (Operation operation : operations) {
                transitions = operation.getTransitionsList();
                for (Transition transition : transitions) {
                    if (state.equals(transition.getY())) {
                        if (state.equals(transition.getX())) {
                            continue;
                        }
                        add = transition.getX() == null;
                        if (add) {
                            break operationsLoop;
                        }
                    }
                }
            }
            if (add) {
                list.add(state);
            }
        }
        return list;
    }

    public List<State> getFinalStatesList() {
        List<State> list = new ArrayList<>();
        List<State> states = getStatesList();
        List<Operation> operations = getOperationsList();
        List<Transition> transitions;
        boolean add;
        for (State state : states) {
            add = true;
            operationsLoop:
            for (Operation operation : operations) {
                transitions = operation.getTransitionsList();
                for (Transition transition : transitions) {
                    if (state.equals(transition.getX())) {
                        if (state.equals(transition.getY())) {
                            continue;
                        }
                        add = transition.getY() == null;
                        if (add) {
                            break operationsLoop;
                        }
                    }
                }
            }
            if (add) {
                list.add(state);
            }
        }
        return list;
    }

    public List<State> getImplicitInitialStatesList() {
        List<State> list = new ArrayList<>();
        List<State> states = getStatesList();
        List<Operation> operations = getOperationsList();
        List<Transition> transitions;
        boolean add;
        for (State state : states) {
            add = true;
            operationsLoop:
            for (Operation operation : operations) {
                transitions = operation.getTransitionsList();
                for (Transition transition : transitions) {
                    if (state.equals(transition.getY())) {
                        if (state.equals(transition.getX())) {
                            continue;
                        }
                        add = false;
                        break operationsLoop;
                    }
                }
            }
            if (add) {
                list.add(state);
            }
        }
        return list;
    }

    public List<State> getImplicitFinalStatesList() {
        List<State> list = new ArrayList<>();
        List<State> states = getStatesList();
        List<Operation> operations = getOperationsList();
        List<Transition> transitions;
        boolean add;
        for (State state : states) {
            add = true;
            operationsLoop:
            for (Operation operation : operations) {
                transitions = operation.getTransitionsList();
                for (Transition transition : transitions) {
                    if (state.equals(transition.getX())) {
                        if (state.equals(transition.getY())) {
                            continue;
                        }
                        add = false;
                        break operationsLoop;
                    }
                }
            }
            if (add) {
                list.add(state);
            }
        }
        return list;
    }

    public Map<String, State> getStatesMap() {
        Map<String, State> map = new LinkedHashMap<>();
        Map<String, Expression> expressions = _atlas.getExpressionsMap();
        Expression value;
        Class<?> fieldType;
        for (String key : expressions.keySet()) {
            value = expressions.get(key);
            if (value instanceof State) {
                fieldType = value.getDeclaringField().getType();
                if (State.class.isAssignableFrom(fieldType)) {
                    map.put(value.getName(), (State) value);
                }
            }
        }
        return map;
    }

    private Entity getClassMainInstance() {
        if (_rootInstance) {
            return this;
        }
        return _project.getEntity(getDataType());
    }

    private Entity getBaseClassMainInstance() {
        return _baseClass == null ? null : _project.getEntity(_baseClass);
    }

    private List<Class<?>> getMainInstanceSubclassesList() {
        Entity main = getClassMainInstance();
        return main == null ? getSubclassesList() : main.getSubclassesList();
    }

    boolean isEntityReference() {
        Field field = getDeclaringField();
        if (field != null) {
            Class<?> type = field.getType();
            if (type != null) {
                return EntityReference.class.isAssignableFrom(type);
            }
        }
        return false;
    }

    @Override
    public boolean isOverlayableEntityReference() {
        if (isEntityReference() && !isHiddenField() && !isEnumerationEntity()) {
            List<Property> list = getRoot().getOverlayPropertiesList();
            return list != null && !list.isEmpty();
        }
        return false;
    }

    /**
     * @return the overlay entities list
     */
    @Override
    public List<Entity> getOverlayEntitiesList() {
        return new ArrayList<>(getOverlayEntitiesMap().values());
    }

    /**
     * @return the overlay entities map
     */
    @Override
    public Map<String, Entity> getOverlayEntitiesMap() {
        Map<String, Entity> map = new LinkedHashMap<>();
        Entity entt;
        Entity root;
        String name;
        List<Property> properties = getPropertiesList();
        for (Property property : properties) {
            if (property.isOverlayableEntityReference()) {
                entt = (Entity) property;
                root = entt.getRoot();
                name = root.getName();
                if (!map.containsKey(name)) {
                    map.put(name, root);
                }
            }
        }
        return map;
    }

    /**
     * @return the accesible operations overlay entities list
     */
    @Override
    public List<Entity> getAccesibleOperationsOverlayEntitiesList() {
        return new ArrayList<>(getAccesibleOperationsOverlayEntitiesMap().values());
    }

    /**
     * @return the accesible operations overlay entities map
     */
    @Override
    public Map<String, Entity> getAccesibleOperationsOverlayEntitiesMap() {
        Map<String, Entity> map = new LinkedHashMap<>();
        Entity entt;
        Entity root;
        String name;
        for (Operation operation : getAccesibleBusinessOperationsList()) {
            for (Parameter parameter : operation.getParametersList()) {
                if (parameter.isOverlayableEntityReference()) {
                    entt = (Entity) parameter;
                    root = entt.getRoot();
                    name = root.getName();
                    if (!map.containsKey(name)) {
                        map.put(name, root);
                    }
                }
            }
        }
        return map;
    }

    /**
     * @return the operation keys
     */
    public Set<String> getOperationKeys() {
        Set<String> set = new TreeSet<>();
        set.addAll(getDefaultCrudOperationKeys());
        set.addAll(getUserDefinedOperationKeys());
        return set;
    }

    /**
     * @return the CRUD operation keys
     */
    public Set<String> getDefaultCrudOperationKeys() {
        Set<String> set = new TreeSet<>();
        String[] operations = Operation.getCrudOperationKeys();
        set.addAll(Arrays.asList(operations));
        return set;
    }

    /**
     * @return the user-defined operation keys
     */
    public Set<String> getUserDefinedOperationKeys() {
        Set<String> set = new TreeSet<>();
        for (Operation operation : getOperationsList()) {
            set.add(operation.getName());
        }
        return set;
    }

    /**
     * @return the CRUD operation list
     */
    @Override
    public List<Operation> getCrudOperationsList() {
        ArrayList<Operation> list = new ArrayList<>();
        if (select != null) {
            list.add(select);
        }
        if (insert != null) {
            list.add(insert);
        }
        if (update != null) {
            list.add(update);
        }
        if (delete != null) {
            list.add(delete);
        }
        return list;
    }

    /**
     * @return the user-defined business operation list
     */
    @Override
    public List<Operation> getBusinessOperationsList() {
        ArrayList<Operation> list = new ArrayList<>();
        OperationType operationType;
        for (Operation operation : getOperationsList()) {
            operationType = operation.getOperationType();
            switch (operationType) {
                case EXPORT:
                case REPORT:
                case PROCEDURE:
                case PROCESS:
                    list.add(operation);
                    break;
            }
        }
        return list;
    }

    /**
     * @return the user-defined accesible business operation list
     */
    @Override
    public List<Operation> getAccesibleBusinessOperationsList() {
        List<Operation> list = getBusinessOperationsList();
        if (list != null && !list.isEmpty()) {
            Collection<Operation> collection = ColUtils.filter(list, new IsAccesibleOperation());
            if (collection != null && !collection.isEmpty()) {
                return new ArrayList<>(collection);
            }
        }
        return new ArrayList<>();
    }

    /**
     * @return the parameter keys
     */
    public Set<String> getParameterKeys() {
        Set<String> set = new TreeSet<>();
        for (Property property : getPropertiesList()) {
            set.add(property.getName());
        }
        for (Operation operation : getOperationsList()) {
            for (Parameter parameter : operation.getParametersList()) {
                set.add(parameter.getName());
            }
        }
        return set;
    }

    /**
     * @return the default wrapper class
     */
    @Override
    public Class<? extends EntityWrapper> getDefaultWrapperClass() {
        return EntityWrapper.class;
    }

    // <editor-fold defaultstate="collapsed" desc="EntityExpression">
    @Override
    public BooleanComparisonX isNull() {
        return XB.Entity.Comparison.isNull(this);
    }

    @Override
    public BooleanComparisonX isNotNull() {
        return XB.Entity.Comparison.isNotNull(this);
    }

    @Override
    public BooleanComparisonX isEqualTo(Entity y) {
        return XB.Entity.Comparison.isEqualTo(this, y);
    }

    @Override
    public BooleanComparisonX isEqualTo(Instance y) {
        return XB.Entity.Comparison.isEqualTo(this, y);
    }

    @Override
    public BooleanComparisonX isEqualTo(EntityExpression y) {
        return XB.Entity.Comparison.isEqualTo(this, y);
    }

    @Override
    public BooleanComparisonX isNotEqualTo(Entity y) {
        return XB.Entity.Comparison.isNotEqualTo(this, y);
    }

    @Override
    public BooleanComparisonX isNotEqualTo(Instance y) {
        return XB.Entity.Comparison.isNotEqualTo(this, y);
    }

    @Override
    public BooleanComparisonX isNotEqualTo(EntityExpression y) {
        return XB.Entity.Comparison.isNotEqualTo(this, y);
    }

    @Override
    public BooleanComparisonX isNullOrEqualTo(Entity y) {
        return XB.Entity.Comparison.isNullOrEqualTo(this, y);
    }

    @Override
    public BooleanComparisonX isNullOrEqualTo(Instance y) {
        return XB.Entity.Comparison.isNullOrEqualTo(this, y);
    }

    @Override
    public BooleanComparisonX isNullOrEqualTo(EntityExpression y) {
        return XB.Entity.Comparison.isNullOrEqualTo(this, y);
    }

    @Override
    public BooleanComparisonX isNullOrNotEqualTo(Entity y) {
        return XB.Entity.Comparison.isNullOrNotEqualTo(this, y);
    }

    @Override
    public BooleanComparisonX isNullOrNotEqualTo(Instance y) {
        return XB.Entity.Comparison.isNullOrNotEqualTo(this, y);
    }

    @Override
    public BooleanComparisonX isNullOrNotEqualTo(EntityExpression y) {
        return XB.Entity.Comparison.isNullOrNotEqualTo(this, y);
    }

    @Override
    public EntityOrderedPairX coalesce(Entity y) {
        return XB.Entity.OrderedPair.coalesce(this, y);
    }

    @Override
    public EntityOrderedPairX coalesce(Instance y) {
        return XB.Entity.OrderedPair.coalesce(this, y);
    }

    @Override
    public EntityOrderedPairX coalesce(EntityExpression y) {
        return XB.Entity.OrderedPair.coalesce(this, y);
    }
    // </editor-fold>

    // <editor-fold defaultstate="collapsed" desc="Special Fields">
    protected static final String EMAIL_REGEX = Constants.EMAIL_REGEX;

    protected static final BooleanScalarX TRUTH = XB.TRUTH;

    protected static final BooleanScalarX UNTRUTH = XB.UNTRUTH;

    protected static final CharacterScalarX EMPTY = XB.EMPTY;

    protected static final CharacterScalarX CURRENT_USER_CODE = XB.CURRENT_USER_CODE;

    protected static final NumericScalarX CURRENT_USER_ID = XB.CURRENT_USER_ID;

    protected static final TemporalScalarX CURRENT_DATE = XB.CURRENT_DATE;

    protected static final TemporalScalarX CURRENT_TIME = XB.CURRENT_TIME;

    protected static final TemporalScalarX CURRENT_TIMESTAMP = XB.CURRENT_TIMESTAMP;
    // </editor-fold>

    // <editor-fold defaultstate="collapsed" desc="Special Expressions">
    protected static BooleanScalarX truth() {
        return TRUTH;
    }

    protected static BooleanScalarX untruth() {
        return UNTRUTH;
    }

    protected static CharacterScalarX empty() {
        return EMPTY;
    }

    protected static CharacterScalarX currentUserCode() {
        return CURRENT_USER_CODE;
    }

    protected static NumericScalarX currentUserId() {
        return CURRENT_USER_ID;
    }

    protected static TemporalScalarX currentDate() {
        return CURRENT_DATE;
    }

    protected static TemporalScalarX currentTime() {
        return CURRENT_TIME;
    }

    protected static TemporalScalarX currentTimestamp() {
        return CURRENT_TIMESTAMP;
    }
    // </editor-fold>

    // <editor-fold defaultstate="collapsed" desc="Supplementary Expressions">
    protected static CharacterOrderedPairX concat(Object x, CharacterExpression y) {
        return XB.Character.OrderedPair.concat(charStringOf(x), y);
    }

    protected static CharacterScalarX charStringOf(Object x) {
        return XB.toCharString(x);
    }

    protected static TemporalScalarX dateOf(Object x) {
        return XB.toDate(x);
    }

    protected static TemporalScalarX timeOf(Object x) {
        return XB.toTime(x);
    }

    protected static TemporalScalarX timestampOf(Object x) {
        return XB.toTimestamp(x);
    }
    // </editor-fold>

    // <editor-fold defaultstate="collapsed" desc="Comparable">
    @Override
    public int compareTo(Entity o) {
        Entity that;
        if (o != null) {
            that = o;
            String thisName = StringUtils.trimToEmpty(this.getName());
            String thatName = StringUtils.trimToEmpty(that.getName());
            return thisName.compareTo(thatName);
        }
        return 0;
    }
    // </editor-fold>

    // <editor-fold defaultstate="collapsed" desc="toString">
    @Override
    protected String fieldsToString(int n, String key, boolean verbose, boolean fields, boolean maps) {
        String tab = verbose ? StringUtils.repeat(" ", 4) : "";
        String fee = verbose ? StringUtils.repeat(tab, n) : "";
        String faa = " = ";
        String foo = verbose ? EOL : ", ";
        String string = super.fieldsToString(n, key, verbose, fields, maps);
        if (fields || verbose) {
            string += _atlas.fieldsToString(n, key, verbose, fields, maps);
            if (verbose) {
//              string += fee + tab + "rootInstance" + faa + _rootInstance + foo;
                string += fee + tab + "explicitlyDeclared" + faa + _explicitlyDeclared + foo;
                string += fee + tab + "implicitlyDeclared" + faa + _implicitlyDeclared + foo;
                if (isEntityReference()) {
                    string += fee + tab + "extension" + faa + isExtension() + foo;
                    string += fee + tab + "oneToOne" + faa + isOneToOne() + foo;
                    string += fee + tab + "manyToOne" + faa + isManyToOne() + foo;
                    string += fee + tab + "fetchType" + faa + _fetchType + foo;
//                  string += fee + tab + "cascadeType" + faa + _cascadeType + foo;
                    string += fee + tab + "navigability" + faa + _navigability + foo;
                    string += fee + tab + "masterDetailView" + faa + _masterDetailView + foo;
                } else {
                    string += fee + tab + "abstract" + faa + _annotatedWithAbstractClass + foo;
                    string += fee + tab + "baseClass" + faa + _baseClass + foo;
                    string += fee + tab + "subclassesList" + faa + getMainInstanceSubclassesList() + foo;
                    string += fee + tab + "primaryKeyProperty" + faa + _primaryKeyProperty + foo;
                    string += fee + tab + "sequenceProperty" + faa + _sequenceProperty + foo;
                    string += fee + tab + "versionProperty" + faa + _versionProperty + foo;
                    string += fee + tab + "numericKeyProperty" + faa + _numericKeyProperty + foo;
                    string += fee + tab + "characterKeyProperty" + faa + _characterKeyProperty + foo;
                    string += fee + tab + "nameProperty" + faa + _nameProperty + foo;
                    string += fee + tab + "descriptionProperty" + faa + _descriptionProperty + foo;
                    string += fee + tab + "imageProperty" + faa + _imageProperty + foo;
                    string += fee + tab + "inactiveIndicatorProperty" + faa + _inactiveIndicatorProperty + foo;
                    string += fee + tab + "urlProperty" + faa + _urlProperty + foo;
                    string += fee + tab + "parentProperty" + faa + _parentProperty + foo;
                    string += fee + tab + "ownerProperty" + faa + _ownerProperty + foo;
                    string += fee + tab + "segmentProperty" + faa + _segmentProperty + foo;
                    string += fee + tab + "businessKeyProperty" + faa + _businessKeyProperty + foo;
                    string += fee + tab + "businessKeyType" + faa + _businessKeyType + foo;
                    string += fee + tab + "existentiallyIndependent" + faa + _existentiallyIndependent + foo;
                    string += fee + tab + "resourceType" + faa + _resourceType + foo;
                    string += fee + tab + "resourceGender" + faa + _resourceGender + foo;
//                  string += fee + tab + "propertiesPrefix" + faa + _propertiesPrefix + foo;
//                  string += fee + tab + "propertiesSuffix" + faa + _propertiesSuffix + foo;
                    string += fee + tab + "selectEnabled" + faa + _selectEnabled + foo;
//                  string += fee + tab + "selectRestricted" + faa + _selectRestricted + foo;
                    string += fee + tab + "selectRowsLimit" + faa + _selectRowsLimit + foo;
                    string += fee + tab + "selectSortOption" + faa + _selectSortOption + foo;
                    string += fee + tab + "insertEnabled" + faa + _insertEnabled + foo;
//                  string += fee + tab + "insertRestricted" + faa + _insertRestricted + foo;
                    string += fee + tab + "updateEnabled" + faa + _updateEnabled + foo;
//                  string += fee + tab + "updateRestricted" + faa + _updateRestricted + foo;
                    string += fee + tab + "deleteEnabled" + faa + _deleteEnabled + foo;
//                  string += fee + tab + "deleteRestricted" + faa + _deleteRestricted + foo;
                    string += fee + tab + "reportEnabled" + faa + _reportEnabled + foo;
                    string += fee + tab + "reportRowsLimit" + faa + _reportRowsLimit + foo;
                    string += fee + tab + "reportSortOption" + faa + _reportSortOption + foo;
                    string += fee + tab + "exportEnabled" + faa + _exportEnabled + foo;
                    string += fee + tab + "exportRowsLimit" + faa + _exportRowsLimit + foo;
                    string += fee + tab + "exportSortOption" + faa + _exportSortOption + foo;
                    string += fee + tab + "tableViewEnabled" + faa + _tableViewEnabled + foo;
                    string += fee + tab + "detailViewEnabled" + faa + _detailViewEnabled + foo;
                    string += fee + tab + "treeViewEnabled" + faa + _treeViewEnabled + foo;
                    string += fee + tab + "consoleViewEnabled" + faa + _consoleViewEnabled + foo;
                    string += fee + tab + "warningsEnabled" + faa + _warningsEnabled + foo;
                }
                string += fee + tab + "searchType" + faa + _searchType + foo;
                string += fee + tab + "listStyle" + faa + _listStyle + foo;
                string += fee + tab + "searchDisplayMode" + faa + _searchDisplayMode + foo;
            }
        }
        return string;
    }

    @Override
    protected String mapsToString(int n, String key, boolean verbose, boolean fields, boolean maps) {
        String string = super.mapsToString(n, key, verbose, fields, maps);
        string += _atlas.mapsToString(n, key, verbose, fields, maps, depth() == 0);
        return string;
    }

    @Override
    protected String getValueString(Object value) {
        if (value == null) {
            return null;
        } else if (value instanceof Instance) {
            Instance instance = (Instance) value;
            return getValueString(instance.getDeclaringArtifact(), instance.getName());
        } else {
            return super.getValueString(value);
        }
    }
    // </editor-fold>

    private Field getPrimaryKeyField(String name, Class<?> type) {
        Class<?>[] validTypes = new Class<?>[]{
            LongProperty.class,
            IntegerProperty.class
        };
        return getKeyPropertyField(KeyProperty.PRIMARY_KEY, name, type, validTypes);
    }

    private Field getSequenceField(String name, Class<?> type) {
        Class<?>[] validTypes = new Class<?>[]{
            LongProperty.class,
            IntegerProperty.class
        };
        return getKeyPropertyField(KeyProperty.SEQUENCE, name, type, validTypes);
    }

    private Field getVersionField(String name, Class<?> type) {
        Class<?>[] validTypes = new Class<?>[]{
            LongProperty.class
        };
        return getKeyPropertyField(KeyProperty.VERSION, name, type, validTypes);
    }

    private Field getNumericKeyField(String name, Class<?> type) {
        Class<?>[] validTypes = new Class<?>[]{
            IntegerProperty.class
        };
        return getKeyPropertyField(KeyProperty.NUMERIC_KEY, name, type, validTypes);
    }

    private Field getCharacterKeyField(String name, Class<?> type) {
        Class<?>[] validTypes = new Class<?>[]{
            StringProperty.class
        };
        return getKeyPropertyField(KeyProperty.CHARACTER_KEY, name, type, validTypes);
    }

    private Field getNameField(String name, Class<?> type) {
        Class<?>[] validTypes = new Class<?>[]{
            StringProperty.class
        };
        return getKeyPropertyField(KeyProperty.NAME, name, type, validTypes);
    }

    private Field getDescriptionField(String name, Class<?> type) {
        Class<?>[] validTypes = new Class<?>[]{
            StringProperty.class
        };
        return getKeyPropertyField(KeyProperty.DESCRIPTION, name, type, validTypes);
    }

    private Field getImageField(String name, Class<?> type) {
        Class<?>[] validTypes = new Class<?>[]{
            BinaryProperty.class
        };
        return getKeyPropertyField(KeyProperty.IMAGE, name, type, validTypes);
    }

    private Field getInactiveIndicatorField(String name, Class<?> type) {
        Class<?>[] validTypes = new Class<?>[]{
            BooleanProperty.class,
            IntegerProperty.class
        };
        return getKeyPropertyField(KeyProperty.INACTIVE_INDICATOR, name, type, validTypes);
    }

    private Field getUrlField(String name, Class<?> type) {
        Class<?>[] validTypes = new Class<?>[]{
            StringProperty.class
        };
        return getKeyPropertyField(KeyProperty.URL, name, type, validTypes);
    }

    private Field getParentField(String name, Class<?> type) {
        Class<?>[] validTypes = new Class<?>[]{
            type
        };
        return getKeyPropertyField(KeyProperty.PARENT, name, type, validTypes);
    }

    private Field getOwnerField(String name, Class<?> type) {
        Class<?>[] validTypes = new Class<?>[]{
            EntityReference.class
        };
        return getKeyPropertyField(KeyProperty.OWNER, name, type, validTypes);
    }

    private Field getSegmentField(String name, Class<?> type) {
        Class<?>[] validTypes = new Class<?>[]{
            EntityReference.class
        };
        return getKeyPropertyField(KeyProperty.SEGMENT, name, type, validTypes);
    }

    private Field getUniqueKeyField(String name, Class<?> type) {
        Class<?>[] validTypes = new Class<?>[]{
            BooleanPrimitive.class,
            CharacterPrimitive.class,
            NumericPrimitive.class,
            TemporalPrimitive.class,
            EntityReference.class
        };
        return getKeyPropertyField(KeyProperty.UNIQUE_KEY, name, type, validTypes);
    }

    private Field getBusinessKeyField(String name, Class<?> type) {
        Class<?>[] validTypes = new Class<?>[]{
            IntegerProperty.class,
            StringProperty.class
        };
        return getKeyPropertyField(KeyProperty.BUSINESS_KEY, name, type, validTypes);
    }

    private Field getStateField(String name, Class<?> type) {
        Class<?>[] validTypes = new Class<?>[]{
            EntityReference.class
        };
        return getKeyPropertyField(KeyProperty.STATE, name, type, validTypes);
    }

    Field getKeyPropertyField(KeyProperty keyProperty, String name, Class<?> type, Class<?>... validTypes) {
        String role = keyProperty.name().replace('_', ' ');
        return XS1.getField(depth() == 0, role, name, type, Entity.class, validTypes);
    }
//
//  private Field getKeyPropertyReferenceField(String name) {
//      if (StringUtils.isNotBlank(name)) {
//          Class<?>[] validTypes = new Class<?>[]{EntityReference.class};
//          return XS1.getField(depth() == 0, name, name, getClass(), Entity.class, validTypes);
//      }
//      return null;
//  }

    private void track(String method) {
        track(method, this);
    }

    private void track(String method, Object... parameters) {
        TLC.getProject().getParser().track(depth(), round(), getClassPath(), method, parameters);
    }

}
