/*
 * Copyright 2017 Jorge Campins y David Uzcategui
 *
 * Este archivo forma parte de Adalid.
 *
 * Adalid es software libre; usted puede redistribuirlo y/o modificarlo bajo los terminos de la
 * licencia "GNU General Public License" publicada por la Fundacion "Free Software Foundation".
 * Adalid se distribuye con la esperanza de que pueda ser util, pero SIN NINGUNA GARANTIA; sin
 * siquiera las garantias implicitas de COMERCIALIZACION e IDONEIDAD PARA UN PROPOSITO PARTICULAR.
 *
 * Para mas detalles vea la licencia "GNU General Public License" en http://www.gnu.org/licenses
 */
package adalid.core;

import adalid.commons.util.StrUtils;
import adalid.core.annotations.ProcessOperationClass;
import adalid.core.enums.Kleenean;
import adalid.core.interfaces.State;
import java.lang.annotation.Annotation;
import java.util.List;
import java.util.Map;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;

/**
 * @author Jorge Campins
 */
public abstract class ProcessOperation extends Operation {

    private static final Logger logger = Logger.getLogger(Operation.class);

    private boolean _annotatedWithProcessOperationClass;

    private String _processName;

    private String _processingGroup;

    private Kleenean _overloading;

    private Kleenean _serviceable;

    /**
     * @return true if annotated with ProcessOperationClass; false otherwise
     */
    public boolean isAnnotatedWithProcessOperationClass() {
        return _annotatedWithProcessOperationClass;
    }

    /**
     * @return the proper name
     */
    @Override
    public String getProperName() {
        return _processName;
    }

    /**
     * @return the process name
     */
    public String getProcessName() {
        return _processName;
    }

    /**
     * @return the processing group identifier
     */
    public String getProcessingGroup() {
        return _processingGroup;
    }

    /**
     * @return the overloading indicator
     */
    public Kleenean getOverloading() {
        return _overloading;
    }

    /**
     * @return the serviceable indicator
     */
    public Kleenean getServiceable() {
        return _serviceable;
    }

    /**
     * El método addTransition se utiliza para agregar una transición a la lista de transiciones que lleva a cabo la operación.
     *
     * @param x estado inicial
     * @param y estado final
     * @return la transición agregada
     */
    public Transition addTransition(State x, State y) {
        Transition transition = new Transition(x, y);
        addTransition(transition);
        return transition;
    }

    /**
     * El método addTransition se utiliza para agregar una transición a la lista de transiciones que lleva a cabo la operación.
     *
     * @param transition la transición a agregar
     */
    public void addTransition(Transition transition) {
        if (transition != null) {
            String key = transition.getName();
            if (key != null) {
                Map<String, Transition> map = getTransitionsMap();
                if (map.containsKey(key)) {
                } else {
                    map.put(key, transition);
                }
            }
        }
    }

    public Trigger trigger(State state) {
        return new Trigger(state, this);
    }

    /**
     * El método addTriggerOn se utiliza para agregar uno o más estados a la lista de estados en los cuales se debe disparar la operación.
     *
     * @param states uno o más estados de la entidad
     */
    public void addTriggerOn(State... states) {
        for (State state : states) {
            Trigger trigger = new Trigger(state, this);
            addTriggerOn(trigger);
        }
    }

    private void addTriggerOn(Trigger trigger) {
        String key = trigger.getName();
        if (key != null) {
            Map<String, Trigger> map = getTriggersMap();
            if (map.containsKey(key)) {
            } else {
                map.put(key, trigger);
            }
        }
    }

    // <editor-fold defaultstate="collapsed" desc="annotate">
    @Override
    void initializeAnnotations() {
        super.initializeAnnotations();
        _annotatedWithProcessOperationClass = false;
        _processName = getName();
        _processingGroup = null;
        _overloading = Kleenean.UNSPECIFIED;
        _serviceable = Kleenean.UNSPECIFIED;
    }

    @Override
    void annotate(Class<?> type) {
        super.annotate(type);
        if (type != null) {
            annotateProcessOperationClass(type);
        }
    }

    @Override
    protected List<Class<? extends Annotation>> getValidTypeAnnotations() {
        List<Class<? extends Annotation>> valid = super.getValidTypeAnnotations();
        valid.add(ProcessOperationClass.class);
        return valid;
    }

    private void annotateProcessOperationClass(Class<?> type) {
        Class<?> annotatedClass = XS1.getAnnotatedClass(type, ProcessOperationClass.class);
        if (annotatedClass != null) {
            ProcessOperationClass annotation = annotatedClass.getAnnotation(ProcessOperationClass.class);
            if (annotation != null) {
                String name = annotation.name();
                if (StringUtils.isNotBlank(name)) {
                    _processName = name;
                }
                String group = annotation.processingGroup();
                if (StringUtils.isNotBlank(group)) {
                    if (StrUtils.esIdentificadorSqlValido(group)) {
                        if (group.length() <= Constants.DEFAULT_CHARACTER_KEY_MAX_LENGTH) {
                            _processingGroup = group;
                            TLC.getProject().getProcessingGroups().add(group);
                        } else {
                            String message = "invalid processing group identifier \"" + group + "\" at " + getFullName();
                            message += "; identifier length cannot be greater than " + Constants.DEFAULT_CHARACTER_KEY_MAX_LENGTH;
                            logger.error(message);
                            TLC.getProject().getParser().increaseErrorCount();
                        }
                    } else {
                        String message = "invalid processing group identifier \"" + group + "\" at " + getFullName();
                        logger.error(message);
                        TLC.getProject().getParser().increaseErrorCount();
                    }
                }
                _overloading = annotation.overloading();
                _serviceable = annotation.serviceable();
                _annotatedWithProcessOperationClass = true;
            }
        }
    }
    // </editor-fold>

}
