/*
 * Copyright 2017 Jorge Campins y David Uzcategui
 *
 * Este archivo forma parte de Adalid.
 *
 * Adalid es software libre; usted puede redistribuirlo y/o modificarlo bajo los terminos de la
 * licencia "GNU General Public License" publicada por la Fundacion "Free Software Foundation".
 * Adalid se distribuye con la esperanza de que pueda ser util, pero SIN NINGUNA GARANTIA; sin
 * siquiera las garantias implicitas de COMERCIALIZACION e IDONEIDAD PARA UN PROPOSITO PARTICULAR.
 *
 * Para mas detalles vea la licencia "GNU General Public License" en http://www.gnu.org/licenses
 */
package adalid.core.annotations;

import adalid.core.enums.Kleenean;
import adalid.core.enums.MimeType;
import adalid.core.enums.UploadStorageOption;
import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

/**
 * La anotación FileReference se utiliza para designar propiedades y parámetros String como referencias a archivos cargados en el servidor.
 *
 * @author Jorge Campins
 */
@Retention(RetentionPolicy.RUNTIME)
@Target(ElementType.FIELD)
public @interface FileReference {

    /**
     * max específica el tamaño máximo (en bytes). Su valor debe ser un número entero, mayor o igual a 0. Utilice 0 para permitir la carga de archivos
     * de cualquier tamaño. Alternativamente, omita el elemento para utilizar el valor predeterminado del atributo. El valor predeterminado es
     * 1.000.000 (1 MB).
     *
     * @return max
     */
    int max() default -1; // Constants.DEFAULT_MAX_INPUT_FILE_SIZE

    /**
     * types especifica la lista de extensiones MIME (Multipurpose Internet Mail Extensions) válidas. Su valor es una lista de elementos de la
     * enumeración MimeType. Alternativamente, omita el elemento permitir la carga de archivos de cualquier tipo.
     *
     * @return types
     */
    MimeType[] types() default {};

    /**
     * regex especifica la expresión regular que deben satisfacer los nombres de los archivos que se pueden cargar. Para más información sobre
     * expresiones regulares consulte la documentación de Java (la página Regular Expressions es un buen punto de partida). Advertencia: la expresión
     * debe ser válida tanto en Java como en JavaScript y, por lo tanto, no debe contener características específicas de ninguno de los dos lenguajes.
     *
     * @return regex
     */
    String regex() default "";

    /**
     * storage especifica el tipo de almacenamiento de los archivos. Su valor es uno de los elementos de la enumeración UploadStorageOption.
     * Seleccione FILE, ROW o ROW_AND_FILE para almacenar el archivo en el servidor de aplicaciones (web), en la base de datos, o en ambos,
     * respectivamente. Alternativamente, omita el elemento para utilizar el valor predeterminado del atributo. El valor predeterminado es
     * ROW_AND_FILE.
     *
     * @return storage
     */
    UploadStorageOption storage() default UploadStorageOption.UNSPECIFIED;

    /**
     * blobField especifica el nombre de la propiedad donde se almacena el contenido del archivo. Este elemento es relevante solo si el valor
     * especificado, o determinado, para el elemento storage es ROW o ROW_AND_FILE.
     *
     * @return blobField
     */
    String blobField() default "";

    /**
     * joinField especifica el nombre de la propiedad que hace referencia a la tabla de la base de datos donde se almacena el contenido del archivo.
     * Este elemento es relevante solo si el valor especificado, o determinado, para el elemento storage es ROW o ROW_AND_FILE.
     *
     * @return joinField
     */
    String joinField() default "";

    /**
     * loadField especifica el nombre de la propiedad donde se almacena la fecha y la hora en que se cargó el archivo. La clase de la propiedad
     * especificada debe ser DateProperty o TimestampProperty.
     *
     * @return loadField
     */
    String loadField() default "";

    /**
     * updateable indica si la referencia puede o no ser actualizada en las vistas (páginas) de consulta y/o registro de la entidad. Su valor es uno
     * de los elementos de la enumeración Kleenean. Seleccione TRUE para permitir la actualización; en caso contrario, seleccione FALSE.
     * Alternativamente, omita el elemento o seleccione UNSPECIFIED para utilizar el valor predeterminado del atributo. El valor predeterminado del
     * atributo es FALSE.
     *
     * @return updateable
     */
    @Deprecated
    Kleenean updateable() default Kleenean.UNSPECIFIED; // FALSE

}
