/*
 * Copyright 2017 Jorge Campins y David Uzcategui
 *
 * Este archivo forma parte de Adalid.
 *
 * Adalid es software libre; usted puede redistribuirlo y/o modificarlo bajo los terminos de la
 * licencia "GNU General Public License" publicada por la Fundacion "Free Software Foundation".
 * Adalid se distribuye con la esperanza de que pueda ser util, pero SIN NINGUNA GARANTIA; sin
 * siquiera las garantias implicitas de COMERCIALIZACION e IDONEIDAD PARA UN PROPOSITO PARTICULAR.
 *
 * Para mas detalles vea la licencia "GNU General Public License" en http://www.gnu.org/licenses
 */
package adalid.core.annotations;

import adalid.core.enums.CascadeType;
import adalid.core.enums.FetchType;
import adalid.core.enums.Navigability;
import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

/**
 * La anotación OneToOne se utiliza para establecer atributos de una referencia (propiedad que hace referencia a otra entidad) para relaciones con
 * cardinalidad uno-a-uno.
 *
 * @author Jorge Campins
 */
@Retention(RetentionPolicy.RUNTIME)
@Target(ElementType.FIELD)
public @interface OneToOne {

    /**
     * fetch especifica si la operación fetch obtiene la entidad relacionada simultáneamente con la entidad (EAGER) o posteriormente, por demanda
     * (LAZY). Su valor es uno de los elementos de la enumeración FetchType. Seleccione EAGER o LAZY o, alternativamente, omita el elemento o
     * seleccione UNSPECIFIED para utilizar el valor predeterminado del atributo. El valor predeterminado del atributo es LAZY.
     *
     * @return fetch
     */
    FetchType fetch() default FetchType.UNSPECIFIED;

    /**
     * cascade especifica las operaciones que deben ser propagadas en cascada a la entidad referenciada. Su valor es uno de los elementos de la
     * enumeración CascadeType. Seleccione PERSIST, MERGE, REMOVE, REFRESH o DETACH para propagar la operación persist, merge, remove, refresh o
     * detach, respectivamente. Puede especificar más de una opción, por ejemplo: cascade = {CascadeType.PERSIST, CascadeType.MERGE}. Seleccione ALL
     * para propagar todas las operaciones Alternativamente, omita el elemento o seleccione UNSPECIFIED para no propagar ninguna de las operaciones.
     *
     * @return cascade
     */
    CascadeType[] cascade() default CascadeType.UNSPECIFIED;

    /**
     * navigability especifica la navegabilidad entre las entidades relacionadas. Su valor es uno de los elementos de la enumeración Navigability.
     * Seleccione BIDIRECTIONAL o UNIDIRECTIONAL para especificar navegabilidad bidireccional o unidireccional, respectivamente. Alternativamente,
     * omita el elemento o seleccione UNSPECIFIED para utilizar el valor predeterminado del atributo. El valor predeterminado del atributo es
     * UNIDIRECTIONAL. La navegabilidad bidireccional utiliza apuntadores en ambas entidades relacionadas para permitir ir de una a otra, en ambos
     * sentidos. La navegabilidad unidireccional solo utiliza apuntadores en la entidad que contiene la referencia hacia la entidad referenciada.
     *
     * @return navigability
     */
    Navigability navigability() default Navigability.UNSPECIFIED;

}
