/*
 * Copyright 2017 Jorge Campins y David Uzcategui
 *
 * Este archivo forma parte de Adalid.
 *
 * Adalid es software libre; usted puede redistribuirlo y/o modificarlo bajo los terminos de la
 * licencia "GNU General Public License" publicada por la Fundacion "Free Software Foundation".
 * Adalid se distribuye con la esperanza de que pueda ser util, pero SIN NINGUNA GARANTIA; sin
 * siquiera las garantias implicitas de COMERCIALIZACION e IDONEIDAD PARA UN PROPOSITO PARTICULAR.
 *
 * Para mas detalles vea la licencia "GNU General Public License" en http://www.gnu.org/licenses
 */
package adalid.core.annotations;

import adalid.core.enums.Checkpoint;
import adalid.core.enums.DefaultCondition;
import adalid.core.enums.Kleenean;
import adalid.core.enums.PropertyAccess;
import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

/**
 * La anotación PropertyField se utiliza para establecer atributos básicos de la propiedad.
 *
 * @author Jorge Campins
 */
@Retention(RetentionPolicy.RUNTIME)
@Target(ElementType.FIELD)
public @interface PropertyField {

    /**
     * access especifica el tipo de control de acceso de la propiedad. Su valor es uno de los elementos de la enumeración PropertyAccess. Seleccione
     * RESTRICTED_WRITING o RESTRICTED_READING para especificar acceso restringido de escritura o lectura, respectivamente. Alternativamente, omita el
     * elemento o seleccione UNSPECIFIED para especificar acceso no restringido.
     *
     * @return access
     */
    PropertyAccess access() default PropertyAccess.UNSPECIFIED;

    /**
     * auditable indica si la propiedad se debe incluir, o no, en las pistas de auditoría de las funciones de insert y update de la tabla de la base
     * de datos que corresponde a la entidad. Su valor es uno de los elementos de la enumeración Kleenean. Seleccione TRUE para incluir la propiedad;
     * en caso contrario, seleccione FALSE. Alternativamente, omita el elemento o seleccione UNSPECIFIED para utilizar el valor predeterminado del
     * atributo. El valor predeterminado del atributo es FALSE para propiedades que corresponden a “objetos binarios” o a contraseñas; y TRUE para las
     * demás propiedades.
     *
     * @return auditable
     */
    Kleenean auditable() default Kleenean.UNSPECIFIED; // TRUE

    /**
     * password indica si la propiedad es, o no, una contraseña. Su valor es uno de los elementos de la enumeración Kleenean. Seleccione TRUE si la
     * propiedad es una contraseña; en caso contrario, seleccione FALSE. Alternativamente, omita el elemento o seleccione UNSPECIFIED para utilizar el
     * valor predeterminado del atributo. El valor predeterminado del atributo es FALSE.
     *
     * @return password
     */
    Kleenean password() default Kleenean.UNSPECIFIED; // FALSE

    /**
     * required indica si la propiedad es, o no, obligatoriamente requerida por las vistas (páginas) de registro. Su valor es uno de los elementos de
     * la enumeración Kleenean. Seleccione TRUE si la propiedad es obligatoriamente requerida; en caso contrario, seleccione FALSE. Alternativamente,
     * omita el elemento o seleccione UNSPECIFIED para utilizar el valor predeterminado del atributo. El valor predeterminado del atributo es FALSE si
     * la propiedad admite nulos (vea Anotación ColumnField) o tiene valor por omisión (vea Método setDefaultValue); en caso contrario es TRUE.
     *
     * @return required
     */
    Kleenean required() default Kleenean.UNSPECIFIED; // FALSE

    /**
     * hidden indica si la propiedad permanece, o no, oculta en las vistas (páginas) e informes. Su valor es uno de los elementos de la enumeración
     * Kleenean. Seleccione TRUE si la propiedad permanece oculta; en caso contrario, seleccione FALSE. Alternativamente, omita el elemento o
     * seleccione UNSPECIFIED para utilizar el valor predeterminado del atributo. El valor predeterminado del atributo es FALSE.
     *
     * @return hidden
     */
    Kleenean hidden() default Kleenean.UNSPECIFIED; // FALSE

    /**
     * create indica si la propiedad es, o no, requerida por la operación insert de las vistas (páginas) de registro. Este elemento es relevante solo
     * si la propiedad es insertable (vea Anotación ColumnField). Su valor es uno de los elementos de la enumeración Kleenean. Seleccione TRUE si la
     * propiedad es requerida por la operación insert; en caso contrario, seleccione FALSE. Alternativamente, omita el elemento o seleccione
     * UNSPECIFIED para utilizar el valor predeterminado del atributo. El valor predeterminado del atributo es FALSE si la propiedad no es requerida
     * (vea el elemento required de esta misma anotación) o no es insertable (vea Anotación ColumnField) o está enlazada a un parámetro de un proceso
     * de instancia (vea Anotación ParameterField); en caso contrario es TRUE.
     *
     * @return create
     */
    Kleenean create() default Kleenean.UNSPECIFIED; // TRUE

    /**
     * update indica si la propiedad es, o no, requerida por la operación update de las vistas (páginas) de registro. Este elemento es relevante solo
     * si la propiedad es actualizable (vea Anotación ColumnField). Su valor es uno de los elementos de la enumeración Kleenean. Seleccione TRUE si la
     * propiedad es requerida por la operación update; en caso contrario, seleccione FALSE. Alternativamente, omita el elemento o seleccione
     * UNSPECIFIED para utilizar el valor predeterminado del atributo. El valor predeterminado del atributo es FALSE si la propiedad no es
     * actualizable (vea Anotación ColumnField) o está enlazada a un parámetro de un proceso de instancia (vea Anotación ParameterField); en caso
     * contrario es TRUE.
     *
     * @return update
     */
    Kleenean update() default Kleenean.UNSPECIFIED; // TRUE

    /**
     * search indica si la propiedad es, o no, un criterio de búsqueda básica en las vistas (páginas) de consulta y registro. Su valor es uno de los
     * elementos de la enumeración Kleenean. Seleccione TRUE si la propiedad es un criterio de búsqueda básica; en caso contrario, seleccione FALSE.
     * Alternativamente, omita el elemento o seleccione UNSPECIFIED para utilizar el valor predeterminado del atributo. El valor predeterminado del
     * atributo es TRUE si la propiedad no admite valores duplicados (vea Anotación ColumnField); o es una clave única (vea Anotación UniqueKey); o es
     * la clave de negocio (vea Anotación BusinessKey); o es la clave numérica (vea Anotación NumericKey); o es la clave alfanumérica (vea Anotación
     * CharacterKey); o es la propiedad nombre (vea Anotación NameProperty); o es la columna discriminadora (vea Anotación DiscriminatorColumn); o es
     * el indicador de inactividad (vea Anotación InactiveIndicator); o es visible en las vistas (páginas) de consulta y registro tabular (vea el
     * elemento table de esta misma anotación); en caso contrario, o si la propiedad es una contraseña (vea el elemento password de esta misma
     * anotación), es FALSE.
     *
     * @return search
     */
    Kleenean search() default Kleenean.UNSPECIFIED; // FALSE

    /**
     * filter indica si la propiedad es, o no, un criterio de búsqueda avanzada en las vistas (páginas) de consulta y registro. Su valor es uno de los
     * elementos de la enumeración Kleenean. Seleccione TRUE si la propiedad es un criterio de búsqueda avanzada; en caso contrario, seleccione FALSE.
     * Alternativamente, omita el elemento o seleccione UNSPECIFIED para utilizar el valor predeterminado del atributo. El valor predeterminado del
     * atributo es FALSE si la propiedad es una contraseña (vea el elemento password de esta misma anotación); en caso contrario es TRUE.
     *
     * @return filter
     */
    Kleenean filter() default Kleenean.UNSPECIFIED; // TRUE

    /**
     * sort indica si la propiedad es, o no, un criterio de ordenamiento en las vistas (páginas) de consulta y registro. Su valor es uno de los
     * elementos de la enumeración Kleenean. Seleccione TRUE si la propiedad es un criterio de ordenamiento; en caso contrario, seleccione FALSE.
     * Alternativamente, omita el elemento o seleccione UNSPECIFIED para utilizar el valor predeterminado del atributo. El valor predeterminado del
     * atributo es TRUE.
     *
     * @return sort
     */
    Kleenean sort() default Kleenean.UNSPECIFIED; // TRUE

    /**
     * table indica si la propiedad es, o no, visible en las vistas (páginas) de consulta y registro tabular. Su valor es uno de los elementos de la
     * enumeración Kleenean. Seleccione TRUE si la propiedad es visible; en caso contrario, seleccione FALSE. Alternativamente, omita el elemento o
     * seleccione UNSPECIFIED para utilizar el valor predeterminado del atributo. El valor predeterminado del atributo es TRUE si la propiedad no
     * admite valores duplicados (vea Anotación ColumnField); o es una clave única (vea Anotación UniqueKey); o es la clave de negocio (vea Anotación
     * BusinessKey); o es la clave numérica (vea Anotación NumericKey); o es la clave alfanumérica (vea Anotación CharacterKey); o es la propiedad
     * nombre (vea Anotación NameProperty); o es la columna discriminadora (vea Anotación DiscriminatorColumn); o es el indicador de inactividad (vea
     * Anotación InactiveIndicator); o es requerida (vea el elemento required de esta misma anotación); en caso contrario es FALSE.
     *
     * @return table
     */
    Kleenean table() default Kleenean.UNSPECIFIED; // FALSE

    /**
     * detail indica si la propiedad es, o no, visible en las vistas (páginas) de consulta y registro detallado. Su valor es uno de los elementos de
     * la enumeración Kleenean. Seleccione TRUE si la propiedad es un criterio de búsqueda avanzada; en caso contrario, seleccione FALSE.
     * Alternativamente, omita el elemento o seleccione UNSPECIFIED para utilizar el valor predeterminado del atributo. El valor predeterminado del
     * atributo es TRUE.
     *
     * @return detail
     */
    Kleenean detail() default Kleenean.UNSPECIFIED; // TRUE

    /**
     * column indica si la propiedad puede ser, o no, una columna de vistas definidas por el usuario final. Su valor es uno de los elementos de la
     * enumeración Kleenean. Seleccione TRUE si la propiedad puede ser una columna; en caso contrario, seleccione FALSE. Alternativamente, omita el
     * elemento o seleccione UNSPECIFIED para utilizar el valor predeterminado del atributo. El valor predeterminado del atributo es FALSE si la
     * propiedad es una contraseña (vea el elemento password de esta misma anotación); en caso contrario es TRUE.
     *
     * @return column
     */
    Kleenean column() default Kleenean.UNSPECIFIED; // TRUE

    /**
     * report indica si la propiedad es, o no, incluida en el informe producido por la operación report de las vistas (páginas) de consulta y
     * registro. Su valor es uno de los elementos de la enumeración Kleenean. Seleccione TRUE si la propiedad es incluida; en caso contrario,
     * seleccione FALSE. Alternativamente, omita el elemento o seleccione UNSPECIFIED para utilizar el valor predeterminado del atributo. El valor
     * predeterminado del atributo es TRUE si la propiedad no admite valores duplicados (vea Anotación ColumnField); o es una clave única (vea
     * Anotación UniqueKey); o es la clave de negocio (vea Anotación BusinessKey); o es la clave numérica (vea Anotación NumericKey); o es la clave
     * alfanumérica (vea Anotación CharacterKey); o es la propiedad nombre (vea Anotación NameProperty); o es la columna discriminadora (vea Anotación
     * DiscriminatorColumn); o es el indicador de inactividad (vea Anotación InactiveIndicator); o es requerida (vea el elemento required de esta
     * misma anotación); en caso contrario, o si la propiedad es una contraseña (vea el elemento password de esta misma anotación), es FALSE.
     *
     * @return report
     */
    Kleenean report() default Kleenean.UNSPECIFIED; // FALSE

    /**
     * export indica si la propiedad es, o no, incluida en el informe producido por la operación export de las vistas (páginas) de consulta y
     * registro. Su valor es uno de los elementos de la enumeración Kleenean. Seleccione TRUE si la propiedad es incluida; en caso contrario,
     * seleccione FALSE. Alternativamente, omita el elemento o seleccione UNSPECIFIED para utilizar el valor predeterminado del atributo. El valor
     * predeterminado del atributo es FALSE si la propiedad es una contraseña (vea el elemento password de esta misma anotación); en caso contrario es
     * TRUE.
     *
     * @return export
     */
    Kleenean export() default Kleenean.UNSPECIFIED; // TRUE

    /**
     * submit indica si propiedad es un disparador, es decir, si las vistas (páginas) de registro envían la información al servidor de aplicaciones
     * inmediatamente que el valor de esta propiedad es modificado. Su valor es uno de los elementos de la enumeración Kleenean. Seleccione TRUE si la
     * propiedad es un disparador; en caso contrario, seleccione FALSE. Alternativamente, omita el elemento o seleccione UNSPECIFIED para utilizar el
     * valor predeterminado del atributo. El valor predeterminado del atributo es FALSE.
     *
     * @return submit
     * @deprecated
     */
    @Deprecated
    Kleenean submit() default Kleenean.UNSPECIFIED; // FALSE

    /**
     * headertextless indica si la propiedad es, o no, una propiedad sin título en las vistas (páginas) de consulta y registro tabular. Su valor es
     * uno de los elementos de la enumeración Kleenean. Seleccione TRUE si la propiedad es una propiedad sin título; en caso contrario, seleccione
     * FALSE. Alternativamente, omita el elemento o seleccione UNSPECIFIED para utilizar el valor predeterminado del atributo. El valor predeterminado
     * del atributo es FALSE
     *
     * @return headertextless
     */
    Kleenean headertextless() default Kleenean.UNSPECIFIED; // FALSE

    /**
     * heading indica si la propiedad es, o no, parte del encabezado en las vistas (páginas) Maestro/Detalle, donde la entidad es el maestro. Su valor
     * es uno de los elementos de la enumeración Kleenean. Seleccione TRUE si la propiedad es una propiedad del encabezado; en caso contrario,
     * seleccione FALSE. Alternativamente, omita el elemento o seleccione UNSPECIFIED para utilizar el valor predeterminado del atributo. El valor
     * predeterminado del atributo es FALSE
     *
     * @return heading
     */
    Kleenean heading() default Kleenean.UNSPECIFIED; // FALSE

    /**
     * overlay indica si la propiedad es, o no, parte del panel de vista rápida en las vistas (páginas) donde la entidad es referenciada. Su valor es
     * uno de los elementos de la enumeración Kleenean. Seleccione TRUE si la propiedad es una propiedad del panel de vista rápida; en caso contrario,
     * seleccione FALSE. Alternativamente, omita el elemento o seleccione UNSPECIFIED para utilizar el valor predeterminado del atributo. El valor
     * predeterminado del atributo es FALSE
     *
     * @return overlay
     */
    Kleenean overlay() default Kleenean.UNSPECIFIED; // FALSE

    /**
     * immutable indica si la propiedad es, o no, inmutable. Solo aplica a propiedades de una enumeración. El valor de las propiedades inmutables se
     * almacena en el bundle de recursos básicos de la aplicación generada, de modo que el valor puede ser obtenido sin acceder a la base de datos. Su
     * valor es uno de los elementos de la enumeración Kleenean. Seleccione TRUE si la propiedad es inmutable; en caso contrario, seleccione FALSE.
     * Alternativamente, omita el elemento o seleccione UNSPECIFIED para utilizar el valor predeterminado del atributo. El valor predeterminado del
     * atributo es FALSE
     *
     * @return immutable
     */
    Kleenean immutable() default Kleenean.UNSPECIFIED; // FALSE

    /**
     * defaultCondition especifica en qué circunstancias aplicar el valor por omisión de la propiedad. Su valor es uno de los elementos de la
     * enumeración DefaultCondition. Seleccione NULL_ON_INSERT, IF_NULL_ON_UPDATE, IF_NULL, UNCONDITIONALLY_ON_INSERT, UNCONDITIONALLY_ON_UPDATE o
     * UNCONDITIONALLY para aplicar el valor por omisión si el valor es nulo en la operación insert; o si el valor es nulo en la operación update; o
     * si el valor es nulo, tanto en insert como en update; o incondicionalmente en la operación insert; o incondicionalmente en la operación update;
     * o incondicionalmente tanto en insert como en update, respectivamente. Alternativamente, omita el elemento para utilizar el valor predeterminado
     * del atributo. El valor predeterminado del atributo es IF_NULL.
     *
     * @return defaultCondition
     */
    DefaultCondition defaultCondition() default DefaultCondition.IF_NULL;

    /**
     * defaultCheckpoint especifica en qué componente o componentes donde se debe aplicar el valor por omisión de la propiedad. Su valor es uno de los
     * elementos de la enumeración Checkpoint. Seleccione DATABASE_TRIGGER o USER_INTERFACE si el valor por omisión se debe aplicar solamente en un
     * disparador (trigger) de la base de datos o en la interfaz de usuario, respectivamente. Seleccione WHEREVER_POSSIBLE para aplicar el valor por
     * omisión en ambos componentes, siempre que sea posible; esta es la opción predeterminada.
     *
     * @return defaultCheckpoint
     */
    Checkpoint defaultCheckpoint() default Checkpoint.WHEREVER_POSSIBLE;

    /**
     * sequence específica el número de secuencia o posición relativa en la que se muestra la propiedad en las vistas (páginas) de consulta y
     * registro. Su valor debe ser un número entero entre 0 y 2.147.483.647. Alternativamente, omita el elemento para utilizar el valor predeterminado
     * del atributo. El valor predeterminado del atributo es 0. Si todas las propiedades tienen el mismo número de secuencia (0 o cualquier otro),
     * entonces las vistas las muestran en el orden en el mismo orden en el que las meta propiedades están definidas en la meta entidad.
     *
     * @return sequence
     */
    int sequence() default 0;

}
