/*
 * Copyright 2017 Jorge Campins y David Uzcategui
 *
 * Este archivo forma parte de Adalid.
 *
 * Adalid es software libre; usted puede redistribuirlo y/o modificarlo bajo los terminos de la
 * licencia "GNU General Public License" publicada por la Fundacion "Free Software Foundation".
 * Adalid se distribuye con la esperanza de que pueda ser util, pero SIN NINGUNA GARANTIA; sin
 * siquiera las garantias implicitas de COMERCIALIZACION e IDONEIDAD PARA UN PROPOSITO PARTICULAR.
 *
 * Para mas detalles vea la licencia "GNU General Public License" en http://www.gnu.org/licenses
 */
package adalid.core.annotations;

import adalid.core.enums.Kleenean;
import adalid.core.enums.LetterCase;
import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

/**
 * La anotación StringField se utiliza para establecer atributos de propiedades y parámetros String.
 *
 * @author Jorge Campins
 */
@Retention(RetentionPolicy.RUNTIME)
@Target(ElementType.FIELD)
public @interface StringField {

    /**
     * maxLength específica la cantidad mínima de caracteres que deben tener los valores de la propiedad o parámetro. Su valor debe ser un número
     * entero entre 0 y 8.000. El valor predeterminado es 0
     *
     * @return maxLength
     */
    int maxLength() default -1;

    /**
     * minLength específica la cantidad máxima de caracteres que pueden tener los valores de la propiedad o parámetro. Su valor debe ser un número
     * entero entre 0 y 8.000. El valor predeterminado es 8.000
     *
     * @return minLength
     */
    int minLength() default -1;

    /**
     * regex especifica la expresión regular que deben satisfacer los valores de la propiedad o parámetro. Para más información sobre expresiones
     * regulares consulte la documentación de Java (la página Regular Expressions es un buen punto de partida).
     *
     * @return regex
     */
    String regex() default "";

    /**
     * letterCase especifica la conversión que se debe realizar al almacenar valores de la propiedad o parámetro en la base de datos. Su valor es uno
     * de los elementos de la enumeración LetterCase. Seleccione LOWER, UPPER o CAPITALIZED para convertir todos los caracteres a minúsculas, todos a
     * mayúsculas, o para capitalizar (convertir el primer carácter de cada palabra a mayúscula y el resto a minúsculas), respectivamente.
     * Alternativamente, omita el elemento o seleccione UNSPECIFIED para no ejecutar conversión alguna.
     *
     * @return letterCase
     */
    LetterCase letterCase() default LetterCase.UNSPECIFIED;

    /**
     * allowDiacritics indica si se permiten, o no, signos diacríticos al almacenar valores de la propiedad o parámetro en la base de datos. Su valor
     * es uno de los elementos de la enumeración Kleenean. Seleccione TRUE para permitir signos diacríticos; en caso contrario, seleccione FALSE.
     * Alternativamente, omita el elemento o seleccione UNSPECIFIED para utilizar el valor predeterminado del atributo. El valor predeterminado del
     * atributo es TRUE.
     *
     * @return allowDiacritics
     */
    Kleenean allowDiacritics() default Kleenean.UNSPECIFIED;

    /**
     * converter específica el nombre de un componente personalizado que permite convertir el valor de la propiedad o parámetro.
     *
     * @return converter
     */
    String converter() default "";

    /**
     * validator específica el nombre de un componente personalizado que permite validar el valor de la propiedad o parámetro.
     *
     * @return validator
     */
    String validator() default "";

}
