/*
 * Copyright 2017 Jorge Campins y David Uzcategui
 *
 * Este archivo forma parte de Adalid.
 *
 * Adalid es software libre; usted puede redistribuirlo y/o modificarlo bajo los terminos de la
 * licencia "GNU General Public License" publicada por la Fundacion "Free Software Foundation".
 * Adalid se distribuye con la esperanza de que pueda ser util, pero SIN NINGUNA GARANTIA; sin
 * siquiera las garantias implicitas de COMERCIALIZACION e IDONEIDAD PARA UN PROPOSITO PARTICULAR.
 *
 * Para mas detalles vea la licencia "GNU General Public License" en http://www.gnu.org/licenses
 */
package adalid.core.data.types;

import adalid.core.Constants;
import adalid.core.XS2;
import adalid.core.enums.Kleenean;
import adalid.core.enums.LetterCase;
import adalid.core.enums.MimeType;
import adalid.core.enums.UploadStorageOption;
import adalid.core.enums.UrlDisplayType;
import adalid.core.enums.UrlType;
import adalid.core.interfaces.Property;
import adalid.core.primitives.CharacterPrimitive;
import java.lang.reflect.Field;
import java.util.LinkedHashMap;
import java.util.Locale;
import java.util.Map;
import java.util.regex.Pattern;
import org.apache.commons.lang.StringUtils;

/**
 * @author Jorge Campins
 */
public class StringData extends CharacterPrimitive {

    private static final String EOL = "\n";

    public static final String EMPTY = "";

    {
        XS2.setDataClass(this, StringData.class);
        XS2.setDataType(this, String.class);
    }

    private int _minLength = 0;

    private Integer _maxLength;

    private Pattern _pattern;

    private final Map<Locale, String> _localizedRegexErrorMessage = new LinkedHashMap<>();

    private LetterCase _letterCase = LetterCase.UNSPECIFIED;

    private boolean _allowDiacritics = true;

    private String _specialConverterName;

    private String _specialValidatorName;

    private UrlType _urlType = UrlType.UNSPECIFIED;

    private UrlDisplayType _urlDisplayType = UrlDisplayType.UNSPECIFIED;

    private int _maxInputFileSize = -1; // UNSPECIFIED

    private MimeType[] _validInputFileTypes = new MimeType[]{};

    private Pattern _validInputFilePattern;

    private UploadStorageOption _uploadStorageOption = UploadStorageOption.UNSPECIFIED;

    private String _blobFieldName;

    private String _joinFieldName;

    private String _loadFieldName;

    private Field _blobField;

    private Field _joinField;

    private Field _loadField;

    private Property _blobProperty;

    private Property _joinProperty;

    private Property _loadProperty;

    private Kleenean _updateableFileReference = Kleenean.UNSPECIFIED; // unused variable

    /**
     * @return the minLength
     */
    public Integer getMinLength() {
        return _minLength;
    }

    /**
     * @param minLength the minLength to set
     */
    public void setMinLength(Integer minLength) {
        XS2.checkAccess();
        _minLength = minLength == null || minLength < 0 ? 0 : _maxLength == null || _maxLength > minLength ? minLength : _maxLength;
    }

    /**
     * @return the maxLength
     */
    public Integer getMaxLength() {
        return _maxLength;
    }

    /**
     * @param maxLength the maxLength to set
     */
    public void setMaxLength(Integer maxLength) {
        XS2.checkAccess();
        _maxLength = maxLength == null || maxLength > 0 ? maxLength : null;
        if (_maxLength != null && _minLength > _maxLength) {
            _minLength = _maxLength;
        }
    }

    /**
     * @return the pattern
     */
    public Pattern getPattern() {
        return _pattern;
    }

    /**
     * @param pattern the pattern to set
     */
    public void setPattern(Pattern pattern) {
        XS2.checkAccess();
        _pattern = pattern;
    }

    /**
     * @return the default regex error message
     */
    public String getDefaultRegexErrorMessage() {
        return getLocalizedRegexErrorMessage(null);
    }

    /**
     * El método setDefaultRegexErrorMessage se utiliza para establecer el mensaje de error asociado a la expresión regular definida mediante el
     * elemento regex de la anotación StringField, que se almacena en el archivo de recursos por defecto. En caso de que el archivo de recursos para
     * el idioma seleccionado por el usuario no esté disponible, la interfaz de la aplicación utiliza el archivo de recursos por defecto para obtener
     * el valor del mensaje.
     *
     * @param message mensaje de error asociado a la expresión regular
     */
    public void setDefaultRegexErrorMessage(String message) {
        setLocalizedRegexErrorMessage(null, message);
    }

    /**
     * @param locale the locale for the regex error message
     * @return the localized regex error message
     */
    public String getLocalizedRegexErrorMessage(Locale locale) {
        Locale l = localeReadingKey(locale);
        return _localizedRegexErrorMessage.get(l);
    }

    /**
     * El método setLocalizedRegexErrorMessage se utiliza para establecer el mensaje de error asociado a la expresión regular definida mediante el
     * elemento regex de la anotación StringField, que se almacena en el archivo de recursos de configuración regional. En caso de que el archivo de
     * recursos para el idioma seleccionado por el usuario no esté disponible, la interfaz de la aplicación utiliza el archivo de recursos por defecto
     * para obtener el valor del mensaje.
     *
     * @param locale configuración regional
     * @param message mensaje de error asociado a la expresión regular
     */
    public void setLocalizedRegexErrorMessage(Locale locale, String message) {
        Locale l = localeWritingKey(locale);
        if (message == null) {
            _localizedRegexErrorMessage.remove(l);
        } else {
            _localizedRegexErrorMessage.put(l, message);
        }
    }

    /**
     * @return the letter case
     */
    public LetterCase getLetterCase() {
        return _letterCase;
    }

    /**
     * @param letterCase the letter case to set
     */
    public void setLetterCase(LetterCase letterCase) {
        XS2.checkAccess();
        _letterCase = letterCase;
    }

    /**
     * @return the allow diacritics indicator
     */
    public boolean getAllowDiacritics() {
        return _allowDiacritics;
    }

    /**
     * @param allowDiacritics the allow diacritics indicator to set
     */
    public void setAllowDiacritics(boolean allowDiacritics) {
        XS2.checkAccess();
        _allowDiacritics = allowDiacritics;
    }

    /**
     * @return the special converter name
     */
    public String getSpecialConverterName() {
        return _specialConverterName;
    }

    public void setSpecialConverterName(String converter) {
        XS2.checkAccess();
        _specialConverterName = converter;
    }

    /**
     * @return the special validator name
     */
    public String getSpecialValidatorName() {
        return _specialValidatorName;
    }

    public void setSpecialValidatorName(String validator) {
        XS2.checkAccess();
        _specialValidatorName = validator;
    }

    /**
     * @return the url type
     */
//  @Override -- Implements method from: URL (StringProperty)
    public UrlType getUrlType() {
        return _urlType;
    }

    /**
     * @param type the url type to set
     */
    public void setUrlType(UrlType type) {
        XS2.checkAccess();
        _urlType = type;
    }

    /**
     * @return the url display type
     */
//  @Override -- Implements method from: URL (StringProperty)
    public UrlDisplayType getUrlDisplayType() {
        return _urlDisplayType;
    }

    /**
     * @param type the url display type to set
     */
    public void setUrlDisplayType(UrlDisplayType type) {
        XS2.checkAccess();
        _urlDisplayType = type;
    }

    /**
     * @return the max input file size
     */
//  @Override -- Implements method from: FileReference (StringProperty/StringParameter)
    public int getMaxInputFileSize() {
        return _maxInputFileSize < 0 ? Constants.DEFAULT_MAX_INPUT_FILE_SIZE : _maxInputFileSize;
    }

    /**
     * @param size the max input file size to set
     */
    public void setMaxInputFileSize(int size) {
        XS2.checkAccess();
        _maxInputFileSize = size;
    }

    protected int maxInputFileSize() {
        return _maxInputFileSize;
    }

    /**
     * @return the valid input file types
     */
//  @Override -- Implements method from: FileReference (StringProperty/StringParameter)
    public MimeType[] getValidInputFileTypes() {
        return _validInputFileTypes;
    }

    /**
     * @param types the valid input file types to set
     */
    public void setValidInputFileTypes(MimeType[] types) {
        XS2.checkAccess();
        _validInputFileTypes = types;
    }

    private final static String MIME_TYPES_REGEX_PREFIX = "^(.*)(\\.)";

    private final static String MIME_TYPES_REGEX_SUFFIX = "$";

//  @Override -- Implements method from: FileReference (StringProperty/StringParameter)
    public String getValidInputFileTypesRegex() {
        MimeType[] validInputFileTypes = getValidInputFileTypes();
        if (validInputFileTypes == null || validInputFileTypes.length == 0) {
            return null;
        }
        String[] strings = new String[validInputFileTypes.length];
        for (int i = 0; i < validInputFileTypes.length; i++) {
            strings[i] = StringUtils.join(validInputFileTypes[i].getExtensions(), "|");
        }
        String join = StringUtils.join(strings, "|");
        return MIME_TYPES_REGEX_PREFIX + "(" + join + ")" + MIME_TYPES_REGEX_SUFFIX;
    }

    /**
     * @return the valid input file pattern
     */
//  @Override -- Implements method from: FileReference (StringProperty/StringParameter)
    public Pattern getValidInputFilePattern() {
        return _validInputFilePattern;
    }

    /**
     * @param pattern the valid input file pattern to set
     */
    public void setValidInputFilePattern(Pattern pattern) {
        XS2.checkAccess();
        _validInputFilePattern = pattern;
    }

    /**
     * @return the upload storage option
     */
//  @Override -- Implements method from: FileReference (StringProperty/StringParameter)
    public UploadStorageOption getUploadStorageOption() {
        return _uploadStorageOption;
    }

    /**
     * @param option the upload storage option to set
     */
    public void setUploadStorageOption(UploadStorageOption option) {
        XS2.checkAccess();
        _uploadStorageOption = option;
    }

    /**
     * @return the blob field name
     */
//  @Override -- Implements method from: FileReference (StringProperty/StringParameter)
    public String getBlobFieldName() {
        return _blobFieldName;
    }

    /**
     * @param fieldName the blob field name to set
     */
    public void setBlobFieldName(String fieldName) {
        XS2.checkAccess();
        _blobFieldName = fieldName;
    }

    /**
     * @return the blob field
     */
//  @Override -- Implements method from: FileReference (StringProperty/StringParameter)
    public Field getBlobField() {
        return _blobField;
    }

    /**
     * @param field the blob field to set
     */
    public void setBlobField(Field field) {
        XS2.checkAccess();
        _blobField = field;
    }

    /**
     * @return the blob property
     */
//  @Override -- Implements method from: FileReference (StringProperty/StringParameter)
    public Property getBlobProperty() {
        return _blobProperty;
    }

    /**
     * @param property the blob property to set
     */
    public void setBlobProperty(Property property) {
        XS2.checkAccess();
        _blobProperty = property;
    }

    /**
     * @return the join field name
     */
//  @Override -- Implements method from: FileReference (StringProperty/StringParameter)
    public String getJoinFieldName() {
        return _joinFieldName;
    }

    /**
     * @param fieldName the join field name to set
     */
    public void setJoinFieldName(String fieldName) {
        XS2.checkAccess();
        _joinFieldName = fieldName;
    }

    /**
     * @return the join field
     */
//  @Override -- Implements method from: FileReference (StringProperty/StringParameter)
    public Field getJoinField() {
        return _joinField;
    }

    /**
     * @param field the join field to set
     */
    public void setJoinField(Field field) {
        XS2.checkAccess();
        _joinField = field;
    }

    /**
     * @return the join property
     */
//  @Override -- Implements method from: FileReference (StringProperty/StringParameter)
    public Property getJoinProperty() {
        return _joinProperty;
    }

    /**
     * @param property the join property to set
     */
    public void setJoinProperty(Property property) {
        XS2.checkAccess();
        _joinProperty = property;
    }

    /**
     * @return the load field name
     */
//  @Override -- Implements method from: FileReference (StringProperty/StringParameter)
    public String getLoadFieldName() {
        return _loadFieldName;
    }

    /**
     * @param fieldName the load field name to set
     */
    public void setLoadFieldName(String fieldName) {
        XS2.checkAccess();
        _loadFieldName = fieldName;
    }

    /**
     * @return the load field
     */
//  @Override -- Implements method from: FileReference (StringProperty/StringParameter)
    public Field getLoadField() {
        return _loadField;
    }

    /**
     * @param field the load field to set
     */
    public void setLoadField(Field field) {
        XS2.checkAccess();
        _loadField = field;
    }

    /**
     * @return the load property
     */
//  @Override -- Implements method from: FileReference (StringProperty/StringParameter)
    public Property getLoadProperty() {
        return _loadProperty;
    }

    /**
     * @param property the load property to set
     */
    public void setLoadProperty(Property property) {
        XS2.checkAccess();
        _loadProperty = property;
    }

    /**
     * @return the updateable file reference indicator
     */
//  @Override -- Implements method from: FileReference (StringProperty/StringParameter)
    public Kleenean getUpdateableFileReference() {
        return _updateableFileReference;
    }

    /**
     * @param updateable the updateable file reference indicator to set
     */
    public void setUpdateableFileReference(Kleenean updateable) {
        XS2.checkAccess();
        _updateableFileReference = updateable;
    }

    /**
     * @return the updateable file reference indicator as boolean
     */
//  @Override -- Implements method from: FileReference (StringProperty/StringParameter)
    public boolean isUpdateableFileReference() {
        Kleenean updateableFileReference = getUpdateableFileReference();
        return updateableFileReference != null && updateableFileReference.toBoolean(false);
    }

    // <editor-fold defaultstate="collapsed" desc="toString">
    @Override
    protected String fieldsToString(int n, String key, boolean verbose, boolean fields, boolean maps) {
        String tab = verbose ? StringUtils.repeat(" ", 4) : "";
        String fee = verbose ? StringUtils.repeat(tab, n) : "";
        String faa = " = ";
        String foo = verbose ? EOL : ", ";
        String string = super.fieldsToString(n, key, verbose, fields, maps);
        if (fields || verbose) {
            if (verbose) {
                string += fee + tab + "minLength" + faa + _minLength + foo;
                if (_maxLength != null) {
                    string += fee + tab + "maxLength" + faa + _maxLength + foo;
                }
            }
        }
        return string;
    }
    // </editor-fold>

}
