/*
 * Copyright 2017 Jorge Campins y David Uzcategui
 *
 * Este archivo forma parte de Adalid.
 *
 * Adalid es software libre; usted puede redistribuirlo y/o modificarlo bajo los terminos de la
 * licencia "GNU General Public License" publicada por la Fundacion "Free Software Foundation".
 * Adalid se distribuye con la esperanza de que pueda ser util, pero SIN NINGUNA GARANTIA; sin
 * siquiera las garantias implicitas de COMERCIALIZACION e IDONEIDAD PARA UN PROPOSITO PARTICULAR.
 *
 * Para mas detalles vea la licencia "GNU General Public License" en http://www.gnu.org/licenses
 */
package adalid.core.interfaces;

import adalid.core.Instance;
import adalid.core.ProcessOperation;
import adalid.core.Trigger;
import adalid.core.enums.SpecialBooleanValue;
import adalid.core.enums.SpecialCharacterValue;
import adalid.core.enums.SpecialEntityValue;
import adalid.core.enums.SpecialNumericValue;
import adalid.core.enums.SpecialTemporalValue;
import adalid.core.expressions.BooleanComparisonX;
import adalid.core.expressions.BooleanConditionalX;
import adalid.core.expressions.BooleanOrderedPairX;
import adalid.core.expressions.BooleanScalarX;
import adalid.core.expressions.CharacterConditionalX;
import adalid.core.expressions.CharacterScalarX;
import adalid.core.expressions.EntityConditionalX;
import adalid.core.expressions.NumericConditionalX;
import adalid.core.expressions.TemporalConditionalX;
import java.util.Locale;

/**
 * @author Jorge Campins
 */
public interface BooleanExpression extends Expression {

    BooleanComparisonX isNull();

    BooleanComparisonX isNotNull();

    BooleanComparisonX isTrue();

    BooleanComparisonX isFalse();

    BooleanComparisonX isNotTrue();

    BooleanComparisonX isNotFalse();

    BooleanComparisonX isNullOrTrue();

    BooleanComparisonX isNullOrFalse();

    BooleanComparisonX isEqualTo(BooleanExpression y);

    BooleanComparisonX isNotEqualTo(BooleanExpression y);

    BooleanComparisonX isNullOrEqualTo(BooleanExpression y);

    BooleanComparisonX isNullOrNotEqualTo(BooleanExpression y);

    /**
     * El método and contruye una expresión lógica que genera la conjunción (AND) de esta expresión (operando X) con la expresión que recibe como
     * argumento (operando Y). La conjunción es un operador lógico que resulta en verdadero si ambos operandos son verdadero.
     *
     * @param y operando Y
     * @return expresión lógica que genera la conjunción de ambos operandos.
     */
    BooleanOrderedPairX and(BooleanExpression y);

    /**
     * El método nand contruye una expresión lógica que genera la negación alternativa (NAND) de esta expresión (operando X) con la expresión que
     * recibe como argumento (operando Y). La negación alternativa es un operador lógico que resulta en verdadero si uno de los operandos es falso. Es
     * equivalente a la negación (NOT) de la conjunción (AND).
     *
     * @param y operando Y
     * @return expresión lógica que genera la negación alternativa de ambos operandos.
     */
    BooleanOrderedPairX nand(BooleanExpression y);

    /**
     * El método or contruye una expresión lógica que genera la disyunción (OR) de esta expresión (operando X) con la expresión que recibe como
     * argumento (operando Y). La disyunción es un operador lógico que resulta en verdadero si uno de los operandos es verdadero.
     *
     * @param y operando Y
     * @return expresión lógica que genera la disyunción de ambos operandos.
     */
    BooleanOrderedPairX or(BooleanExpression y);

    /**
     * El método nor contruye una expresión lógica que genera la negación conjunta (NOR) de esta expresión (operando X) con la expresión que recibe
     * como argumento (operando Y). La negación conjunta es un operador lógico que resulta en verdadero si ambos operandos son falso. Es equivalente a
     * la negación (NOT) de la disyunción (OR).
     *
     * @param y operando Y
     * @return expresión lógica que genera la negación conjunta de ambos operandos.
     */
    BooleanOrderedPairX nor(BooleanExpression y);

    /**
     * El método xor contruye una expresión lógica que genera la disyunción exclusiva (XOR) de esta expresión (operando X) con la expresión que recibe
     * como argumento (operando Y). La disyunción exclusiva es un operador lógico que resulta en verdadero si uno y solo uno de los operandos es
     * verdadero.
     *
     * @param y operando Y
     * @return expresión lógica que genera la disyunción exclusiva de ambos operandos.
     */
    BooleanOrderedPairX xor(BooleanExpression y);

    /**
     * El método xnor contruye una expresión lógica que genera la equivalencia (XNOR) de esta expresión (operando X) con la expresión que recibe como
     * argumento (operando Y). La equivalencia es un operador lógico que resulta en verdadero si ambos operandos son verdadero ó si ambos son falso.
     * Es equivalente a la negación (NOT) de la disyunción exclusiva (XOR).
     *
     * @param y operando Y
     * @return expresión lógica que genera la equivalencia de ambos operandos.
     */
    BooleanOrderedPairX xnor(BooleanExpression y);

    /**
     * El método implies contruye una expresión lógica que genera la condicional material (X_IMPLIES_Y) de esta expresión (operando X) con la
     * expresión que recibe como argumento (operando Y). La condicional material es un operador lógico que resulta en falso solo si X es verdadero y Y
     * es falso. Es equivalente a la disyunción (OR) de la negación (NOT) de X con Y (NOT X OR Y).
     *
     * @param y operando Y
     * @return expresión lógica que genera la condicional material de ambos operandos.
     */
    BooleanOrderedPairX implies(BooleanExpression y);

    /**
     * El método not contruye una expresión lógica que genera la negación (NOT) de esta expresión (operando X). La negación es un operador lógico que
     * resulta en verdadero si X es falso y en falso si X es verdadero.
     *
     * @return expresión lógica que genera la negación de esta expresión.
     */
    BooleanScalarX not();

    CharacterScalarX toCharString();

    EntityConditionalX then(Entity value);

    EntityConditionalX then(Instance value);

    EntityConditionalX then(SpecialEntityValue value);

    EntityConditionalX then(EntityExpression value);

    BooleanConditionalX then(Boolean value);

    BooleanConditionalX then(SpecialBooleanValue value);

    BooleanConditionalX then(BooleanExpression value);

    CharacterConditionalX then(String value);

    CharacterConditionalX then(SpecialCharacterValue value);

    CharacterConditionalX then(CharacterExpression value);

    NumericConditionalX then(Number value);

    NumericConditionalX then(SpecialNumericValue value);

    NumericConditionalX then(NumericExpression value);

    TemporalConditionalX then(java.util.Date value);

    TemporalConditionalX then(SpecialTemporalValue value);

    TemporalConditionalX then(TemporalExpression value);

    Trigger trigger(ProcessOperation operation);

    String getDefaultErrorMessage();

    /**
     * El método setDefaultErrorMessage se utiliza para establecer el mensaje de error asociado a la expresión que se almacena en el archivo de
     * recursos por defecto. En caso de que el archivo de recursos para el idioma seleccionado por el usuario no esté disponible, la interfaz de la
     * aplicación utiliza el archivo de recursos por defecto para obtener el valor del mensaje.
     *
     * @param defaultErrorMessage mensaje de error asociado a la expresión
     */
    void setDefaultErrorMessage(String defaultErrorMessage);

    String getLocalizedErrorMessage(Locale locale);

    /**
     * El método setLocalizedErrorMessage se utiliza para establecer el mensaje de error asociado a la expresión que se almacena en el archivo de
     * recursos de configuración regional. En caso de que el archivo de recursos para el idioma seleccionado por el usuario no esté disponible, la
     * interfaz de la aplicación utiliza el archivo de recursos por defecto para obtener el valor del mensaje.
     *
     * @param locale configuración regional
     * @param localizedErrorMessage mensaje de error asociado a la expresión
     */
    void setLocalizedErrorMessage(Locale locale, String localizedErrorMessage);

}
