/*
 * Copyright 2017 Jorge Campins y David Uzcategui
 *
 * Este archivo forma parte de Adalid.
 *
 * Adalid es software libre; usted puede redistribuirlo y/o modificarlo bajo los terminos de la
 * licencia "GNU General Public License" publicada por la Fundacion "Free Software Foundation".
 * Adalid se distribuye con la esperanza de que pueda ser util, pero SIN NINGUNA GARANTIA; sin
 * siquiera las garantias implicitas de COMERCIALIZACION e IDONEIDAD PARA UN PROPOSITO PARTICULAR.
 *
 * Para mas detalles vea la licencia "GNU General Public License" en http://www.gnu.org/licenses
 */
package adalid.core.interfaces;

import adalid.core.Tab;
import adalid.core.enums.AggregateFunction;
import adalid.core.enums.Checkpoint;
import adalid.core.enums.DefaultCondition;
import adalid.core.enums.PropertyAccess;
import java.util.List;
import java.util.Map;

/**
 * @author Jorge Campins
 */
public interface Property extends DataArtifact, ValuedArtifact {

    /**
     * @return the corresponding property at the declaring entity's root instance
     */
    Property getPropertyAtRoot();

    /**
     * @return the base field indicator
     */
    boolean isBaseField();

    /**
     * @return the key field indicator
     */
    boolean isKeyField();

    /**
     * @return the property access
     */
    PropertyAccess getPropertyAccess();

    /**
     * @return the auditable indicator
     */
    boolean isAuditable();

    /**
     * @return the password indicator
     */
    boolean isPassword();

    /**
     * @return the required indicator
     */
    boolean isReadOnly();

    /**
     * @return the required indicator
     */
    boolean isRequiredProperty();

    /**
     * @return the hidden field indicator
     */
    boolean isHiddenField();

    /**
     * @return the create field indicator
     */
    boolean isCreateField();

    /**
     * @return the update field indicator
     */
    boolean isUpdateField();

    /**
     * @return the search field indicator
     */
    boolean isSearchField();

    /**
     * @return the filter field indicator
     */
    boolean isFilterField();

    /**
     * @return the sort field indicator
     */
    boolean isSortField();

    /**
     * @return the table field indicator
     */
    boolean isTableField();

    /**
     * @return the detail field indicator
     */
    boolean isDetailField();

    /**
     * @return the column field indicator
     */
    boolean isColumnField();

    /**
     * @return the report field indicator
     */
    boolean isReportField();

    /**
     * @return the export field indicator
     */
    boolean isExportField();

    /**
     * @return the submit field indicator
     */
    boolean isSubmitField();

    /**
     * @return the headertextless field indicator
     */
    boolean isHeadertextlessField();

    /**
     * @return the heading field indicator
     */
    boolean isHeadingField();

    /**
     * @return the overlay field indicator
     */
    boolean isOverlayField();

    /**
     * @return the immutable field indicator
     */
    boolean isImmutableField();

    /**
     * @return the default condition
     */
    DefaultCondition getDefaultCondition();

    /**
     * @return the default checkpoint
     */
    Checkpoint getDefaultCheckpoint();

    /**
     * @return the sequence number
     */
    int getSequenceNumber();

    /**
     * @return the aggregate function
     */
    AggregateFunction getAggregateFunction();

    /**
     * @return the aggregate title
     */
    String getAggregateTitle();

    /**
     * @return the calculable indicator
     */
    boolean isCalculable();

    /**
     * @return the nullable indicator
     */
    boolean isNullable();

    /**
     * @return the insertable indicator
     */
    boolean isInsertable();

    /**
     * @return the updateable indicator
     */
    boolean isUpdateable();

    /**
     * @return the unique indicator
     */
    boolean isUnique();

    /**
     * @return the rendering filter
     */
    BooleanExpression getRenderingFilter();

    /**
     * El método setRenderingFilter se utiliza para establecer el filtro de presentación de propiedades en vistas (páginas) de registro, y de
     * parámetros en vistas (páginas) de ejecución de operaciones de negocio. Solo si se cumplen los criterios del filtro, el valor de la propiedad o
     * el parámetro será visible en las vistas (páginas) de registro o ejecución de operaciones de negocio.
     *
     * @param renderingFilter expresión booleana que se utiliza como filtro
     */
    void setRenderingFilter(BooleanExpression renderingFilter);

    /**
     * @return the requiring filter
     */
    BooleanExpression getRequiringFilter();

    /**
     * El método setRequiringFilter se utiliza para establecer el filtro de obligatoriedad de propiedades en vistas (páginas) de registro, y de
     * parámetros en vistas (páginas) de ejecución de operaciones de negocio. Solo si se cumplen los criterios del filtro, el valor de la propiedad o
     * el parámetro será obligatoriamente requerido en las vistas (páginas) de registro o ejecución de operaciones de negocio.
     *
     * @param requiringFilter expresión booleana que se utiliza como filtro
     */
    void setRequiringFilter(BooleanExpression requiringFilter);

    /**
     * @return the modifying filter
     */
    BooleanExpression getModifyingFilter();

    /**
     * El método setModifyingFilter se utiliza para establecer el filtro de modificación de propiedades en vistas (páginas) de registro, y de
     * parámetros en vistas (páginas) de ejecución de operaciones de negocio. Solo si se cumplen los criterios del filtro, el valor de la propiedad o
     * el parámetro podrá ser modificado en las vistas (páginas) de registro o ejecución de operaciones de negocio.
     *
     * @param modifyingFilter expresión booleana que se utiliza como filtro
     */
    void setModifyingFilter(BooleanExpression modifyingFilter);

    /**
     * @return the nullifying filter
     */
    BooleanExpression getNullifyingFilter();

    /**
     * El método setNullifyingFilter se utiliza para establecer el filtro de anulación de propiedades en vistas (páginas) de registro, y de parámetros
     * en vistas (páginas) de ejecución de operaciones de negocio. Solo si se cumplen los criterios del filtro, el valor de la propiedad o el
     * parámetro será anulado en las vistas (páginas) de registro o ejecución de operaciones de negocio.
     *
     * @param nullifyingFilter expresión booleana que se utiliza como filtro
     */
    void setNullifyingFilter(BooleanExpression nullifyingFilter);

    /**
     * @return the enclosing tabs list
     */
    List<Tab> getEnclosingTabs();

    /**
     * @return the file reference indicator
     */
    boolean isFileReferenceField();

    /**
     * @return the graphic image indicator
     */
    boolean isGraphicImageField();

    /**
     * @return the URL indicator
     */
    boolean isUniformResourceLocatorField();

    /**
     * @return the variant string indicator
     */
    boolean isVariantStringField();

    /**
     * @return true if it is the Primary Key property; otherwise false
     */
    boolean isPrimaryKeyProperty();

    /**
     * @return true if it is the Sequence property; otherwise false
     */
    boolean isSequenceProperty();

    /**
     * @return true if it is the Version property; otherwise false
     */
    boolean isVersionProperty();

    /**
     * @return true if it is the Numeric Key property; otherwise false
     */
    boolean isNumericKeyProperty();

    /**
     * @return true if it is the Character Key property; otherwise false
     */
    boolean isCharacterKeyProperty();

    /**
     * @return true if it is the Name property; otherwise false
     */
    boolean isNameProperty();

    /**
     * @return true if it is the Description property; otherwise false
     */
    boolean isDescriptionProperty();

    /**
     * @return true if it is the Image property; otherwise false
     */
    boolean isImageProperty();

    /**
     * @return true if it is the Inactive Indicator property; otherwise false
     */
    boolean isInactiveIndicatorProperty();

    /**
     * @return true if it is the URL property; otherwise false
     */
    boolean isUrlProperty();

    /**
     * @return true if it is the Parent property; otherwise false
     */
    boolean isParentProperty();

    /**
     * @return true if it is the Owner property; otherwise false
     */
    boolean isOwnerProperty();

    /**
     * @return true if it is the segment property; otherwise false
     */
    boolean isSegmentProperty();

    /**
     * @return true if it is a Unique Key property; otherwise false
     */
    boolean isUniqueKeyProperty();

    /**
     * @return true if it is the Business Key property; otherwise false
     */
    boolean isBusinessKeyProperty();

    /**
     * @return true if it is the Discriminator property; otherwise false
     */
    boolean isDiscriminatorProperty();

    /**
     * @return true if it is the State property; otherwise false
     */
    boolean isStateProperty();

    /**
     * @return true if it is a Primitive; otherwise false
     */
    boolean isPrimitive();

    /**
     * @return true if it is a BinaryPrimitive; otherwise false
     */
    boolean isBinaryPrimitive();

    /**
     * @return true if it is a BooleanPrimitive; otherwise false
     */
    boolean isBooleanPrimitive();

    /**
     * @return true if it is a CharacterPrimitive; otherwise false
     */
    boolean isCharacterPrimitive();

    /**
     * @return true if it is a NumericPrimitive; otherwise false
     */
    boolean isNumericPrimitive();

    /**
     * @return true if it is a TemporalPrimitive; otherwise false
     */
    boolean isTemporalPrimitive();

    /**
     * @return true if it is a BigDecimalData; otherwise false
     */
    boolean isBigDecimalData();

    /**
     * @return true if it is a BigIntegerData; otherwise false
     */
    boolean isBigIntegerData();

    /**
     * @return true if it is a BinaryData; otherwise false
     */
    boolean isBinaryData();

    /**
     * @return true if it is a BooleanData; otherwise false
     */
    boolean isBooleanData();

    /**
     * @return true if it is a ByteData; otherwise false
     */
    boolean isByteData();

    /**
     * @return true if it is a CharacterData; otherwise false
     */
    boolean isCharacterData();

    /**
     * @return true if it is a DateData; otherwise false
     */
    boolean isDateData();

    /**
     * @return true if it is a DoubleData; otherwise false
     */
    boolean isDoubleData();

    /**
     * @return true if it is a FloatData; otherwise false
     */
    boolean isFloatData();

    /**
     * @return true if it is a IntegerData; otherwise false
     */
    boolean isIntegerData();

    /**
     * @return true if it is a LongData; otherwise false
     */
    boolean isLongData();

    /**
     * @return true if it is a ShortData; otherwise false
     */
    boolean isShortData();

    /**
     * @return true if it is a StringData; otherwise false
     */
    boolean isStringData();

    /**
     * @return true if it is a TimeData; otherwise false
     */
    boolean isTimeData();

    /**
     * @return true if it is a TimestampData; otherwise false
     */
    boolean isTimestampData();

    /**
     * @return true if it is an entity; otherwise false
     */
    boolean isEntity();

    /**
     * @return true if it is a contextual entity; otherwise false
     */
    boolean isContextualEntity();

    /**
     * @return true if it is a enumeration entity; otherwise false
     */
    boolean isEnumerationEntity();

    /**
     * @return true if it is a database entity; otherwise false
     */
    boolean isDatabaseEntity();

    /**
     * @return true if it is a persistent entity; otherwise false
     */
    boolean isPersistentEntity();

    /**
     * @return true if it is a persistent enumeration entity; otherwise false
     */
    boolean isPersistentEnumerationEntity();

    /**
     * @return true if is a overlayable entity reference; otherwise false
     */
    boolean isOverlayableEntityReference();

    /**
     * @return the property path list
     */
    List<Artifact> getPropertyPathList();

    /**
     * @return the list of properties referencing this property in their rendering filter
     */
    Map<String, Property> getMaxValueReferencingProperties();

    /**
     * @param recursively
     * @return the list of properties referencing this property in their rendering filter
     */
    Map<String, Property> getMaxValueReferencingProperties(boolean recursively);

    /**
     * @return the list of properties referencing this property in their rendering filter
     */
    Map<String, Property> getMinValueReferencingProperties();

    /**
     * @param recursively
     * @return the list of properties referencing this property in their rendering filter
     */
    Map<String, Property> getMinValueReferencingProperties(boolean recursively);

    /**
     * @return the list of properties referencing this property in their modifying filter
     */
    Map<String, Property> getModifyingFilterReferencingProperties();

    /**
     * @param recursively
     * @return the list of properties referencing this property in their modifying filter
     */
    Map<String, Property> getModifyingFilterReferencingProperties(boolean recursively);

    /**
     * @return the list of properties referencing this property in their rendering filter
     */
    Map<String, Property> getRenderingFilterReferencingProperties();

    /**
     * @param recursively
     * @return the list of properties referencing this property in their rendering filter
     */
    Map<String, Property> getRenderingFilterReferencingProperties(boolean recursively);

    /**
     * @return the list of properties referencing this property in their requiring filter
     */
    Map<String, Property> getRequiringFilterReferencingProperties();

    /**
     * @param recursively
     * @return the list of properties referencing this property in their requiring filter
     */
    Map<String, Property> getRequiringFilterReferencingProperties(boolean recursively);

    /**
     * @return the list of properties referencing this property in their search query filter
     */
    Map<String, Property> getSearchQueryFilterReferencingProperties();

    /**
     * @param recursively
     * @return the list of properties referencing this property in their search query filter
     */
    Map<String, Property> getSearchQueryFilterReferencingProperties(boolean recursively);

    /**
     * @return the property size in pixels
     */
    int getPixels();

    /**
     * @return the column size in pixels
     */
    int getColumnPixels();

}
