/*
 * Copyright 2017 Jorge Campins y David Uzcategui
 *
 * Este archivo forma parte de Adalid.
 *
 * Adalid es software libre; usted puede redistribuirlo y/o modificarlo bajo los terminos de la
 * licencia "GNU General Public License" publicada por la Fundacion "Free Software Foundation".
 * Adalid se distribuye con la esperanza de que pueda ser util, pero SIN NINGUNA GARANTIA; sin
 * siquiera las garantias implicitas de COMERCIALIZACION e IDONEIDAD PARA UN PROPOSITO PARTICULAR.
 *
 * Para mas detalles vea la licencia "GNU General Public License" en http://www.gnu.org/licenses
 */
package adalid.core.parameters;

import adalid.core.annotations.FileReference;
import adalid.core.annotations.ParameterField;
import adalid.core.annotations.StringField;
import adalid.core.data.types.StringData;
import adalid.core.enums.Kleenean;
import adalid.core.enums.MimeType;
import adalid.core.enums.UploadStorageOption;
import adalid.core.interfaces.Parameter;
import adalid.core.interfaces.Property;
import adalid.core.properties.StringProperty;
import java.lang.annotation.Annotation;
import java.lang.reflect.Field;
import java.util.List;
import java.util.regex.Pattern;

/**
 * @author Jorge Campins
 */
public class StringParameter extends StringData implements Parameter {

    @Override
    protected List<Class<? extends Annotation>> getValidFieldAnnotations() {
        List<Class<? extends Annotation>> valid = super.getValidFieldAnnotations();
        valid.add(FileReference.class);
        valid.add(ParameterField.class);
        valid.add(StringField.class);
        return valid;
    }

    /**
     * @return the max input file size
     */
    @Override // Implements method from: FileReference (StringProperty/StringParameter)
    public int getMaxInputFileSize() {
        int maxInputFileSize = maxInputFileSize();
        if (maxInputFileSize < 0) {
            StringProperty linkedProperty = getParameterLinkedProperty();
            if (linkedProperty != null) {
                return linkedProperty.getMaxInputFileSize();
            }
        }
        return super.getMaxInputFileSize();
    }

    /**
     * @return the valid input file types
     */
    @Override // Implements method from: FileReference (StringProperty/StringParameter)
    public MimeType[] getValidInputFileTypes() {
        MimeType[] validInputFileTypes = super.getValidInputFileTypes();
        if (validInputFileTypes == null || validInputFileTypes.length == 0) {
            StringProperty linkedProperty = getParameterLinkedProperty();
            if (linkedProperty != null) {
                return linkedProperty.getValidInputFileTypes();
            }
        }
        return validInputFileTypes;
    }

    /**
     * @return the valid input file pattern
     */
    @Override // Implements method from: FileReference (StringProperty/StringParameter)
    public Pattern getValidInputFilePattern() {
        Pattern _validInputFilePattern = super.getValidInputFilePattern();
        if (_validInputFilePattern == null) {
            StringProperty linkedProperty = getParameterLinkedProperty();
            if (linkedProperty != null) {
                return linkedProperty.getValidInputFilePattern();
            }
        }
        return _validInputFilePattern;
    }

    /**
     * @return the upload storage option
     */
    @Override // Implements method from: FileReference (StringProperty/StringParameter)
    public UploadStorageOption getUploadStorageOption() {
        UploadStorageOption uploadStorageOption = super.getUploadStorageOption();
        if (uploadStorageOption == null || uploadStorageOption.equals(UploadStorageOption.UNSPECIFIED)) {
            StringProperty linkedProperty = getParameterLinkedProperty();
            if (linkedProperty != null) {
                return linkedProperty.getUploadStorageOption();
            }
        }
        return uploadStorageOption;
    }

    /**
     * @return the blob field name
     */
    @Override // Implements method from: FileReference (StringProperty/StringParameter)
    public String getBlobFieldName() {
        String blobFieldName = super.getBlobFieldName();
        if (blobFieldName == null || blobFieldName.length() == 0) {
            StringProperty linkedProperty = getParameterLinkedProperty();
            if (linkedProperty != null) {
                return linkedProperty.getBlobFieldName();
            }
        }
        return blobFieldName;
    }

    /**
     * @return the blob field
     */
    @Override // Implements method from: FileReference (StringProperty/StringParameter)
    public Field getBlobField() {
        Field blobField = super.getBlobField();
        if (blobField == null) {
            StringProperty linkedProperty = getParameterLinkedProperty();
            if (linkedProperty != null) {
                return linkedProperty.getBlobField();
            }
        }
        return blobField;
    }

    /**
     * @return the blob property
     */
    @Override // Implements method from: FileReference (StringProperty/StringParameter)
    public Property getBlobProperty() {
        Property blobProperty = super.getBlobProperty();
        if (blobProperty == null) {
            StringProperty linkedProperty = getParameterLinkedProperty();
            if (linkedProperty != null) {
                return linkedProperty.getBlobProperty();
            }
        }
        return blobProperty;
    }

    /**
     * @return the join field name
     */
    @Override // Implements method from: FileReference (StringProperty/StringParameter)
    public String getJoinFieldName() {
        String joinFieldName = super.getJoinFieldName();
        if (joinFieldName == null || joinFieldName.length() == 0) {
            StringProperty linkedProperty = getParameterLinkedProperty();
            if (linkedProperty != null) {
                return linkedProperty.getJoinFieldName();
            }
        }
        return joinFieldName;
    }

    /**
     * @return the join field
     */
    @Override // Implements method from: FileReference (StringProperty/StringParameter)
    public Field getJoinField() {
        Field joinField = super.getJoinField();
        if (joinField == null) {
            StringProperty linkedProperty = getParameterLinkedProperty();
            if (linkedProperty != null) {
                return linkedProperty.getJoinField();
            }
        }
        return joinField;
    }

    /**
     * @return the join property
     */
    @Override // Implements method from: FileReference (StringProperty/StringParameter)
    public Property getJoinProperty() {
        Property joinProperty = super.getJoinProperty();
        if (joinProperty == null) {
            StringProperty linkedProperty = getParameterLinkedProperty();
            if (linkedProperty != null) {
                return linkedProperty.getJoinProperty();
            }
        }
        return joinProperty;
    }

    /**
     * @return the load field name
     */
    @Override // Implements method from: FileReference (StringProperty/StringParameter)
    public String getLoadFieldName() {
        String loadFieldName = super.getLoadFieldName();
        if (loadFieldName == null || loadFieldName.length() == 0) {
            StringProperty linkedProperty = getParameterLinkedProperty();
            if (linkedProperty != null) {
                return linkedProperty.getLoadFieldName();
            }
        }
        return loadFieldName;
    }

    /**
     * @return the load field
     */
    @Override // Implements method from: FileReference (StringProperty/StringParameter)
    public Field getLoadField() {
        Field loadField = super.getLoadField();
        if (loadField == null) {
            StringProperty linkedProperty = getParameterLinkedProperty();
            if (linkedProperty != null) {
                return linkedProperty.getLoadField();
            }
        }
        return loadField;
    }

    /**
     * @return the load property
     */
    @Override // Implements method from: FileReference (StringProperty/StringParameter)
    public Property getLoadProperty() {
        Property loadProperty = super.getLoadProperty();
        if (loadProperty == null) {
            StringProperty linkedProperty = getParameterLinkedProperty();
            if (linkedProperty != null) {
                return linkedProperty.getLoadProperty();
            }
        }
        return loadProperty;
    }

    /**
     * @return the updateable file reference option
     */
    @Override // Implements method from: FileReference (StringProperty/StringParameter)
    public Kleenean getUpdateableFileReference() {
        Kleenean updateableFileReference = super.getUpdateableFileReference();
        if (updateableFileReference == null || updateableFileReference.equals(Kleenean.UNSPECIFIED)) {
            StringProperty linkedProperty = getParameterLinkedProperty();
            if (linkedProperty != null) {
                return linkedProperty.getUpdateableFileReference();
            }
        }
        return updateableFileReference;
    }

    private StringProperty getParameterLinkedProperty() {
        Property linkedProperty = getLinkedProperty();
        return linkedProperty instanceof StringProperty ? (StringProperty) linkedProperty : null;
    }

}
