/*
 * Copyright 2017 Jorge Campins y David Uzcategui
 *
 * Este archivo forma parte de Adalid.
 *
 * Adalid es software libre; usted puede redistribuirlo y/o modificarlo bajo los terminos de la
 * licencia "GNU General Public License" publicada por la Fundacion "Free Software Foundation".
 * Adalid se distribuye con la esperanza de que pueda ser util, pero SIN NINGUNA GARANTIA; sin
 * siquiera las garantias implicitas de COMERCIALIZACION e IDONEIDAD PARA UN PROPOSITO PARTICULAR.
 *
 * Para mas detalles vea la licencia "GNU General Public License" en http://www.gnu.org/licenses
 */
package adalid.jee2.meta.proyecto.base;

import adalid.commons.ProjectObjectModelReader;
import adalid.commons.properties.PropertiesHandler;
import adalid.commons.properties.SortedProperties;
import adalid.commons.util.StrUtils;
import adalid.core.Operation;
import adalid.core.Project;
import adalid.core.annotations.ProjectModule;
import adalid.core.enums.DatabaseLockingMechanism;
import adalid.core.enums.Kleenean;
import adalid.core.interfaces.Artifact;
import adalid.core.interfaces.Entity;
import adalid.core.interfaces.Parameter;
import adalid.core.interfaces.Property;
import adalid.jee2.ProjectObjectModel;
import java.util.Arrays;
import java.util.List;
import java.util.Properties;
import java.util.ResourceBundle;
import java.util.Set;
import java.util.TreeSet;
import meta.proyecto.comun.EntidadesBasicas;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Level;

/**
 * @author Jorge Campins
 */
public class ProyectoBase extends Project {

    // <editor-fold defaultstate="collapsed" desc="static fields">
    private static final String BASE_NAME = ProyectoBase.class.getName();

    private static final ResourceBundle RB = ResourceBundle.getBundle(BASE_NAME);

    private static final String DIR = System.getProperty("user.dir");

    private static final String SEP = System.getProperty("file.separator");

    private static final String PROPERTIES_SUFFIX = ".properties";

    private static final String DICTIONARY_DIR = DIR + SEP + "dictionary";

    private static final String DICTIONARY_NEXT_ID_KEY = "$next$id$";

    private static final String ENTITIES_DICTIONARY = DICTIONARY_DIR + SEP + "entities" + PROPERTIES_SUFFIX;

    private static final String OPERATIONS_DICTIONARY = DICTIONARY_DIR + SEP + "operations" + PROPERTIES_SUFFIX;

    private static final String PAGES_DICTIONARY = DICTIONARY_DIR + SEP + "pages" + PROPERTIES_SUFFIX;

    private static final String PARAMETERS_DICTIONARY = DICTIONARY_DIR + SEP + "parameters" + PROPERTIES_SUFFIX;

    private static final String ADALID_SCHEMA = StringUtils.trimToEmpty(getString("adalid.schema"));

    private static final List<String> SQL_FOLDERS_LIST = getStringList("sql.folders");
    // </editor-fold>

    // <editor-fold defaultstate="collapsed" desc="static fields' public getters">
    public static String getEsquemaEntidadesComunes() {
        return ADALID_SCHEMA;
    }

    public static List<String> getSqlFoldersList() {
        return SQL_FOLDERS_LIST;
    }
    // </editor-fold>

    // <editor-fold defaultstate="collapsed" desc="static fields' private getters">
    private static List<String> getStringList(String key) {
        String string = getString(key);
        if (string == null) {
            return null;
        }
        String[] strings = StringUtils.split(string, ", ");
        return Arrays.asList(strings);
    }

    private static String getString(String key) {
        try {
            String string = RB.getString(key);
            return StringUtils.trimToNull(string);
        } catch (Exception e) {
            return null;
        }
    }
    // </editor-fold>

    static {
        setLocale(SPANISH);
    }

    public ProyectoBase() {
        super();
        init();
    }

    private void init() {
//      setSupportedLocales(ENGLISH, SPANISH);
        setUserEntityClass(meta.entidad.comun.control.acceso.Usuario.class);
        setUploadedFileEntityClass(meta.entidad.comun.auditoria.ArchivoAdjunto.class);
    }

    @Override
    public boolean analyze() {
        boolean analyzed = super.analyze();
        if (analyzed) {
//          loadDictionary();
            initDictionary();
            buildDictionary();
//          storeDictionary();
        }
        return analyzed;
    }

    private final ProjectObjectModel pom = new ProjectObjectModel();

    @Override
    public ProjectObjectModelReader getProjectObjectModel() {
        return pom;
    }

    @Override
    public String getAdalidProjectVersion() {
        return pom.getProjectVersionNumber();
    }

    @Override
    protected void logAdalidProjectVersion() {
        pom.logProjectVersion();
    }

    @ProjectModule(menu = Kleenean.FALSE, role = Kleenean.FALSE)
    EntidadesBasicas entidadesBasicas;

    // <editor-fold defaultstate="collapsed" desc="private fields">
    private boolean _dictionaryEnabled;

    private Properties _entitiesDictionary;

    private Properties _operationsDictionary;

    private Properties _pagesDictionary;

    private Properties _parametersDictionary;

    private String _baseFolderName;

    private String _databaseName;

    private String _databaseFormerSchemaName;

    private String _rootFolderName;

    private String _roleCodePrefix;

    private String _messageDigestAlgorithm;

    private boolean _sqlBusinessAuditTrail;

    private DatabaseLockingMechanism _databaseLockingMechanism;
    // </editor-fold>

    /**
     * @return true if the dictionary is enabled; false otherwise
     */
    public boolean isDictionaryEnabled() {
        return _dictionaryEnabled;
    }

    /**
     * enables the dictionary
     */
    public void enableDictionary() {
        _dictionaryEnabled = true;
    }

    /**
     * @return the entities dictionary
     */
    public Properties getEntitiesDictionary() {
        return _entitiesDictionary;
    }

    /**
     * @return the operations dictionary
     */
    public Properties getOperationsDictionary() {
        return _operationsDictionary;
    }

    /**
     * @return the pages dictionary
     */
    public Properties getPagesDictionary() {
        return _pagesDictionary;
    }

    /**
     * @return the parameters dictionary
     */
    public Properties getParametersDictionary() {
        return _parametersDictionary;
    }

    /**
     * @return the entity keys
     */
    public Set<String> getEntityKeys() {
        Set<String> set = new TreeSet<>();
        set.addAll(getEntitiesMap().keySet());
        return set;
    }

    /**
     * @return the operation keys
     */
    public Set<String> getOperationKeys() {
        Set<String> set = new TreeSet<>();
        set.addAll(getDefaultCrudOperationKeys());
        set.addAll(getUserDefinedOperationKeys());
        return set;
    }

    /**
     * @return the CRUD operation keys
     */
    public Set<String> getDefaultCrudOperationKeys() {
        Set<String> set = new TreeSet<>();
        List<Entity> entities = getEntitiesList();
        String[] operations = Operation.getCrudOperationKeys();
        String simpleName;
        for (Entity entity : entities) {
            simpleName = entity.getDataType().getSimpleName();
            for (String name : operations) {
                set.add(simpleName + "." + name);
            }
        }
        return set;
    }

    /**
     * @return the user-defined operation keys
     */
    public Set<String> getUserDefinedOperationKeys() {
        Set<String> set = new TreeSet<>();
        List<Entity> entities = getEntitiesList();
        String simpleName;
        for (Entity entity : entities) {
            simpleName = entity.getDataType().getSimpleName();
            for (Operation operation : entity.getOperationsList()) {
                set.add(simpleName + "." + operation.getName());
            }
        }
        return set;
    }

    /**
     * @return the page keys
     */
    public Set<String> getPageKeys() {
        Set<String> set = new TreeSet<>();
        set.addAll(getDisplaysMap().keySet());
        return set;
    }

    /**
     * @return the parameter keys
     */
    public Set<String> getParameterKeys() {
        Set<String> set = new TreeSet<>();
        List<Entity> entities = getEntitiesList();
        String simpleName;
        for (Entity entity : entities) {
            simpleName = entity.getDataType().getSimpleName();
            for (Property property : entity.getPropertiesList()) {
                set.add(simpleName + "." + property.getName());
            }
            for (Operation operation : entity.getOperationsList()) {
                for (Parameter parameter : operation.getParametersList()) {
                    set.add(simpleName + "." + operation.getName() + "." + parameter.getName());
                }
            }
        }
        return set;
    }

    /**
     * @param entity
     * @return the entity number
     */
    public String getEntityNumber(Entity entity) {
        if (entity == null) {
            return "?";
        }
//      String key = entity.getName();
        String key = entity.getDataType().getSimpleName();
        return getEntityNumber(key);
    }

    /**
     * @param key
     * @return the entity number
     */
    public String getEntityNumber(String key) {
        if (StringUtils.isBlank(key)) {
            return "?";
        }
        if (_dictionaryEnabled) {
            return _entitiesDictionary.getProperty(key, "?");
        }
        return StrUtils.getLongNumericCode(StrUtils.getHumplessCase(key, '_'));
    }

    /**
     * @param key
     * @return the entity parameter number
     */
    public String getEntityParameterNumber(String key) {
        if (StringUtils.isBlank(key)) {
            return "?";
        }
//      if (_dictionaryEnabled) {
//          return _entityParametersDictionary.getProperty(key, "?");
//      }
        return StrUtils.getLongNumericCode(StrUtils.getHumplessCase(key, '_'));
    }

    /**
     * @param operation
     * @return the operation number
     */
    public String getOperationNumber(Operation operation) {
        if (operation == null) {
            return "?";
        }
        String name = operation.getName();
        Entity declaringEntity = operation.getDeclaringEntity();
        return getOperationNumber(name, declaringEntity);
    }

    /**
     * @param name
     * @param declaringEntity
     * @return the operation number
     */
    public String getOperationNumber(String name, Entity declaringEntity) {
        if (StringUtils.isBlank(name) || declaringEntity == null) {
            return "?";
        }
        String simpleName = declaringEntity.getDataType().getSimpleName();
        String key = simpleName + "." + name;
        return getOperationNumber(key);
    }

    /**
     * @param key
     * @return the operation number
     */
    public String getOperationNumber(String key) {
        if (StringUtils.isBlank(key)) {
            return "?";
        }
        if (_dictionaryEnabled) {
            return _operationsDictionary.getProperty(key, "?");
        }
        return StrUtils.getLongNumericCode(StrUtils.getHumplessCase(key, '_'));
    }

    /**
     * @param key
     * @return the operation parameter number
     */
    public String getOperationParameterNumber(String key) {
        if (StringUtils.isBlank(key)) {
            return "?";
        }
//      if (_dictionaryEnabled) {
//          return _operationParametersDictionary.getProperty(key, "?");
//      }
        return StrUtils.getLongNumericCode(StrUtils.getHumplessCase(key, '_'));
    }

    /**
     * @param key
     * @return the page number
     */
    public String getPageNumber(String key) {
        if (StringUtils.isBlank(key)) {
            return "?";
        }
        if (_dictionaryEnabled) {
            return _pagesDictionary.getProperty(key, "?");
        }
        return StrUtils.getLongNumericCode(StrUtils.getHumplessCase(key, '_'));
    }

    /**
     * @param artifact
     * @return the parameter number
     */
    public String getParameterNumber(Artifact artifact) {
        if (artifact == null) {
            return "?";
        }
        Artifact declaringArtifact = artifact.getDeclaringArtifact();
        if (artifact instanceof Property && declaringArtifact instanceof Entity) {
            String name = artifact.getName();
            Entity declaringEntity = (Entity) declaringArtifact;
            return getParameterNumber(name, declaringEntity);
        }
        if (artifact instanceof Parameter && declaringArtifact instanceof Operation) {
            String name = artifact.getName();
            Operation declaringOperation = (Operation) declaringArtifact;
            return getParameterNumber(name, declaringOperation);
        }
        return "?";
    }

    /**
     * @param name
     * @param declaringEntity
     * @return the parameter number
     */
    public String getParameterNumber(String name, Entity declaringEntity) {
        if (StringUtils.isBlank(name) || declaringEntity == null) {
            return "?";
        }
        String simpleName = declaringEntity.getDataType().getSimpleName();
        String key = simpleName + "." + name;
        return getParameterNumber(key);
    }

    /**
     * @param name
     * @param declaringOperation
     * @return the parameter number
     */
    public String getParameterNumber(String name, Operation declaringOperation) {
        if (StringUtils.isBlank(name) || declaringOperation == null) {
            return "?";
        }
        Entity declaringEntity = declaringOperation.getDeclaringEntity();
        if (declaringEntity == null) {
            return "?";
        }
        String simpleName = declaringEntity.getDataType().getSimpleName();
        String key = simpleName + "." + declaringOperation.getName() + "." + name;
        return getParameterNumber(key);
    }

    /**
     * @param key
     * @return the parameter number
     */
    public String getParameterNumber(String key) {
        if (StringUtils.isBlank(key)) {
            return "?";
        }
        if (_dictionaryEnabled) {
            return _parametersDictionary.getProperty(key, "?");
        }
        return StrUtils.getLongNumericCode(StrUtils.getHumplessCase(key, '_'));
    }

    private void loadDictionary() {
        _entitiesDictionary = PropertiesHandler.loadProperties(ENTITIES_DICTIONARY, true, Level.WARN);
        _operationsDictionary = PropertiesHandler.loadProperties(OPERATIONS_DICTIONARY, true, Level.WARN);
        _pagesDictionary = PropertiesHandler.loadProperties(PAGES_DICTIONARY, true, Level.WARN);
        _parametersDictionary = PropertiesHandler.loadProperties(PARAMETERS_DICTIONARY, true, Level.WARN);
    }

    private void initDictionary() {
        _entitiesDictionary = new SortedProperties();
        _operationsDictionary = new SortedProperties();
        _pagesDictionary = new SortedProperties();
        _parametersDictionary = new SortedProperties();
    }

    private void buildDictionary() {
        addKeys(_entitiesDictionary, getEntityKeys());
        addKeys(_operationsDictionary, getOperationKeys());
        addKeys(_pagesDictionary, getPageKeys());
        addKeys(_parametersDictionary, getParameterKeys());
    }

    private void addKeys(Properties properties, Set<String> keys) {
        String id$ = properties.getProperty(DICTIONARY_NEXT_ID_KEY);
        long id = StringUtils.isNumeric(id$) ? new Long(id$) : 1;
        for (String key : keys) {
            if (properties.containsKey(key)) {
            } else {
                properties.setProperty(key, "" + id++);
            }
        }
        properties.setProperty(DICTIONARY_NEXT_ID_KEY, "" + id);
    }

    private void storeDictionary() {
        PropertiesHandler.storeProperties(_entitiesDictionary, ENTITIES_DICTIONARY, getName());
        PropertiesHandler.storeProperties(_operationsDictionary, OPERATIONS_DICTIONARY, getName());
        PropertiesHandler.storeProperties(_pagesDictionary, PAGES_DICTIONARY, getName());
        PropertiesHandler.storeProperties(_parametersDictionary, PARAMETERS_DICTIONARY, getName());
    }

    /**
     * @return the base folder name
     */
    public String getBaseFolderName() {
        return StringUtils.defaultIfBlank(_baseFolderName, getDefaultBaseFolderName());
    }

    /**
     * @param baseFolderName the base folder name to set
     */
    public void setBaseFolderName(String baseFolderName) {
        _baseFolderName = StrUtils.getFileName(baseFolderName);
    }

    /**
     * @return the database name
     */
    public String getDatabaseName() {
        return StringUtils.defaultIfBlank(_databaseName, getDefaultDatabaseName());
    }

    /**
     * El método setDatabaseName se utiliza para establecer el nombre de la base de datos del proyecto, en caso de que se deba utilizar un nombre
     * diferente al predeterminado; el nombre predeterminado es el alias del proyecto
     *
     * @param databaseName nombre de la base de datos
     */
    public void setDatabaseName(String databaseName) {
        _databaseName = StrUtils.getLowerCaseIdentifier(databaseName, '-');
    }

    /**
     * @return the database former schema name
     */
    public String getDatabaseFormerSchemaName() {
        return StringUtils.defaultIfBlank(_databaseFormerSchemaName, getDefaultDatabaseFormerSchemaName());
    }

    /**
     * El método setDatabaseFormerSchemaName se utiliza para establecer el nombre del esquema previo de la base de datos del proyecto, en caso de que
     * se deba utilizar un nombre diferente al predeterminado; el nombre predeterminado es former.
     *
     * @param databaseFormerSchemaName nombre del esquema previo
     */
    public void setDatabaseFormerSchemaName(String databaseFormerSchemaName) {
        _databaseFormerSchemaName = StrUtils.getLowerCaseIdentifier(databaseFormerSchemaName, '_');
    }

    /**
     * @return the root folder name
     */
    public String getRootFolderName() {
        return StringUtils.defaultIfBlank(_rootFolderName, getDefaultRootFolderName());
    }

    /**
     * @param rootFolderName the root folder name to set
     */
    public void setRootFolderName(String rootFolderName) {
        _rootFolderName = StrUtils.getFileName(rootFolderName);
    }

    /**
     * @return the role code prefix
     */
    public String getRoleCodePrefix() {
        return StringUtils.trimToEmpty(_roleCodePrefix);
    }

    /**
     * El método setRoleCodePrefix se utiliza para establecer el prefijo de código de rol del controlador de seguridad del proyecto. Esta propiedad
     * solo se utiliza si el controlador de seguridad del proyecto cumple el protocolo LDAP. El prefijo de código de rol se antepone a los códigos de
     * los roles definidos en la aplicación para determinar los correspondientes códigos en el directorio LDAP. Dado que esta propiedad no tiene valor
     * predeterminado, si no se establece su valor, el código de cada rol en el directorio LDAP debe ser igual al código definido en la aplicación.
     *
     * @param roleCodePrefix prefijo de código de rol del controlador de seguridad del proyecto.
     */
    public void setRoleCodePrefix(String roleCodePrefix) {
        _roleCodePrefix = StrUtils.getIdentifier(roleCodePrefix);
    }

    /**
     * Returns a string that identifies the algorithm, independent of implementation details. The name should be a standard Java Security name (such
     * as "SHA", "MD5", and so on). See the MessageDigest section in the Java Cryptography Architecture Standard Algorithm Name Documentation for
     * information about standard algorithm names.
     *
     * @return the message digest algorithm
     */
    public String getMessageDigestAlgorithm() {
        return StringUtils.defaultIfBlank(_messageDigestAlgorithm, getDefaultMessageDigestAlgorithm());
    }

    /**
     * El método setMessageDigestAlgorithm se utiliza para especificar el nombre del algoritmo de encriptación del proyecto generado. El valor
     * predeterminado de esta propiedad es MD5.
     *
     * @param messageDigestAlgorithm nombre del algoritmo de encriptación del proyecto generado. Utilice como argumento el nombre de un algoritmo
     * soportado por la plataforma; por ejemplo: MD2, MD5, SHA-1, SHA-256, SHA-384, SHA-512.
     */
    public void setMessageDigestAlgorithm(String messageDigestAlgorithm) {
        _messageDigestAlgorithm = StrUtils.getIdentifier(messageDigestAlgorithm, '-');
    }

    /**
     * @return true if SQL business functions audit trail should be enabled; false otherwise
     */
    public boolean isSqlBusinessAuditTrail() {
        return _sqlBusinessAuditTrail;
    }

    /**
     * @param enabled true if SQL business functions audit trail should be enabled; false otherwise
     */
    public void setSqlBusinessAuditTrail(boolean enabled) {
        _sqlBusinessAuditTrail = enabled;
    }

    /**
     * @return the database locking mechanism
     */
    public DatabaseLockingMechanism getDatabaseLockingMechanism() {
        return _databaseLockingMechanism == null ? getDefaultDatabaseLockingMechanism() : _databaseLockingMechanism;
    }

    /**
     * @param databaseLockingMechanism the database locking mechanism to set
     */
    public void setDatabaseLockingMechanism(DatabaseLockingMechanism databaseLockingMechanism) {
        _databaseLockingMechanism = databaseLockingMechanism;
    }

    protected String getDefaultBaseFolderName() {
        return getAlias();
    }

    protected String getDefaultDatabaseName() {
        return getAlias();
    }

    protected String getDefaultDatabaseFormerSchemaName() {
        return "former";
    }

    protected String getDefaultRootFolderName() {
        return getAlias();
    }

    protected String getDefaultMessageDigestAlgorithm() {
        return "MD5";
    }

    protected DatabaseLockingMechanism getDefaultDatabaseLockingMechanism() {
        return DatabaseLockingMechanism.ENTITY_VERSIONING;
    }

}
